"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable dot-notation */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const asset_constants_1 = require("../../core/test/asset-constants");
const lib_1 = require("../lib");
const asset_constants_2 = require("./asset-constants");
describe('Test WorkerInstanceConfiguration for Linux', () => {
    let stack;
    let vpc;
    let instance;
    beforeEach(() => {
        stack = new aws_cdk_lib_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
    });
    test('basic setup', () => {
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Instance', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        [
                            '#!/bin/bash\n' +
                                `mkdir -p $(dirname '/tmp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py')\naws s3 cp 's3://`,
                            {
                                'Fn::Sub': asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Bucket,
                            },
                            `/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py' '/tmp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py'\n` +
                                `mkdir -p $(dirname '/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh')\naws s3 cp 's3://`,
                            {
                                'Fn::Sub': asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Bucket,
                            },
                            `/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh' '/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh'\n` +
                                'set -e\n' +
                                `chmod +x '/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh'\n` +
                                `'/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh' '' '' '' '${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}' ${lib_1.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} /tmp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py`,
                        ],
                    ],
                },
            },
        });
    });
    test('custom listener port', () => {
        const otherListenerPort = 55555;
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            workerSettings: {
                listenerPort: otherListenerPort,
            },
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Instance', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        [
                            '#!/bin/bash\n' +
                                `mkdir -p $(dirname '/tmp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py')\naws s3 cp 's3://`,
                            {
                                'Fn::Sub': asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Bucket,
                            },
                            `/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py' '/tmp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py'\n` +
                                `mkdir -p $(dirname '/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh')\naws s3 cp 's3://`,
                            {
                                'Fn::Sub': asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Bucket,
                            },
                            `/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh' '/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh'\n` +
                                'set -e\n' +
                                `chmod +x '/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh'\n` +
                                `'/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh' '' '' '' '${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}' ${otherListenerPort} /tmp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py`,
                        ],
                    ],
                },
            },
        });
    });
    test('groups, pools, region setup', () => {
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            workerSettings: {
                groups: ['g1', 'g2'],
                pools: ['p1', 'p2'],
                region: 'r1',
            },
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Instance', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        [
                            '#!/bin/bash\n' +
                                `mkdir -p $(dirname '/tmp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py')\naws s3 cp 's3://`,
                            {
                                'Fn::Sub': asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Bucket,
                            },
                            `/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py' '/tmp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py'\n` +
                                `mkdir -p $(dirname '/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh')\naws s3 cp 's3://`,
                            {
                                'Fn::Sub': asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Bucket,
                            },
                            `/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh' '/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh'\n` +
                                'set -e\n' +
                                `chmod +x '/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh'\n` +
                                `'/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh' 'g1,g2' 'p1,p2' 'r1' '${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}' ${lib_1.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} /tmp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py`,
                        ],
                    ],
                },
            },
        });
    });
    test('log setup', () => {
        // GIVEN
        const logGroupProps = {
            logGroupPrefix: '/test-prefix/',
        };
        // WHEN
        const config = new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            cloudWatchLogSettings: logGroupProps,
        });
        const logGroup = config.node.findChild('ConfigLogGroup');
        const logGroupName = stack.resolve(logGroup.logGroupName);
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Instance', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        [
                            `#!/bin/bash\nmkdir -p $(dirname '/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh')\naws s3 cp 's3://`,
                            {
                                'Fn::Sub': asset_constants_1.CWA_ASSET_LINUX.Bucket,
                            },
                            `/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh' '/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh'\nset -e\nchmod +x '/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh'\n'/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh' -i `,
                            {
                                Ref: 'AWS::Region',
                            },
                            ' ',
                            {
                                Ref: assertions_1.Match.stringLikeRegexp('^ConfigStringParameter.*'),
                            },
                            `\nmkdir -p $(dirname '/tmp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py')\naws s3 cp 's3://`,
                            {
                                'Fn::Sub': asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Bucket,
                            },
                            `/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py' '/tmp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py'\n` +
                                `mkdir -p $(dirname '/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh')\naws s3 cp 's3://`,
                            {
                                'Fn::Sub': asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Bucket,
                            },
                            `/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh' '/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh'\n` +
                                'set -e\n' +
                                `chmod +x '/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh'\n` +
                                `'/tmp/${asset_constants_2.CONFIG_WORKER_ASSET_LINUX.Key}.sh' '' '' '' '${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}' ${lib_1.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} /tmp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_LINUX.Key}.py`,
                        ],
                    ],
                },
            },
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::SSM::Parameter', {
            Value: {
                'Fn::Join': [
                    '',
                    [
                        '{\"logs\":{\"logs_collected\":{\"files\":{\"collect_list\":[{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"cloud-init-output-{instance_id}\",\"file_path\":\"/var/log/cloud-init-output.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"WorkerLogs-{instance_id}\",\"file_path\":\"/var/log/Thinkbox/Deadline10/deadlineslave*.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"LauncherLogs-{instance_id}\",\"file_path\":\"/var/log/Thinkbox/Deadline10/deadlinelauncher*.log\",\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}',
                    ],
                ],
            },
        });
    });
});
describe('Test WorkerInstanceConfiguration for Windows', () => {
    let stack;
    let vpc;
    let instance;
    beforeEach(() => {
        stack = new aws_cdk_lib_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
        });
    });
    test('basic setup', () => {
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Instance', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        [
                            `<powershell>mkdir (Split-Path -Path 'C:/temp/${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py' ) -ea 0\nRead-S3Object -BucketName '`,
                            {
                                'Fn::Sub': asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Bucket,
                            },
                            `' -key '${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py' -file 'C:/temp/${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py' -ErrorAction Stop\n` +
                                `mkdir (Split-Path -Path 'C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' ) -ea 0\nRead-S3Object -BucketName '`,
                            {
                                'Fn::Sub': asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Bucket,
                            },
                            `' -key '${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' -file 'C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' -ErrorAction Stop\n` +
                                `&'C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' '' '' '' '${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}' ${lib_1.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} C:/temp/${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py\n` +
                                `if (!$?) { Write-Error 'Failed to execute the file \"C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1\"' -ErrorAction Stop }</powershell>`,
                        ],
                    ],
                },
            },
        });
    });
    test('groups, pools, region setup', () => {
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            workerSettings: {
                groups: ['g1', 'g2'],
                pools: ['p1', 'p2'],
                region: 'r1',
            },
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Instance', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        [
                            `<powershell>mkdir (Split-Path -Path 'C:/temp/${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py' ) -ea 0\nRead-S3Object -BucketName '`,
                            {
                                'Fn::Sub': asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Bucket,
                            },
                            `' -key '${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py' -file 'C:/temp/${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py' -ErrorAction Stop\n` +
                                `mkdir (Split-Path -Path 'C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' ) -ea 0\nRead-S3Object -BucketName '`,
                            {
                                'Fn::Sub': asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Bucket,
                            },
                            `' -key '${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' -file 'C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' -ErrorAction Stop\n` +
                                `&'C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' 'g1,g2' 'p1,p2' 'r1' '${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}' ${lib_1.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} C:/temp/${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py\n` +
                                `if (!$?) { Write-Error 'Failed to execute the file \"C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1\"' -ErrorAction Stop }</powershell>`,
                        ],
                    ],
                },
            },
        });
    });
    test('custom listner port', () => {
        const otherListenerPort = 55555;
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            workerSettings: {
                listenerPort: otherListenerPort,
            },
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Instance', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        [
                            `<powershell>mkdir (Split-Path -Path 'C:/temp/${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py' ) -ea 0\nRead-S3Object -BucketName '`,
                            {
                                'Fn::Sub': asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Bucket,
                            },
                            `' -key '${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py' -file 'C:/temp/${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py' -ErrorAction Stop\n` +
                                `mkdir (Split-Path -Path 'C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' ) -ea 0\nRead-S3Object -BucketName '`,
                            {
                                'Fn::Sub': asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Bucket,
                            },
                            `' -key '${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' -file 'C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' -ErrorAction Stop\n` +
                                `&'C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' '' '' '' '${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}' ${otherListenerPort} C:/temp/${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py\n` +
                                `if (!$?) { Write-Error 'Failed to execute the file \"C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1\"' -ErrorAction Stop }</powershell>`,
                        ],
                    ],
                },
            },
        });
    });
    test('log setup', () => {
        // GIVEN
        const logGroupProps = {
            logGroupPrefix: '/test-prefix/',
        };
        // WHEN
        const config = new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            cloudWatchLogSettings: logGroupProps,
        });
        const logGroup = config.node.findChild('ConfigLogGroup');
        const logGroupName = stack.resolve(logGroup.logGroupName);
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Instance', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        [
                            `<powershell>mkdir (Split-Path -Path 'C:/temp/${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1' ) -ea 0\nRead-S3Object -BucketName '`,
                            {
                                'Fn::Sub': asset_constants_1.CWA_ASSET_WINDOWS.Bucket,
                            },
                            `' -key '${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1' -file 'C:/temp/${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1' -ErrorAction Stop\n&'C:/temp/${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1' -i `,
                            {
                                Ref: 'AWS::Region',
                            },
                            ' ',
                            {
                                Ref: assertions_1.Match.stringLikeRegexp('^ConfigStringParameter.*'),
                            },
                            `\nif (!$?) { Write-Error 'Failed to execute the file \"C:/temp/${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1\"' -ErrorAction Stop }\n` +
                                `mkdir (Split-Path -Path 'C:/temp/${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py' ) -ea 0\nRead-S3Object -BucketName '`,
                            {
                                'Fn::Sub': asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Bucket,
                            },
                            `' -key '${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py' -file 'C:/temp/${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py' -ErrorAction Stop\n` +
                                `mkdir (Split-Path -Path 'C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' ) -ea 0\nRead-S3Object -BucketName '`,
                            {
                                'Fn::Sub': asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Bucket,
                            },
                            `' -key '${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' -file 'C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' -ErrorAction Stop\n` +
                                `&'C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1' '' '' '' '${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}' ${lib_1.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} C:/temp/${asset_constants_2.CONFIG_WORKER_ASSET_WINDOWS.Key}.py\n` +
                                `if (!$?) { Write-Error 'Failed to execute the file \"C:/temp/${asset_constants_2.CONFIG_WORKER_PORT_ASSET_WINDOWS.Key}.ps1\"' -ErrorAction Stop }</powershell>`,
                        ],
                    ],
                },
            },
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::SSM::Parameter', {
            Value: {
                'Fn::Join': [
                    '',
                    [
                        '{\"logs\":{\"logs_collected\":{\"files\":{\"collect_list\":[{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"UserdataExecution-{instance_id}\",\"file_path\":\"C:\\\\ProgramData\\\\Amazon\\\\EC2-Windows\\\\Launch\\\\Log\\\\UserdataExecution.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"WorkerLogs-{instance_id}\",\"file_path\":\"C:\\\\ProgramData\\\\Thinkbox\\\\Deadline10\\\\logs\\\\deadlineslave*.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"LauncherLogs-{instance_id}\",\"file_path\":\"C:\\\\ProgramData\\\\Thinkbox\\\\Deadline10\\\\logs\\\\deadlinelauncher*.log\",\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}',
                    ],
                ],
            },
        });
    });
});
describe('Test WorkerInstanceConfiguration connect to RenderQueue', () => {
    let stack;
    let vpc;
    let renderQueue;
    let renderQueueSGId;
    beforeEach(() => {
        stack = new aws_cdk_lib_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        const rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
        const version = new lib_1.VersionQuery(stack, 'Version');
        renderQueue = new lib_1.RenderQueue(stack, 'RQ', {
            version,
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_1.Repository(stack, 'Repository', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            }),
            trafficEncryption: { externalTLS: { enabled: false } },
        });
        const rqSecGrp = renderQueue.connections.securityGroups[0];
        renderQueueSGId = stack.resolve(rqSecGrp.securityGroupId);
    });
    test('For Linux', () => {
        // GIVEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            renderQueue,
        });
        const instanceSG = instance.connections.securityGroups[0];
        const instanceSGId = stack.resolve(instanceSG.securityGroupId);
        // THEN
        // Open-box testing. We know that we invoked the connection method on the
        // render queue if the security group for the instance has an ingress rule to the RQ.
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            ToPort: 8080,
            SourceSecurityGroupId: instanceSGId,
            GroupId: renderQueueSGId,
        });
    });
    test('For Windows', () => {
        // GIVEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
        });
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            renderQueue,
        });
        const instanceSG = instance.connections.securityGroups[0];
        const instanceSGId = stack.resolve(instanceSG.securityGroupId);
        // THEN
        // Open-box testing. We know that we invoked the connection method on the
        // render queue if the security group for the instance has an ingress rule to the RQ.
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            ToPort: 8080,
            SourceSecurityGroupId: instanceSGId,
            GroupId: renderQueueSGId,
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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