"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const wait_for_stable_service_1 = require("../lib/wait-for-stable-service");
const test_helper_1 = require("./test-helper");
describe('WaitForStableService', () => {
    let app;
    let stack;
    let isolatedStack;
    let cluster;
    let taskDefinition;
    let service;
    beforeEach(() => {
        app = new aws_cdk_lib_1.App();
        stack = new aws_cdk_lib_1.Stack(app, 'Stack');
        isolatedStack = new aws_cdk_lib_1.Stack(app, 'IsolatedStack');
        cluster = new aws_ecs_1.Cluster(stack, 'Cluster');
        cluster.addCapacity('ASG', {
            vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT },
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.C5, aws_ec2_1.InstanceSize.LARGE),
            minCapacity: 1,
            maxCapacity: 1,
        });
        taskDefinition = new aws_ecs_1.Ec2TaskDefinition(stack, 'RCSTask');
        taskDefinition.addContainer('Test', {
            image: aws_ecs_1.ContainerImage.fromAsset(__dirname),
            memoryLimitMiB: 7500,
        });
        service = new aws_ecs_1.Ec2Service(stack, 'Service', {
            cluster,
            taskDefinition,
        });
    });
    test('creates a custom resource', () => {
        // WHEN
        new wait_for_stable_service_1.WaitForStableService(isolatedStack, 'WaitForStableService', {
            service,
        });
        // THEN
        assertions_1.Template.fromStack(isolatedStack).hasResourceProperties('Custom::RFDK_WaitForStableService', {
            cluster: isolatedStack.resolve(cluster.clusterArn),
            services: [isolatedStack.resolve(service.serviceArn)],
        });
    });
    test('creates lambda correctly', () => {
        // WHEN
        new wait_for_stable_service_1.WaitForStableService(isolatedStack, 'WaitForStableService', {
            service,
        });
        test_helper_1.resourcePropertiesCountIs(isolatedStack, 'AWS::Lambda::Function', {
            Handler: 'wait-for-stable-service.wait',
            Environment: {
                Variables: {
                    DEBUG: 'false',
                },
            },
            Runtime: 'nodejs16.x',
            Timeout: 900,
        }, 1);
    });
    test('adds policies to the lambda role', () => {
        // WHEN
        new wait_for_stable_service_1.WaitForStableService(isolatedStack, 'WaitForStableService', {
            service,
        });
        // THEN
        assertions_1.Template.fromStack(isolatedStack).hasResourceProperties('AWS::IAM::Role', {
            ManagedPolicyArns: assertions_1.Match.arrayWith([
                isolatedStack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole').managedPolicyArn),
            ]),
            Policies: [{
                    PolicyDocument: assertions_1.Match.objectLike({
                        Statement: [{
                                Action: 'ecs:DescribeServices',
                                Effect: 'Allow',
                                Resource: assertions_1.Match.arrayWith([
                                    isolatedStack.resolve(cluster.clusterArn),
                                    isolatedStack.resolve(service.serviceArn),
                                ]),
                            }],
                    }),
                }],
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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