"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const lib_1 = require("../lib");
test('VersionQuery constructor full version', () => {
    const app = new aws_cdk_lib_1.App();
    const stack = new aws_cdk_lib_1.Stack(app, 'Stack');
    new lib_1.VersionQuery(stack, 'VersionQuery', { version: '10.1.9.2' });
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_DEADLINE_INSTALLERS', {
        forceRun: assertions_1.Match.absent(),
        versionString: '10.1.9.2',
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                    Principal: {
                        Service: 'lambda.amazonaws.com',
                    },
                },
            ],
        },
        ManagedPolicyArns: [
            {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole',
                    ],
                ],
            },
        ],
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
        Handler: 'version-provider.handler',
        Role: {
            'Fn::GetAtt': [
                assertions_1.Match.stringLikeRegexp('SingletonLambda.*ServiceRole.*'),
                'Arn',
            ],
        },
        Runtime: 'nodejs16.x',
    });
});
test('VersionQuery constructor no versionString', () => {
    const app = new aws_cdk_lib_1.App();
    const stack = new aws_cdk_lib_1.Stack(app, 'Stack');
    new lib_1.VersionQuery(stack, 'VersionQuery');
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_DEADLINE_INSTALLERS', {
        forceRun: assertions_1.Match.anyValue(),
        versionString: assertions_1.Match.absent(),
    });
});
test.each([
    ['10.1.9'],
    ['10.1'],
    ['10'],
])('VersionQuery constructor partial version: %s', (version) => {
    const app = new aws_cdk_lib_1.App();
    const stack = new aws_cdk_lib_1.Stack(app, 'Stack');
    new lib_1.VersionQuery(stack, 'VersionQuery', { version });
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_DEADLINE_INSTALLERS', {
        versionString: version,
        forceRun: assertions_1.Match.anyValue(),
    });
});
describe('VersionQuery.linuxInstallers', () => {
    let customResource;
    let versionQuery;
    let stack;
    beforeAll(() => {
        // GIVEN
        const app = new aws_cdk_lib_1.App();
        stack = new aws_cdk_lib_1.Stack(app, 'Stack');
        versionQuery = new lib_1.VersionQuery(stack, 'VersionQuery');
        customResource = versionQuery.node.findChild('DeadlineResource');
    });
    describe('.repository', () => {
        let repoInstaller;
        beforeAll(() => {
            // WHEN
            repoInstaller = versionQuery.linuxInstallers.repository;
        });
        test('S3 bucket from Custom::RFDK_DEADLINE_INSTALLERS "S3Bucket" attribute', () => {
            // THEN
            expect(stack.resolve(repoInstaller.s3Bucket.bucketName))
                .toEqual(stack.resolve(customResource.getAtt('S3Bucket')));
        });
        test('S3 object key from Custom::RFDK_DEADLINE_INSTALLERS "LinuxRepositoryInstaller" attribute', () => {
            // THEN
            expect(stack.resolve(repoInstaller.objectKey))
                .toEqual(stack.resolve(customResource.getAtt('LinuxRepositoryInstaller')));
        });
    });
    describe('.client', () => {
        let clientInstaller;
        beforeAll(() => {
            // WHEN
            clientInstaller = versionQuery.linuxInstallers.client;
        });
        test('S3 bucket from Custom::RFDK_DEADLINE_INSTALLERS "S3Bucket" attribute', () => {
            // THEN
            expect(stack.resolve(clientInstaller.s3Bucket.bucketName))
                .toEqual(stack.resolve(customResource.getAtt('S3Bucket')));
        });
        test('S3 object key from Custom::RFDK_DEADLINE_INSTALLERS "LinuxClientInstaller" attribute', () => {
            // THEN
            expect(stack.resolve(clientInstaller.objectKey))
                .toEqual(stack.resolve(customResource.getAtt('LinuxClientInstaller')));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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