"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const lib_1 = require("../lib");
describe('ThinkboxDockerRecipes', () => {
    let app;
    let stack;
    let images;
    let userAwsCustomerAgreementAndIpLicenseAcceptance;
    describe('defaults', () => {
        beforeEach(() => {
            // GIVEN
            app = new aws_cdk_lib_1.App();
            stack = new aws_cdk_lib_1.Stack(app, 'Stack');
            userAwsCustomerAgreementAndIpLicenseAcceptance = lib_1.AwsCustomerAgreementAndIpLicenseAcceptance.USER_ACCEPTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE;
            // WHEN
            images = new lib_1.ThinkboxDockerImages(stack, 'Images', {
                userAwsCustomerAgreementAndIpLicenseAcceptance,
            });
        });
        test('fails validation when terms are not accepted', () => {
            // GIVEN
            const newStack = new aws_cdk_lib_1.Stack(app, 'NewStack');
            const expectedError = `
The ThinkboxDockerImages will install Deadline onto one or more EC2 instances.

By downloading or using the Deadline software, you agree to the AWS Customer Agreement (https://aws.amazon.com/agreement/)
and AWS Intellectual Property License (https://aws.amazon.com/legal/aws-ip-license-terms/). You acknowledge that Deadline
is AWS Content as defined in those Agreements.

Please set the userAcceptsAwsCustomerAgreementAndIpLicense property to
USER_ACCEPTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE to signify your acceptance of these terms.
`;
            userAwsCustomerAgreementAndIpLicenseAcceptance = lib_1.AwsCustomerAgreementAndIpLicenseAcceptance.USER_REJECTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE;
            new lib_1.ThinkboxDockerImages(newStack, 'Images', {
                userAwsCustomerAgreementAndIpLicenseAcceptance,
            });
            // WHEN
            function synth() {
                app.synth();
            }
            // THEN
            expect(synth).toThrow(expectedError);
        });
        test('creates Custom::RFDK_ECR_PROVIDER', () => {
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_EcrProvider', {
                ForceRun: assertions_1.Match.anyValue(),
            });
        });
        describe('provides container images for', () => {
            test.each([
                [
                    'RCS',
                    () => {
                        return images.remoteConnectionServer;
                    },
                    lib_1.ThinkboxManagedDeadlineDockerRecipes.REMOTE_CONNECTION_SERVER,
                ],
                [
                    'License Forwarder',
                    () => {
                        return images.licenseForwarder;
                    },
                    lib_1.ThinkboxManagedDeadlineDockerRecipes.LICENSE_FORWARDER,
                ],
            ])('%s', (_label, imageGetter, recipe) => {
                // GIVEN
                const taskDefStack = new aws_cdk_lib_1.Stack(app, 'TaskDefStack');
                const image = imageGetter();
                const taskDefinition = new aws_ecs_1.TaskDefinition(taskDefStack, 'TaskDef', {
                    compatibility: aws_ecs_1.Compatibility.EC2,
                });
                const ecrProvider = images.node.defaultChild;
                const expectedImage = `${ecrProvider.getAtt('EcrURIPrefix')}${recipe}`;
                // WHEN
                new aws_ecs_1.ContainerDefinition(taskDefStack, 'ContainerDef', {
                    image,
                    taskDefinition,
                    memoryReservationMiB: 1024,
                });
                // THEN
                assertions_1.Template.fromStack(taskDefStack).hasResourceProperties('AWS::ECS::TaskDefinition', {
                    ContainerDefinitions: assertions_1.Match.arrayWith([assertions_1.Match.objectLike({
                            Image: taskDefStack.resolve(expectedImage),
                        })]),
                });
            });
        });
        describe('.forRenderQueue()', () => {
            let rcsImage;
            let rqImages;
            beforeEach(() => {
                // GIVEN
                rcsImage = images.remoteConnectionServer;
                // WHEN
                rqImages = images.forRenderQueue();
            });
            test('returns correct RCS image', () => {
                // THEN
                expect(rqImages.remoteConnectionServer).toBe(rcsImage);
            });
        });
        describe('.forUsageBasedLicensing()', () => {
            let lfImage;
            let ublImages;
            beforeEach(() => {
                // GIVEN
                lfImage = images.licenseForwarder;
                // WHEN
                ublImages = images.forUsageBasedLicensing();
            });
            test('returns correct RCS image', () => {
                // THEN
                expect(ublImages.licenseForwarder).toBe(lfImage);
            });
        });
    });
    describe('with version', () => {
        let version;
        beforeEach(() => {
            // GIVEN
            app = new aws_cdk_lib_1.App();
            stack = new aws_cdk_lib_1.Stack(app, 'Stack');
            version = new lib_1.VersionQuery(stack, 'Version');
            // WHEN
            images = new lib_1.ThinkboxDockerImages(stack, 'Images', {
                version,
                userAwsCustomerAgreementAndIpLicenseAcceptance,
            });
        });
        describe('provides container images for', () => {
            test.each([
                [
                    'RCS',
                    () => {
                        return images.remoteConnectionServer;
                    },
                    lib_1.ThinkboxManagedDeadlineDockerRecipes.REMOTE_CONNECTION_SERVER,
                ],
                [
                    'License Forwarder',
                    () => {
                        return images.licenseForwarder;
                    },
                    lib_1.ThinkboxManagedDeadlineDockerRecipes.LICENSE_FORWARDER,
                ],
            ])('%s', (_label, imageGetter, recipe) => {
                // GIVEN
                const taskDefStack = new aws_cdk_lib_1.Stack(app, 'TaskDefStack');
                const image = imageGetter();
                const taskDefinition = new aws_ecs_1.TaskDefinition(taskDefStack, 'TaskDef', {
                    compatibility: aws_ecs_1.Compatibility.EC2,
                });
                const ecrProvider = images.node.defaultChild;
                const expectedImage = `${ecrProvider.getAtt('EcrURIPrefix')}${recipe}:${aws_cdk_lib_1.Token.asString(version.majorVersion)}.${aws_cdk_lib_1.Token.asString(version.minorVersion)}.${aws_cdk_lib_1.Token.asString(version.releaseVersion)}`;
                // WHEN
                new aws_ecs_1.ContainerDefinition(taskDefStack, 'ContainerDef', {
                    image,
                    taskDefinition,
                    memoryReservationMiB: 1024,
                });
                // THEN
                assertions_1.Template.fromStack(taskDefStack).hasResourceProperties('AWS::ECS::TaskDefinition', {
                    ContainerDefinitions: assertions_1.Match.arrayWith([assertions_1.Match.objectLike({
                            Image: taskDefStack.resolve(expectedImage),
                        })]),
                });
            });
        });
        test('validates VersionQuery is not in a different stack', () => {
            // GIVEN
            const newStack = new aws_cdk_lib_1.Stack(app, 'NewStack');
            new lib_1.ThinkboxDockerImages(newStack, 'Images', {
                version,
                userAwsCustomerAgreementAndIpLicenseAcceptance,
            });
            // WHEN
            function synth() {
                app.synth();
            }
            // THEN
            expect(synth).toThrow('A VersionQuery can not be supplied from a different stack');
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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