"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const lib_1 = require("../lib");
describe('Stage', () => {
    // GIVEN
    const STAGE_PATH = path.join(__dirname, 'assets');
    /**
     * A sub-class of Stage that makes the constructor public for testing purposes.
     *
     * This is to be able to specify a manifest rather than load it from the file-system via {@link Stage.fromDirectory}.
     */
    class StageWithPulicConstructor extends lib_1.Stage {
        constructor(props) {
            super(props);
        }
    }
    beforeEach(() => {
        jest.resetModules();
    });
    describe('.fromDirectory', () => {
        // GIVEN
        const manifest = {
            schema: 1,
            version: '10.1.9.2',
            recipes: {},
        };
        const readFileSync = jest.spyOn(fs, 'readFileSync');
        let stage;
        beforeAll(() => {
            readFileSync.mockReturnValue(JSON.stringify(manifest));
            stage = require('../lib').Stage.fromDirectory(STAGE_PATH); // eslint-disable-line
        });
        afterAll(() => {
            readFileSync.mockRestore();
            jest.resetModules();
        });
        test('has dirPath', () => {
            // THEN
            expect(stage.dirPath).toBe(STAGE_PATH);
        });
        test('loads manifest.json from directory', () => {
            // THEN
            const expectedPath = path.join(STAGE_PATH, 'manifest.json');
            expect(readFileSync).toHaveBeenCalledWith(expectedPath, expect.objectContaining({
                encoding: 'utf8',
            }));
            expect(stage.manifest).toEqual(manifest);
        });
    });
    test('.fromDirectory not supported version failure', () => {
        // GIVEN
        const manifest = {
            schema: 1,
            version: '10.1.8.0',
            recipes: {},
        };
        // WHEN
        const readFileSync = jest.spyOn(fs, 'readFileSync');
        readFileSync.mockReturnValue(JSON.stringify(manifest));
        // THEN
        expect(() => {
            require('../lib').Stage.fromDirectory(STAGE_PATH); // eslint-disable-line
        }).toThrow('Staged Deadline Version (10.1.8.0) is less than the minimum supported version (10.1.9.2)');
        readFileSync.mockRestore();
        jest.resetModules();
    });
    test('has manifest', () => {
        const manifest = {
            schema: 1,
            version: '10.1.9.2',
            recipes: {
                a: {
                    title: 'a-title',
                    description: 'a-description',
                    buildArgs: {
                        argone: 'a-argone-value',
                        argtwo: 'a-argtwo-value',
                    },
                },
                b: {
                    title: 'b-title',
                    description: 'b-description',
                    buildArgs: {
                        argone: 'b-argone-value',
                        argtwo: 'b-argtwo-value',
                    },
                },
            },
        };
        const stage = new StageWithPulicConstructor({
            manifest,
            path: STAGE_PATH,
        });
        // THEN
        expect(stage.manifest).toEqual(manifest);
    });
    describe('manifest validation', () => {
        test.each([
            [
                'mainfest wrong type',
                {
                    manifest: 1,
                    expectedError: /Expected object but got/,
                },
            ],
            [
                'manifest version not string',
                {
                    manifest: {
                        version: 1,
                        recipes: {},
                    },
                },
            ],
            [
                'missing schema',
                {
                    manifest: {
                        version: '10.1.9.2',
                        recipes: {},
                    },
                    expectedError: /Manifest contains no "schema" key/,
                },
            ],
            [
                'wrong schema type', {
                    manifest: {
                        schema: 'a',
                        version: '10.1.9.2',
                        recipes: {},
                    },
                    expectedError: /Expected a numeric "schema" but got:/,
                },
            ],
            [
                'missing version', {
                    manifest: {
                        schema: 1,
                        recipes: {},
                    },
                    expectedError: /Manifest contains no "version" key/,
                },
            ],
            [
                'incorrect version format', {
                    manifest: {
                        schema: 1,
                        version: '10.1.',
                        recipes: {},
                    },
                    expectedError: /Invalid version format/,
                },
            ],
            [
                'version not supported', {
                    manifest: {
                        schema: 1,
                        version: '10.1.0.0',
                        recipes: {},
                    },
                    expectedError: 'Staged Deadline Version (10.1.0.0) is less than the minimum supported version (10.1.9.2)',
                },
            ],
        ])('%s', (_name, testcase) => {
            const { manifest, expectedError } = testcase;
            // WHEN
            function when() {
                new StageWithPulicConstructor({
                    path: STAGE_PATH,
                    manifest,
                });
            }
            expect(when).toThrow(expectedError);
        });
    });
    describe('valid recipes', () => {
        test.each([
            [
                'build args and no target',
                {
                    title: 'Eugene\'s Favorite Cookies',
                    description: 'C is for cookie. That\'s good enough for me',
                    buildArgs: {
                        flour: '3 cups',
                        butter: '1 cup',
                        sugar: '1/2 cup',
                        egg: '2',
                        bakingpowder: '1 tsp',
                        bakingsoda: '1 tsb',
                        vanilla: '1 tsp',
                        salt: '1 pinch',
                    },
                },
            ],
            [
                'buildArgs and a target',
                {
                    title: 'Josh\'s S\'mores',
                    description: 'a secret family recipe',
                    buildArgs: {
                        grahamcracker: '2',
                        marshmellow: '1',
                        chocolate: '2',
                        campfire: '1',
                    },
                    target: 'simplicity',
                },
            ],
            [
                'target and no buildArgs',
                {
                    title: 'Jericho\s special brownie batch with a strawberry milkshake',
                    description: 'a slight improvement to the original recipe',
                    target: 'target',
                },
            ],
            [
                'no target or buildArgs',
                {
                    title: 'Yash\'s Tequila Lime Shot (TLS)',
                    description: 'A sure-fire way to brighten your day',
                },
            ],
        ])('%s', (_scenario, recipe) => {
            // GIVEN
            const manifest = {
                recipes: {
                    recipeName: recipe,
                },
                schema: 1,
                version: '10.1.9.2',
            };
            // WHEN
            const stage = new StageWithPulicConstructor({
                manifest,
                path: STAGE_PATH,
            });
            // THEN
            expect(stage.manifest.recipes.recipeName).toEqual(recipe);
        });
    });
    describe('.imageFromRecipe()', () => {
        let app;
        let stack;
        beforeEach(() => {
            // GIVEN
            app = new aws_cdk_lib_1.App();
            stack = new aws_cdk_lib_1.Stack(app, 'Stack');
        });
        test('creates DockerImageAssets from existing recipes', () => {
            jest.resetModules();
            const mockDockerImageAssetConstructor = jest.fn();
            jest.mock('aws-cdk-lib/aws-ecr-assets', () => {
                class DockerImageAsset {
                    constructor(...args) {
                        mockDockerImageAssetConstructor(...args);
                    }
                }
                return {
                    DockerImageAsset,
                };
            });
            // Import the class under test now that the ec2 mock is in place
            const ReloadedStage = jest.requireActual('../lib').Stage; // eslint-disable-line
            class ReloadedStageWithPublicConstructor extends ReloadedStage {
                constructor(props) {
                    super(props);
                }
            }
            // GIVEN
            const recipeName = 'myname';
            const recipe = {
                description: 'some description',
                title: 'some title',
                buildArgs: {
                    a: 'a',
                    b: 'b',
                },
            };
            const stage = new ReloadedStageWithPublicConstructor({
                path: STAGE_PATH,
                manifest: {
                    version: '10.1.9.2',
                    schema: 1,
                    recipes: {
                        [recipeName]: recipe,
                    },
                },
            });
            // WHEN
            stage.imageFromRecipe(stack, 'Image', recipeName);
            // THEN
            expect(mockDockerImageAssetConstructor).toHaveBeenCalledWith(stack, 'Image', {
                ...recipe,
                directory: STAGE_PATH,
            });
            expect(mockDockerImageAssetConstructor.mock.calls[0][2]).not.toMatchObject({
                target: expect.anything(),
            });
        });
        test('throws exception when recipe does not exist', () => {
            // GIVEN
            const manifest = {
                schema: 1,
                version: '10.1.9.2',
                recipes: {},
            };
            const invalidRecipeName = 'this-recipe-does-not-exist';
            const stage = new StageWithPulicConstructor({
                manifest,
                path: STAGE_PATH,
            });
            // WHEN
            function when() {
                stage.imageFromRecipe(stack, 'Image', invalidRecipeName);
            }
            // THEN
            expect(when).toThrowError(`No such recipe: ${invalidRecipeName}`);
        });
    });
    describe('.getVersion()', () => {
        test('returns a version in the success case', () => {
            // GIVEN
            const app = new aws_cdk_lib_1.App();
            const stack = new aws_cdk_lib_1.Stack(app, 'Stack');
            const manifest = {
                schema: 1,
                version: '10.1.9.2',
                recipes: {},
            };
            const stage = new StageWithPulicConstructor({
                manifest,
                path: STAGE_PATH,
            });
            // WHEN
            stage.getVersion(stack, 'Version');
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_DEADLINE_INSTALLERS', {
                forceRun: assertions_1.Match.anyValue(),
                versionString: '10.1.9',
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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