"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_docdb_1 = require("aws-cdk-lib/aws-docdb");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_efs_1 = require("aws-cdk-lib/aws-efs");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const core_1 = require("../../core");
const asset_constants_1 = require("../../core/test/asset-constants");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const lib_1 = require("../lib");
const asset_constants_2 = require("./asset-constants");
const test_helper_1 = require("./test-helper");
let app;
let stack;
let vpc;
let version;
function escapeTokenRegex(s) {
    // A CDK Token looks like: ${Token[TOKEN.12]}
    // This contains the regex special characters: ., $, {, }, [, and ]
    // Escape those for use in a regex.
    return s.replace(/[.${}[\]]/g, '\\$&');
}
beforeEach(() => {
    app = new aws_cdk_lib_1.App();
    stack = new aws_cdk_lib_1.Stack(app, 'Stack');
    vpc = new aws_ec2_1.Vpc(stack, 'VPC', {
        subnetConfiguration: [
            {
                name: 'Public',
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            },
            {
                name: 'Private',
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
            },
            {
                name: 'Isolated',
                subnetType: aws_ec2_1.SubnetType.PRIVATE_ISOLATED,
            },
        ],
    });
    class MockVersion extends lib_1.Version {
        constructor() {
            super(...arguments);
            this.linuxInstallers = {
                patchVersion: 0,
                repository: {
                    objectKey: 'testInstaller',
                    s3Bucket: new aws_s3_1.Bucket(stack, 'LinuxInstallerBucket'),
                },
                client: {
                    objectKey: 'testClientInstaller',
                    s3Bucket: new aws_s3_1.Bucket(stack, 'LinuxClientInstallerBucket'),
                },
            };
        }
        linuxFullVersionString() {
            return this.toString();
        }
    }
    version = new MockVersion([10, 1, 19, 4]);
});
test('can create two repositories', () => {
    // GIVEN
    new lib_1.Repository(stack, 'Repo1', {
        vpc,
        version,
    });
    // THEN
    expect(() => {
        new lib_1.Repository(stack, 'Repo2', {
            vpc,
            version,
        });
    }).not.toThrow();
});
test('repository installer instance is created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResource('AWS::AutoScaling::AutoScalingGroup', {
        Properties: {
            MaxSize: '1',
            MinSize: '1',
            VPCZoneIdentifier: [
                {
                    Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                },
                {
                    Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                },
            ],
        },
        CreationPolicy: {
            ResourceSignal: {
                Count: 1,
                Timeout: 'PT15M',
            },
        },
        UpdatePolicy: {
            AutoScalingReplacingUpdate: {
                WillReplace: true,
            },
            AutoScalingScheduledAction: {
                IgnoreUnmodifiedGroupSizeProperties: true,
            },
        },
        DependsOn: assertions_1.Match.arrayWith([
            'repositoryInstallerDocumentDatabaseInstance11A6F8C8E',
        ]),
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't3.large',
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: 2049,
        ToPort: 2049,
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerFileSystemEfsSecurityGroup289D043C',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    });
});
test('repository installer honors vpcSubnet', () => {
    // Note: Default is private subnets, so it's sufficient to test something other than
    // private subnets.
    // WHEN
    const isolatedSubnetIds = ['IsolatedSubnet1', 'IsolatedSubnet2'];
    const attrVpc = aws_ec2_1.Vpc.fromVpcAttributes(stack, 'TestVpc', {
        availabilityZones: ['us-east-1a', 'us-east-1b'],
        vpcId: 'vpcid',
        isolatedSubnetIds,
    });
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc: attrVpc,
        version,
        vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PRIVATE_ISOLATED },
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::AutoScalingGroup', {
        VPCZoneIdentifier: isolatedSubnetIds,
    });
});
test('repository installer security groups created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: 2049,
        ToPort: 2049,
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerFileSystemEfsSecurityGroup289D043C',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseA36CE7FE',
                'Port',
            ],
        },
        ToPort: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseA36CE7FE',
                'Port',
            ],
        },
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseSecurityGroupBEFDC58F',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    });
});
/*
IAM Policy document tests. The policy for the installer instance is:
  expectCDK(stack).to(haveResourceLike('AWS::IAM::Policy', {
    PolicyDocument: {
      Statement: [
        {}, // CloudWatch log group put
        {}, // cloudwatch agent install script
        {}, // cloudwatch agent string parameters
        {}, // cloudwatch agent get installer permissions
        {}, // gpg get installer permissions
        {}, // DocDB secret get
        {}, // filesystem mount script get
        {}, // installer get
        {}, // repository installation script asset get
        {}, // update autoscaling policy
        {}, // describe tags
      ],
    },
  }));

  We only explicitly test for the permissions we explicitly add:
    - docDB secret get
    - installer get
    - autoscaling policy (handled by: 'alidate instance self-termination' test)
    - describe tags (handled by: 'alidate instance self-termination' test)
  The others are side-effects of adding the cloudwatch agent, and mounting the filesystem.
  We make sure that we do those things in other ways, and trust to their constructs to be
  verifying proper permissions.
 */
test('repository installer iam permissions: db secret access', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assertions_1.Match.arrayWith([{
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'repositoryInstallerDocumentDatabaseSecretAttachment29753B7C',
                    },
                }]),
        },
        PolicyName: assertions_1.Match.stringLikeRegexp('^repositoryInstallerInstanceRoleDefaultPolicy.*'),
    });
});
test('repository installer iam permissions: installer get', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assertions_1.Match.arrayWith([
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Resource: assertions_1.Match.arrayWith([
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        'Fn::Sub': asset_constants_1.CWA_ASSET_LINUX.Bucket,
                                    },
                                ],
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        'Fn::Sub': asset_constants_1.CWA_ASSET_LINUX.Bucket,
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ]),
                },
            ]),
        },
        PolicyName: assertions_1.Match.stringLikeRegexp('^repositoryInstallerInstanceRoleDefaultPolicy.*'),
    });
});
test('default repository installer log group created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: '/renderfarm/repositoryInstaller',
    });
});
test('repository installer logs all required files', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    // The CloudWatchAgent stores the agent configuration in an SSM Parameter. Check it for the required setup.
    // Note: This would be better implemented using the tools in: https://github.com/aws/aws-cdk/pull/8444
    // but that is not yet available.
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::SSM::Parameter', {
        Type: 'String',
        Value: {
            'Fn::Join': [
                '',
                [
                    '{\"logs\":{\"logs_collected\":{\"files\":{\"collect_list\":[{\"log_group_name\":\"',
                    {},
                    '\",\"log_stream_name\":\"cloud-init-output-{instance_id}\",\"file_path\":\"/var/log/cloud-init-output.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                    {},
                    '\",\"log_stream_name\":\"deadlineRepositoryInstallationLogs-{instance_id}\",\"file_path\":\"/tmp/bitrock_installer.log\",\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}',
                ],
            ],
        },
    });
});
test('repository mounts repository filesystem', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // WHEN
    const userData = repo.node.defaultChild.userData.render();
    // THEN
    expect(userData).toMatch(new RegExp(escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 false rw')));
});
test.each([
    [aws_cdk_lib_1.RemovalPolicy.DESTROY, 'Delete'],
    [aws_cdk_lib_1.RemovalPolicy.RETAIN, 'Retain'],
    [aws_cdk_lib_1.RemovalPolicy.SNAPSHOT, 'Snapshot'],
])('repository honors database removal policy: %p', (policy, expected) => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            database: policy,
        },
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResource('AWS::DocDB::DBCluster', {
        DeletionPolicy: expected,
    });
});
test.each([
    [aws_cdk_lib_1.RemovalPolicy.DESTROY, 'Delete'],
    [aws_cdk_lib_1.RemovalPolicy.RETAIN, 'Retain'],
    [aws_cdk_lib_1.RemovalPolicy.SNAPSHOT, 'Snapshot'],
])('repository honors filesystem removal policy: %p', (policy, expected) => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            filesystem: policy,
        },
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResource('AWS::EFS::FileSystem', {
        DeletionPolicy: expected,
    });
});
test('repository warns if removal policy for filesystem when filesystem provided', () => {
    // GIVEN
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testAP = new aws_efs_1.AccessPoint(stack, 'TestAccessPoint', {
        fileSystem: testEFS,
    });
    const testFS = new core_1.MountableEfs(stack, {
        filesystem: testEFS,
        accessPoint: testAP,
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
        removalPolicy: {
            filesystem: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        },
    });
    // THEN
    assertions_1.Annotations.fromStack(stack).hasWarning(`/${repo.node.path}`, 'RemovalPolicy for filesystem will not be applied since a filesystem is not being created by this construct');
});
test('repository warns if removal policy for database when database provided', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
        removalPolicy: {
            database: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        },
    });
    // THEN
    assertions_1.Annotations.fromStack(stack).hasWarning(`/${repo.node.path}`, assertions_1.Match.stringLikeRegexp('RemovalPolicy for database will not be applied since a database is not being created by this construct.*'));
});
test('repository creates deadlineDatabase if none provided', () => {
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testAP = new aws_efs_1.AccessPoint(stack, 'TestAccessPoint', {
        fileSystem: testEFS,
    });
    const testFS = new core_1.MountableEfs(stack, {
        filesystem: testEFS,
        accessPoint: testAP,
    });
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
    });
    // THEN
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::DocDB::DBCluster', 1);
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::DocDB::DBInstance', 1);
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::DocDB::DBCluster', {
        EnableCloudwatchLogsExports: ['audit'],
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::DocDB::DBClusterParameterGroup', {
        Parameters: {
            audit_logs: 'enabled',
        },
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::DocDB::DBInstance', {
        AutoMinorVersionUpgrade: true,
    });
});
test('disabling Audit logging does not enable Cloudwatch audit logs', () => {
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testAP = new aws_efs_1.AccessPoint(stack, 'TestAccessPoint', {
        fileSystem: testEFS,
    });
    const testFS = new core_1.MountableEfs(stack, {
        filesystem: testEFS,
        accessPoint: testAP,
    });
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
        databaseAuditLogging: false,
    });
    // THEN
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::DocDB::DBCluster', 1);
    test_helper_1.resourcePropertiesCountIs(stack, 'AWS::DocDB::DBCluster', {
        EnableCloudwatchLogsExports: assertions_1.Match.arrayWith(['audit']),
    }, 0);
    test_helper_1.resourcePropertiesCountIs(stack, 'AWS::DocDB::DBClusterParameterGroup', {
        Parameters: {
            audit_logs: 'enabled',
        },
    }, 0);
});
test('repository warns if databaseAuditLogging defined and database is specified', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            filesystem: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        },
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        databaseAuditLogging: true,
    });
    const warningMsg = 'The parameter databaseAuditLogging only has an effect when the Repository is creating its own database.\n' +
        'Please ensure that the Database provided is configured correctly.';
    // THEN
    assertions_1.Annotations.fromStack(stack).hasWarning(`/${repo.node.path}`, warningMsg);
});
test('honors subnet specification', () => {
    // GIVEN
    const dependencyStack = new aws_cdk_lib_1.Stack(app, 'DepStack');
    const dependencyVpc = new aws_ec2_1.Vpc(dependencyStack, 'DepVpc');
    const subnets = [
        aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet1', {
            subnetId: 'SubnetID1',
            availabilityZone: 'us-west-2a',
        }),
        aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet2', {
            subnetId: 'SubnetID2',
            availabilityZone: 'us-west-2b',
        }),
    ];
    const isolatedStack = new aws_cdk_lib_1.Stack(app, 'IsolatedStack');
    // WHEN
    new lib_1.Repository(isolatedStack, 'repositoryInstaller', {
        vpc: dependencyVpc,
        version,
        vpcSubnets: {
            subnets,
        },
    });
    // THEN
    assertions_1.Template.fromStack(isolatedStack).hasResourceProperties('AWS::DocDB::DBSubnetGroup', {
        SubnetIds: [
            'SubnetID1',
            'SubnetID2',
        ],
    });
    assertions_1.Template.fromStack(isolatedStack).hasResourceProperties('AWS::EFS::MountTarget', { SubnetId: 'SubnetID1' });
    assertions_1.Template.fromStack(isolatedStack).hasResourceProperties('AWS::EFS::MountTarget', { SubnetId: 'SubnetID2' });
});
test('repository honors database instance count', () => {
    // GIVEN
    const instanceCount = 2;
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        documentDbInstanceCount: instanceCount,
    });
    // THEN
    test_helper_1.resourcePropertiesCountIs(stack, 'AWS::DocDB::DBInstance', {
        AutoMinorVersionUpgrade: true,
    }, instanceCount);
});
test('repository honors database retention period', () => {
    // GIVEN
    const period = 20;
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        backupOptions: {
            databaseRetention: aws_cdk_lib_1.Duration.days(period),
        },
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::DocDB::DBCluster', {
        BackupRetentionPeriod: period,
    });
});
test('warns if both retention period and database provided', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
        backupOptions: {
            databaseRetention: aws_cdk_lib_1.Duration.days(20),
        },
    });
    // THEN
    assertions_1.Annotations.fromStack(stack).hasWarning(`/${repo.node.path}`, 'Backup retention for database will not be applied since a database is not being created by this construct');
});
test('repository creates filesystem if none provided', () => {
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
        },
        backup: {
            retention: aws_cdk_lib_1.Duration.days(15),
        },
    });
    if (!fsDatabase.secret) {
        throw new Error('secret cannot be null');
    }
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
    });
    // THEN
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::EFS::FileSystem', 1);
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::EFS::MountTarget', 2);
    expect(repo.node.tryFindChild('PadEfsStorage')).toBeDefined();
    expect(repo.node.findChild('FileSystem').node.tryFindChild('PaddingAccessPoint')).toBeDefined();
});
test('default repository instance is created using user defined installation path prefix', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        repositoryInstallationPrefix: 'xyz',
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toMatch(/\/mnt\/efs\/fs1\/xyz/);
});
test('default repository instance is created using user defined installation path prefix with extra slashes in path', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        repositoryInstallationPrefix: '/xyz//',
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toMatch(/\/mnt\/efs\/fs1\/xyz/);
});
test('repository instance is created with user defined timeout', () => {
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: aws_cdk_lib_1.Duration.minutes(30),
        version,
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResource('AWS::AutoScaling::AutoScalingGroup', {
        CreationPolicy: {
            ResourceSignal: {
                Count: 1,
                Timeout: 'PT30M',
            },
        },
    });
});
test('repository instance is created with correct installer path version', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: aws_cdk_lib_1.Duration.minutes(30),
        version,
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toEqual(expect.stringContaining(version.versionString));
});
test.each([
    'test-prefix/',
    '',
])('repository instance is created with correct LogGroup prefix %s', (testPrefix) => {
    // GIVEN
    const id = 'repositoryInstaller';
    // WHEN
    new lib_1.Repository(stack, id, {
        vpc,
        version,
        logGroupProps: {
            logGroupPrefix: testPrefix,
        },
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
        LogGroupName: testPrefix + id,
    });
});
test('validate instance self-termination', () => {
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: aws_cdk_lib_1.Duration.minutes(30),
        version,
    });
    // THEN
    const regionToken = escapeTokenRegex('${Token[AWS.Region.\\d+]}');
    const asgLogicalIdToken = escapeTokenRegex('${Token[Stack.repositoryInstaller.Installer.ASG.LogicalID.\\d+]}');
    const expectedString = `function exitTrap\\(\\)\\{\nexitCode=\\$\\?\nsleep 1m\nTOKEN=\\$\\(curl -X PUT "http:\\/\\/169\\.254\\.169\\.254\\/latest\\/api\\/token" -H "X-aws-ec2-metadata-token-ttl-seconds: 30" 2> \\/dev\\/null\\)\nINSTANCE="\\$\\(curl -s -H "X-aws-ec2-metadata-token: \\$TOKEN" http:\\/\\/169\\.254\\.169\\.254\\/latest\\/meta-data\\/instance-id  2> \\/dev\\/null\\)"\nASG="\\$\\(aws --region ${regionToken} ec2 describe-tags --filters "Name=resource-id,Values=\\$\\{INSTANCE\\}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags\\[0\\]\\.Value" --output text\\)"\naws --region ${regionToken} autoscaling update-auto-scaling-group --auto-scaling-group-name \\$\\{ASG\\} --min-size 0 --max-size 0 --desired-capacity 0\n\\/opt\\/aws\\/bin\\/cfn-signal --stack ${stack.stackName} --resource ${asgLogicalIdToken} --region ${regionToken} -e \\$exitCode \\|\\| echo 'Failed to send Cloudformation Signal'\n\\}`;
    expect(repo.node.defaultChild.userData.render()).toMatch(new RegExp(expectedString));
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assertions_1.Match.arrayWith([
                {
                    Action: 'autoscaling:UpdateAutoScalingGroup',
                    Condition: {
                        StringEquals: {
                            'autoscaling:ResourceTag/resourceLogicalId': aws_cdk_lib_1.Names.uniqueId(repo),
                        },
                    },
                    Effect: 'Allow',
                    Resource: '*',
                },
                {
                    Action: 'ec2:DescribeTags',
                    Effect: 'Allow',
                    Resource: '*',
                },
            ]),
        },
    });
});
test('repository configure client instance', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: aws_cdk_lib_1.Duration.minutes(30),
        version,
    });
    const instance = new aws_ec2_1.Instance(stack, 'Instance', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    const instanceRole = (instance
        .node.findChild('InstanceRole')
        .node.defaultChild);
    const db = (repo
        .node.findChild('DocumentDatabase'));
    // WHEN
    repo.configureClientInstance({
        host: instance,
        mountPoint: '/mnt/repository',
    });
    const userData = instance.userData.render();
    // THEN
    // white-box testing. If we mount the filesystem, then we've called: setupDirectConnect()
    expect(userData).toMatch(new RegExp(escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/repository false rw')));
    // Make sure we added the DB connection args
    expect(userData).toMatch(/.*export -f configure_deadline_database.*/);
    // Make sure we call the configureRepositoryDirectConnect script with appropriate argument.
    expect(userData).toContain(`'/tmp/${asset_constants_2.CONFIG_REPO_DIRECT_CONNECT_LINUX.Key}.sh' "/mnt/repository/DeadlineRepository"`);
    // Assert the IAM instance profile is given read access to the database credentials secret
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assertions_1.Match.arrayWith([{
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: stack.resolve(db.secret.secretArn),
                }]),
        },
        Roles: [
            stack.resolve(instanceRole.ref),
        ],
    });
});
test('configureClientInstance uses singleton for repo config script', () => {
    // Note: If this test fails, then check the asset hash for REPO_DC_ASSET -- it may have changed.
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: aws_cdk_lib_1.Duration.minutes(30),
        version,
    });
    const instance1 = new aws_ec2_1.Instance(stack, 'Instance1', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    const instance2 = new aws_ec2_1.Instance(stack, 'Instance2', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    // WHEN
    repo.configureClientInstance({
        host: instance1,
        mountPoint: '/mnt/repository',
    });
    repo.configureClientInstance({
        host: instance2,
        mountPoint: '/mnt/repository',
    });
    // THEN
    // Make sure that both instances have access to the same Asset for the configureRepositoryDirectConnect script
    test_helper_1.resourcePropertiesCountIs(stack, 'AWS::IAM::Policy', {
        PolicyDocument: assertions_1.Match.objectLike({
            Statement: assertions_1.Match.arrayWith([
                {
                    Effect: 'Allow',
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Resource: [
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        'Fn::Sub': asset_constants_2.REPO_DC_ASSET.Bucket,
                                    },
                                ],
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        'Fn::Sub': asset_constants_2.REPO_DC_ASSET.Bucket,
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                },
            ]),
        }),
        PolicyName: assertions_1.Match.stringLikeRegexp('Instance[1-2]InstanceRoleDefaultPolicy.*'),
    }, 2);
});
test('windows client cannot direct connect to repository', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: aws_cdk_lib_1.Duration.minutes(30),
        version,
    });
    const instance = new aws_ec2_1.Instance(stack, 'Instance', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
    });
    // THEN
    expect(() => {
        repo.configureClientInstance({
            host: instance,
            mountPoint: 'd:\\',
        });
    }).toThrowError('Deadline direct connect on Windows hosts is not yet supported by the RFDK.');
});
describe('tagging', () => {
    tag_helpers_1.testConstructTags({
        constructName: 'Repository',
        createConstruct: () => {
            // GIVEN
            const isolatedStack = new aws_cdk_lib_1.Stack(app, 'IsolatedStack');
            new lib_1.Repository(isolatedStack, 'Repository', {
                vpc,
                version,
            });
            return isolatedStack;
        },
        resourceTypeCounts: {
            'AWS::EC2::SecurityGroup': 3,
            'AWS::DocDB::DBClusterParameterGroup': 1,
            'AWS::DocDB::DBSubnetGroup': 1,
            'AWS::SecretsManager::Secret': 2,
            'AWS::DocDB::DBCluster': 1,
            'AWS::DocDB::DBInstance': 1,
            'AWS::IAM::Role': 1,
            'AWS::AutoScaling::AutoScalingGroup': 1,
            'AWS::SSM::Parameter': 1,
        },
    });
});
describe('Security Groups', () => {
    let repositorySecurityGroup;
    beforeEach(() => {
        repositorySecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'AdditionalSecurityGroup', { vpc });
    });
    describe('DocDB', () => {
        test('adds security groups on construction', () => {
            // WHEN
            new lib_1.Repository(stack, 'Repository', {
                version,
                vpc,
                securityGroupsOptions: {
                    database: repositorySecurityGroup,
                },
            });
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::DocDB::DBCluster', {
                VpcSecurityGroupIds: assertions_1.Match.arrayWith([stack.resolve(repositorySecurityGroup.securityGroupId)]),
            });
        });
    });
    describe('EFS', () => {
        test('adds security groups on construction', () => {
            // WHEN
            new lib_1.Repository(stack, 'Repository', {
                version,
                vpc,
                securityGroupsOptions: {
                    fileSystem: repositorySecurityGroup,
                },
            });
            // THEN
            // The EFS construct adds the security group to each mount target, and one mount target is generated per subnet.
            const numMountTargets = vpc.selectSubnets().subnets.length;
            test_helper_1.resourcePropertiesCountIs(stack, 'AWS::EFS::MountTarget', {
                SecurityGroups: assertions_1.Match.arrayWith([stack.resolve(repositorySecurityGroup.securityGroupId)]),
            }, numMountTargets);
        });
    });
    describe('Installer', () => {
        test('adds security groups on construction', () => {
            // WHEN
            new lib_1.Repository(stack, 'Repository', {
                version,
                vpc,
                securityGroupsOptions: {
                    installer: repositorySecurityGroup,
                },
            });
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
                SecurityGroups: assertions_1.Match.arrayWith([stack.resolve(repositorySecurityGroup.securityGroupId)]),
            });
        });
    });
});
test('validates VersionQuery is not in a different stack', () => {
    // GIVEN
    const newStack = new aws_cdk_lib_1.Stack(app, 'NewStack');
    version = new lib_1.VersionQuery(stack, 'Version');
    new lib_1.Repository(newStack, 'Repository', {
        vpc,
        version,
    });
    // WHEN
    function synth() {
        app.synth();
    }
    // THEN
    expect(synth).toThrow('A VersionQuery can not be supplied from a different stack');
});
test('creates an EFS AccessPoint when no filesystem is supplied', () => {
    // WHEN
    const repo = new lib_1.Repository(stack, 'Repository', {
        version,
        vpc,
    });
    // THEN
    const efsResource = repo.node.findChild('FileSystem').node.defaultChild;
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EFS::AccessPoint', {
        FileSystemId: stack.resolve(efsResource.ref),
        PosixUser: {
            Gid: '0',
            Uid: '0',
        },
        RootDirectory: {},
    });
});
test('throws an error if supplied a MountableEfs with no Access Point', () => {
    // GIVEN
    const newStack = new aws_cdk_lib_1.Stack(app, 'NewStack');
    const fs = new aws_efs_1.FileSystem(newStack, 'FileSystem', {
        vpc,
    });
    const mountableFs = new core_1.MountableEfs(newStack, {
        filesystem: fs,
    });
    // WHEN
    function when() {
        new lib_1.Repository(newStack, 'Repo', {
            version,
            vpc,
            fileSystem: mountableFs,
        });
    }
    // THEN
    expect(when).toThrow('When using EFS with the Repository, you must provide an EFS Access Point');
});
test('disable Secrets Management by default when Deadline version is old', () => {
    // GIVEN
    const newStack = new aws_cdk_lib_1.Stack(app, 'NewStack');
    const oldVersion = new lib_1.VersionQuery(newStack, 'OldDeadlineVersion', { version: '10.0.0.0' });
    // WHEN
    const repository = new lib_1.Repository(newStack, 'Repo', {
        vpc,
        version: oldVersion,
    });
    // THEN
    expect(repository.secretsManagementSettings.enabled).toBeFalsy();
    expect(repository.secretsManagementSettings.credentials).toBeUndefined();
});
test('throws when Secrets Management is enabled but deadline version is too low', () => {
    // GIVEN
    const newStack = new aws_cdk_lib_1.Stack(app, 'NewStack');
    const oldVersion = new lib_1.VersionQuery(newStack, 'OldDeadlineVersion', { version: '10.0.0.0' });
    // WHEN
    function when() {
        new lib_1.Repository(newStack, 'Repo', {
            version: oldVersion,
            vpc,
            secretsManagementSettings: {
                enabled: true,
            },
        });
    }
    // THEN
    expect(when).toThrow(`The supplied Deadline version (${oldVersion.versionString}) does not support Deadline Secrets Management in RFDK. Either upgrade Deadline to the minimum required version (${lib_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION.versionString}) or disable the feature in the Repository's construct properties.`);
});
test('imports repository settings', () => {
    // GIVEN
    const repositorySettings = new aws_s3_assets_1.Asset(stack, 'RepositorySettingsAsset', {
        path: __filename,
    });
    // WHEN
    const repository = new lib_1.Repository(stack, 'Repository', {
        vpc,
        version,
        repositorySettings,
    });
    // THEN
    const installerGroup = repository.node.tryFindChild('Installer');
    // Note: The repostory settings is the js file that this compiles in to. The hash for that is fragile in that any
    // change to this file will change it. So, we search for an s3 cp of a js file (this is the only one).
    expect(installerGroup.userData.render()).toMatch(new RegExp('aws s3 cp \'s3://[^\']+/[a-f0-9]+.js\''));
});
test('IMountableLinuxFilesystem.usesUserPosixPermissions() = true changes ownership of repository files', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'Repository', {
        version,
        vpc,
        fileSystem: {
            mountToLinuxInstance: (_target, _mount) => { },
            usesUserPosixPermissions: () => true,
        },
    });
    // WHEN
    const script = repo.node.defaultChild.userData.render();
    // THEN
    expect(script).toMatch('-o 1000:1000');
});
test('IMountableLinuxFilesystem.usesUserPosixPermissions() = false does not change ownership of repository files', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'Repository', {
        version,
        vpc,
        fileSystem: {
            mountToLinuxInstance: (_target, _mount) => { },
            usesUserPosixPermissions: () => false,
        },
    });
    // WHEN
    const script = repo.node.defaultChild.userData.render();
    // THEN
    expect(script).not.toMatch('-o 1000:1000');
});
test('secret manager enabled', () => {
    // GIVEN
    const expectedCredentials = new aws_secretsmanager_1.Secret(stack, 'CustomSMAdminUser', {
        description: 'Custom admin credentials for the Secret Management',
        generateSecretString: {
            excludeCharacters: '\"$&\'()-/<>[\\]\`{|}',
            includeSpace: false,
            passwordLength: 24,
            requireEachIncludedType: true,
            generateStringKey: 'password',
            secretStringTemplate: JSON.stringify({ username: 'admin' }),
        },
    });
    // WHEN
    const repository = new lib_1.Repository(stack, 'Repository', {
        vpc,
        version,
        secretsManagementSettings: {
            enabled: true,
            credentials: expectedCredentials,
        },
    });
    // THEN
    expect(repository.secretsManagementSettings.credentials).toBe(expectedCredentials);
    const installerGroup = repository.node.tryFindChild('Installer');
    expect(installerGroup.userData.render()).toContain(`-r ${stack.region} -c ${expectedCredentials.secretArn}`);
});
test('secret manager is enabled by default', () => {
    // WHEN
    const repository = new lib_1.Repository(stack, 'Repository', {
        vpc,
        version,
    });
    // THEN
    expect(repository.secretsManagementSettings.enabled).toBeTruthy();
    expect(repository.secretsManagementSettings.credentials).toBeDefined();
});
test('credentials are undefined when secrets management is disabled', () => {
    // WHEN
    const repository = new lib_1.Repository(stack, 'Repository', {
        vpc,
        version,
        secretsManagementSettings: {
            enabled: false,
        },
    });
    // THEN
    expect(repository.secretsManagementSettings.credentials).toBeUndefined();
});
test('throws an error if credentials are undefined and database is imported', () => {
    // GIVEN
    const sg = new aws_ec2_1.SecurityGroup(stack, 'SG', {
        vpc,
    });
    const secret = new aws_secretsmanager_1.Secret(stack, 'Secret');
    const database = aws_docdb_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'DbCluster', {
        clusterEndpointAddress: '1.2.3.4',
        clusterIdentifier: 'foo',
        instanceEndpointAddresses: ['1.2.3.5'],
        instanceIdentifiers: ['i0'],
        port: 27001,
        readerEndpointAddress: '1.2.3.6',
        securityGroup: sg,
    });
    const databaseConnection = lib_1.DatabaseConnection.forDocDB({ database, login: secret });
    // WHEN
    function when() {
        new lib_1.Repository(stack, 'Repository', {
            vpc,
            version,
            database: databaseConnection,
        });
    }
    // THEN
    expect(when).toThrow('Admin credentials for Deadline Secrets Management cannot be generated when using an imported database. For setting up your own credentials, please refer to https://github.com/aws/aws-rfdk/tree/mainline/packages/aws-rfdk/lib/deadline#configuring-deadline-secrets-management-on-the-repository.');
});
//# sourceMappingURL=data:application/json;base64,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