"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_route53_1 = require("aws-cdk-lib/aws-route53");
const core_1 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const lib_1 = require("../lib");
const spot_event_plugin_fleet_1 = require("../lib/spot-event-plugin-fleet");
const test_helper_1 = require("./test-helper");
describe('ConfigureSpotEventPlugin', () => {
    let stack;
    let vpc;
    let region;
    let renderQueue;
    let version;
    let app;
    let fleet;
    let groupName;
    const workerMachineImage = new aws_ec2_1.GenericWindowsImage({
        'us-east-1': 'ami-any',
    });
    beforeEach(() => {
        region = 'us-east-1';
        app = new aws_cdk_lib_1.App();
        stack = new aws_cdk_lib_1.Stack(app, 'stack', {
            env: {
                region,
            },
        });
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        version = new lib_1.VersionQuery(stack, 'Version');
        renderQueue = new lib_1.RenderQueue(stack, 'RQ', {
            vpc,
            images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
            repository: new lib_1.Repository(stack, 'Repository', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            }),
            trafficEncryption: { externalTLS: { enabled: false } },
            version,
        });
        groupName = 'group_name1';
        fleet = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotFleet', {
            vpc,
            renderQueue: renderQueue,
            deadlineGroups: [
                groupName,
            ],
            instanceTypes: [
                aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL),
            ],
            workerMachineImage,
            maxCapacity: 1,
        });
    });
    describe('creates a custom resource', () => {
        test('with default spot event plugin properties', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_ConfigureSpotEventPlugin', assertions_1.Match.objectLike({
                spotPluginConfigurations: assertions_1.Match.objectLike({
                    AWSInstanceStatus: 'Disabled',
                    DeleteInterruptedSlaves: false,
                    DeleteTerminatedSlaves: false,
                    IdleShutdown: 10,
                    Logging: 'Standard',
                    PreJobTaskMode: 'Conservative',
                    Region: aws_cdk_lib_1.Stack.of(renderQueue).region,
                    ResourceTracker: true,
                    StaggerInstances: 50,
                    State: 'Global Enabled',
                    StrictHardCap: false,
                }),
            }));
        });
        test('with custom spot event plugin properties', () => {
            // GIVEN
            const configuration = {
                awsInstanceStatus: lib_1.SpotEventPluginDisplayInstanceStatus.EXTRA_INFO_0,
                deleteEC2SpotInterruptedWorkers: true,
                deleteSEPTerminatedWorkers: true,
                idleShutdown: aws_cdk_lib_1.Duration.minutes(20),
                loggingLevel: lib_1.SpotEventPluginLoggingLevel.VERBOSE,
                preJobTaskMode: lib_1.SpotEventPluginPreJobTaskMode.NORMAL,
                region: 'us-west-2',
                enableResourceTracker: false,
                maximumInstancesStartedPerCycle: 10,
                state: lib_1.SpotEventPluginState.DISABLED,
                strictHardCap: true,
            };
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
                configuration,
            });
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_ConfigureSpotEventPlugin', assertions_1.Match.objectLike({
                spotPluginConfigurations: assertions_1.Match.objectLike({
                    AWSInstanceStatus: 'ExtraInfo0',
                    DeleteInterruptedSlaves: true,
                    DeleteTerminatedSlaves: true,
                    IdleShutdown: 20,
                    Logging: 'Verbose',
                    PreJobTaskMode: 'Normal',
                    Region: 'us-west-2',
                    ResourceTracker: false,
                    StaggerInstances: 10,
                    State: 'Disabled',
                    StrictHardCap: true,
                }),
            }));
        });
        test('without spot fleets', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
            });
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_ConfigureSpotEventPlugin', {
                spotFleetRequestConfigurations: assertions_1.Match.absent(),
            });
        });
        test('provides RQ connection parameters to custom resource', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_ConfigureSpotEventPlugin', assertions_1.Match.objectLike({
                connection: assertions_1.Match.objectLike({
                    hostname: stack.resolve(renderQueue.endpoint.hostname),
                    port: stack.resolve(renderQueue.endpoint.portAsString()),
                    protocol: stack.resolve(renderQueue.endpoint.applicationProtocol.toString()),
                }),
            }));
        });
        test('with default spot fleet request configuration', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            const rfdkTag = runtime_info_1.tagFields(fleet);
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_ConfigureSpotEventPlugin', {
                spotFleetRequestConfigurations: {
                    [groupName]: {
                        AllocationStrategy: 'lowestPrice',
                        IamFleetRole: stack.resolve(fleet.fleetRole.roleArn),
                        LaunchTemplateConfigs: assertions_1.Match.arrayWith([
                            assertions_1.Match.objectLike({
                                LaunchTemplateSpecification: {
                                    Version: '$Latest',
                                    LaunchTemplateId: stack.resolve(fleet.launchTemplate.launchTemplateId),
                                },
                            }),
                        ]),
                        TagSpecifications: assertions_1.Match.arrayWith([
                            assertions_1.Match.objectLike({
                                ResourceType: 'spot-fleet-request',
                                Tags: assertions_1.Match.arrayWith([
                                    {
                                        Key: rfdkTag.name,
                                        Value: rfdkTag.value,
                                    },
                                ]),
                            }),
                        ]),
                    },
                },
            });
        });
        test('adds policies to the render queue', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            test_helper_1.resourcePropertiesCountIs(stack, 'AWS::IAM::Role', {
                ManagedPolicyArns: assertions_1.Match.arrayWith([
                    stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginAdminPolicy').managedPolicyArn),
                    stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy').managedPolicyArn),
                ]),
            }, 1);
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'iam:PassRole',
                            Condition: {
                                StringLike: {
                                    'iam:PassedToService': 'ec2.amazonaws.com',
                                },
                            },
                            Effect: 'Allow',
                            Resource: [
                                stack.resolve(fleet.fleetRole.roleArn),
                                stack.resolve(fleet.fleetInstanceRole.roleArn),
                            ],
                        },
                        {
                            Action: 'ec2:CreateTags',
                            Effect: 'Allow',
                            Resource: [
                                'arn:aws:ec2:*:*:spot-fleet-request/*',
                                'arn:aws:ec2:*:*:volume/*',
                            ],
                        },
                    ],
                },
                Roles: [{
                        Ref: 'RQRCSTaskTaskRole00DC9B43',
                    }],
            });
        });
        test('adds resource tracker policy even if rt disabled', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
                configuration: {
                    enableResourceTracker: false,
                },
            });
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Role', {
                ManagedPolicyArns: assertions_1.Match.arrayWith([
                    stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy').managedPolicyArn),
                ]),
            });
        });
        test.each([
            undefined,
            [],
        ])('without spot fleet', (noFleets) => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: noFleets,
            });
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_ConfigureSpotEventPlugin', assertions_1.Match.objectLike({
                spotFleetRequestConfigurations: assertions_1.Match.absent(),
            }));
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Role', assertions_1.Match.not({
                ManagedPolicyArns: assertions_1.Match.arrayWith([
                    stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginAdminPolicy').managedPolicyArn),
                    stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy').managedPolicyArn),
                ]),
            }));
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', assertions_1.Match.not({
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'iam:PassRole',
                            Condition: {
                                StringLike: {
                                    'iam:PassedToService': 'ec2.amazonaws.com',
                                },
                            },
                            Effect: 'Allow',
                            Resource: [
                                stack.resolve(fleet.fleetRole.roleArn),
                                stack.resolve(fleet.fleetInstanceRole.roleArn),
                            ],
                        },
                        {
                            Action: 'ec2:CreateTags',
                            Effect: 'Allow',
                            Resource: 'arn:aws:ec2:*:*:spot-fleet-request/*',
                        },
                    ],
                },
                Roles: [{
                        Ref: 'RQRCSTaskTaskRole00DC9B43',
                    }],
            }));
        });
        test('fleet with validUntil', () => {
            // GIVEN
            const validUntil = aws_cdk_lib_1.Expiration.atDate(new Date(2022, 11, 17));
            const fleetWithCustomProps = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                vpc,
                renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                ],
                workerMachineImage,
                maxCapacity: 1,
                validUntil,
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleetWithCustomProps,
                ],
            });
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_ConfigureSpotEventPlugin', assertions_1.Match.objectLike({
                spotFleetRequestConfigurations: assertions_1.Match.objectLike({
                    [groupName]: assertions_1.Match.objectLike({
                        ValidUntil: validUntil.date.toISOString(),
                    }),
                }),
            }));
        });
    });
    test('only one object allowed per render queue', () => {
        // GIVEN
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        // WHEN
        function createConfigureSpotEventPlugin() {
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin2', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).toThrowError(/Only one ConfigureSpotEventPlugin construct is allowed per render queue./);
    });
    test('can create multiple objects with different render queues', () => {
        // GIVEN
        const renderQueue2 = new lib_1.RenderQueue(stack, 'RQ2', {
            vpc,
            images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
            repository: new lib_1.Repository(stack, 'Repository2', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            }),
            trafficEncryption: { externalTLS: { enabled: false } },
            version,
        });
        // WHEN
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin2', {
            vpc,
            renderQueue: renderQueue2,
            spotFleets: [
                fleet,
            ],
        });
        // THEN
        assertions_1.Template.fromStack(stack).resourceCountIs('Custom::RFDK_ConfigureSpotEventPlugin', 2);
    });
    test('throws with not supported render queue', () => {
        // GIVEN
        const invalidRenderQueue = {};
        // WHEN
        function createConfigureSpotEventPlugin() {
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin2', {
                vpc,
                renderQueue: invalidRenderQueue,
                spotFleets: [
                    fleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).toThrowError(/The provided render queue is not an instance of RenderQueue class. Some functionality is not supported./);
    });
    test('tagSpecifications returns undefined if fleet does not have tags', () => {
        // GIVEN
        const mockFleet = {
            tags: {
                hasTags: jest.fn().mockReturnValue(false),
            },
        };
        const mockedFleet = mockFleet;
        const config = new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        // WHEN
        // eslint-disable-next-line dot-notation
        const result = stack.resolve(config['tagSpecifications'](mockedFleet, lib_1.SpotFleetResourceType.INSTANCE));
        // THEN
        expect(result).toBeUndefined();
    });
    describe('with TLS', () => {
        let renderQueueWithTls;
        let caCert;
        beforeEach(() => {
            const host = 'renderqueue';
            const zoneName = 'deadline-test.internal';
            caCert = new core_1.X509CertificatePem(stack, 'RootCA', {
                subject: {
                    cn: 'SampleRootCA',
                },
            });
            renderQueueWithTls = new lib_1.RenderQueue(stack, 'RQ with TLS', {
                vpc,
                images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
                repository: new lib_1.Repository(stack, 'Repository2', {
                    vpc,
                    version,
                }),
                version,
                hostname: {
                    zone: new aws_route53_1.PrivateHostedZone(stack, 'DnsZone', {
                        vpc,
                        zoneName: zoneName,
                    }),
                    hostname: host,
                },
                trafficEncryption: {
                    externalTLS: {
                        rfdkCertificate: new core_1.X509CertificatePem(stack, 'RQCert', {
                            subject: {
                                cn: `${host}.${zoneName}`,
                            },
                            signingCertificate: caCert,
                        }),
                    },
                },
            });
        });
        test('Lambda role can get the ca secret', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueueWithTls,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: stack.resolve(renderQueueWithTls.certChain.secretArn),
                        },
                    ],
                },
                Roles: [
                    {
                        Ref: 'ConfigureSpotEventPluginConfiguratorServiceRole341B4735',
                    },
                ],
            });
        });
        test('creates a custom resource with connection', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueueWithTls,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_ConfigureSpotEventPlugin', assertions_1.Match.objectLike({
                connection: assertions_1.Match.objectLike({
                    hostname: stack.resolve(renderQueueWithTls.endpoint.hostname),
                    port: stack.resolve(renderQueueWithTls.endpoint.portAsString()),
                    protocol: stack.resolve(renderQueueWithTls.endpoint.applicationProtocol.toString()),
                    caCertificateArn: stack.resolve(renderQueueWithTls.certChain.secretArn),
                }),
            }));
        });
    });
    test('throws with the same group name', () => {
        // WHEN
        function createConfigureSpotEventPlugin() {
            const duplicateFleet = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'DuplicateSpotFleet', {
                vpc,
                renderQueue,
                workerMachineImage: fleet.machineImage,
                instanceTypes: fleet.instanceTypes,
                maxCapacity: fleet.maxCapacity,
                deadlineGroups: fleet.deadlineGroups,
            });
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                    duplicateFleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).toThrowError(`Bad Group Name: ${groupName}. Group names in Spot Fleet Request Configurations should be unique.`);
    });
    test('uses selected subnets', () => {
        // WHEN
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            vpcSubnets: { subnets: [vpc.privateSubnets[0]] },
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
            Handler: 'configure-spot-event-plugin.configureSEP',
            VpcConfig: {
                SubnetIds: [
                    stack.resolve(vpc.privateSubnets[0].subnetId),
                ],
            },
        });
    });
    describe('throws with wrong deadline version', () => {
        test.each([
            ['10.1.9'],
            ['10.1.10'],
        ])('%s', (versionString) => {
            // GIVEN
            const newStack = new aws_cdk_lib_1.Stack(app, 'NewStack');
            version = new lib_1.VersionQuery(newStack, 'OldVersion', {
                version: versionString,
            });
            renderQueue = new lib_1.RenderQueue(newStack, 'OldRenderQueue', {
                vpc,
                images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
                repository: new lib_1.Repository(newStack, 'Repository', {
                    vpc,
                    version,
                    secretsManagementSettings: { enabled: false },
                }),
                trafficEncryption: { externalTLS: { enabled: false } },
                version,
            });
            // WHEN
            function createConfigureSpotEventPlugin() {
                new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                    vpc,
                    renderQueue: renderQueue,
                    spotFleets: [
                        fleet,
                    ],
                });
            }
            // THEN
            expect(createConfigureSpotEventPlugin).toThrowError(`Minimum supported Deadline version for ConfigureSpotEventPlugin is 10.1.12.0. Received: ${versionString}.`);
        });
    });
    test('does not throw with min deadline version', () => {
        // GIVEN
        const versionString = '10.1.12';
        const newStack = new aws_cdk_lib_1.Stack(app, 'NewStack');
        version = new lib_1.VersionQuery(newStack, 'OldVersion', {
            version: versionString,
        });
        renderQueue = new lib_1.RenderQueue(newStack, 'OldRenderQueue', {
            vpc,
            images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
            repository: new lib_1.Repository(newStack, 'Repository', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            }),
            trafficEncryption: { externalTLS: { enabled: false } },
            version,
        });
        // WHEN
        function createConfigureSpotEventPlugin() {
            new lib_1.ConfigureSpotEventPlugin(newStack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).not.toThrow();
    });
    describe('secrets management enabled', () => {
        beforeEach(() => {
            region = 'us-east-1';
            app = new aws_cdk_lib_1.App();
            stack = new aws_cdk_lib_1.Stack(app, 'stack', {
                env: {
                    region,
                },
            });
            vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
            version = new lib_1.VersionQuery(stack, 'Version');
            renderQueue = new lib_1.RenderQueue(stack, 'RQ', {
                vpc,
                images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
                repository: new lib_1.Repository(stack, 'Repository', {
                    vpc,
                    version,
                }),
                version,
            });
            groupName = 'group_name1';
        });
        test('a fleet without vpcSubnets specified => warns about dedicated subnets', () => {
            // GIVEN
            fleet = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue: renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL),
                ],
                workerMachineImage,
                maxCapacity: 1,
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                renderQueue,
                vpc,
                spotFleets: [fleet],
            });
            // THEN
            assertions_1.Annotations.fromStack(stack).hasWarning(`/${fleet.node.path}`, 'Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components');
        });
        test('a fleet with vpcSubnets specified => does not warn about dedicated subnets', () => {
            // GIVEN
            fleet = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotFleetWithSubnets', {
                vpc,
                vpcSubnets: {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
                },
                renderQueue: renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL),
                ],
                workerMachineImage,
                maxCapacity: 1,
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                renderQueue,
                vpc,
                spotFleets: [fleet],
            });
            // THEN
            assertions_1.Annotations.fromStack(stack).hasNoWarning(`/${fleet.node.path}`, assertions_1.Match.stringLikeRegexp('.*dedicated subnet.*'));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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