"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.WaitForStableService = void 0;
const crypto_1 = require("crypto");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
/**
 * Depend on this construct to wait until the ECS Service becomes stable.
 * See https://docs.aws.amazon.com/cli/latest/reference/ecs/wait/services-stable.html.
 */
class WaitForStableService extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const lambdaRole = new aws_iam_1.Role(this, 'ECSWaitLambdaRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
            inlinePolicies: {
                describeServices: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                'ecs:DescribeServices',
                            ],
                            effect: aws_iam_1.Effect.ALLOW,
                            resources: [props.service.cluster.clusterArn, props.service.serviceArn],
                        }),
                    ],
                }),
            },
        });
        const waitingFunction = new aws_lambda_1.Function(this, 'ECSWait', {
            role: lambdaRole,
            description: `Used by a WaitForStableService ${this.node.addr} to wait until ECS service becomes stable`,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'), {}),
            environment: {
                DEBUG: 'false',
            },
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            handler: 'wait-for-stable-service.wait',
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        const properties = {
            cluster: props.service.cluster.clusterArn,
            services: [props.service.serviceArn],
            forceRun: this.forceRun(),
        };
        const resource = new aws_cdk_lib_1.CustomResource(this, 'Default', {
            serviceToken: waitingFunction.functionArn,
            resourceType: 'Custom::RFDK_WaitForStableService',
            properties,
        });
        // Prevents a race during a stack-update.
        resource.node.addDependency(lambdaRole);
        resource.node.addDependency(props.service);
        this.node.defaultChild = resource;
    }
    forceRun() {
        return crypto_1.randomBytes(32).toString('base64').slice(0, 32);
    }
}
exports.WaitForStableService = WaitForStableService;
//# sourceMappingURL=data:application/json;base64,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