/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { AutoScalingGroup } from 'aws-cdk-lib/aws-autoscaling';
import { IConnectable, InstanceType, ISecurityGroup, IVpc, Port, SubnetSelection } from 'aws-cdk-lib/aws-ec2';
import { Cluster, ContainerImage, Ec2Service } from 'aws-cdk-lib/aws-ecs';
import { IGrantable, IPrincipal } from 'aws-cdk-lib/aws-iam';
import { ISecret } from 'aws-cdk-lib/aws-secretsmanager';
import { Construct } from 'constructs';
import { LogGroupFactoryProps } from '../../core';
import { IRenderQueue } from './render-queue';
/**
 * Properties for constructing a {@link UsageBasedLicense} instance.
 */
export interface UsageBasedLicenseProps {
    /**
     * The name of the product that the usage-based license applies to.
     */
    readonly licenseName: string;
    /**
     * The set of ports that are used for licensing traffic
     */
    readonly ports: Port[];
    /**
     * The maximum number of usage-based licenses that can be used concurrently.
     */
    readonly limit?: number;
}
/**
 * Instances of this class represent a usage-based license for a particular product.
 * It encapsulates all of the information specific to a product that the UsageBasedLicensing
 * construct requires to interoperate with that product.
 */
export declare class UsageBasedLicense {
    /**
     * Constant used to signify unlimited overage.
     */
    static readonly UNLIMITED: number;
    /**
     * Method for 3dsMax license limit.
     *
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static for3dsMax(limit?: number): UsageBasedLicense;
    /**
     * Method for Arnold license limit
     *
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forArnold(limit?: number): UsageBasedLicense;
    /**
     * Method for Cinema 4D license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forCinema4D(limit?: number): UsageBasedLicense;
    /**
     * Method for Clarisse license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forClarisse(limit?: number): UsageBasedLicense;
    /**
     * Method for Houdini license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forHoudini(limit?: number): UsageBasedLicense;
    /**
     * Method for Katana license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forKatana(limit?: number): UsageBasedLicense;
    /**
     * Method for KeyShot license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forKeyShot(limit?: number): UsageBasedLicense;
    /**
     * Method for krakatoa license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forKrakatoa(limit?: number): UsageBasedLicense;
    /**
     * Method for Mantra license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forMantra(limit?: number): UsageBasedLicense;
    /**
     * Method for maxwell license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forMaxwell(limit?: number): UsageBasedLicense;
    /**
     * Method for Maya license limit
     *
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forMaya(limit?: number): UsageBasedLicense;
    /**
     * Method for Nuke license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forNuke(limit?: number): UsageBasedLicense;
    /**
     * Method for RealFlow license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forRealFlow(limit?: number): UsageBasedLicense;
    /**
     * Method for RedShift license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forRedShift(limit?: number): UsageBasedLicense;
    /**
     * Method for V-Ray license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forVray(limit?: number): UsageBasedLicense;
    /**
     * Method for Yeti license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forYeti(limit?: number): UsageBasedLicense;
    /**
     * The name of license limit
     */
    readonly licenseName: string;
    /**
     * Ports that will be used for this license
     */
    readonly ports: Port[];
    /**
     * Maximum count of licenses that will be used
     */
    readonly limit?: number;
    constructor(props: UsageBasedLicenseProps);
}
/**
 * Set of container images used to serve the {@link UsageBasedLicensing} construct
 */
export interface UsageBasedLicensingImages {
    /**
     * The container image for the Deadline License Forwarder
     */
    readonly licenseForwarder: ContainerImage;
}
/**
 * Properties for the UsageBasedLicensing construct
 */
export interface UsageBasedLicensingProps {
    /**
     * VPC to launch the License Forwarder In
     */
    readonly vpc: IVpc;
    /**
     * Subnets within the VPC in which to host the UBLLicesing servers.
     *
     * @default All private subnets in the VPC.
     */
    readonly vpcSubnets?: SubnetSelection;
    /**
     * The Deadline Render Queue, to which the License Forwarder needs to be connected.
     */
    readonly renderQueue: IRenderQueue;
    /**
     * Type of instance that will be added to an AutoScalingGroup.
     *
     * @default - Will be used C5 Large instance
     */
    readonly instanceType?: InstanceType;
    /**
     * Docker Image for License Forwarder
     */
    readonly images: UsageBasedLicensingImages;
    /**
     * A secret with with 3rd Party Licensing Certificates.
     *
     * If you want to use 3rd Party Licensing Certificates you need to purchase render time on Thinkbox Marketplace
     * and download file with certificates.
     * File with certificates should be put in in a secret.
     */
    readonly certificateSecret: ISecret;
    /**
     * The desired number of Deadline License Forwarders that this construct keeps running.
     *
     * @default 1
     */
    readonly desiredCount?: number;
    /**
     * License limits that will be set in repository configuration
     */
    readonly licenses: UsageBasedLicense[];
    /**
     * Properties for setting up the Deadline License Forwarder's LogGroup in CloudWatch
     * @default - LogGroup will be created with all properties' default values to the LogGroup: /renderfarm/<construct id>
     */
    readonly logGroupProps?: LogGroupFactoryProps;
    /**
     * The security group to use for the License Forwarder
     * @default - A new security group will be created
     */
    readonly securityGroup?: ISecurityGroup;
}
/**
 * This construct is an implementation of the Deadline component that is required for Usage-based Licensing (UBL)
 * (see: https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/licensing-usage-based.html )
 * in a render farm.
 *
 * Internally this is implemented as one or more instances of the Deadline License Forwarder application set up
 * to communicate to the render queue and Thinkbox’s licensing system, and to allow ingress connections
 * from the worker nodes so that they can acquire licenses as needed.
 *
 * The Deadline License Forwarder is set up to run within an AWS ECS task.
 *
 * Access to the running License Forwarder is gated by a security group that, by default, only allows ingress from the
 * Render Queue (in order to register Workers for license forwarding).
 *
 * When a Deadline Worker requires access to licensing via `UsageBasedLicensing.grantPortAccess(...)`, then the RFDK
 * constructs will grant that worker’s security group ingress on TCP port 17004 as well as other ports as required by
 * the specific licenses being used.
 *
 * Note: This construct does not currently implement the Deadline License Forwarder's Web Forwarding functionality.
 * This construct is not usable in any China region.
 *
 * ![architecture diagram](/diagrams/deadline/UsageBasedLicensing.svg)
 *
 * Resources Deployed
 * ------------------------
 * - The Auto Scaling Group (ASG) added to the Amazon Elastic Container Service cluster that is hosting the Deadline
 *   License Forwarder for UBL. This creates one C5 Large instance by default.
 * - Amazon Elastic Block Store (EBS) device(s) associated with the EC2 instance(s) in the ASG. The default volume size is 30 GiB.
 * - An Amazon CloudWatch log group that contains the logs from the Deadline License Forwarder application.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The Deadline License Forwarder is designed to be secured by restricting network access to it. For security, only the Deadline
 *   Workers that require access to Usage-based Licenses should be granted network access to the instances deployed by this construct.
 *   Futhermore, you should restrict that access to only the product(s) that those workers require when deploying this construct.
 */
export declare class UsageBasedLicensing extends Construct implements IGrantable {
    /**
     * The port that the License Forwarder listens on
     */
    private static readonly LF_PORT;
    /**
     * Default prefix for a LogGroup if one isn't provided in the props.
     */
    private static readonly DEFAULT_LOG_GROUP_PREFIX;
    /**
     * The Amazon ECS cluster that is hosting the Deadline License Forwarder for UBL.
     */
    readonly cluster: Cluster;
    /**
     * Autoscaling group for license forwarder instances
     */
    readonly asg: AutoScalingGroup;
    /**
     * The principal to grant permissions to.
     */
    readonly grantPrincipal: IPrincipal;
    /**
     * The ECS service that serves usage based licensing.
     */
    readonly service: Ec2Service;
    constructor(scope: Construct, id: string, props: UsageBasedLicensingProps);
    /**
     * This method grant access of worker fleet to ports that required
     *
     * @param workerFleet - worker fleet
     * @param licenses - UBL licenses
     */
    grantPortAccess(workerFleet: IConnectable, licenses: UsageBasedLicense[]): void;
    /**
     * The connections object that allows you to control network egress/ingress to the License Forwarder.
     */
    get connections(): import("aws-cdk-lib/aws-ec2").Connections;
}
