"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ThinkboxDockerImages = exports.AwsCustomerAgreementAndIpLicenseAcceptance = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto_1 = require("crypto");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
const _1 = require(".");
/**
 * The ThinkboxDockerImages will install Deadline onto one or more EC2 instances.
 *
 * By downloading or using the Deadline software, you agree to the AWS Customer Agreement (https://aws.amazon.com/agreement/)
 * and AWS Intellectual Property License (https://aws.amazon.com/legal/aws-ip-license-terms/). You acknowledge that Deadline
 * is AWS Content as defined in those Agreements.
 *
 * This enum is used to signify acceptance or rejection of these terms.
 */
var AwsCustomerAgreementAndIpLicenseAcceptance;
(function (AwsCustomerAgreementAndIpLicenseAcceptance) {
    /**
     * The user signifies their explicit rejection of the terms.
     */
    AwsCustomerAgreementAndIpLicenseAcceptance[AwsCustomerAgreementAndIpLicenseAcceptance["USER_REJECTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE"] = 0] = "USER_REJECTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE";
    /**
     * The user signifies their explicit acceptance of the terms.
     */
    AwsCustomerAgreementAndIpLicenseAcceptance[AwsCustomerAgreementAndIpLicenseAcceptance["USER_ACCEPTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE"] = 1] = "USER_ACCEPTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE";
})(AwsCustomerAgreementAndIpLicenseAcceptance = exports.AwsCustomerAgreementAndIpLicenseAcceptance || (exports.AwsCustomerAgreementAndIpLicenseAcceptance = {}));
/**
 * An API for interacting with publicly available Deadline container images published by AWS Thinkbox.
 *
 * This provides container images as required by RFDK's Deadline constructs such as
 *
 * * {@link @aws-rfdk/deadline#RenderQueue}
 * * {@link @aws-rfdk/deadline#UsageBasedLicensing}
 *
 * Successful usage of the published Deadline container images with this class requires:
 *
 * 1) Explicit acceptance of the terms of the AWS Thinkbox End User License Agreement, under which Deadline is
 *    distributed; and
 * 2) The lambda on which the custom resource looks up the Thinkbox container images is able to make HTTPS
 *    requests to the official AWS Thinbox download site: https://downloads.thinkboxsoftware.com
 *
 * Resources Deployed
 * ------------------------
 * - A Lambda function containing a script to look up the AWS Thinkbox container image registry
 *
 * Security Considerations
 * ------------------------
 * - CDK deploys the code for this lambda as an S3 object in the CDK bootstrap bucket. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly
 *   recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS
 *   CloudTrail on your account to assist in post-incident analysis of compromised production environments.
 *
 * For example, to construct a RenderQueue using the images:
 *
 * ```ts
 * import { App, Stack, Vpc } from '@aws-rfdk/core';
 * import { AwsCustomerAgreementAndIpLicenseAcceptance, RenderQueue, Repository, ThinkboxDockerImages, VersionQuery } from '@aws-rfdk/deadline';
 * const app = new App();
 * const stack = new Stack(app, 'Stack');
 * const vpc = new Vpc(stack, 'Vpc');
 * const version = new VersionQuery(stack, 'Version', {
 *   version: '10.1.12',
 * });
 * const images = new ThinkboxDockerImages(stack, 'Image', {
 *   version,
 *   // Change this to AwsCustomerAgreementAndIpLicenseAcceptance.USER_ACCEPTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE to accept the terms
 *   // of the AWS Customer Agreement and AWS Intellectual Property License.
 *   userAwsCustomerAgreementAndIpLicenseAcceptance: AwsCustomerAgreementAndIpLicenseAcceptance.USER_REJECTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE,
 * });
 * const repository = new Repository(stack, 'Repository', {
 *   vpc,
 *   version,
 * });
 *
 * const renderQueue = new RenderQueue(stack, 'RenderQueue', {
 *   images: images.forRenderQueue(),
 *   // ...
 * });
 * ```
 */
class ThinkboxDockerImages extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.userAwsCustomerAgreementAndIpLicenseAcceptance = props.userAwsCustomerAgreementAndIpLicenseAcceptance;
        this.version = props?.version;
        const lambdaCode = aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'));
        const lambdaFunc = new aws_lambda_1.SingletonFunction(this, 'VersionProviderFunction', {
            uuid: '08553416-1fc9-4be9-a818-609a31ae1b5b',
            description: 'Used by the ThinkboxDockerImages construct to look up the ECR repositories where AWS Thinkbox publishes Deadline container images.',
            code: lambdaCode,
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            handler: 'ecr-provider.handler',
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        const ecrProvider = new aws_cdk_lib_1.CustomResource(this, 'ThinkboxEcrProvider', {
            serviceToken: lambdaFunc.functionArn,
            properties: {
                // create a random string that will force the Lambda to "update" on each deployment. Changes to its output will
                // be propagated to any CloudFormation resource providers that reference the output ARN
                ForceRun: this.forceRun(),
            },
            resourceType: 'Custom::RFDK_EcrProvider',
        });
        this.node.defaultChild = ecrProvider;
        this.ecrBaseURI = ecrProvider.getAtt('EcrURIPrefix').toString();
        this.remoteConnectionServer = this.ecrImageForRecipe(_1.ThinkboxManagedDeadlineDockerRecipes.REMOTE_CONNECTION_SERVER);
        this.licenseForwarder = this.ecrImageForRecipe(_1.ThinkboxManagedDeadlineDockerRecipes.LICENSE_FORWARDER);
        const thisConstruct = this;
        this.node.addValidation({
            validate() {
                const errors = [];
                // Users must accept the AWS Customer Agreement and AWS Intellectual Property License to use the container images
                if (thisConstruct.userAwsCustomerAgreementAndIpLicenseAcceptance !==
                    AwsCustomerAgreementAndIpLicenseAcceptance.USER_ACCEPTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE) {
                    errors.push(ThinkboxDockerImages.AWS_CONTENT_NOTICE);
                }
                // Using the output of VersionQuery across stacks can cause issues. CloudFormation stack outputs cannot change if
                // a resource in another stack is referencing it.
                if (thisConstruct.version instanceof _1.VersionQuery) {
                    const versionStack = aws_cdk_lib_1.Stack.of(thisConstruct.version);
                    const thisStack = aws_cdk_lib_1.Stack.of(thisConstruct);
                    if (versionStack != thisStack) {
                        errors.push('A VersionQuery can not be supplied from a different stack');
                    }
                }
                return errors;
            },
        });
    }
    ecrImageForRecipe(recipe) {
        let registryName = `${this.ecrBaseURI}${recipe}`;
        if (this.versionString) {
            registryName += `:${this.versionString}`;
        }
        return aws_ecs_1.ContainerImage.fromRegistry(registryName);
    }
    /**
     * Returns container images for use with the {@link RenderQueue} construct
     */
    forRenderQueue() {
        return this;
    }
    /**
     * Returns container images for use with the {@link UsageBasedLicensing} construct
     */
    forUsageBasedLicensing() {
        return this;
    }
    /**
     * A string representation of the Deadline version to retrieve images for.
     *
     * This can be undefined - in which case the latest available version of Deadline is used.
     */
    get versionString() {
        function numAsString(num) {
            return aws_cdk_lib_1.Token.isUnresolved(num) ? aws_cdk_lib_1.Token.asString(num) : num.toString();
        }
        const version = this.version;
        if (version) {
            const major = numAsString(version.majorVersion);
            const minor = numAsString(version.minorVersion);
            const release = numAsString(version.releaseVersion);
            return `${major}.${minor}.${release}`;
        }
        return undefined;
    }
    forceRun() {
        return crypto_1.randomBytes(32).toString('base64').slice(0, 32);
    }
}
exports.ThinkboxDockerImages = ThinkboxDockerImages;
_a = JSII_RTTI_SYMBOL_1;
ThinkboxDockerImages[_a] = { fqn: "aws-rfdk.deadline.ThinkboxDockerImages", version: "1.0.0" };
/**
 * The Deadline licensing message that is presented to the user if they create an instance of
 * this class without explicitly accepting the AWS Content Agreement and AWS Intellectual Property License.
 *
 * Note to developers: The text of this string is a legal requirement, and must not be altered
 * without approval.
 */
ThinkboxDockerImages.AWS_CONTENT_NOTICE = `
The ThinkboxDockerImages will install Deadline onto one or more EC2 instances.

By downloading or using the Deadline software, you agree to the AWS Customer Agreement (https://aws.amazon.com/agreement/)
and AWS Intellectual Property License (https://aws.amazon.com/legal/aws-ip-license-terms/). You acknowledge that Deadline
is AWS Content as defined in those Agreements.

Please set the userAcceptsAwsCustomerAgreementAndIpLicense property to
USER_ACCEPTS_AWS_CUSTOMER_AGREEMENT_AND_IP_LICENSE to signify your acceptance of these terms.
`;
//# sourceMappingURL=data:application/json;base64,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