"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RenderQueue = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_ecs_patterns_1 = require("aws-cdk-lib/aws-ecs-patterns");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_route53_1 = require("aws-cdk-lib/aws-route53");
const constructs_1 = require("constructs");
const _1 = require(".");
const core_1 = require("../../core");
const deployment_instance_1 = require("../../core/lib/deployment-instance");
const runtime_info_1 = require("../../core/lib/runtime-info");
const rq_connection_1 = require("./rq-connection");
const secrets_management_1 = require("./secrets-management");
const version_1 = require("./version");
const wait_for_stable_service_1 = require("./wait-for-stable-service");
/**
 * Base class for Render Queue providers
 */
class RenderQueueBase extends constructs_1.Construct {
}
/**
 * The RenderQueue construct deploys an Elastic Container Service (ECS) service that serves Deadline's REST HTTP API
 * to Deadline Clients.
 *
 * Most Deadline clients will connect to a Deadline render farm via the the RenderQueue. The API provides Deadline
 * clients access to Deadline's database and repository file-system in a way that is secure, performant, and scalable.
 *
 * ![architecture diagram](/diagrams/deadline/RenderQueue.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An Amazon Elastic Container Service (ECS) cluster.
 * - An AWS EC2 auto-scaling group that provides the instances that host the ECS service.
 * - An ECS service with a task definition that deploys the Deadline Remote Connetion Server (RCS) in a container.
 * - A Amazon CloudWatch log group for streaming logs from the Deadline RCS.
 * - An application load balancer, listener and target group that balance incoming traffic among the RCS containers.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - Care must be taken to secure what can connect to the RenderQueue. The RenderQueue does not authenticate API
 *   requests made against it. You must limit access to the RenderQueue endpoint to only trusted hosts. Those hosts
 *   should be governed carefully, as malicious software could use the API to remotely execute code across the entire render farm.
 * - The RenderQueue can be deployed with network encryption through Transport Layer Security (TLS) or without it. Unencrypted
 *   network communications can be eavesdropped upon or modified in transit. We strongly recommend deploying the RenderQueue
 *   with TLS enabled in production environments and it is configured to be on by default.
 */
class RenderQueue extends RenderQueueBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        /**
         * Whether SEP policies have been added
         */
        this.haveAddedSEPPolicies = false;
        /**
         * Whether Resource Tracker policies have been added
         */
        this.haveAddedResourceTrackerPolicies = false;
        this.repository = props.repository;
        this.renderQueueSize = props?.renderQueueSize ?? { min: 1, max: 1 };
        if (props.version.isLessThan(RenderQueue.MINIMUM_LOAD_BALANCING_VERSION)) {
            // Deadline versions earlier than 10.1.10 do not support horizontal scaling behind a load-balancer, so we limit to at most one instance
            if ((this.renderQueueSize.min ?? 0) > 1) {
                throw new Error(`renderQueueSize.min for Deadline version less than ${RenderQueue.MINIMUM_LOAD_BALANCING_VERSION.toString()} cannot be greater than 1 - got ${this.renderQueueSize.min}`);
            }
            if ((this.renderQueueSize.desired ?? 0) > 1) {
                throw new Error(`renderQueueSize.desired for Deadline version less than ${RenderQueue.MINIMUM_LOAD_BALANCING_VERSION.toString()} cannot be greater than 1 - got ${this.renderQueueSize.desired}`);
            }
            if ((this.renderQueueSize.max ?? 0) > 1) {
                throw new Error(`renderQueueSize.max for Deadline version less than ${RenderQueue.MINIMUM_LOAD_BALANCING_VERSION.toString()} cannot be greater than 1 - got ${this.renderQueueSize.max}`);
            }
        }
        this.version = props?.version;
        const externalProtocol = props.trafficEncryption?.externalTLS?.enabled === false ? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
        let loadBalancerFQDN;
        let domainZone;
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            const tlsInfo = this.getOrCreateTlsInfo(props);
            this.certChain = tlsInfo.certChain;
            this.clientCert = tlsInfo.serverCert;
            loadBalancerFQDN = tlsInfo.fullyQualifiedDomainName;
            domainZone = tlsInfo.domainZone;
        }
        else {
            if (props.hostname) {
                loadBalancerFQDN = this.generateFullyQualifiedDomainName(props.hostname.zone, props.hostname.hostname);
                domainZone = props.hostname.zone;
            }
        }
        this.version = props.version;
        const internalProtocol = props.trafficEncryption?.internalProtocol ?? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
        this.cluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            vpc: props.vpc,
        });
        const minCapacity = props.renderQueueSize?.min ?? 1;
        if (minCapacity < 1) {
            throw new Error(`renderQueueSize.min capacity must be at least 1: got ${minCapacity}`);
        }
        const maxCapacity = this.renderQueueSize.max ?? this.renderQueueSize?.desired;
        if (this.renderQueueSize?.desired && maxCapacity && this.renderQueueSize?.desired > maxCapacity) {
            throw new Error(`renderQueueSize.desired capacity cannot be more than ${maxCapacity}: got ${this.renderQueueSize.desired}`);
        }
        this.asg = this.cluster.addCapacity('RCS Capacity', {
            vpcSubnets: props.vpcSubnets ?? RenderQueue.DEFAULT_VPC_SUBNETS_OTHER,
            instanceType: props.instanceType ?? new aws_ec2_1.InstanceType('c5.large'),
            minCapacity,
            desiredCapacity: this.renderQueueSize?.desired,
            maxCapacity,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    // See: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-ami-storage-config.html
                    // We want the volume to be encrypted. The default AMI size is 30-GiB.
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(30, { encrypted: true }),
                }],
            ...{ updateType: undefined },
            updatePolicy: aws_autoscaling_1.UpdatePolicy.rollingUpdate(),
            // addCapacity doesn't specifically take a securityGroup, but it passes on its properties to the ASG it creates,
            // so this security group will get applied there
            // @ts-ignore
            securityGroup: props.securityGroups?.backend,
        });
        this.backendConnections = this.asg.connections;
        /**
         * The ECS-optimized AMI that is defaulted to when adding capacity to a cluster does not include the awscli or unzip
         * packages as is the case with the standard Amazon Linux AMI. These are required by RFDK scripts to configure the
         * direct connection on the host container instances.
         */
        this.asg.userData.addCommands('yum install -yq awscli unzip');
        if (props.enableLocalFileCaching ?? false) {
            // Has to be done before any filesystems mount.
            this.enableFilecaching(this.asg);
        }
        const externalPortNumber = RenderQueue.RCS_PROTO_PORTS[externalProtocol];
        const internalPortNumber = RenderQueue.RCS_PROTO_PORTS[internalProtocol];
        this.logGroup = core_1.LogGroupFactory.createOrFetch(this, 'LogGroupWrapper', id, {
            logGroupPrefix: '/renderfarm/',
            ...props.logGroupProps,
        });
        this.logGroup.grantWrite(this.asg);
        if (props.repository.secretsManagementSettings.enabled) {
            const errors = [];
            if (props.version.isLessThan(version_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION)) {
                errors.push(`The supplied Deadline version (${props.version.versionString}) does not support Deadline Secrets Management in RFDK. Either upgrade Deadline to the minimum required version (${version_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION.versionString}) or disable the feature in the Repository's construct properties.`);
            }
            if (props.repository.secretsManagementSettings.credentials === undefined) {
                errors.push('The Repository does not have Secrets Management credentials');
            }
            if (internalProtocol !== aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
                errors.push('The internal protocol on the Render Queue is not HTTPS.');
            }
            if (externalProtocol !== aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
                errors.push('External TLS on the Render Queue is not enabled.');
            }
            if (errors.length > 0) {
                throw new Error(`Deadline Secrets Management is enabled on the supplied Repository but cannot be enabled on the Render Queue for the following reasons:\n${errors.join('\n')}`);
            }
        }
        const taskDefinition = this.createTaskDefinition({
            image: props.images.remoteConnectionServer,
            portNumber: internalPortNumber,
            protocol: internalProtocol,
            repository: props.repository,
            runAsUser: RenderQueue.RCS_USER,
            secretsManagementOptions: props.repository.secretsManagementSettings.enabled ? {
                credentials: props.repository.secretsManagementSettings.credentials,
                posixUsername: RenderQueue.RCS_USER.username,
            } : undefined,
        });
        this.taskDefinition = taskDefinition;
        // The fully-qualified domain name to use for the ALB
        const loadBalancer = new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, 'LB', {
            vpc: this.cluster.vpc,
            vpcSubnets: props.vpcSubnetsAlb ?? RenderQueue.DEFAULT_VPC_SUBNETS_ALB,
            internetFacing: false,
            deletionProtection: props.deletionProtection ?? true,
            securityGroup: props.securityGroups?.frontend,
        });
        this.pattern = new aws_ecs_patterns_1.ApplicationLoadBalancedEc2Service(this, 'AlbEc2ServicePattern', {
            certificate: this.clientCert,
            cluster: this.cluster,
            desiredCount: this.renderQueueSize?.desired ?? minCapacity,
            domainZone,
            domainName: loadBalancerFQDN,
            listenerPort: externalPortNumber,
            loadBalancer,
            protocol: externalProtocol,
            taskDefinition,
            // This is required to right-size our host capacity and not have the ECS service block on updates. We set a memory
            // reservation, but no memory limit on the container. This allows the container's memory usage to grow unbounded.
            // We want 1:1 container to container instances to not over-spend, but this comes at the price of down-time during
            // cloudformation updates.
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
            // This is required to ensure that the ALB listener's security group does not allow any ingress by default.
            openListener: false,
        });
        // An explicit dependency is required from the Service to the Client certificate
        // Otherwise cloud formation will try to remove the cert before the ALB using it is disposed.
        if (this.clientCert) {
            this.pattern.node.addDependency(this.clientCert);
        }
        // An explicit dependency is required from the service to the ASG providing its capacity.
        // See: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html
        this.pattern.service.node.addDependency(this.asg);
        this.loadBalancer = this.pattern.loadBalancer;
        // Enabling dropping of invalid HTTP header fields on the load balancer to prevent http smuggling attacks.
        this.loadBalancer.setAttribute('routing.http.drop_invalid_header_fields.enabled', 'true');
        if (props.accessLogs) {
            const accessLogsBucket = props.accessLogs.destinationBucket;
            // Policies are applied according to
            // https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-access-logs.html
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:PutObject'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [`${accessLogsBucket.bucketArn}/*`],
                conditions: {
                    StringEquals: {
                        's3:x-amz-acl': 'bucket-owner-full-control',
                    },
                },
            }));
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:GetBucketAcl'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [accessLogsBucket.bucketArn],
            }));
            this.loadBalancer.logAccessLogs(accessLogsBucket, props.accessLogs.prefix);
        }
        // Ensure tasks are run on separate container instances
        this.pattern.service.addPlacementConstraints(aws_ecs_1.PlacementConstraint.distinctInstances());
        /**
         * Uses an escape-hatch to set the target group protocol to HTTPS. We cannot configure server certificate
         * validation, but at least traffic is encrypted and terminated at the application layer.
         */
        const listener = this.loadBalancer.node.findChild('PublicListener');
        this.listener = listener;
        const targetGroup = listener.node.findChild('ECSGroup');
        const targetGroupResource = targetGroup.node.defaultChild;
        targetGroupResource.protocol = aws_elasticloadbalancingv2_1.ApplicationProtocol[internalProtocol];
        targetGroupResource.port = internalPortNumber;
        this.grantPrincipal = taskDefinition.taskRole;
        this.connections = new aws_ec2_1.Connections({
            defaultPort: aws_ec2_1.Port.tcp(externalPortNumber),
            securityGroups: this.pattern.loadBalancer.connections.securityGroups,
        });
        this.endpoint = new core_1.ConnectableApplicationEndpoint({
            address: loadBalancerFQDN ?? this.pattern.loadBalancer.loadBalancerDnsName,
            port: externalPortNumber,
            connections: this.connections,
            protocol: externalProtocol,
        });
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP) {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttp({
                endpoint: this.endpoint,
            });
        }
        else {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttps({
                endpoint: this.endpoint,
                caCert: this.certChain,
            });
        }
        props.repository.secretsManagementSettings.credentials?.grantRead(this);
        this.ecsServiceStabilized = new wait_for_stable_service_1.WaitForStableService(this, 'WaitForStableService', {
            service: this.pattern.service,
        });
        this.node.defaultChild = taskDefinition;
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
        const thisConstruct = this;
        this.node.addValidation({
            validate() {
                const validationErrors = [];
                // Using the output of VersionQuery across stacks can cause issues. CloudFormation stack outputs cannot change if
                // a resource in another stack is referencing it.
                if (thisConstruct.version instanceof _1.VersionQuery) {
                    const versionStack = aws_cdk_lib_1.Stack.of(thisConstruct.version);
                    const thisStack = aws_cdk_lib_1.Stack.of(thisConstruct);
                    if (versionStack != thisStack) {
                        validationErrors.push('A VersionQuery can not be supplied from a different stack');
                    }
                }
                return validationErrors;
            },
        });
    }
    /**
     * @inheritdoc
     */
    configureClientECS(param) {
        param.hosts.forEach(host => this.addChildDependency(host));
        return this.rqConnection.configureClientECS(param);
    }
    /**
     * @inheritdoc
     */
    configureClientInstance(param) {
        this.addChildDependency(param.host);
        this.rqConnection.configureClientInstance(param);
    }
    /**
     * @inheritdoc
     */
    configureSecretsManagementAutoRegistration(props) {
        if (!this.repository.secretsManagementSettings.enabled) {
            // Secrets management is not enabled, so do nothing
            return;
        }
        this.identityRegistrationSettings.addSubnetIdentityRegistrationSetting(props);
    }
    /**
     * Adds AWS Managed Policies to the Render Queue so it is able to control Deadline's Spot Event Plugin.
     *
     * See: https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html for additonal information.
     *
     * @param includeResourceTracker Whether or not the Resource tracker admin policy should also be added (Default: True)
     */
    addSEPPolicies(includeResourceTracker = true) {
        if (!this.haveAddedSEPPolicies) {
            const sepPolicy = aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginAdminPolicy');
            this.taskDefinition.taskRole.addManagedPolicy(sepPolicy);
            this.haveAddedSEPPolicies = true;
        }
        if (!this.haveAddedResourceTrackerPolicies) {
            if (includeResourceTracker) {
                const rtPolicy = aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy');
                this.taskDefinition.taskRole.addManagedPolicy(rtPolicy);
                this.haveAddedResourceTrackerPolicies = true;
            }
        }
    }
    /**
     * Add an ordering dependency to another Construct.
     *
     * All constructs in the child's scope will be deployed after the RenderQueue has been deployed and is ready to recieve traffic.
     *
     * This can be used to ensure that the RenderQueue is fully up and serving queries before a client attempts to connect to it.
     *
     * @param child The child to make dependent upon this RenderQueue.
     */
    addChildDependency(child) {
        // Narrowly define the dependencies to reduce the probability of cycles
        // ex: cycles that involve the security group of the RenderQueue & child.
        child.node.addDependency(this.listener);
        child.node.addDependency(this.taskDefinition);
        child.node.addDependency(this.pattern.service);
        child.node.addDependency(this.ecsServiceStabilized);
    }
    /**
     * Adds security groups to the frontend of the Render Queue, which is its load balancer.
     * @param securityGroups The security groups to add.
     */
    addFrontendSecurityGroups(...securityGroups) {
        securityGroups.forEach(securityGroup => this.loadBalancer.addSecurityGroup(securityGroup));
    }
    /**
     * Adds security groups to the backend components of the Render Queue, which consists of the AutoScalingGroup for the Deadline RCS.
     * @param securityGroups The security groups to add.
     */
    addBackendSecurityGroups(...securityGroups) {
        securityGroups.forEach(securityGroup => this.asg.addSecurityGroup(securityGroup));
    }
    enableFilecaching(asg) {
        const script = core_1.ScriptAsset.fromPathConvention(this, 'FilecachingScript', {
            osType: asg.osType,
            baseName: 'enableCacheFilesd',
            rootDir: path_1.join(__dirname, '..', 'scripts'),
        });
        // A comment in userData to make this easier to test.
        asg.userData.addCommands('# RenderQueue file caching enabled');
        script.executeOn({
            host: asg,
        });
    }
    createTaskDefinition(props) {
        const { image, portNumber, protocol, repository } = props;
        const taskDefinition = new aws_ecs_1.Ec2TaskDefinition(this, 'RCSTask');
        // Mount the repo filesystem to RenderQueue.HOST_REPO_FS_MOUNT_PATH
        const connection = repository.configureClientECS({
            containerInstances: {
                hosts: [this.asg],
            },
            containers: {
                taskDefinition,
            },
        });
        const environment = connection.containerEnvironment;
        if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            // Generate a self-signed X509 certificate, private key and passphrase for use by the RCS containers.
            // Note: the Application Load Balancer does not validate the certificate in any way.
            const rcsCertPem = new core_1.X509CertificatePem(this, 'TlsCaCertPem', {
                subject: {
                    cn: 'renderfarm.local',
                },
            });
            const rcsCertPkcs = new core_1.X509CertificatePkcs12(this, 'TlsRcsCertBundle', {
                sourceCertificate: rcsCertPem,
            });
            [rcsCertPem.cert, rcsCertPkcs.cert, rcsCertPkcs.passphrase].forEach(secret => {
                secret.grantRead(taskDefinition.taskRole);
            });
            environment.RCS_TLS_CA_CERT_URI = rcsCertPem.cert.secretArn;
            environment.RCS_TLS_CERT_URI = rcsCertPkcs.cert.secretArn;
            environment.RCS_TLS_CERT_PASSPHRASE_URI = rcsCertPkcs.passphrase.secretArn;
            environment.RCS_TLS_REQUIRE_CLIENT_CERT = 'no';
        }
        if (props.secretsManagementOptions !== undefined) {
            environment.RCS_SM_CREDENTIALS_URI = props.secretsManagementOptions.credentials.secretArn;
        }
        // We can ignore this in test coverage because we always use RenderQueue.RCS_USER
        /* istanbul ignore next */
        const user = props.runAsUser ? `${props.runAsUser.uid}:${props.runAsUser.gid}` : undefined;
        const containerDefinition = taskDefinition.addContainer('ContainerDefinition', {
            image,
            memoryReservationMiB: 2048,
            environment,
            logging: aws_ecs_1.LogDriver.awsLogs({
                logGroup: this.logGroup,
                streamPrefix: 'RCS',
            }),
            user,
        });
        containerDefinition.addMountPoints(connection.readWriteMountPoint);
        if (props.secretsManagementOptions !== undefined) {
            // Create volume to persist the RSA keypairs generated by Deadline between ECS tasks
            // This makes it so subsequent ECS tasks use the same initial Secrets Management identity
            const volumeName = 'deadline-user-keypairs';
            taskDefinition.addVolume({
                name: volumeName,
                dockerVolumeConfiguration: {
                    scope: aws_ecs_1.Scope.SHARED,
                    autoprovision: true,
                    driver: 'local',
                },
            });
            // Mount the volume into the container at the location where Deadline expects it
            containerDefinition.addMountPoints({
                readOnly: false,
                sourceVolume: volumeName,
                containerPath: `/home/${props.secretsManagementOptions.posixUsername}/.config/.mono/keypairs`,
            });
        }
        // Increase ulimits
        containerDefinition.addUlimits({
            name: aws_ecs_1.UlimitName.NOFILE,
            softLimit: 200000,
            hardLimit: 200000,
        }, {
            name: aws_ecs_1.UlimitName.NPROC,
            softLimit: 64000,
            hardLimit: 64000,
        });
        containerDefinition.addPortMappings({
            containerPort: portNumber,
            hostPort: portNumber,
        });
        return taskDefinition;
    }
    /**
     * Checks if the user supplied any certificate to use for TLS and uses them, or creates defaults to use.
     * @param props
     * @returns TlsInfo either based on input to the render queue, or the created defaults
     */
    getOrCreateTlsInfo(props) {
        if ((props.trafficEncryption?.externalTLS?.acmCertificate !== undefined) ||
            (props.trafficEncryption?.externalTLS?.rfdkCertificate !== undefined)) {
            if (props.hostname === undefined) {
                throw new Error('The hostname for the render queue must be defined if supplying your own certificates.');
            }
            return this.getTlsInfoFromUserProps(props.trafficEncryption.externalTLS, props.hostname);
        }
        return this.createDefaultTlsInfo(props.vpc, props.hostname);
    }
    /**
     * Creates a default certificate to use for TLS and a PrivateHostedZone to put the load balancer in.
     * @param vpc
     * @param hostname
     * @returns default TlsInfo
     */
    createDefaultTlsInfo(vpc, hostname) {
        const domainZone = hostname?.zone ?? new aws_route53_1.PrivateHostedZone(this, 'DnsZone', {
            vpc: vpc,
            zoneName: RenderQueue.DEFAULT_DOMAIN_NAME,
        });
        const fullyQualifiedDomainName = this.generateFullyQualifiedDomainName(domainZone, hostname?.hostname);
        const rootCa = new core_1.X509CertificatePem(this, 'RootCA', {
            subject: {
                cn: 'RenderQueueRootCA',
            },
        });
        const rfdkCert = new core_1.X509CertificatePem(this, 'RenderQueuePemCert', {
            subject: {
                cn: fullyQualifiedDomainName,
            },
            signingCertificate: rootCa,
        });
        const serverCert = new core_1.ImportedAcmCertificate(this, 'AcmCert', rfdkCert);
        const certChain = rfdkCert.certChain;
        return {
            domainZone,
            fullyQualifiedDomainName,
            serverCert,
            certChain,
        };
    }
    /**
     * Gets the certificate and PrivateHostedZone provided in the Render Queue's construct props.
     * @param externalTLS
     * @param hostname
     * @returns The provided certificate and domain info
     */
    getTlsInfoFromUserProps(externalTLS, hostname) {
        let serverCert;
        let certChain;
        if ((externalTLS.acmCertificate !== undefined) &&
            (externalTLS.rfdkCertificate !== undefined)) {
            throw new Error('Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS.');
        }
        if (!hostname.hostname) {
            throw new Error('A hostname must be supplied if a certificate is supplied, '
                + 'with the common name of the certificate matching the hostname + domain name.');
        }
        const fullyQualifiedDomainName = this.generateFullyQualifiedDomainName(hostname.zone, hostname.hostname);
        if (externalTLS.acmCertificate) {
            if (externalTLS.acmCertificateChain === undefined) {
                throw new Error('externalTLS.acmCertificateChain must be provided when using externalTLS.acmCertificate.');
            }
            serverCert = externalTLS.acmCertificate;
            certChain = externalTLS.acmCertificateChain;
        }
        else { // Using externalTLS.rfdkCertificate
            if (externalTLS.rfdkCertificate.certChain === undefined) {
                throw new Error('Provided rfdkCertificate does not contain a certificate chain.');
            }
            serverCert = new core_1.ImportedAcmCertificate(this, 'AcmCert', externalTLS.rfdkCertificate);
            certChain = externalTLS.rfdkCertificate.certChain;
        }
        return {
            domainZone: hostname.zone,
            fullyQualifiedDomainName,
            serverCert,
            certChain,
        };
    }
    /**
     * Helper method to create the fully qualified domain name for the given hostname and PrivateHostedZone.
     * @param hostname
     * @param zone
     * @returns The fully qualified domain name
     */
    generateFullyQualifiedDomainName(zone, hostname = RenderQueue.DEFAULT_HOSTNAME) {
        if (!RenderQueue.RE_VALID_HOSTNAME.test(hostname)) {
            throw new Error(`Invalid RenderQueue hostname: ${hostname}`);
        }
        return `${hostname}.${zone.zoneName}`;
    }
    /**
     * The instance that runs commands during the deployment.
     */
    get deploymentInstance() {
        const CONFIGURE_REPOSITORY_CONSTRUCT_ID = 'ConfigureRepository';
        const deploymentInstanceNode = this.node.tryFindChild(CONFIGURE_REPOSITORY_CONSTRUCT_ID);
        if (deploymentInstanceNode === undefined) {
            return new deployment_instance_1.DeploymentInstance(this, CONFIGURE_REPOSITORY_CONSTRUCT_ID, {
                securityGroup: this.backendConnections.securityGroups[0],
                vpc: this.props.vpc,
                vpcSubnets: this.props.vpcSubnets ?? RenderQueue.DEFAULT_VPC_SUBNETS_OTHER,
                logGroupProps: this.props.logGroupProps,
            });
        }
        else if (deploymentInstanceNode instanceof deployment_instance_1.DeploymentInstance) {
            return deploymentInstanceNode;
        }
        else {
            throw new Error(`Unexpected type for ${deploymentInstanceNode.node.path}. Expected ${deployment_instance_1.DeploymentInstance.name}, but found ${typeof (deploymentInstanceNode)}.`);
        }
    }
    /**
     * The construct that manages Deadline Secrets Management identity registration settings
     */
    get identityRegistrationSettings() {
        const IDENTITY_REGISTRATION_CONSTRUCT_ID = 'SecretsManagementIdentityRegistration';
        const secretsManagementIdentityRegistration = this.node.tryFindChild(IDENTITY_REGISTRATION_CONSTRUCT_ID);
        if (!secretsManagementIdentityRegistration) {
            return new secrets_management_1.SecretsManagementIdentityRegistration(this, IDENTITY_REGISTRATION_CONSTRUCT_ID, {
                deploymentInstance: this.deploymentInstance,
                repository: this.repository,
                renderQueueSubnets: this.props.vpc.selectSubnets(this.props.vpcSubnetsAlb ?? RenderQueue.DEFAULT_VPC_SUBNETS_ALB),
                version: this.props.version,
            });
        }
        else if (secretsManagementIdentityRegistration instanceof secrets_management_1.SecretsManagementIdentityRegistration) {
            return secretsManagementIdentityRegistration;
        }
        else {
            throw new Error(`Unexpected type for ${secretsManagementIdentityRegistration.node.path}. Expected ${secrets_management_1.SecretsManagementIdentityRegistration.name}, but found ${typeof (secretsManagementIdentityRegistration)}.`);
        }
    }
}
exports.RenderQueue = RenderQueue;
_a = JSII_RTTI_SYMBOL_1;
RenderQueue[_a] = { fqn: "aws-rfdk.deadline.RenderQueue", version: "1.0.0" };
/**
 * Container listening ports for each protocol.
 */
RenderQueue.RCS_PROTO_PORTS = {
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP]: 8080,
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS]: 4433,
};
RenderQueue.DEFAULT_HOSTNAME = 'renderqueue';
RenderQueue.DEFAULT_DOMAIN_NAME = 'aws-rfdk.com';
RenderQueue.DEFAULT_VPC_SUBNETS_ALB = { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT, onePerAz: true };
RenderQueue.DEFAULT_VPC_SUBNETS_OTHER = { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT };
/**
* The minimum Deadline version required for the Remote Connection Server to support load-balancing
*/
RenderQueue.MINIMUM_LOAD_BALANCING_VERSION = new version_1.Version([10, 1, 10, 0]);
/**
 * Regular expression that validates a hostname (portion in front of the subdomain).
 */
RenderQueue.RE_VALID_HOSTNAME = /^[a-z](?:[a-z0-9-]{0,61}[a-z0-9])?$/i;
/**
 * Information for the RCS user.
 */
RenderQueue.RCS_USER = { uid: 1000, gid: 1000, username: 'ec2-user' };
//# sourceMappingURL=data:application/json;base64,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