"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseConnection = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_docdb_1 = require("aws-cdk-lib/aws-docdb");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const core_1 = require("../../core");
/**
 * Helper class for connecting Thinkbox's Deadline to a specific Database.
 */
class DatabaseConnection {
    /**
     * Creates a DatabaseConnection which allows Deadline to connect to Amazon DocumentDB.
     * Note: Deadline officially supports only databases that are compatible with MongoDB 3.6.
     *
     * Resources Deployed
     * ------------------------
     * This construct does not deploy any resources
     */
    static forDocDB(options) {
        return new DocDBDatabaseConnection(options);
    }
    /**
     * Creates a DatabaseConnection which allows Deadline to connect to MongoDB.
     * Note: Deadline officially supports only databases that are compatible with MongoDB 3.6.
     *
     * Resources Deployed
     * ------------------------
     * This construct does not deploy any resources
     */
    static forMongoDbInstance(options) {
        return new MongoDbInstanceDatabaseConnection(options);
    }
}
exports.DatabaseConnection = DatabaseConnection;
_a = JSII_RTTI_SYMBOL_1;
DatabaseConnection[_a] = { fqn: "aws-rfdk.deadline.DatabaseConnection", version: "1.0.0" };
/**
 * Specialization of {@link DatabaseConnection} targetting Amazon DocumentDB.
 */
class DocDBDatabaseConnection extends DatabaseConnection {
    constructor(props) {
        super();
        this.props = props;
        if (!this.isCompatibleDocDBVersion()) {
            aws_cdk_lib_1.Annotations.of(props.database).addError('engineVersion must be 3.6.0 to be compatible with Deadline');
        }
        this.containerEnvironment = {
            // The container must fetch the credentials from Secrets Manager
            DB_CREDENTIALS_URI: this.props.login.secretArn,
        };
        if (props.database instanceof aws_docdb_1.DatabaseCluster) {
            this.databaseConstruct = props.database;
        }
    }
    /**
     * @inheritdoc
     */
    addInstallerDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Can only install Deadline from a Linux instance.');
        }
        host.userData.addCommands('configure_database_installation_args(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export SECRET_STRING=\`aws secretsmanager get-secret-value --secret-id ${this.props.login.secretArn} --region ${aws_cdk_lib_1.Stack.of(this.props.login).region} | getJsonVal 'SecretString'\``, "DB_USERNAME=`printenv SECRET_STRING | getJsonVal 'username'`", "DB_PASSWORD=`printenv SECRET_STRING | getJsonVal 'password'`", 'unset SECRET_STRING', `INSTALLER_DB_ARGS=( ["--dbuser"]=$DB_USERNAME ["--dbpassword"]=$DB_PASSWORD ["--dbhost"]=${this.props.database.clusterEndpoint.hostname}` +
            ` ["--dbport"]=${this.props.database.clusterEndpoint.portAsString()} ["--dbtype"]=DocumentDB )`, 'unset DB_USERNAME', 'unset DB_PASSWORD', 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_database_installation_args');
    }
    /**
     * @inheritdoc
     */
    addConnectionDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Connecting to the Deadline Database is currently only supported for Linux.');
        }
        host.userData.addCommands('configure_deadline_database(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export SECRET_STRING=\`aws secretsmanager get-secret-value --secret-id ${this.props.login.secretArn} --region ${aws_cdk_lib_1.Stack.of(this.props.login).region} | getJsonVal 'SecretString'\``, "DB_USERNAME=`printenv SECRET_STRING | getJsonVal 'username'`", "DB_PASSWORD=`printenv SECRET_STRING | getJsonVal 'password'`", 'unset SECRET_STRING', 'sudo -u ec2-user "${deadlinecommand}" -StoreDatabasecredentials "${DB_USERNAME}" "${DB_PASSWORD}"', 'unset DB_USERNAME', 'unset DB_PASSWORD', 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_deadline_database');
    }
    /**
     * @inheritdoc
     */
    allowConnectionsFrom(other) {
        other.connections.allowTo(this.props.database, this.props.database.connections.defaultPort);
    }
    /**
     * @inheritdoc
     */
    grantRead(grantee) {
        this.props.login.grantRead(grantee);
    }
    /**
     * @inheritdoc
     */
    addChildDependency(child) {
        // To depend on document DB it is not sufficient to depend on the Cluster. The instances are what serves data, so
        // we must add a dependency to an instance in the DocDB cluster.
        // The DocDB L2 does not expose any of its instances as properties, so we have to escape-hatch to gain access.
        const docdbInstance = this.props.database.node.tryFindChild('Instance1');
        // We won't find an instance in two situations:
        //  1) The DocDB Cluster was created from attributes. In this case, the DocDB pre-exists the stack and there's no need
        //     to depend on anything.
        //  2) The DocDB Cluster was constructed, but the internal name for the instance has been changed from 'Instance1'; this is
        //     unlikely, but not impossible.
        // We can differentiate cases (1) & (2) by looking for the defaultChild on the cluster. The version from attributes will not have one.
        if (docdbInstance) {
            child.node.addDependency(docdbInstance);
        }
        else if (this.props.database.node.defaultChild) {
            throw new Error('The internal implementation of the AWS CDK\'s DocumentDB cluster construct may have changed. Please update to a newer RFDK for an updated implementation, or file a ticket if this is the latest release.');
        }
    }
    /**
     * @inheritdoc
     */
    addSecurityGroup(...securityGroups) {
        let added = false;
        const errorReasons = [];
        if (this.props.database instanceof aws_docdb_1.DatabaseCluster) {
            const resource = this.props.database.node.tryFindChild('Resource');
            // TODO: Replace this code with the addSecurityGroup method of DatabaseCluster once this PR is merged:
            // https://github.com/aws/aws-cdk/pull/13290
            if (resource instanceof aws_docdb_1.CfnDBCluster) {
                const cfnCluster = resource;
                const securityGroupIds = securityGroups.map(sg => sg.securityGroupId);
                if (cfnCluster.vpcSecurityGroupIds === undefined) {
                    cfnCluster.vpcSecurityGroupIds = securityGroupIds;
                }
                else {
                    cfnCluster.vpcSecurityGroupIds.push(...securityGroupIds);
                }
                added = true;
            }
            else {
                errorReasons.push('The internal implementation of AWS CDK\'s DocumentDB cluster construct has changed.');
            }
        }
        else {
            errorReasons.push('The "database" property passed to this class is not an instance of AWS CDK\'s DocumentDB cluster construct.');
        }
        if (!added) {
            aws_cdk_lib_1.Annotations.of(this.props.database).addWarning(`Failed to add the following security groups to ${this.props.database.node.id}: ${securityGroups.map(sg => sg.node.id).join(', ')}. ` +
                errorReasons.join(' '));
        }
    }
    /**
     * Deadline is only compatible with MongoDB 3.6. This function attempts to determine whether
     * the given DocDB version is compatible.
     */
    isCompatibleDocDBVersion() {
        // The defaultChild of a DocDB DatabaseCluster is a CfnDBCluster, but we only have this
        // child if the customer didn't import from attributes. We can check the DB version by
        // checking the value of the engineVersion property of that object.
        if (this.props.database.node.defaultChild) {
            const cluster = this.props.database.node.defaultChild;
            return cluster.engineVersion?.startsWith('3.6') ?? false;
        }
        return true; // No information, assume it's compatible.
    }
}
/**
 * Specialization of {@link DatabaseConnection} targetting MongoDB.
 */
class MongoDbInstanceDatabaseConnection extends DatabaseConnection {
    constructor(props) {
        super();
        this.props = props;
        this.containerEnvironment = {
            DB_TLS_CLIENT_CERT_URI: props.clientCertificate.cert.secretArn,
            DB_TLS_CLIENT_CERT_PASSWORD_URI: props.clientCertificate.passphrase.secretArn,
        };
        if (props.database instanceof core_1.MongoDbInstance) {
            if (props.database.mongoDataVolume instanceof aws_ec2_1.Volume) {
                this.databaseConstruct = props.database.mongoDataVolume;
            }
        }
    }
    /**
     * @inheritdoc
     */
    allowConnectionsFrom(other) {
        other.connections.allowTo(this.props.database, aws_ec2_1.Port.tcp(this.props.database.port));
    }
    /**
     * @inheritdoc
     */
    addInstallerDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Can only install Deadline from a Linux instance.');
        }
        this.downloadCertificate(host);
        const certPwSecret = this.props.clientCertificate.passphrase;
        host.userData.addCommands('configure_database_installation_args(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 
        // Suppress -x, so no secrets go to the logs
        'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `CERT_PASSWORD=$(aws secretsmanager get-secret-value --secret-id ${certPwSecret.secretArn} --region ${aws_cdk_lib_1.Stack.of(certPwSecret).region} | getJsonVal 'SecretString')`, 'INSTALLER_DB_ARGS=( ["--dbssl"]=true ["--dbauth"]=true ["--dbsslauth"]=true ' +
            `["--dbhost"]="${this.props.database.fullHostname}" ["--dbport"]=${this.props.database.port} ` +
            `["--dbclientcert"]="${MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION}" ["--dbcertpass"]=$CERT_PASSWORD )`, 'unset CERT_PASSWORD', 
        // Restore -x, if it was set.
        'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_database_installation_args');
    }
    /**
     * @inheritdoc
     */
    addConnectionDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Connecting to the Deadline Database is currently only supported for Linux.');
        }
        this.downloadCertificate(host);
        const certPwSecret = this.props.clientCertificate.passphrase;
        host.userData.addCommands('configure_deadline_database(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export DB_CERT_FILE="${MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION}"`, `export DB_CERT_PASSWORD=$(aws secretsmanager get-secret-value --secret-id ${certPwSecret.secretArn} --region ${aws_cdk_lib_1.Stack.of(certPwSecret).region} | getJsonVal 'SecretString')`, 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_deadline_database');
    }
    /**
     * @inheritdoc
     */
    grantRead(grantee) {
        this.props.clientCertificate.cert.grantRead(grantee);
        this.props.clientCertificate.passphrase.grantRead(grantee);
    }
    /**
     * @inheritdoc
     */
    addChildDependency(child) {
        if (this.props.database.hasOwnProperty('server')) {
            const db = this.props.database;
            child.node.addDependency(db.server.autoscalingGroup.node.defaultChild);
        }
    }
    /**
     * @inheritdoc
     */
    addSecurityGroup(...securityGroups) {
        this.props.database.addSecurityGroup(...securityGroups);
    }
    /**
     * Download the client PKCS#12 certificate for authenticating to the MongoDB, and place it into
     * the path defined by: DB_CERT_LOCATION
     * @param host
     */
    downloadCertificate(host) {
        const stack = aws_cdk_lib_1.Stack.of(host);
        const uuid = 'e8125dd2-ab2c-4861-8ee4-998c26b30ee0';
        const uniqueId = 'GetSecretToFile' + host.osType + uuid.replace(/[-]/g, '');
        const getSecretsScript = stack.node.tryFindChild(uniqueId) ??
            core_1.ScriptAsset.fromPathConvention(stack, uniqueId, {
                osType: host.osType,
                baseName: 'getSecretToFile',
                rootDir: path.join(__dirname, '..', 'scripts'),
            });
        getSecretsScript.executeOn({
            host,
            args: [
                this.props.clientCertificate.cert.secretArn,
                MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION,
            ],
        });
    }
}
MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION = '/opt/Thinkbox/certs/mongo_client.pfx';
//# sourceMappingURL=data:application/json;base64,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