"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const x509_certificate_1 = require("../lib/x509-certificate");
test('Generate cert', () => {
    const stack = new aws_cdk_lib_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
    });
    // Expect the custom resource for cert generation
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'AWS',
            OU: 'Thinkbox',
        },
    });
    // Cannot have a CertificateValidFor property if not given one. Adding one
    // would cause existing certificates to be re-generated on re-deploy, and thus
    // risk breaking customer's setups.
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_X509Generator', assertions_1.Match.not({
        CertificateValidFor: assertions_1.Match.anyValue(),
    }));
    // Expect the resource for converting to PKCS #12 not to be created
    assertions_1.Template.fromStack(stack).resourceCountIs('Custom::RFDK_X509_PKCS12', 0);
    // Expect the DynamoDB table used for custom resource tracking
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::DynamoDB::Table', 1);
    // Expect a Secret used to store the cert passphrase
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::SecretsManager::Secret', 1);
    // Expect a policy that can interact with DynamoDB and SecretsManager
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assertions_1.Match.arrayWith([
                assertions_1.Match.objectLike({
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:ConditionCheckItem',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                        'dynamodb:DescribeTable',
                    ],
                }),
                assertions_1.Match.objectLike({
                    Action: 'dynamodb:DescribeTable',
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: assertions_1.Match.stringLikeRegexp('^CertPassphrase.*'),
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-F53F5427': 'f53f5427b2e9eb4739661fcc0b249b6e',
                        },
                    },
                }),
            ]),
        },
    });
    // Expect no KMS key for encryption
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::KMS::Key', 0);
    // Expect Lambda for doing the cert generation to use the generate() handler and openssl layer
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
        Handler: 'x509-certificate.generate',
        Layers: assertions_1.Match.arrayWith([
            assertions_1.Match.stringLikeRegexp('^arn:aws:lambda:us-west-2:224375009292:layer:openssl-al2:.*'),
        ]),
        Environment: {
            Variables: {
                DATABASE: {
                    Ref: assertions_1.Match.stringLikeRegexp('^CertTable.*'),
                },
            },
        },
    });
    // Should not be any errors.
    assertions_1.Annotations.fromStack(stack).hasNoInfo(`/${cert.node.path}`, assertions_1.Match.anyValue());
    assertions_1.Annotations.fromStack(stack).hasNoWarning(`/${cert.node.path}`, assertions_1.Match.anyValue());
    assertions_1.Annotations.fromStack(stack).hasNoError(`/${cert.node.path}`, assertions_1.Match.anyValue());
});
test('Generate cert, all options set', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const subject = {
        cn: 'testCN',
        o: 'testO',
        ou: 'testOu',
    };
    const encryptionKey = new aws_kms_1.Key(stack, 'Key');
    const signingCertificate = new x509_certificate_1.X509CertificatePem(stack, 'SigningCert', { subject });
    new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
        encryptionKey,
        signingCertificate,
        validFor: 3000,
    });
    // Expect the custom resource for cert generation
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'testO',
            OU: 'testOu',
        },
        SigningCertificate: {
            Cert: {
                'Fn::GetAtt': [
                    'SigningCert',
                    'Cert',
                ],
            },
            Key: {
                'Fn::GetAtt': [
                    'SigningCert',
                    'Key',
                ],
            },
            Passphrase: {
                Ref: assertions_1.Match.stringLikeRegexp('^SigningCertPassphrase.*'),
            },
            CertChain: '',
        },
        CertificateValidFor: '3000',
    });
    // Expect the resource for converting to PKCS #12 not to be created
    assertions_1.Template.fromStack(stack).resourceCountIs('Custom::RFDK_X509_PKCS12', 0);
    // Expect a policy that can interact with DynamoDB and SecretsManager for the signing cert's custom resource
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assertions_1.Match.arrayWith([
                assertions_1.Match.objectLike({
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:ConditionCheckItem',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                        'dynamodb:DescribeTable',
                    ],
                }),
                assertions_1.Match.objectLike({
                    Action: 'dynamodb:DescribeTable',
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'kms:Encrypt',
                        'kms:ReEncrypt*',
                        'kms:GenerateDataKey*',
                    ],
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: assertions_1.Match.stringLikeRegexp('^CertPassphrase.*'),
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-B2B09A60': 'b2b09a6086e87fe14005f4e0b800e4f0',
                        },
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        'Fn::GetAtt': [
                            'SigningCert',
                            'Cert',
                        ],
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        'Fn::GetAtt': [
                            'SigningCert',
                            'Key',
                        ],
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: 'SigningCertPassphrase42F0BC4F',
                    },
                }),
            ]),
        },
    });
    // Expect a policy that can interact with DynamoDB and SecretsManager for the cert's custom resource
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: assertions_1.Match.objectLike({
            Statement: assertions_1.Match.arrayWith([
                assertions_1.Match.objectLike({
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:ConditionCheckItem',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                        'dynamodb:DescribeTable',
                    ],
                }),
                assertions_1.Match.objectLike({
                    Action: 'dynamodb:DescribeTable',
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: assertions_1.Match.stringLikeRegexp('^SigningCertPassphrase.*'),
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-BA0FA489': 'ba0fa4898b2088c5b25f15075f605300',
                        },
                    },
                }),
            ]),
        }),
    });
    // Expect Lambda for doing the cert generation to use the generate() handler
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
        Handler: 'x509-certificate.generate',
    });
});
test('Grant cert read', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const grantable = new aws_iam_1.Role(stack, 'TestRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
    });
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
    });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    cert.grantCertRead(grantable);
    // Expect the custom resource to be created
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'AWS',
            OU: 'Thinkbox',
        },
    });
    // Expect the grantCertRead() to add this policy
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                },
            ],
        },
    });
    // Expect the grantCertRead() not to add this full read policy
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', assertions_1.Match.not({
        PolicyDocument: {
            Statement: assertions_1.Match.arrayWith([
                assertions_1.Match.objectLike({
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Key',
                        ],
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        Ref: certPassphraseID,
                    },
                }),
            ]),
        },
    }));
    // Expect the PKCS #12 generator not to be created
    assertions_1.Template.fromStack(stack).resourceCountIs('Custom::RFDK_X509_PKCS12', 0);
});
test('Grant full read', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const grantable = new aws_iam_1.Role(stack, 'TestRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
    });
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
    });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    cert.grantFullRead(grantable);
    // Expect the custom resource to be created
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'AWS',
            OU: 'Thinkbox',
        },
    });
    // Expect the grantFullRead() to add this policy
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', assertions_1.Match.not({
        PolicyDocument: {
            Statement: assertions_1.Match.arrayWith([
                assertions_1.Match.objectLike({
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Key',
                        ],
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        Ref: certPassphraseID,
                    },
                }),
            ]),
        },
    }));
    // Expect the PKCS #12 generator not to be created
    assertions_1.Template.fromStack(stack).resourceCountIs('Custom::RFDK_X509_PKCS12', 0);
});
test('Validating expiry', () => {
    // GIVEN
    const stack = new aws_cdk_lib_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
    const subject = { cn: 'testCN' };
    // WHEN
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
        validFor: 0,
    });
    // THEN
    assertions_1.Annotations.fromStack(stack).hasError(`/${cert.node.path}`, 'Certificates must be valid for at least one day.');
});
test('Validating expiry with token', () => {
    // GIVEN
    const stack = new aws_cdk_lib_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
    const subject = { cn: 'testCN' };
    // WHEN
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
        validFor: aws_cdk_lib_1.Lazy.number({
            produce() {
                return 0;
            },
        }),
    });
    // THEN
    assertions_1.Annotations.fromStack(stack).hasNoInfo(`/${cert.node.path}`, assertions_1.Match.anyValue());
    assertions_1.Annotations.fromStack(stack).hasNoWarning(`/${cert.node.path}`, assertions_1.Match.anyValue());
    assertions_1.Annotations.fromStack(stack).hasNoError(`/${cert.node.path}`, assertions_1.Match.anyValue());
});
test('Convert to PKCS #12', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', { subject });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    const pkcs12Cert = new x509_certificate_1.X509CertificatePkcs12(stack, 'CertPkcs12', { sourceCertificate: cert });
    const pkcs12CertPassphraseID = stack.getLogicalId(pkcs12Cert.passphrase.node.defaultChild);
    // Expect the PKCS #12 custom resource
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_X509_PKCS12', {
        Passphrase: {
            Ref: 'CertPkcs12Passphrase1E3DF360',
        },
        Secret: {
            NamePrefix: 'Default/CertPkcs12',
            Description: 'Default/CertPkcs12',
            Tags: [
                {
                    Key: 'X509SecretGrant-71090F78',
                    Value: '71090f7809ce64f7c970cb645d4d473c',
                },
            ],
        },
        Certificate: {
            Cert: {
                'Fn::GetAtt': [
                    'Cert',
                    'Cert',
                ],
            },
            Key: {
                'Fn::GetAtt': [
                    'Cert',
                    'Key',
                ],
            },
            Passphrase: {
                Ref: certPassphraseID,
            },
        },
    });
    // Expect the source certificate (custom resource)
    assertions_1.Template.fromStack(stack).resourceCountIs('Custom::RFDK_X509Generator', 1);
    // Expect the PKCS #12 to have a password secret
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::SecretsManager::Secret', {
        Description: 'Passphrase for the private key of the X509Certificate CertPkcs12',
        GenerateSecretString: {
            ExcludeCharacters: '"()$\'',
            ExcludePunctuation: true,
            IncludeSpace: false,
            PasswordLength: 24,
            RequireEachIncludedType: true,
        },
    });
    // Expect the PKCS #12 resource to have a policy with access to the X.509 resource
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: assertions_1.Match.objectLike({
            Statement: assertions_1.Match.arrayWith([
                assertions_1.Match.objectLike({
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:ConditionCheckItem',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                        'dynamodb:DescribeTable',
                    ],
                }),
                assertions_1.Match.objectLike({
                    Action: 'dynamodb:DescribeTable',
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: pkcs12CertPassphraseID,
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-71090F78': '71090f7809ce64f7c970cb645d4d473c',
                        },
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Key',
                        ],
                    },
                }),
                assertions_1.Match.objectLike({
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: certPassphraseID,
                    },
                }),
            ]),
        }),
    });
    // Expect no KMS key
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::KMS::Key', 0);
    // Expect the Lambda for converting the PEM to PKCS 12
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
        Handler: 'x509-certificate.convert',
    });
});
test('Convert to PKCS #12, use KMS', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const subject = { cn: 'testCN' };
    const sourceCertificate = new x509_certificate_1.X509CertificatePem(stack, 'Cert', { subject });
    const certPassphraseID = stack.getLogicalId(sourceCertificate.passphrase.node.defaultChild);
    const encryptionKey = new aws_kms_1.Key(stack, 'Key');
    new x509_certificate_1.X509CertificatePkcs12(stack, 'CertPkcs12', {
        sourceCertificate,
        encryptionKey,
    });
    // Expect the PKCS #12 custom resource
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_X509_PKCS12', {
        Passphrase: {
            Ref: 'CertPkcs12Passphrase1E3DF360',
        },
        Secret: {
            NamePrefix: 'Default/CertPkcs12',
            Description: 'Default/CertPkcs12',
            Tags: [
                {
                    Key: 'X509SecretGrant-71090F78',
                    Value: '71090f7809ce64f7c970cb645d4d473c',
                },
            ],
        },
        Certificate: {
            Cert: {
                'Fn::GetAtt': [
                    'Cert',
                    'Cert',
                ],
            },
            Key: {
                'Fn::GetAtt': [
                    'Cert',
                    'Key',
                ],
            },
            Passphrase: {
                Ref: certPassphraseID,
            },
        },
    });
    // Expect the source certificate (custom resource)
    assertions_1.Template.fromStack(stack).resourceCountIs('Custom::RFDK_X509Generator', 1);
    // Expect the PKCS #12 to have a password secret
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::SecretsManager::Secret', {
        Description: 'Passphrase for the private key of the X509Certificate CertPkcs12',
        GenerateSecretString: {
            ExcludeCharacters: '"()$\'',
            ExcludePunctuation: true,
            IncludeSpace: false,
            PasswordLength: 24,
            RequireEachIncludedType: true,
        },
    });
    // Expect a KMS key for encryption
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::KMS::Key', 1);
    // Expect the Lambda for converting the PEM to PKCS #12
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
        Handler: 'x509-certificate.convert',
    });
});
//# sourceMappingURL=data:application/json;base64,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