"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.testConstructTags = exports.resourceTagMatcher = void 0;
const assertions_1 = require("aws-cdk-lib/assertions");
const test_helper_1 = require("./test-helper");
/**
 * The name of the tag that RFDK is expected to use
 */
const RFDK_TAG_NAME = 'aws-rfdk';
/**
 * The current version of the RFDK package
 */
const RFDK_VERSION = require('../../../package.json').version; // eslint-disable-line @typescript-eslint/no-require-imports
/**
 * Returns the resource properties for an expected RFDK tag for a given resource
 * type
 *
 * @param resourceType The L1 CloudFormation resource type (e.g. "AWS::EC2::Instance")
 * @param constructName The name of the L2 (or higher) RFDK construct class
 */
function getExpectedRfdkTagProperties(resourceType, constructName) {
    const expectedValue = `${RFDK_VERSION}:${constructName}`;
    return resourceTagMatcher(resourceType, RFDK_TAG_NAME, expectedValue);
}
/**
 * Returns a CDK matcher for an expected tag key/value pair for a given Cfn resource type.
 * This is known to support the following resource types:
 *
 * * `AWS::AutoScaling::AutoScalingGroup`
 * * `AWS::EC2::SecurityGroup`
 * * `AWS::IAM::Role`
 * * `AWS::SSM::Parameter`
 *
 * All other resources are assumed to allow passing the following tag properties:
 *
 * ```js
 * {
 *   Tags: [
 *     {
 *       Key: 'key',
 *       Value: 'value',
 *     },
 *     // ...
 *   ]
 * }
 * ```
 */
/* eslint-disable-next-line jest/no-export */
function resourceTagMatcher(resourceType, tagName, tagValue) {
    if (resourceType === 'AWS::SSM::Parameter') {
        return {
            Tags: {
                [tagName]: tagValue,
            },
        };
    }
    else if (resourceType === 'AWS::AutoScaling::AutoScalingGroup') {
        return {
            Tags: assertions_1.Match.arrayWith([{
                    Key: tagName,
                    PropagateAtLaunch: true,
                    Value: tagValue,
                }]),
        };
    }
    else {
        return {
            Tags: assertions_1.Match.arrayWith([{
                    Key: tagName,
                    Value: tagValue,
                }]),
        };
    }
}
exports.resourceTagMatcher = resourceTagMatcher;
/**
 * Creates tests for the specified resources created by a L2 (or higher construct) to ensure that the resources it
 * creates are created with the RFDK tagging convention.
 *
 * The convention is to create a tag named "aws-rfdk" whose value follows:
 *
 * <VERSION>:<CONSTRUCT_NAME>
 *
 * @param args Arguments to configure the creation of construct tagging tests
 */
/* eslint-disable-next-line jest/no-export */
function testConstructTags(args) {
    const { constructName, createConstruct, resourceTypeCounts } = args;
    const entries = Object.entries(resourceTypeCounts);
    test.each(entries)('tags %s x%d', (resourceType, expectedCount) => {
        // GIVEN
        const expectedProps = getExpectedRfdkTagProperties(resourceType, constructName);
        // WHEN
        const stack = createConstruct();
        // THEN
        test_helper_1.resourcePropertiesCountIs(stack, resourceType, expectedProps, expectedCount);
    });
}
exports.testConstructTags = testConstructTags;
//# sourceMappingURL=data:application/json;base64,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