"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const lib_1 = require("../lib");
const test_helper_1 = require("./test-helper");
describe('Test StaticIpServer', () => {
    let stack;
    let vpc;
    beforeEach(() => {
        const app = new aws_cdk_lib_1.App();
        stack = new aws_cdk_lib_1.Stack(app, 'StackName');
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
    });
    test('basic setup', () => {
        // WHEN
        new lib_1.StaticPrivateIpServer(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResource('AWS::AutoScaling::AutoScalingGroup', {
            Properties: {
                MinSize: '1',
                MaxSize: '1',
                LifecycleHookSpecificationList: assertions_1.Match.arrayWith([
                    assertions_1.Match.objectLike({
                        DefaultResult: 'ABANDON',
                        HeartbeatTimeout: 120,
                        LifecycleHookName: 'NewStaticPrivateIpServer',
                        LifecycleTransition: 'autoscaling:EC2_INSTANCE_LAUNCHING',
                        NotificationMetadata: {
                            'Fn::Join': [
                                '',
                                assertions_1.Match.arrayWith([
                                    '{"eniId":"',
                                    {
                                        Ref: 'InstanceEniA230F5FE',
                                    },
                                    '"}',
                                ]),
                            ],
                        },
                    }),
                ]),
                Tags: assertions_1.Match.arrayWith([{
                        Key: 'RfdkStaticPrivateIpServerGrantConditionKey',
                        PropagateAtLaunch: true,
                        Value: 'StackNameAttachEniToInstance83a5dca5db544aa485d28d419cdf85ceF20CDF73',
                    }]),
            },
            DependsOn: assertions_1.Match.arrayWith([
                'AttachEniToInstance83a5dca5db544aa485d28d419cdf85ceAttachEniNotificationTopicc8b1e9a6783c4954b191204dd5e3b9e0695D3E7F',
                'InstanceEniA230F5FE',
            ]),
            UpdatePolicy: {
                AutoScalingScheduledAction: {
                    IgnoreUnmodifiedGroupSizeProperties: true,
                },
            },
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::NetworkInterface', {
            Description: 'Static ENI for StackName/Instance',
            GroupSet: assertions_1.Match.arrayWith([{
                    'Fn::GetAtt': [
                        'InstanceAsgInstanceSecurityGroup2DB1DA8B',
                        'GroupId',
                    ],
                }]),
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
            Handler: 'index.handler',
            Runtime: 'nodejs16.x',
            Description: 'Created by RFDK StaticPrivateIpServer to process instance launch lifecycle events in stack \'StackName\'. This lambda attaches an ENI to newly launched instances.',
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::SNS::Topic', {
            DisplayName: 'For RFDK instance-launch notifications for stack \'StackName\'',
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::SNS::Subscription', {
            Protocol: 'lambda',
            TopicArn: {
                Ref: 'AttachEniNotificationTopicc8b1e9a6783c4954b191204dd5e3b9e0F5D22665',
            },
            Endpoint: {
                'Fn::GetAtt': [
                    'AttachEniToInstance83a5dca5db544aa485d28d419cdf85ce70724E62',
                    'Arn',
                ],
            },
        });
        // The Lambda's policy should allow ENI attachment & condition-limited CompleteLifecycle.
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: assertions_1.Match.arrayWith([
                    assertions_1.Match.objectLike({
                        Action: 'autoscaling:CompleteLifecycleAction',
                        Effect: 'Allow',
                        Condition: {
                            'ForAnyValue:StringEquals': {
                                'autoscaling:ResourceTag/RfdkStaticPrivateIpServerGrantConditionKey': 'StackNameAttachEniToInstance83a5dca5db544aa485d28d419cdf85ceF20CDF73',
                            },
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':autoscaling:',
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    ':',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    ':autoScalingGroup:*:autoScalingGroupName/*',
                                ],
                            ],
                        },
                    }),
                    assertions_1.Match.objectLike({
                        Effect: 'Allow',
                        Action: [
                            'ec2:DescribeNetworkInterfaces',
                            'ec2:AttachNetworkInterface',
                        ],
                        Resource: '*',
                    }),
                ]),
            },
        });
        // Count singleton objects
        assertions_1.Template.fromStack(stack).resourceCountIs('AWS::Lambda::Function', 2); // Log retention & event handler.
        assertions_1.Template.fromStack(stack).resourceCountIs('AWS::SNS::Topic', 1);
        test_helper_1.resourcePropertiesCountIs(stack, 'AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Effect: 'Allow',
                        Action: 'sts:AssumeRole',
                        Principal: {
                            Service: 'autoscaling.amazonaws.com',
                        },
                    },
                ],
            },
        }, 1);
        test_helper_1.resourcePropertiesCountIs(stack, 'AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'sns:Publish',
                        Effect: 'Allow',
                        Resource: {
                            Ref: 'AttachEniNotificationTopicc8b1e9a6783c4954b191204dd5e3b9e0F5D22665',
                        },
                    },
                ],
            },
        }, 1);
    });
    test('creates singleton resources', () => {
        // WHEN
        new lib_1.StaticPrivateIpServer(stack, 'Instance1', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        new lib_1.StaticPrivateIpServer(stack, 'Instance2', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // THEN
        // Make sure both ASGs are tagged to allow CompleteLifeCycle by the singleton lambda.
        test_helper_1.resourcePropertiesCountIs(stack, 'AWS::AutoScaling::AutoScalingGroup', {
            Tags: assertions_1.Match.arrayWith([{
                    Key: 'RfdkStaticPrivateIpServerGrantConditionKey',
                    PropagateAtLaunch: true,
                    Value: 'StackNameAttachEniToInstance83a5dca5db544aa485d28d419cdf85ceF20CDF73',
                }]),
        }, 2);
        // Count singleton objects
        assertions_1.Template.fromStack(stack).resourceCountIs('AWS::Lambda::Function', 2); // Log retention & event handler.
        assertions_1.Template.fromStack(stack).resourceCountIs('AWS::SNS::Topic', 1);
        test_helper_1.resourcePropertiesCountIs(stack, 'AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Effect: 'Allow',
                        Action: 'sts:AssumeRole',
                        Principal: {
                            Service: 'autoscaling.amazonaws.com',
                        },
                    },
                ],
            },
        }, 1);
    });
    test('throw exception when no available subnets', () => {
        // THEN
        expect(() => {
            new lib_1.StaticPrivateIpServer(stack, 'Instance', {
                vpc,
                instanceType: new aws_ec2_1.InstanceType('t3.small'),
                machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                vpcSubnets: {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
                    availabilityZones: ['dummy zone'],
                },
            });
        }).toThrowError(/Did not find any subnets matching/);
    });
    test('resource signal count', () => {
        // WHEN
        new lib_1.StaticPrivateIpServer(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            resourceSignalTimeout: aws_cdk_lib_1.Duration.hours(12),
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResource('AWS::AutoScaling::AutoScalingGroup', {
            CreationPolicy: {
                ResourceSignal: {
                    Count: 1,
                    Timeout: 'PT12H',
                },
            },
        });
        expect(() => {
            new lib_1.StaticPrivateIpServer(stack, 'InstanceFail', {
                vpc,
                instanceType: new aws_ec2_1.InstanceType('t3.small'),
                machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                resourceSignalTimeout: aws_cdk_lib_1.Duration.seconds(12 * 60 * 60 + 1),
            });
        }).toThrowError('Resource signal timeout cannot exceed 12 hours.');
    });
});
//# sourceMappingURL=data:application/json;base64,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