"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const script_assets_1 = require("../lib/script-assets");
const asset_constants_1 = require("./asset-constants");
const instanceType = aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.MICRO);
const linuxImage = new aws_ec2_1.AmazonLinuxImage();
const windowsImage = new aws_ec2_1.WindowsImage(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE);
describe('executeScriptAsset', () => {
    let stack;
    let vpc;
    beforeEach(() => {
        // Create a new stack, vpc, and instance for each test
        stack = new aws_cdk_lib_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'vpc');
    });
    it.each([
        [linuxImage, '../scripts/bash/configureCloudWatchAgent.sh', asset_constants_1.CWA_ASSET_LINUX.Bucket],
        [windowsImage, '../scripts/powershell/configureCloudWatchAgent.ps1', asset_constants_1.CWA_ASSET_WINDOWS.Bucket],
    ])('grants read permissions', (machineImage, scriptLocation, bucketKey) => {
        // GIVEN
        const instance = new aws_ec2_1.Instance(stack, 'inst', {
            vpc,
            instanceType,
            machineImage,
        });
        const asset = new script_assets_1.ScriptAsset(stack, 'asset', {
            path: path.join(__dirname, scriptLocation),
        });
        // WHEN
        asset.executeOn({ host: instance });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: assertions_1.Match.arrayWith([
                    {
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':s3:::',
                                        { 'Fn::Sub': bucketKey },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':s3:::',
                                        { 'Fn::Sub': bucketKey },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ]),
                Version: '2012-10-17',
            },
            PolicyName: 'instInstanceRoleDefaultPolicyCB9E402C',
            Roles: [{ Ref: 'instInstanceRoleFE783FB1' }],
        });
    });
    test('downloads and executes script for linux', () => {
        // GIVEN
        const instance = new aws_ec2_1.Instance(stack, 'inst', {
            vpc,
            instanceType,
            machineImage: linuxImage,
        });
        const asset = new script_assets_1.ScriptAsset(stack, 'asset', {
            path: path.join(__dirname, '../scripts/bash/configureCloudWatchAgent.sh'),
        });
        // WHEN
        asset.executeOn({
            host: instance,
            args: ['arg1'],
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Instance', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        [
                            `#!/bin/bash\nmkdir -p $(dirname '/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh')\naws s3 cp 's3://`,
                            {
                                'Fn::Sub': asset_constants_1.CWA_ASSET_LINUX.Bucket,
                            },
                            `/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh' '/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh'\nset -e\nchmod +x '/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh'\n'/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh' arg1`,
                        ],
                    ],
                },
            },
        });
    });
    test('downloads and executes script for windows', () => {
        // GIVEN
        const instance = new aws_ec2_1.Instance(stack, 'inst', {
            vpc,
            instanceType,
            machineImage: windowsImage,
        });
        const asset = new script_assets_1.ScriptAsset(stack, 'asset', {
            path: path.join(__dirname, '../scripts/powershell/configureCloudWatchAgent.ps1'),
        });
        // WHEN
        asset.executeOn({
            host: instance,
            args: ['arg1'],
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Instance', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        [
                            `<powershell>mkdir (Split-Path -Path 'C:/temp/${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1' ) -ea 0\nRead-S3Object -BucketName '`,
                            {
                                'Fn::Sub': asset_constants_1.CWA_ASSET_WINDOWS.Bucket,
                            },
                            `' -key '${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1' -file 'C:/temp/${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1' -ErrorAction Stop\n` +
                                `&'C:/temp/${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1' arg1\nif (!$?) { Write-Error 'Failed to execute the file "C:/temp/${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1"' -ErrorAction Stop }</powershell>`,
                        ],
                    ],
                },
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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