"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_route53_1 = require("aws-cdk-lib/aws-route53");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
const tag_helpers_1 = require("./tag-helpers");
describe('Test MongoDbInstance', () => {
    let app;
    let stack;
    let vpc;
    let dnsZone;
    let caCert;
    let serverCert;
    const hostname = 'hostname';
    const zoneName = 'testZone';
    const version = lib_1.MongoDbVersion.COMMUNITY_3_6;
    const userSsplAcceptance = lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL;
    beforeEach(() => {
        app = new aws_cdk_lib_1.App();
        stack = new aws_cdk_lib_1.Stack(app, 'Stack');
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        dnsZone = new aws_route53_1.PrivateHostedZone(stack, 'PrivateHostedZone', {
            vpc,
            zoneName,
        });
        caCert = new lib_1.X509CertificatePem(stack, 'CaCert', {
            subject: {
                cn: 'DistinguishedName',
            },
        });
        serverCert = new lib_1.X509CertificatePem(stack, 'ServerCert', {
            subject: {
                cn: `${hostname}.${zoneName}`,
            },
            signingCertificate: caCert,
        });
    });
    test('default mongodb instance is created correctly', () => {
        // WHEN
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // THEN
        assertions_1.Template.fromStack(stack).resourceCountIs('AWS::AutoScaling::AutoScalingGroup', 1);
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: 'r5.large',
            BlockDeviceMappings: assertions_1.Match.arrayWith([
                assertions_1.Match.objectLike({
                    Ebs: assertions_1.Match.objectLike({
                        Encrypted: true,
                    }),
                }),
            ]),
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Route53::RecordSet', {
            Name: hostname + '.' + zoneName + '.',
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::SecretsManager::Secret', {
            Description: `Admin credentials for the MongoDB database ${aws_cdk_lib_1.Names.uniqueId(instance)}`,
            GenerateSecretString: {
                ExcludeCharacters: '\"()$\'',
                ExcludePunctuation: true,
                GenerateStringKey: 'password',
                IncludeSpace: false,
                PasswordLength: 24,
                RequireEachIncludedType: true,
                SecretStringTemplate: '{\"username\":\"admin\"}',
            },
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Volume', {
            Encrypted: true,
            Tags: assertions_1.Match.arrayWith([
                assertions_1.Match.objectLike({
                    Key: 'VolumeGrantAttach-6238D22B12',
                    Value: '6238d22b121db8094cb816e2a49d2b61',
                }),
            ]),
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
            PolicyDocument: assertions_1.Match.objectLike({
                Statement: assertions_1.Match.arrayWith([
                    {
                        Action: 'cloudformation:SignalResource',
                        Effect: 'Allow',
                        Resource: {
                            Ref: 'AWS::StackId',
                        },
                    },
                    {
                        Action: [
                            'logs:CreateLogStream',
                            'logs:PutLogEvents',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':logs:',
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    ':',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    ':log-group:',
                                    {
                                        'Fn::GetAtt': [
                                            'MongoDbInstanceMongoDbInstanceLogGroupWrapperEAF733BB',
                                            'LogGroupName',
                                        ],
                                    },
                                    ':*',
                                ],
                            ],
                        },
                    },
                    {
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':s3:::',
                                        {
                                            'Fn::Sub': asset_constants_1.CWA_ASSET_LINUX.Bucket,
                                        },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':s3:::',
                                        {
                                            'Fn::Sub': asset_constants_1.CWA_ASSET_LINUX.Bucket,
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                    {
                        Action: [
                            'ssm:DescribeParameters',
                            'ssm:GetParameters',
                            'ssm:GetParameter',
                            'ssm:GetParameterHistory',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':ssm:',
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    ':',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    ':parameter/',
                                    {
                                        Ref: assertions_1.Match.stringLikeRegexp('^MongoDbInstanceStringParameter.*'),
                                    },
                                ],
                            ],
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'Cert',
                            ],
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'CertChain',
                            ],
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'Key',
                            ],
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            Ref: assertions_1.Match.stringLikeRegexp('^ServerCertPassphrase.*'),
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            Ref: assertions_1.Match.stringLikeRegexp('^MongoDbInstanceAdminUser.*'),
                        },
                    },
                ]),
            }),
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
            LogGroupName: '/renderfarm/MongoDbInstance',
        });
        const cloudInitLogPath = '/var/log/cloud-init-output.log';
        const cloudInitLogPrefix = 'cloud-init-output';
        const mongoLogPath = '/var/log/mongodb/mongod.log';
        const mongoLogPrefix = 'MongoDB';
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::SSM::Parameter', {
            Description: 'config file for Repository logs config',
            Value: {
                'Fn::Join': [
                    '',
                    assertions_1.Match.arrayWith([
                        `\",\"log_stream_name\":\"${cloudInitLogPrefix}-{instance_id}\",\"file_path\":\"${cloudInitLogPath}\",` +
                            '\"timezone\":\"Local\"},{\"log_group_name\":\"',
                        `\",\"log_stream_name\":\"${mongoLogPrefix}-{instance_id}\",\"file_path\":\"${mongoLogPath}\"` +
                            ',\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}',
                    ]),
                ],
            },
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        assertions_1.Match.arrayWith([
                            // Make sure we add signal on exit
                            '#!/bin/bash\n' +
                                'function exitTrap(){\n' +
                                'exitCode=$?\n' +
                                '/opt/aws/bin/cfn-signal --stack Stack --resource MongoDbInstanceServerAsgASG2643AD1D --region ',
                            {
                                Ref: 'AWS::Region',
                            },
                            ' -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n' +
                                'test "${MONGO_SETUP_DIR}" != "" && sudo umount "${MONGO_SETUP_DIR}"\n' +
                                '}\n' +
                                // Set up the exit trap
                                'trap exitTrap EXIT\n' +
                                // Script settings
                                'set -xefuo pipefail\n' +
                                // Setup CloudWatch agent
                                `mkdir -p $(dirname '/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh')\n` +
                                'aws s3 cp \'s3://',
                            {
                                'Fn::Sub': asset_constants_1.CWA_ASSET_LINUX.Bucket,
                            },
                            `/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh' '/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh'\n` +
                                'set -e\n' +
                                `chmod +x '/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh'\n'/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh' -i `,
                            {
                                Ref: 'AWS::Region',
                            },
                            ' ',
                            {
                                Ref: assertions_1.Match.stringLikeRegexp('^MongoDbInstanceStringParameter.*'),
                            },
                            // Make sure we mount the EBS Volume
                            `\nmkdir -p $(dirname '/tmp/${asset_constants_1.MOUNT_EBS_SCRIPT_LINUX.Key}.zip')\n` +
                                'aws s3 cp \'s3://',
                            {
                                'Fn::Sub': asset_constants_1.MOUNT_EBS_SCRIPT_LINUX.Bucket,
                            },
                            `/${asset_constants_1.MOUNT_EBS_SCRIPT_LINUX.Key}.zip' '/tmp/${asset_constants_1.MOUNT_EBS_SCRIPT_LINUX.Key}.zip'\n` +
                                'TMPDIR=$(mktemp -d)\n' +
                                'pushd "$TMPDIR"\n' +
                                `unzip /tmp/${asset_constants_1.MOUNT_EBS_SCRIPT_LINUX.Key}.zip\n` +
                                'bash ./mountEbsBlockVolume.sh ',
                            {
                                Ref: assertions_1.Match.stringLikeRegexp('^MongoDbInstanceMongoDbData.*'),
                            },
                            ' xfs /var/lib/mongo rw ""\n' +
                                'popd\n' +
                                `rm -f /tmp/${asset_constants_1.MOUNT_EBS_SCRIPT_LINUX.Key}.zip\n` +
                                // Install MongoDB
                                `mkdir -p $(dirname '/tmp/${asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Key}.sh')\n` +
                                'aws s3 cp \'s3://',
                            {
                                'Fn::Sub': asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Bucket,
                            },
                            `/${asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Key}.sh' '/tmp/${asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Key}.sh'\n` +
                                `bash /tmp/${asset_constants_1.INSTALL_MONGODB_3_6_SCRIPT_LINUX.Key}.sh\n` +
                                // Fetching the MongoDB configuration scripts
                                `mkdir -p $(dirname '/tmp/${asset_constants_1.MONGODB_3_6_CONFIGURATION_SCRIPTS.Key}.zip')\n` +
                                'aws s3 cp \'s3://',
                            {
                                'Fn::Sub': asset_constants_1.MONGODB_3_6_CONFIGURATION_SCRIPTS.Bucket,
                            },
                            `/${asset_constants_1.MONGODB_3_6_CONFIGURATION_SCRIPTS.Key}.zip' '/tmp/${asset_constants_1.MONGODB_3_6_CONFIGURATION_SCRIPTS.Key}.zip'\n` +
                                // Configure MongoDB
                                'which mongod && test -f /etc/mongod.conf\n' +
                                'sudo service mongod stop\n' +
                                'MONGO_SETUP_DIR=$(mktemp -d)\n' +
                                'mkdir -p "${MONGO_SETUP_DIR}"\n' +
                                'sudo mount -t tmpfs -o size=50M tmpfs "${MONGO_SETUP_DIR}"\n' +
                                'pushd "${MONGO_SETUP_DIR}"\n' +
                                `unzip /tmp/${asset_constants_1.MONGODB_3_6_CONFIGURATION_SCRIPTS.Key}.zip\n` +
                                'cp /etc/mongod.conf .\n' +
                                // Getting the server certificate
                                'bash serverCertFromSecrets.sh \"',
                            {
                                'Fn::GetAtt': [
                                    'ServerCert',
                                    'Cert',
                                ],
                            },
                            '" "',
                            {
                                'Fn::GetAtt': [
                                    'ServerCert',
                                    'CertChain',
                                ],
                            },
                            '" "',
                            {
                                'Fn::GetAtt': [
                                    'ServerCert',
                                    'Key',
                                ],
                            },
                            '" "',
                            {
                                Ref: assertions_1.Match.stringLikeRegexp('^ServerCertPassphrase.*'),
                            },
                            '"\n' +
                                // Set mongodb certificates and credentials
                                'sudo mkdir -p /etc/mongod_certs\n' +
                                'sudo mv ./ca.crt ./key.pem /etc/mongod_certs\n' +
                                'sudo chown root.mongod -R /etc/mongod_certs/\n' +
                                'sudo chmod 640 -R /etc/mongod_certs/\n' +
                                'sudo chmod 750 /etc/mongod_certs/\n' +
                                'sudo chown mongod.mongod -R /var/lib/mongo\n' +
                                'bash ./setMongoLimits.sh\n' +
                                'bash ./setStoragePath.sh "/var/lib/mongo"\n' +
                                'bash ./setMongoNoAuth.sh\n' +
                                'sudo service mongod start\n' +
                                'bash ./setAdminCredentials.sh \"',
                            {
                                Ref: assertions_1.Match.stringLikeRegexp('^MongoDbInstanceAdminUser.*'),
                            },
                            '"\n' +
                                // Setup for live deployment, and start mongod
                                'sudo service mongod stop\n' +
                                'bash ./setLiveConfiguration.sh\n' +
                                'sudo systemctl enable mongod\n' +
                                'sudo service mongod start\n' +
                                'popd',
                        ]),
                    ],
                },
            },
        });
        // Make sure all the required public members are set
        expect(instance.version).toBe(version);
        expect(instance.connections).toBeDefined();
        expect(instance.connections).toBe(instance.server.connections);
        expect(instance.grantPrincipal).toBeDefined();
        expect(instance.grantPrincipal).toBe(instance.server.grantPrincipal);
        expect(instance.port).toBeDefined();
        expect(instance.role).toBeDefined();
        expect(instance.role).toBe(instance.server.role);
        expect(instance.userData).toBeDefined();
        expect(instance.userData).toBe(instance.server.userData);
        expect(instance.fullHostname).toBeDefined();
    });
    test('throw exception when no available subnets', () => {
        // GIVEN
        const invalidSubnets = {
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
            availabilityZones: ['dummy zone'],
        };
        // THEN
        expect(() => {
            new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
                mongoDb: {
                    version,
                    dnsZone,
                    hostname,
                    serverCertificate: serverCert,
                    userSsplAcceptance,
                },
                vpc,
                vpcSubnets: invalidSubnets,
            });
        }).toThrowError(/Did not find any subnets matching/);
    });
    test('changing instance type works correctly', () => {
        // GIVEN
        const expectedInstanceType = 'm4.micro';
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            instanceType: new aws_ec2_1.InstanceType(expectedInstanceType),
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: expectedInstanceType,
        });
    });
    test('allowing ssh connection with key name', () => {
        // GIVEN
        const expectedKeyName = 'someKeyName';
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            keyName: expectedKeyName,
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
            KeyName: expectedKeyName,
        });
    });
    test('using custom admin user works correctly', () => {
        // GIVEN
        const expectedAdminUser = new aws_secretsmanager_1.Secret(stack, 'AdminUser', {
            description: 'Custom admin credentials for the MongoDB database',
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
                generateStringKey: 'test_password',
                secretStringTemplate: JSON.stringify({ username: 'admin' }),
            },
        });
        // WHEN
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                adminUser: expectedAdminUser,
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // THEN
        expect(instance.adminUser).toBe(expectedAdminUser);
    });
    test('setting security group works correctly', () => {
        // GIVEN
        const actualSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'SecurityGroup', {
            securityGroupName: 'CustomSecurityGroup',
            vpc,
        });
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            securityGroup: actualSecurityGroup,
        });
        // THEN
        assertions_1.Template.fromStack(stack).resourceCountIs('AWS::EC2::SecurityGroup', 1);
    });
    test('setting role works correctly', () => {
        // GIVEN
        const expectedRole = new aws_iam_1.Role(stack, 'Role', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            roleName: 'CustomRole',
        });
        // WHEN
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            role: expectedRole,
        });
        // THEN
        expect(instance.server.role).toBe(expectedRole);
        expect(instance.role).toBe(expectedRole);
    });
    test('setting custom data volume works correctly', () => {
        // GIVEN
        const actualVolume = new aws_ec2_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: aws_cdk_lib_1.Size.gibibytes(50),
        });
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
                mongoDataVolume: {
                    volume: actualVolume,
                },
            },
            vpc,
        });
        // THEN
        assertions_1.Template.fromStack(stack).resourceCountIs('AWS::EC2::Volume', 1);
    });
    test('setting custom encryption key for data volume works correctly', () => {
        // GIVEN
        // KmsKeyId is Key961B73FD
        const actualEncryptionKey = new aws_kms_1.Key(stack, 'Key', {
            description: 'Key for testing',
        });
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
                mongoDataVolume: {
                    volumeProps: {
                        encryptionKey: actualEncryptionKey,
                    },
                },
            },
            vpc,
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Volume', {
            Encrypted: true,
            KmsKeyId: assertions_1.Match.objectLike({
                'Fn::GetAtt': assertions_1.Match.arrayWith([
                    'Key961B73FD',
                ]),
            }),
        });
    });
    test('setting custom size for data volume works correctly', () => {
        // GIVEN
        const volumeSize = 123;
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
                mongoDataVolume: {
                    volumeProps: {
                        size: aws_cdk_lib_1.Size.gibibytes(volumeSize),
                    },
                },
            },
            vpc,
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::EC2::Volume', {
            Size: volumeSize,
        });
    });
    test('setting LogGroup bucket name enables export to S3', () => {
        // GIVEN
        const bucketName = 'test-bucket';
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                bucketName,
            },
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Events::Rule', {
            Targets: assertions_1.Match.arrayWith([assertions_1.Match.objectLike({
                    Input: '{\"BucketName\":\"' + bucketName + '\",\"ExportFrequencyInHours\":1,\"LogGroupName\":\"/renderfarm/MongoDbInstance\",\"RetentionInHours\":72}',
                })]),
        });
    });
    test.each([
        'test-prefix/',
        '',
    ])('is created with correct LogGroup prefix %s', (testPrefix) => {
        // GIVEN
        const id = 'MongoDbInstance';
        // WHEN
        new lib_1.MongoDbInstance(stack, id, {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                logGroupPrefix: testPrefix,
            },
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
            LogGroupName: testPrefix + id,
        });
    });
    test('is created with correct LogGroup retention', () => {
        // GIVEN
        const retention = aws_logs_1.RetentionDays.ONE_DAY;
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                retention,
            },
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
            RetentionInDays: retention,
        });
    });
    test('adds security group', () => {
        // GIVEN
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'NewSecurityGroup', {
            vpc,
        });
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // WHEN
        instance.addSecurityGroup(securityGroup);
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::AutoScaling::LaunchConfiguration', {
            SecurityGroups: assertions_1.Match.arrayWith([stack.resolve(securityGroup.securityGroupId)]),
        });
    });
    tag_helpers_1.testConstructTags({
        constructName: 'MongoDbInstance',
        createConstruct: () => {
            const isolatedStack = new aws_cdk_lib_1.Stack(app, 'IsolatedStack');
            new lib_1.MongoDbInstance(isolatedStack, 'MongoDbInstance', {
                mongoDb: {
                    version,
                    dnsZone,
                    hostname,
                    serverCertificate: serverCert,
                    userSsplAcceptance,
                },
                vpc,
            });
            return isolatedStack;
        },
        resourceTypeCounts: {
            'AWS::EC2::SecurityGroup': 1,
            'AWS::IAM::Role': 1,
            'AWS::AutoScaling::AutoScalingGroup': 1,
            'AWS::EC2::NetworkInterface': 1,
            'AWS::SecretsManager::Secret': 1,
            'AWS::EC2::Volume': 1,
            'AWS::SSM::Parameter': 1,
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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