"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const log_group_factory_1 = require("../lib/log-group-factory");
describe('log group', () => {
    test('created correctly with defaults', () => {
        const stack = new aws_cdk_lib_1.Stack();
        // WHEN
        log_group_factory_1.LogGroupFactory.createOrFetch(stack, 'TestId', 'testLogGroup');
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
            LogGroupName: 'testLogGroup',
            RetentionInDays: 3,
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', assertions_1.Match.not({
            Role: {
                'Fn::GetAtt': [
                    'LogGroupExporter6382448ce4b242e9b14fa0a9ccdb198eServiceRoleB67C808B',
                    'Arn',
                ],
            },
        }));
    });
    test('created correctly with prefix', () => {
        const stack = new aws_cdk_lib_1.Stack();
        // WHEN
        log_group_factory_1.LogGroupFactory.createOrFetch(stack, 'TestId', 'testLogGroup', {
            logGroupPrefix: 'prefix-',
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
            LogGroupName: 'prefix-testLogGroup',
            RetentionInDays: 3,
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', assertions_1.Match.not({
            Role: {
                'Fn::GetAtt': [
                    'LogGroupExporter6382448ce4b242e9b14fa0a9ccdb198eServiceRoleB67C808B',
                    'Arn',
                ],
            },
        }));
    });
    test('created correctly with custom retention', () => {
        const stack = new aws_cdk_lib_1.Stack();
        // WHEN
        log_group_factory_1.LogGroupFactory.createOrFetch(stack, 'TestId', 'testLogGroup', {
            retention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
            LogGroupName: 'testLogGroup',
            RetentionInDays: 7,
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', assertions_1.Match.not({
            Role: {
                'Fn::GetAtt': [
                    'LogGroupExporter6382448ce4b242e9b14fa0a9ccdb198eServiceRoleB67C808B',
                    'Arn',
                ],
            },
        }));
    });
});
describe('exporting log group', () => {
    test('created correctly with defaults', () => {
        const stack = new aws_cdk_lib_1.Stack();
        // WHEN
        log_group_factory_1.LogGroupFactory.createOrFetch(stack, 'TestId', 'testLogGroup', {
            bucketName: 'test-bucket',
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
            LogGroupName: 'testLogGroup',
            RetentionInDays: 3,
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
            Role: {
                'Fn::GetAtt': [
                    'LogGroupExporter6382448ce4b242e9b14fa0a9ccdb198eServiceRoleB67C808B',
                    'Arn',
                ],
            },
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Events::Rule', {
            Targets: [
                {
                    Input: '{\"BucketName\":\"test-bucket\",\"ExportFrequencyInHours\":1,\"LogGroupName\":\"testLogGroup\",\"RetentionInHours\":72}',
                },
            ],
        });
    });
    test('created correctly with prefix', () => {
        const stack = new aws_cdk_lib_1.Stack();
        // WHEN
        log_group_factory_1.LogGroupFactory.createOrFetch(stack, 'TestId', 'testLogGroup', {
            bucketName: 'test-bucket',
            logGroupPrefix: 'prefix-',
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
            LogGroupName: 'prefix-testLogGroup',
            RetentionInDays: 3,
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
            Role: {
                'Fn::GetAtt': [
                    'LogGroupExporter6382448ce4b242e9b14fa0a9ccdb198eServiceRoleB67C808B',
                    'Arn',
                ],
            },
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Events::Rule', {
            Targets: [
                {
                    Input: '{\"BucketName\":\"test-bucket\",\"ExportFrequencyInHours\":1,\"LogGroupName\":\"prefix-testLogGroup\",\"RetentionInHours\":72}',
                },
            ],
        });
    });
    test('created correctly with custom retention', () => {
        const stack = new aws_cdk_lib_1.Stack();
        // WHEN
        log_group_factory_1.LogGroupFactory.createOrFetch(stack, 'TestId', 'testLogGroup', {
            bucketName: 'test-bucket',
            retention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
            LogGroupName: 'testLogGroup',
            RetentionInDays: 7,
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
            Role: {
                'Fn::GetAtt': [
                    'LogGroupExporter6382448ce4b242e9b14fa0a9ccdb198eServiceRoleB67C808B',
                    'Arn',
                ],
            },
        });
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Events::Rule', {
            Targets: [
                {
                    Input: '{\"BucketName\":\"test-bucket\",\"ExportFrequencyInHours\":1,\"LogGroupName\":\"testLogGroup\",\"RetentionInHours\":168}',
                },
            ],
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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