"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const imported_acm_certificate_1 = require("../lib/imported-acm-certificate");
const x509_certificate_1 = require("../lib/x509-certificate");
let stack;
let secretCert;
let certPassphraseID;
let importedAcmCertificate;
describe('ImportedAcmCertificate', () => {
    beforeEach(() => {
        // GIVEN
        stack = new aws_cdk_lib_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
        secretCert = new x509_certificate_1.X509CertificatePem(stack, 'Pem', {
            subject: { cn: 'Server' },
        });
        certPassphraseID = stack.getLogicalId(secretCert.passphrase.node.defaultChild);
        // WHEN
        importedAcmCertificate = new imported_acm_certificate_1.ImportedAcmCertificate(stack, 'AcmCert', {
            cert: secretCert.cert,
            certChain: secretCert.certChain,
            key: secretCert.key,
            passphrase: secretCert.passphrase,
        });
    });
    test('creates Custom::RFDK_AcmImportedCertificate', () => {
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::RFDK_AcmImportedCertificate', {
            X509CertificatePem: {
                Cert: {
                    'Fn::GetAtt': [
                        'Pem',
                        'Cert',
                    ],
                },
                Key: {
                    'Fn::GetAtt': [
                        'Pem',
                        'Key',
                    ],
                },
                Passphrase: {
                    Ref: certPassphraseID,
                },
                CertChain: '',
            },
            Tags: [
                {
                    Key: 'AcmCertImport-F4E2ABF9',
                    Value: 'f4e2abf974443234fdb095fafcfa9ee2',
                },
                {
                    Key: 'Name',
                    Value: 'f4e2abf974443234fdb095fafcfa9ee2',
                },
            ],
        });
    });
    describe('creates AWS::DynamoDB::Table for database', () => {
        test('with PhysicalID partition key', () => {
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::DynamoDB::Table', {
                AttributeDefinitions: assertions_1.Match.arrayWith([
                    {
                        AttributeName: 'PhysicalId',
                        AttributeType: 'S',
                    },
                ]),
                KeySchema: assertions_1.Match.arrayWith([
                    {
                        AttributeName: 'PhysicalId',
                        KeyType: 'HASH',
                    },
                ]),
            });
        });
        test('with CustomResource sort key', () => {
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::DynamoDB::Table', {
                AttributeDefinitions: assertions_1.Match.arrayWith([
                    {
                        AttributeName: 'CustomResource',
                        AttributeType: 'S',
                    },
                ]),
                KeySchema: assertions_1.Match.arrayWith([
                    {
                        AttributeName: 'CustomResource',
                        KeyType: 'RANGE',
                    },
                ]),
            });
        });
    });
    test('creates AWS::IAM::Policy', () => {
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    assertions_1.Match.objectLike({
                        Action: [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                            'dynamodb:ConditionCheckItem',
                            'dynamodb:BatchWriteItem',
                            'dynamodb:PutItem',
                            'dynamodb:UpdateItem',
                            'dynamodb:DeleteItem',
                            'dynamodb:DescribeTable',
                        ],
                    }),
                    assertions_1.Match.objectLike({
                        Action: 'dynamodb:DescribeTable',
                    }),
                    assertions_1.Match.objectLike({
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            'Fn::GetAtt': [
                                'Pem',
                                'Cert',
                            ],
                        },
                    }),
                    assertions_1.Match.objectLike({
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            'Fn::GetAtt': [
                                'Pem',
                                'Key',
                            ],
                        },
                    }),
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            Ref: certPassphraseID,
                        },
                    },
                    assertions_1.Match.objectLike({
                        Action: [
                            'acm:AddTagsToCertificate',
                            'acm:ImportCertificate',
                        ],
                        Condition: {
                            StringEquals: {
                                'aws:RequestTag/AcmCertImport-F4E2ABF9': 'f4e2abf974443234fdb095fafcfa9ee2',
                            },
                        },
                        Resource: '*',
                    }),
                    assertions_1.Match.objectLike({
                        Action: [
                            'acm:DeleteCertificate',
                            'acm:DescribeCertificate',
                            'acm:GetCertificate',
                        ],
                        Resource: '*',
                    }),
                ],
            },
        });
    });
    describe('custom resource lambda function', () => {
        test('uses correct handler', () => {
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
                Handler: 'x509-certificate.importCert',
            });
        });
        test('uses RFDK lambda layer', () => {
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
                Layers: assertions_1.Match.arrayWith([
                    assertions_1.Match.stringLikeRegexp('arn:aws:lambda:us-west-2:224375009292:layer:openssl-al2:.*'),
                ]),
            });
        });
        test('sets DATABASE environment variable', () => {
            // GIVEN
            const table = importedAcmCertificate.node.findChild('Table');
            // THEN
            assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Lambda::Function', {
                Environment: {
                    Variables: assertions_1.Match.objectLike({
                        DATABASE: stack.resolve(table.tableName),
                    }),
                },
            });
        });
    });
    describe('applyRemovalPolicy', () => {
        test('default RemovalPolicy is Delete', () => {
            assertions_1.Template.fromStack(stack).hasResource('Custom::RFDK_AcmImportedCertificate', {
                DeletionPolicy: 'Delete',
                UpdateReplacePolicy: 'Delete',
            });
        });
        test('Different policy can be applied', () => {
            importedAcmCertificate.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.RETAIN);
            assertions_1.Template.fromStack(stack).hasResource('Custom::RFDK_AcmImportedCertificate', {
                DeletionPolicy: 'Retain',
                UpdateReplacePolicy: 'Retain',
            });
        });
    });
    describe('metricDaysToExpiry', () => {
        let metricExpiry;
        beforeEach(() => {
            // GIVEN
            metricExpiry = importedAcmCertificate.metricDaysToExpiry();
        });
        test('uses certificate ARN', () => {
            // THEN
            expect(metricExpiry.dimensions?.CertificateArn).toEqual(importedAcmCertificate.certificateArn);
        });
        test('uses correct metric', () => {
            // THEN
            expect(metricExpiry.metricName).toEqual('DaysToExpiry');
            expect(metricExpiry.namespace).toEqual('AWS/CertificateManager');
            expect(metricExpiry.statistic).toEqual(aws_cloudwatch_1.Statistic.MINIMUM);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW1wb3J0ZWQtYWNtLWNlcnRpZmljYXRlLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbXBvcnRlZC1hY20tY2VydGlmaWNhdGUudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7OztHQUdHOztBQUVILDZDQUFtRDtBQUNuRCx1REFHZ0M7QUFDaEMsK0RBR29DO0FBSXBDLDhFQUF5RTtBQUN6RSw4REFBNkQ7QUFFN0QsSUFBSSxLQUFZLENBQUM7QUFDakIsSUFBSSxVQUE4QixDQUFDO0FBQ25DLElBQUksZ0JBQXdCLENBQUM7QUFDN0IsSUFBSSxzQkFBOEMsQ0FBQztBQUVuRCxRQUFRLENBQUMsd0JBQXdCLEVBQUUsR0FBRyxFQUFFO0lBQ3RDLFVBQVUsQ0FBQyxHQUFHLEVBQUU7UUFDZCxRQUFRO1FBQ1IsS0FBSyxHQUFHLElBQUksbUJBQUssQ0FBQyxTQUFTLEVBQUUsT0FBTyxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUV4RSxVQUFVLEdBQUcsSUFBSSxxQ0FBa0IsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO1lBQ2hELE9BQU8sRUFBRSxFQUFFLEVBQUUsRUFBRSxRQUFRLEVBQUU7U0FDMUIsQ0FBQyxDQUFDO1FBQ0gsZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxZQUF5QixDQUFDLENBQUM7UUFFNUYsT0FBTztRQUNQLHNCQUFzQixHQUFHLElBQUksaURBQXNCLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtZQUNwRSxJQUFJLEVBQUUsVUFBVSxDQUFDLElBQUk7WUFDckIsU0FBUyxFQUFFLFVBQVUsQ0FBQyxTQUFTO1lBQy9CLEdBQUcsRUFBRSxVQUFVLENBQUMsR0FBRztZQUNuQixVQUFVLEVBQUUsVUFBVSxDQUFDLFVBQVU7U0FDbEMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsNkNBQTZDLEVBQUUsR0FBRyxFQUFFO1FBQ3ZELE9BQU87UUFDUCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxxQ0FBcUMsRUFBRTtZQUNyRixrQkFBa0IsRUFBRTtnQkFDbEIsSUFBSSxFQUFFO29CQUNKLFlBQVksRUFBRTt3QkFDWixLQUFLO3dCQUNMLE1BQU07cUJBQ1A7aUJBQ0Y7Z0JBQ0QsR0FBRyxFQUFFO29CQUNILFlBQVksRUFBRTt3QkFDWixLQUFLO3dCQUNMLEtBQUs7cUJBQ047aUJBQ0Y7Z0JBQ0QsVUFBVSxFQUFFO29CQUNWLEdBQUcsRUFBRSxnQkFBZ0I7aUJBQ3RCO2dCQUNELFNBQVMsRUFBRSxFQUFFO2FBQ2Q7WUFDRCxJQUFJLEVBQUU7Z0JBQ0o7b0JBQ0UsR0FBRyxFQUFFLHdCQUF3QjtvQkFDN0IsS0FBSyxFQUFFLGtDQUFrQztpQkFDMUM7Z0JBQ0Q7b0JBQ0UsR0FBRyxFQUFFLE1BQU07b0JBQ1gsS0FBSyxFQUFFLGtDQUFrQztpQkFDMUM7YUFDRjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLDJDQUEyQyxFQUFFLEdBQUcsRUFBRTtRQUN6RCxJQUFJLENBQUMsK0JBQStCLEVBQUUsR0FBRyxFQUFFO1lBQ3pDLE9BQU87WUFDUCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxzQkFBc0IsRUFBRTtnQkFDdEUsb0JBQW9CLEVBQUUsa0JBQUssQ0FBQyxTQUFTLENBQUM7b0JBQ3BDO3dCQUNFLGFBQWEsRUFBRSxZQUFZO3dCQUMzQixhQUFhLEVBQUUsR0FBRztxQkFDbkI7aUJBQ0YsQ0FBQztnQkFDRixTQUFTLEVBQUUsa0JBQUssQ0FBQyxTQUFTLENBQUM7b0JBQ3pCO3dCQUNFLGFBQWEsRUFBRSxZQUFZO3dCQUMzQixPQUFPLEVBQUUsTUFBTTtxQkFDaEI7aUJBQ0YsQ0FBQzthQUNILENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLDhCQUE4QixFQUFFLEdBQUcsRUFBRTtZQUN4QyxPQUFPO1lBQ1AscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUMscUJBQXFCLENBQUMsc0JBQXNCLEVBQUU7Z0JBQ3RFLG9CQUFvQixFQUFFLGtCQUFLLENBQUMsU0FBUyxDQUFDO29CQUNwQzt3QkFDRSxhQUFhLEVBQUUsZ0JBQWdCO3dCQUMvQixhQUFhLEVBQUUsR0FBRztxQkFDbkI7aUJBQ0YsQ0FBQztnQkFDRixTQUFTLEVBQUUsa0JBQUssQ0FBQyxTQUFTLENBQUM7b0JBQ3pCO3dCQUNFLGFBQWEsRUFBRSxnQkFBZ0I7d0JBQy9CLE9BQU8sRUFBRSxPQUFPO3FCQUNqQjtpQkFDRixDQUFDO2FBQ0gsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQywwQkFBMEIsRUFBRSxHQUFHLEVBQUU7UUFDcEMsT0FBTztRQUNQLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLHFCQUFxQixDQUFDLGtCQUFrQixFQUFFO1lBQ2xFLGNBQWMsRUFBRTtnQkFDZCxTQUFTLEVBQUU7b0JBQ1Qsa0JBQUssQ0FBQyxVQUFVLENBQUM7d0JBQ2YsTUFBTSxFQUFFOzRCQUNOLHVCQUF1Qjs0QkFDdkIscUJBQXFCOzRCQUNyQiwyQkFBMkI7NEJBQzNCLGdCQUFnQjs0QkFDaEIsa0JBQWtCOzRCQUNsQixlQUFlOzRCQUNmLDZCQUE2Qjs0QkFDN0IseUJBQXlCOzRCQUN6QixrQkFBa0I7NEJBQ2xCLHFCQUFxQjs0QkFDckIscUJBQXFCOzRCQUNyQix3QkFBd0I7eUJBQ3pCO3FCQUNGLENBQUM7b0JBQ0Ysa0JBQUssQ0FBQyxVQUFVLENBQUM7d0JBQ2YsTUFBTSxFQUFFLHdCQUF3QjtxQkFDakMsQ0FBQztvQkFDRixrQkFBSyxDQUFDLFVBQVUsQ0FBQzt3QkFDZixNQUFNLEVBQUU7NEJBQ04sK0JBQStCOzRCQUMvQiwrQkFBK0I7eUJBQ2hDO3dCQUNELFFBQVEsRUFBRTs0QkFDUixZQUFZLEVBQUU7Z0NBQ1osS0FBSztnQ0FDTCxNQUFNOzZCQUNQO3lCQUNGO3FCQUNGLENBQUM7b0JBQ0Ysa0JBQUssQ0FBQyxVQUFVLENBQUM7d0JBQ2YsTUFBTSxFQUFFOzRCQUNOLCtCQUErQjs0QkFDL0IsK0JBQStCO3lCQUNoQzt3QkFDRCxRQUFRLEVBQUU7NEJBQ1IsWUFBWSxFQUFFO2dDQUNaLEtBQUs7Z0NBQ0wsS0FBSzs2QkFDTjt5QkFDRjtxQkFDRixDQUFDO29CQUNGO3dCQUNFLE1BQU0sRUFBRTs0QkFDTiwrQkFBK0I7NEJBQy9CLCtCQUErQjt5QkFDaEM7d0JBQ0QsUUFBUSxFQUFFOzRCQUNSLEdBQUcsRUFBRSxnQkFBZ0I7eUJBQ3RCO3FCQUNGO29CQUNELGtCQUFLLENBQUMsVUFBVSxDQUFDO3dCQUNmLE1BQU0sRUFBRTs0QkFDTiwwQkFBMEI7NEJBQzFCLHVCQUF1Qjt5QkFDeEI7d0JBQ0QsU0FBUyxFQUFFOzRCQUNULFlBQVksRUFBRTtnQ0FDWix1Q0FBdUMsRUFBRSxrQ0FBa0M7NkJBQzVFO3lCQUNGO3dCQUNELFFBQVEsRUFBRSxHQUFHO3FCQUNkLENBQUM7b0JBQ0Ysa0JBQUssQ0FBQyxVQUFVLENBQUM7d0JBQ2YsTUFBTSxFQUFFOzRCQUNOLHVCQUF1Qjs0QkFDdkIseUJBQXlCOzRCQUN6QixvQkFBb0I7eUJBQ3JCO3dCQUNELFFBQVEsRUFBRSxHQUFHO3FCQUNkLENBQUM7aUJBQ0g7YUFDRjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLGlDQUFpQyxFQUFFLEdBQUcsRUFBRTtRQUMvQyxJQUFJLENBQUMsc0JBQXNCLEVBQUUsR0FBRyxFQUFFO1lBQ2hDLE9BQU87WUFDUCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyx1QkFBdUIsRUFBRTtnQkFDdkUsT0FBTyxFQUFFLDZCQUE2QjthQUN2QyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyx3QkFBd0IsRUFBRSxHQUFHLEVBQUU7WUFDbEMsT0FBTztZQUNQLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLHFCQUFxQixDQUFDLHVCQUF1QixFQUFFO2dCQUN2RSxNQUFNLEVBQUUsa0JBQUssQ0FBQyxTQUFTLENBQUM7b0JBQ3RCLGtCQUFLLENBQUMsZ0JBQWdCLENBQUMsNERBQTRELENBQUM7aUJBQ3JGLENBQUM7YUFDSCxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxvQ0FBb0MsRUFBRSxHQUFHLEVBQUU7WUFDOUMsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLHNCQUFzQixDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFVLENBQUM7WUFFdEUsT0FBTztZQUNQLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLHFCQUFxQixDQUFDLHVCQUF1QixFQUFFO2dCQUN2RSxXQUFXLEVBQUU7b0JBQ1gsU0FBUyxFQUFFLGtCQUFLLENBQUMsVUFBVSxDQUFDO3dCQUMxQixRQUFRLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDO3FCQUN6QyxDQUFDO2lCQUNIO2FBQ0YsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxvQkFBb0IsRUFBRSxHQUFHLEVBQUU7UUFDbEMsSUFBSSxDQUFDLGlDQUFpQyxFQUFFLEdBQUcsRUFBRTtZQUMzQyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxXQUFXLENBQUMscUNBQXFDLEVBQUU7Z0JBQzNFLGNBQWMsRUFBRSxRQUFRO2dCQUN4QixtQkFBbUIsRUFBRSxRQUFRO2FBQzlCLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGlDQUFpQyxFQUFFLEdBQUcsRUFBRTtZQUMzQyxzQkFBc0IsQ0FBQyxrQkFBa0IsQ0FBQywyQkFBYSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQ2hFLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLFdBQVcsQ0FBQyxxQ0FBcUMsRUFBRTtnQkFDM0UsY0FBYyxFQUFFLFFBQVE7Z0JBQ3hCLG1CQUFtQixFQUFFLFFBQVE7YUFDOUIsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxvQkFBb0IsRUFBRSxHQUFHLEVBQUU7UUFDbEMsSUFBSSxZQUFvQixDQUFDO1FBRXpCLFVBQVUsQ0FBQyxHQUFHLEVBQUU7WUFDZCxRQUFRO1lBQ1IsWUFBWSxHQUFHLHNCQUFzQixDQUFDLGtCQUFrQixFQUFFLENBQUM7UUFDN0QsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsc0JBQXNCLEVBQUUsR0FBRyxFQUFFO1lBQ2hDLE9BQU87WUFDUCxNQUFNLENBQUMsWUFBWSxDQUFDLFVBQVUsRUFBRSxjQUFjLENBQUMsQ0FBQyxPQUFPLENBQUMsc0JBQXNCLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDakcsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMscUJBQXFCLEVBQUUsR0FBRyxFQUFFO1lBQy9CLE9BQU87WUFDUCxNQUFNLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsQ0FBQztZQUN4RCxNQUFNLENBQUMsWUFBWSxDQUFDLFNBQVMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1lBQ2pFLE1BQU0sQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLENBQUMsT0FBTyxDQUFDLDBCQUFTLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDNUQsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuaW1wb3J0IHsgUmVtb3ZhbFBvbGljeSwgU3RhY2sgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQge1xuICBNYXRjaCxcbiAgVGVtcGxhdGUsXG59IGZyb20gJ2F3cy1jZGstbGliL2Fzc2VydGlvbnMnO1xuaW1wb3J0IHtcbiAgTWV0cmljLFxuICBTdGF0aXN0aWMsXG59IGZyb20gJ2F3cy1jZGstbGliL2F3cy1jbG91ZHdhdGNoJztcbmltcG9ydCB7IFRhYmxlIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWR5bmFtb2RiJztcbmltcG9ydCB7IENmblNlY3JldCB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1zZWNyZXRzbWFuYWdlcic7XG5cbmltcG9ydCB7IEltcG9ydGVkQWNtQ2VydGlmaWNhdGUgfSBmcm9tICcuLi9saWIvaW1wb3J0ZWQtYWNtLWNlcnRpZmljYXRlJztcbmltcG9ydCB7IFg1MDlDZXJ0aWZpY2F0ZVBlbSB9IGZyb20gJy4uL2xpYi94NTA5LWNlcnRpZmljYXRlJztcblxubGV0IHN0YWNrOiBTdGFjaztcbmxldCBzZWNyZXRDZXJ0OiBYNTA5Q2VydGlmaWNhdGVQZW07XG5sZXQgY2VydFBhc3NwaHJhc2VJRDogc3RyaW5nO1xubGV0IGltcG9ydGVkQWNtQ2VydGlmaWNhdGU6IEltcG9ydGVkQWNtQ2VydGlmaWNhdGU7XG5cbmRlc2NyaWJlKCdJbXBvcnRlZEFjbUNlcnRpZmljYXRlJywgKCkgPT4ge1xuICBiZWZvcmVFYWNoKCgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIHN0YWNrID0gbmV3IFN0YWNrKHVuZGVmaW5lZCwgJ1N0YWNrJywgeyBlbnY6IHsgcmVnaW9uOiAndXMtd2VzdC0yJyB9IH0pO1xuXG4gICAgc2VjcmV0Q2VydCA9IG5ldyBYNTA5Q2VydGlmaWNhdGVQZW0oc3RhY2ssICdQZW0nLCB7XG4gICAgICBzdWJqZWN0OiB7IGNuOiAnU2VydmVyJyB9LFxuICAgIH0pO1xuICAgIGNlcnRQYXNzcGhyYXNlSUQgPSBzdGFjay5nZXRMb2dpY2FsSWQoc2VjcmV0Q2VydC5wYXNzcGhyYXNlLm5vZGUuZGVmYXVsdENoaWxkIGFzIENmblNlY3JldCk7XG5cbiAgICAvLyBXSEVOXG4gICAgaW1wb3J0ZWRBY21DZXJ0aWZpY2F0ZSA9IG5ldyBJbXBvcnRlZEFjbUNlcnRpZmljYXRlKHN0YWNrLCAnQWNtQ2VydCcsIHtcbiAgICAgIGNlcnQ6IHNlY3JldENlcnQuY2VydCxcbiAgICAgIGNlcnRDaGFpbjogc2VjcmV0Q2VydC5jZXJ0Q2hhaW4sXG4gICAgICBrZXk6IHNlY3JldENlcnQua2V5LFxuICAgICAgcGFzc3BocmFzZTogc2VjcmV0Q2VydC5wYXNzcGhyYXNlLFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdjcmVhdGVzIEN1c3RvbTo6UkZES19BY21JbXBvcnRlZENlcnRpZmljYXRlJywgKCkgPT4ge1xuICAgIC8vIFRIRU5cbiAgICBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spLmhhc1Jlc291cmNlUHJvcGVydGllcygnQ3VzdG9tOjpSRkRLX0FjbUltcG9ydGVkQ2VydGlmaWNhdGUnLCB7XG4gICAgICBYNTA5Q2VydGlmaWNhdGVQZW06IHtcbiAgICAgICAgQ2VydDoge1xuICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgJ1BlbScsXG4gICAgICAgICAgICAnQ2VydCcsXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgICAgS2V5OiB7XG4gICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAnUGVtJyxcbiAgICAgICAgICAgICdLZXknLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICAgIFBhc3NwaHJhc2U6IHtcbiAgICAgICAgICBSZWY6IGNlcnRQYXNzcGhyYXNlSUQsXG4gICAgICAgIH0sXG4gICAgICAgIENlcnRDaGFpbjogJycsXG4gICAgICB9LFxuICAgICAgVGFnczogW1xuICAgICAgICB7XG4gICAgICAgICAgS2V5OiAnQWNtQ2VydEltcG9ydC1GNEUyQUJGOScsXG4gICAgICAgICAgVmFsdWU6ICdmNGUyYWJmOTc0NDQzMjM0ZmRiMDk1ZmFmY2ZhOWVlMicsXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBLZXk6ICdOYW1lJyxcbiAgICAgICAgICBWYWx1ZTogJ2Y0ZTJhYmY5NzQ0NDMyMzRmZGIwOTVmYWZjZmE5ZWUyJyxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSk7XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCdjcmVhdGVzIEFXUzo6RHluYW1vREI6OlRhYmxlIGZvciBkYXRhYmFzZScsICgpID0+IHtcbiAgICB0ZXN0KCd3aXRoIFBoeXNpY2FsSUQgcGFydGl0aW9uIGtleScsICgpID0+IHtcbiAgICAgIC8vIFRIRU5cbiAgICAgIFRlbXBsYXRlLmZyb21TdGFjayhzdGFjaykuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkR5bmFtb0RCOjpUYWJsZScsIHtcbiAgICAgICAgQXR0cmlidXRlRGVmaW5pdGlvbnM6IE1hdGNoLmFycmF5V2l0aChbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQXR0cmlidXRlTmFtZTogJ1BoeXNpY2FsSWQnLFxuICAgICAgICAgICAgQXR0cmlidXRlVHlwZTogJ1MnLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0pLFxuICAgICAgICBLZXlTY2hlbWE6IE1hdGNoLmFycmF5V2l0aChbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQXR0cmlidXRlTmFtZTogJ1BoeXNpY2FsSWQnLFxuICAgICAgICAgICAgS2V5VHlwZTogJ0hBU0gnLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0pLFxuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCd3aXRoIEN1c3RvbVJlc291cmNlIHNvcnQga2V5JywgKCkgPT4ge1xuICAgICAgLy8gVEhFTlxuICAgICAgVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RHluYW1vREI6OlRhYmxlJywge1xuICAgICAgICBBdHRyaWJ1dGVEZWZpbml0aW9uczogTWF0Y2guYXJyYXlXaXRoKFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBdHRyaWJ1dGVOYW1lOiAnQ3VzdG9tUmVzb3VyY2UnLFxuICAgICAgICAgICAgQXR0cmlidXRlVHlwZTogJ1MnLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0pLFxuICAgICAgICBLZXlTY2hlbWE6IE1hdGNoLmFycmF5V2l0aChbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQXR0cmlidXRlTmFtZTogJ0N1c3RvbVJlc291cmNlJyxcbiAgICAgICAgICAgIEtleVR5cGU6ICdSQU5HRScsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSksXG4gICAgICB9KTtcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnY3JlYXRlcyBBV1M6OklBTTo6UG9saWN5JywgKCkgPT4ge1xuICAgIC8vIFRIRU5cbiAgICBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgIE1hdGNoLm9iamVjdExpa2Uoe1xuICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICdkeW5hbW9kYjpCYXRjaEdldEl0ZW0nLFxuICAgICAgICAgICAgICAnZHluYW1vZGI6R2V0UmVjb3JkcycsXG4gICAgICAgICAgICAgICdkeW5hbW9kYjpHZXRTaGFyZEl0ZXJhdG9yJyxcbiAgICAgICAgICAgICAgJ2R5bmFtb2RiOlF1ZXJ5JyxcbiAgICAgICAgICAgICAgJ2R5bmFtb2RiOkdldEl0ZW0nLFxuICAgICAgICAgICAgICAnZHluYW1vZGI6U2NhbicsXG4gICAgICAgICAgICAgICdkeW5hbW9kYjpDb25kaXRpb25DaGVja0l0ZW0nLFxuICAgICAgICAgICAgICAnZHluYW1vZGI6QmF0Y2hXcml0ZUl0ZW0nLFxuICAgICAgICAgICAgICAnZHluYW1vZGI6UHV0SXRlbScsXG4gICAgICAgICAgICAgICdkeW5hbW9kYjpVcGRhdGVJdGVtJyxcbiAgICAgICAgICAgICAgJ2R5bmFtb2RiOkRlbGV0ZUl0ZW0nLFxuICAgICAgICAgICAgICAnZHluYW1vZGI6RGVzY3JpYmVUYWJsZScsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0pLFxuICAgICAgICAgIE1hdGNoLm9iamVjdExpa2Uoe1xuICAgICAgICAgICAgQWN0aW9uOiAnZHluYW1vZGI6RGVzY3JpYmVUYWJsZScsXG4gICAgICAgICAgfSksXG4gICAgICAgICAgTWF0Y2gub2JqZWN0TGlrZSh7XG4gICAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgICAgJ3NlY3JldHNtYW5hZ2VyOkdldFNlY3JldFZhbHVlJyxcbiAgICAgICAgICAgICAgJ3NlY3JldHNtYW5hZ2VyOkRlc2NyaWJlU2VjcmV0JyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICAgICAnUGVtJyxcbiAgICAgICAgICAgICAgICAnQ2VydCcsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0pLFxuICAgICAgICAgIE1hdGNoLm9iamVjdExpa2Uoe1xuICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICdzZWNyZXRzbWFuYWdlcjpHZXRTZWNyZXRWYWx1ZScsXG4gICAgICAgICAgICAgICdzZWNyZXRzbWFuYWdlcjpEZXNjcmliZVNlY3JldCcsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgICAgJ1BlbScsXG4gICAgICAgICAgICAgICAgJ0tleScsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0pLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAnc2VjcmV0c21hbmFnZXI6R2V0U2VjcmV0VmFsdWUnLFxuICAgICAgICAgICAgICAnc2VjcmV0c21hbmFnZXI6RGVzY3JpYmVTZWNyZXQnLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICAgIFJlZjogY2VydFBhc3NwaHJhc2VJRCxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICBNYXRjaC5vYmplY3RMaWtlKHtcbiAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAnYWNtOkFkZFRhZ3NUb0NlcnRpZmljYXRlJyxcbiAgICAgICAgICAgICAgJ2FjbTpJbXBvcnRDZXJ0aWZpY2F0ZScsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgQ29uZGl0aW9uOiB7XG4gICAgICAgICAgICAgIFN0cmluZ0VxdWFsczoge1xuICAgICAgICAgICAgICAgICdhd3M6UmVxdWVzdFRhZy9BY21DZXJ0SW1wb3J0LUY0RTJBQkY5JzogJ2Y0ZTJhYmY5NzQ0NDMyMzRmZGIwOTVmYWZjZmE5ZWUyJyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBSZXNvdXJjZTogJyonLFxuICAgICAgICAgIH0pLFxuICAgICAgICAgIE1hdGNoLm9iamVjdExpa2Uoe1xuICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICdhY206RGVsZXRlQ2VydGlmaWNhdGUnLFxuICAgICAgICAgICAgICAnYWNtOkRlc2NyaWJlQ2VydGlmaWNhdGUnLFxuICAgICAgICAgICAgICAnYWNtOkdldENlcnRpZmljYXRlJyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBSZXNvdXJjZTogJyonLFxuICAgICAgICAgIH0pLFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ2N1c3RvbSByZXNvdXJjZSBsYW1iZGEgZnVuY3Rpb24nLCAoKSA9PiB7XG4gICAgdGVzdCgndXNlcyBjb3JyZWN0IGhhbmRsZXInLCAoKSA9PiB7XG4gICAgICAvLyBUSEVOXG4gICAgICBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJywge1xuICAgICAgICBIYW5kbGVyOiAneDUwOS1jZXJ0aWZpY2F0ZS5pbXBvcnRDZXJ0JyxcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgdGVzdCgndXNlcyBSRkRLIGxhbWJkYSBsYXllcicsICgpID0+IHtcbiAgICAgIC8vIFRIRU5cbiAgICAgIFRlbXBsYXRlLmZyb21TdGFjayhzdGFjaykuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkxhbWJkYTo6RnVuY3Rpb24nLCB7XG4gICAgICAgIExheWVyczogTWF0Y2guYXJyYXlXaXRoKFtcbiAgICAgICAgICBNYXRjaC5zdHJpbmdMaWtlUmVnZXhwKCdhcm46YXdzOmxhbWJkYTp1cy13ZXN0LTI6MjI0Mzc1MDA5MjkyOmxheWVyOm9wZW5zc2wtYWwyOi4qJyksXG4gICAgICAgIF0pLFxuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdzZXRzIERBVEFCQVNFIGVudmlyb25tZW50IHZhcmlhYmxlJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHRhYmxlID0gaW1wb3J0ZWRBY21DZXJ0aWZpY2F0ZS5ub2RlLmZpbmRDaGlsZCgnVGFibGUnKSBhcyBUYWJsZTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6TGFtYmRhOjpGdW5jdGlvbicsIHtcbiAgICAgICAgRW52aXJvbm1lbnQ6IHtcbiAgICAgICAgICBWYXJpYWJsZXM6IE1hdGNoLm9iamVjdExpa2Uoe1xuICAgICAgICAgICAgREFUQUJBU0U6IHN0YWNrLnJlc29sdmUodGFibGUudGFibGVOYW1lKSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgnYXBwbHlSZW1vdmFsUG9saWN5JywgKCkgPT4ge1xuICAgIHRlc3QoJ2RlZmF1bHQgUmVtb3ZhbFBvbGljeSBpcyBEZWxldGUnLCAoKSA9PiB7XG4gICAgICBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spLmhhc1Jlc291cmNlKCdDdXN0b206OlJGREtfQWNtSW1wb3J0ZWRDZXJ0aWZpY2F0ZScsIHtcbiAgICAgICAgRGVsZXRpb25Qb2xpY3k6ICdEZWxldGUnLFxuICAgICAgICBVcGRhdGVSZXBsYWNlUG9saWN5OiAnRGVsZXRlJyxcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgdGVzdCgnRGlmZmVyZW50IHBvbGljeSBjYW4gYmUgYXBwbGllZCcsICgpID0+IHtcbiAgICAgIGltcG9ydGVkQWNtQ2VydGlmaWNhdGUuYXBwbHlSZW1vdmFsUG9saWN5KFJlbW92YWxQb2xpY3kuUkVUQUlOKTtcbiAgICAgIFRlbXBsYXRlLmZyb21TdGFjayhzdGFjaykuaGFzUmVzb3VyY2UoJ0N1c3RvbTo6UkZES19BY21JbXBvcnRlZENlcnRpZmljYXRlJywge1xuICAgICAgICBEZWxldGlvblBvbGljeTogJ1JldGFpbicsXG4gICAgICAgIFVwZGF0ZVJlcGxhY2VQb2xpY3k6ICdSZXRhaW4nLFxuICAgICAgfSk7XG4gICAgfSk7XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCdtZXRyaWNEYXlzVG9FeHBpcnknLCAoKSA9PiB7XG4gICAgbGV0IG1ldHJpY0V4cGlyeTogTWV0cmljO1xuXG4gICAgYmVmb3JlRWFjaCgoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgbWV0cmljRXhwaXJ5ID0gaW1wb3J0ZWRBY21DZXJ0aWZpY2F0ZS5tZXRyaWNEYXlzVG9FeHBpcnkoKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3VzZXMgY2VydGlmaWNhdGUgQVJOJywgKCkgPT4ge1xuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KG1ldHJpY0V4cGlyeS5kaW1lbnNpb25zPy5DZXJ0aWZpY2F0ZUFybikudG9FcXVhbChpbXBvcnRlZEFjbUNlcnRpZmljYXRlLmNlcnRpZmljYXRlQXJuKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3VzZXMgY29ycmVjdCBtZXRyaWMnLCAoKSA9PiB7XG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3QobWV0cmljRXhwaXJ5Lm1ldHJpY05hbWUpLnRvRXF1YWwoJ0RheXNUb0V4cGlyeScpO1xuICAgICAgZXhwZWN0KG1ldHJpY0V4cGlyeS5uYW1lc3BhY2UpLnRvRXF1YWwoJ0FXUy9DZXJ0aWZpY2F0ZU1hbmFnZXInKTtcbiAgICAgIGV4cGVjdChtZXRyaWNFeHBpcnkuc3RhdGlzdGljKS50b0VxdWFsKFN0YXRpc3RpYy5NSU5JTVVNKTtcbiAgICB9KTtcbiAgfSk7XG59KTtcbiJdfQ==