"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_autoscaling_1 = require("aws-cdk-lib/aws-autoscaling");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const constructs_1 = require("constructs");
const lib_1 = require("../lib");
const tag_helpers_1 = require("./tag-helpers");
const test_helper_1 = require("./test-helper");
let app;
let infraStack;
let hmStack;
let wfStack;
let vpc;
let healthMonitor;
class TestMonitorableFleet extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const fleet = new aws_autoscaling_1.AutoScalingGroup(this, 'ASG', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc: props.vpc,
            minCapacity: props.minCapacity,
        });
        this.connections = new aws_ec2_1.Connections();
        this.targetCapacity = parseInt(fleet.node.defaultChild.maxSize, 10);
        this.targetScope = this;
        this.targetToMonitor = fleet;
        this.targetCapacityMetric = new aws_cloudwatch_1.Metric({
            namespace: 'AWS/AutoScaling',
            metricName: 'GroupDesiredCapacity',
            dimensionsMap: {
                AutoScalingGroupName: fleet.autoScalingGroupName,
            },
            label: 'GroupDesiredCapacity',
        });
        this.targetUpdatePolicy = new aws_iam_1.Policy(this, 'ASGUpdatePolicy', {
            statements: [new aws_iam_1.PolicyStatement({
                    actions: ['autoscaling:UpdateAutoScalingGroup'],
                    resources: [fleet.autoScalingGroupArn],
                })],
        });
    }
}
describe('HealthMonitor', () => {
    beforeEach(() => {
        app = new aws_cdk_lib_1.App();
        infraStack = new aws_cdk_lib_1.Stack(app, 'infraStack');
        hmStack = new aws_cdk_lib_1.Stack(app, 'hmStack');
        wfStack = new aws_cdk_lib_1.Stack(app, 'wfStack');
        vpc = new aws_ec2_1.Vpc(infraStack, 'VPC');
    });
    test('validating default health monitor properties', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        // THEN
        assertions_1.Template.fromStack(hmStack).resourceCountIs('AWS::ElasticLoadBalancingV2::LoadBalancer', 0);
        assertions_1.Template.fromStack(hmStack).hasResourceProperties('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {
                        Action: 'kms:*',
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':root',
                                    ],
                                ],
                            },
                        },
                        Resource: '*',
                    },
                    {
                        Action: [
                            'kms:Decrypt',
                            'kms:GenerateDataKey',
                        ],
                        Effect: 'Allow',
                        Principal: {
                            Service: 'cloudwatch.amazonaws.com',
                        },
                        Resource: '*',
                    },
                ],
            },
            Description: `This key is used to encrypt SNS messages for ${aws_cdk_lib_1.Names.uniqueId(healthMonitor)}.`,
            EnableKeyRotation: true,
        });
        assertions_1.Template.fromStack(hmStack).hasResourceProperties('AWS::SNS::TopicPolicy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'sns:Publish',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'cloudwatch.amazonaws.com',
                        },
                        Resource: {
                            Ref: hmStack.getLogicalId(healthMonitor.unhealthyFleetActionTopic.node.defaultChild),
                        },
                        Sid: '0',
                    },
                ],
            },
            Topics: [
                {
                    Ref: hmStack.getLogicalId(healthMonitor.unhealthyFleetActionTopic.node.defaultChild),
                },
            ],
        });
        assertions_1.Template.fromStack(hmStack).hasResourceProperties('AWS::SNS::Topic', {
            KmsMasterKeyId: {
                'Fn::GetAtt': [
                    `${hmStack.getLogicalId(healthMonitor.node.findChild('SNSEncryptionKey').node.defaultChild)}`,
                    'Arn',
                ],
            },
        });
        assertions_1.Template.fromStack(hmStack).hasResourceProperties('AWS::Lambda::Function', {});
        assertions_1.Template.fromStack(hmStack).hasResourceProperties('AWS::SNS::Subscription', {
            Protocol: 'lambda',
            TopicArn: {
                Ref: `${infraStack.getLogicalId(healthMonitor.node.findChild('UnhealthyFleetTopic').node.defaultChild)}`,
            },
            Endpoint: {
                'Fn::GetAtt': [
                    'unhealthyFleetTermination28bccf6aaa76478c9239e2f5bcc0254c8C612A5E',
                    'Arn',
                ],
            },
        });
    });
    test('validating health monitor properties while passing a key', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
            encryptionKey: aws_kms_1.Key.fromKeyArn(hmStack, 'importedKey', 'arn:aws:kms:us-west-2:123456789012:key/testarn'),
        });
        // THEN
        assertions_1.Template.fromStack(hmStack).resourceCountIs('AWS::ElasticLoadBalancingV2::LoadBalancer', 0);
        assertions_1.Template.fromStack(hmStack).resourceCountIs('AWS::KMS::Key', 0);
        assertions_1.Template.fromStack(hmStack).hasResourceProperties('AWS::SNS::Topic', {
            KmsMasterKeyId: 'arn:aws:kms:us-west-2:123456789012:key/testarn',
        });
        assertions_1.Template.fromStack(hmStack).hasResourceProperties('AWS::Lambda::Function', {});
        assertions_1.Template.fromStack(hmStack).hasResourceProperties('AWS::SNS::Subscription', {
            Protocol: 'lambda',
            TopicArn: {
                Ref: `${infraStack.getLogicalId(healthMonitor.node.findChild('UnhealthyFleetTopic').node.defaultChild)}`,
            },
            Endpoint: {
                'Fn::GetAtt': [
                    'unhealthyFleetTermination28bccf6aaa76478c9239e2f5bcc0254c8C612A5E',
                    'Arn',
                ],
            },
        });
    });
    test('validating the target with default health config', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        assertions_1.Template.fromStack(wfStack).hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {});
        assertions_1.Template.fromStack(hmStack).hasResourceProperties('AWS::EC2::SecurityGroup', assertions_1.Match.not({
            SecurityGroupIngress: assertions_1.Match.arrayWith([{
                    CidrIp: '0.0.0.0/0',
                    FromPort: 8081,
                    IpProtocol: 'tcp',
                    ToPort: 8081,
                }]),
        }));
        assertions_1.Template.fromStack(wfStack).hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '8081',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        });
        assertions_1.Template.fromStack(wfStack).hasResourceProperties('AWS::CloudWatch::Alarm', {});
    });
    test('validating the target with custom health config', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {
            port: 7171,
        });
        // THEN
        assertions_1.Template.fromStack(wfStack).hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {});
        assertions_1.Template.fromStack(wfStack).hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '7171',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        });
        assertions_1.Template.fromStack(wfStack).hasResourceProperties('AWS::CloudWatch::Alarm', {});
    });
    test('2 ASG gets registered to same LB', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, { port: 7171 });
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
        });
        healthMonitor.registerFleet(fleet2, { port: 7171 });
        // THEN
        test_helper_1.resourcePropertiesCountIs(hmStack, 'AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assertions_1.Match.arrayWith([
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ]),
            Scheme: 'internal',
        }, 1);
        assertions_1.Template.fromStack(wfStack).resourceCountIs('AWS::ElasticLoadBalancingV2::Listener', 2);
        assertions_1.Template.fromStack(wfStack).hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '7171',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        });
        assertions_1.Template.fromStack(wfStack).hasResourceProperties('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanThreshold',
            EvaluationPeriods: 8,
            ActionsEnabled: true,
            DatapointsToAlarm: 8,
            Threshold: 0,
            TreatMissingData: 'notBreaching',
        });
        assertions_1.Template.fromStack(wfStack).hasResourceProperties('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanThreshold',
            EvaluationPeriods: 1,
            ActionsEnabled: true,
            DatapointsToAlarm: 1,
            Threshold: 35,
            TreatMissingData: 'notBreaching',
        });
    });
    test('validating LB target limit', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'targets-per-application-load-balancer',
                    max: 50,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
            minCapacity: 50,
        });
        healthMonitor.registerFleet(fleet, {});
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
            minCapacity: 50,
        });
        healthMonitor.registerFleet(fleet2, {});
        // THEN
        test_helper_1.resourcePropertiesCountIs(hmStack, 'AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assertions_1.Match.arrayWith([
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ]),
            Scheme: 'internal',
            Type: 'application',
        }, 2);
        assertions_1.Template.fromStack(wfStack).resourceCountIs('AWS::ElasticLoadBalancingV2::Listener', 2);
        assertions_1.Template.fromStack(wfStack).hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 8081,
            Protocol: 'HTTP',
        });
    });
    test('validating LB listener limit', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'listeners-per-application-load-balancer',
                    max: 1,
                }, {
                    name: 'target-groups-per-action-on-application-load-balancer',
                    max: 1,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
        });
        healthMonitor.registerFleet(fleet2, {});
        // THEN
        test_helper_1.resourcePropertiesCountIs(hmStack, 'AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assertions_1.Match.arrayWith([
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ]),
            Scheme: 'internal',
            Type: 'application',
        }, 2);
        assertions_1.Template.fromStack(wfStack).resourceCountIs('AWS::ElasticLoadBalancingV2::Listener', 2);
        assertions_1.Template.fromStack(wfStack).hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 8081,
            Protocol: 'HTTP',
        });
    });
    test('validating target group limit per lb', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'target-groups-per-application-load-balancer',
                    max: 1,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
        });
        healthMonitor.registerFleet(fleet2, {});
        // THEN
        test_helper_1.resourcePropertiesCountIs(hmStack, 'AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Scheme: 'internal',
            Type: 'application',
        }, 2);
        assertions_1.Template.fromStack(wfStack).resourceCountIs('AWS::ElasticLoadBalancingV2::Listener', 2);
        assertions_1.Template.fromStack(wfStack).hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 8081,
            Protocol: 'HTTP',
        });
    });
    test('validating target limit exhaustion', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'targets-per-application-load-balancer',
                    max: 1,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
            minCapacity: 2,
        });
        expect(() => {
            healthMonitor.registerFleet(fleet, {});
        }).toThrowError(/AWS service limit \"targets-per-application-load-balancer\" reached. Limit: 1/);
    });
    test('validating deletion protection', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            deletionProtection: false,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        assertions_1.Template.fromStack(hmStack).hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', assertions_1.Match.not(assertions_1.Match.objectLike({
            Properties: {
                LoadBalancerAttributes: assertions_1.Match.arrayWith([
                    {
                        Key: 'deletion_protection.enabled',
                        Value: 'true',
                    },
                ]),
                Scheme: assertions_1.Match.absent,
                Type: assertions_1.Match.absent,
            },
        })));
    });
    test('drop invalid http header fields enabled', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        assertions_1.Template.fromStack(hmStack).hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assertions_1.Match.arrayWith([
                {
                    Key: 'routing.http.drop_invalid_header_fields.enabled',
                    Value: 'true',
                },
            ]),
        });
    });
    test('specifying a subnet', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            vpcSubnets: {
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            },
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        // Make sure it has the public subnets
        assertions_1.Template.fromStack(hmStack).hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Subnets: [
                { 'Fn::ImportValue': assertions_1.Match.stringLikeRegexp('.*PublicSubnet.*') },
                { 'Fn::ImportValue': assertions_1.Match.stringLikeRegexp('.*PublicSubnet.*') },
            ],
        });
        // Make sure the private subnets aren't present
        assertions_1.Template.fromStack(hmStack).hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Subnets: [
                { 'Fn::ImportValue': assertions_1.Match.not(assertions_1.Match.stringLikeRegexp('.*PrivateSubnet.*')) },
                { 'Fn::ImportValue': assertions_1.Match.not(assertions_1.Match.stringLikeRegexp('.*PrivateSubnet.*')) },
            ],
        });
    });
    test('specifying a security group', () => {
        // GIVEN
        const securityGroup = new aws_ec2_1.SecurityGroup(infraStack, 'LBSecurityGroup', { vpc });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            securityGroup,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        // Make sure it has the security group
        assertions_1.Template.fromStack(hmStack).hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            SecurityGroups: assertions_1.Match.arrayWith([
                hmStack.resolve(securityGroup.securityGroupId),
            ]),
        });
        // HealthMonitor should not create its own security group
        assertions_1.Template.fromStack(hmStack).resourceCountIs('AWS::EC2::SecurityGroup', 0);
    });
    describe('tagging', () => {
        tag_helpers_1.testConstructTags({
            constructName: 'HealthMonitor',
            createConstruct: () => {
                // GIVEN
                const fleetStack = new aws_cdk_lib_1.Stack(app, 'FleetStack');
                const fleet = new TestMonitorableFleet(fleetStack, 'workerFleet', {
                    vpc,
                });
                // WHEN
                healthMonitor = new lib_1.HealthMonitor(hmStack, 'HealthMonitor', {
                    vpc,
                });
                healthMonitor.registerFleet(fleet, {});
                return hmStack;
            },
            resourceTypeCounts: {
                'AWS::KMS::Key': 1,
                'AWS::SNS::Topic': 1,
                'AWS::ElasticLoadBalancingV2::LoadBalancer': 1,
                'AWS::EC2::SecurityGroup': 1,
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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