"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const exporting_log_group_1 = require("../lib/exporting-log-group");
test('default exporting log group is created correctly', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const bucket = new aws_s3_1.Bucket(stack, 'DestinationBucket', {
        bucketName: 'test-bucket',
    });
    // WHEN
    new exporting_log_group_1.ExportingLogGroup(stack, 'ExportingLogGroup', {
        bucketName: bucket.bucketName,
        logGroupName: 'logGroup',
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
        ServiceToken: {
            'Fn::GetAtt': [
                'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8aFD4BFC8A',
                'Arn',
            ],
        },
        LogGroupName: 'logGroup',
        RetentionInDays: 3,
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'logs:DescribeExportTasks',
                    Effect: 'Allow',
                    Resource: '*',
                },
                {
                    Action: 'logs:CreateExportTask',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':logs:',
                                {
                                    Ref: 'AWS::Region',
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':log-group:',
                                {
                                    'Fn::GetAtt': [
                                        'ExportingLogGroupLogRetention80FFADE8',
                                        'LogGroupName',
                                    ],
                                },
                                ':*',
                            ],
                        ],
                    },
                },
            ],
        },
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::CloudWatch::Alarm', {
        ComparisonOperator: 'GreaterThanOrEqualToThreshold',
        EvaluationPeriods: 1,
        Dimensions: [
            {
                Name: 'FunctionName',
                Value: {
                    Ref: 'LogGroupExporter6382448ce4b242e9b14fa0a9ccdb198eC741F553',
                },
            },
        ],
        MetricName: 'Errors',
        Namespace: 'AWS/Lambda',
        Period: 300,
        Statistic: 'Sum',
        Threshold: 1,
    });
    assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::Events::Rule', {
        ScheduleExpression: 'rate(1 hour)',
        State: 'ENABLED',
        Targets: [
            {
                Arn: {
                    'Fn::GetAtt': [
                        'LogGroupExporter6382448ce4b242e9b14fa0a9ccdb198eC741F553',
                        'Arn',
                    ],
                },
                Id: 'Target0',
                Input: {
                    'Fn::Join': [
                        '',
                        [
                            '{\"BucketName\":\"',
                            {
                                Ref: 'DestinationBucket4BECDB47',
                            },
                            '\",\"ExportFrequencyInHours\":1,\"LogGroupName\":\"logGroup\",\"RetentionInHours\":72}',
                        ],
                    ],
                },
            },
        ],
    });
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::Lambda::Function', 2);
});
test('custom set retention is created correctly', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const bucket = new aws_s3_1.Bucket(stack, 'DestinationBucket', {
        bucketName: 'test-bucket',
    });
    // WHEN
    new exporting_log_group_1.ExportingLogGroup(stack, 'ExportingLogGroup', {
        bucketName: bucket.bucketName,
        logGroupName: 'logGroup',
        retention: aws_logs_1.RetentionDays.ONE_WEEK,
    });
    // THEN
    assertions_1.Template.fromStack(stack).hasResourceProperties('Custom::LogRetention', {
        ServiceToken: {
            'Fn::GetAtt': [
                'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8aFD4BFC8A',
                'Arn',
            ],
        },
        LogGroupName: 'logGroup',
        RetentionInDays: 7,
    });
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::Lambda::Function', 2);
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::CloudWatch::Alarm', 1);
    assertions_1.Template.fromStack(stack).resourceCountIs('AWS::Events::Rule', 1);
});
//# sourceMappingURL=data:application/json;base64,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