"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
describe('CloudWatchAgent', () => {
    let stack;
    let vpc;
    let cloudWatchConfig;
    beforeEach(() => {
        stack = new aws_cdk_lib_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        // Generate CloudWatch Agent configuration JSON
        const configBuilder = new lib_1.CloudWatchConfigBuilder();
        cloudWatchConfig = configBuilder.generateCloudWatchConfiguration();
    });
    test('creates an SSM parameter containing the configuration', () => {
        // GIVEN
        const host = new aws_ec2_1.Instance(stack, 'Instance', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc,
        });
        // WHEN
        new lib_1.CloudWatchAgent(stack, 'testResource', {
            cloudWatchConfig,
            host,
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::SSM::Parameter', {
            Type: 'String',
            Value: cloudWatchConfig,
        });
    });
    test('creates an IAM policy to access the SSM parameter, CDK asset bucket, and CloudWatch agent bucket', () => {
        // GIVEN
        const host = new aws_ec2_1.Instance(stack, 'Instance', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc,
        });
        // WHEN
        new lib_1.CloudWatchAgent(stack, 'testResource', {
            cloudWatchConfig,
            host,
        });
        // THEN
        assertions_1.Template.fromStack(stack).hasResourceProperties('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':s3:::',
                                        { 'Fn::Sub': asset_constants_1.CWA_ASSET_LINUX.Bucket },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':s3:::',
                                        { 'Fn::Sub': asset_constants_1.CWA_ASSET_LINUX.Bucket },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                    {
                        Action: [
                            'ssm:DescribeParameters',
                            'ssm:GetParameters',
                            'ssm:GetParameter',
                            'ssm:GetParameterHistory',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':ssm:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':parameter/',
                                    { Ref: 'StringParameter472EED0E' },
                                ],
                            ],
                        },
                    },
                    {
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:aws:s3:::amazoncloudwatch-agent-',
                                        { Ref: 'AWS::Region' },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:aws:s3:::amazoncloudwatch-agent-',
                                        { Ref: 'AWS::Region' },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                    {
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:aws:s3:::rfdk-external-dependencies-',
                                        { Ref: 'AWS::Region' },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:aws:s3:::rfdk-external-dependencies-',
                                        { Ref: 'AWS::Region' },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
            PolicyName: 'InstanceInstanceRoleDefaultPolicy4ACE9290',
            Roles: [{ Ref: 'InstanceInstanceRoleE9785DE5' }],
        });
    });
    test.each([
        [' -i', undefined],
        [' -i', true],
        ['', false],
    ])('adds user data commands to fetch and execute the script (linux). installFlag: %s shouldInstallAgent: %p', (installFlag, shouldInstallAgent) => {
        // GIVEN
        const host = new aws_ec2_1.Instance(stack, 'Instance', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc,
        });
        // WHEN
        new lib_1.CloudWatchAgent(stack, 'testResource', {
            cloudWatchConfig,
            host,
            shouldInstallAgent,
        });
        // THEN
        const userData = stack.resolve(host.userData.render());
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    `#!/bin/bash\nmkdir -p $(dirname '/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh')\naws s3 cp 's3://`,
                    { 'Fn::Sub': asset_constants_1.CWA_ASSET_LINUX.Bucket },
                    `/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh' '/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh'\nset -e\nchmod +x '/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh'\n'/tmp/${asset_constants_1.CWA_ASSET_LINUX.Key}.sh'${installFlag} `,
                    { Ref: 'AWS::Region' },
                    ' ',
                    { Ref: 'StringParameter472EED0E' },
                ],
            ],
        });
    });
    test.each([
        [' -i', undefined],
        [' -i', true],
        ['', false],
    ])('adds user data commands to fetch and execute the script (windows). installFlag: %s shouldInstallAgent: %p', (installFlag, shouldInstallAgent) => {
        // GIVEN
        const host = new aws_ec2_1.Instance(stack, 'Instance', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.WindowsImage(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
            vpc,
        });
        // WHEN
        new lib_1.CloudWatchAgent(stack, 'testResource', {
            cloudWatchConfig,
            host,
            shouldInstallAgent,
        });
        // THEN
        const userData = stack.resolve(host.userData.render());
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    `<powershell>mkdir (Split-Path -Path 'C:/temp/${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1' ) -ea 0\nRead-S3Object -BucketName '`,
                    { 'Fn::Sub': asset_constants_1.CWA_ASSET_WINDOWS.Bucket },
                    `' -key '${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1' -file 'C:/temp/${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1' -ErrorAction Stop\n&'C:/temp/${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1'${installFlag} `,
                    { Ref: 'AWS::Region' },
                    ' ',
                    { Ref: 'StringParameter472EED0E' },
                    `\nif (!$?) { Write-Error 'Failed to execute the file \"C:/temp/${asset_constants_1.CWA_ASSET_WINDOWS.Key}.ps1\"' -ErrorAction Stop }</powershell>`,
                ],
            ],
        });
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2xvdWR3YXRjaC1hZ2VudC50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY2xvdWR3YXRjaC1hZ2VudC50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7O0dBR0c7O0FBRUgsNkNBQWtDO0FBQ2xDLHVEQUVnQztBQUNoQyxpREFVNkI7QUFDN0IsZ0NBR2dCO0FBQ2hCLHVEQUcyQjtBQUUzQixRQUFRLENBQUMsaUJBQWlCLEVBQUUsR0FBRyxFQUFFO0lBQy9CLElBQUksS0FBWSxDQUFDO0lBQ2pCLElBQUksR0FBUSxDQUFDO0lBQ2IsSUFBSSxnQkFBd0IsQ0FBQztJQUU3QixVQUFVLENBQUMsR0FBRyxFQUFFO1FBQ2QsS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO1FBQ3BCLEdBQUcsR0FBRyxJQUFJLGFBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFNUIsK0NBQStDO1FBQy9DLE1BQU0sYUFBYSxHQUFHLElBQUksNkJBQXVCLEVBQUUsQ0FBQztRQUNwRCxnQkFBZ0IsR0FBRyxhQUFhLENBQUMsK0JBQStCLEVBQUUsQ0FBQztJQUNyRSxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyx1REFBdUQsRUFBRSxHQUFHLEVBQUU7UUFDakUsUUFBUTtRQUNSLE1BQU0sSUFBSSxHQUFHLElBQUksa0JBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQzNDLFlBQVksRUFBRSxzQkFBWSxDQUFDLEVBQUUsQ0FBQyx1QkFBYSxDQUFDLEVBQUUsRUFBRSxzQkFBWSxDQUFDLEtBQUssQ0FBQztZQUNuRSxZQUFZLEVBQUUsSUFBSSwwQkFBZ0IsQ0FBQztnQkFDakMsVUFBVSxFQUFFLCtCQUFxQixDQUFDLGNBQWM7YUFDakQsQ0FBQztZQUNGLEdBQUc7U0FDSixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsSUFBSSxxQkFBZSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7WUFDekMsZ0JBQWdCO1lBQ2hCLElBQUk7U0FDTCxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUMscUJBQXFCLENBQUMscUJBQXFCLEVBQUU7WUFDckUsSUFBSSxFQUFFLFFBQVE7WUFDZCxLQUFLLEVBQUUsZ0JBQWdCO1NBQ3hCLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLGtHQUFrRyxFQUFFLEdBQUcsRUFBRTtRQUM1RyxRQUFRO1FBQ1IsTUFBTSxJQUFJLEdBQUcsSUFBSSxrQkFBUSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDM0MsWUFBWSxFQUFFLHNCQUFZLENBQUMsRUFBRSxDQUFDLHVCQUFhLENBQUMsRUFBRSxFQUFFLHNCQUFZLENBQUMsS0FBSyxDQUFDO1lBQ25FLFlBQVksRUFBRSxJQUFJLDBCQUFnQixDQUFDO2dCQUNqQyxVQUFVLEVBQUUsK0JBQXFCLENBQUMsY0FBYzthQUNqRCxDQUFDO1lBQ0YsR0FBRztTQUNKLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLHFCQUFlLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtZQUN6QyxnQkFBZ0I7WUFDaEIsSUFBSTtTQUNMLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxrQkFBa0IsRUFBRTtZQUNsRSxjQUFjLEVBQUU7Z0JBQ2QsU0FBUyxFQUFFO29CQUNUO3dCQUNFLE1BQU0sRUFBRTs0QkFDTixlQUFlOzRCQUNmLGVBQWU7NEJBQ2YsVUFBVTt5QkFDWDt3QkFDRCxNQUFNLEVBQUUsT0FBTzt3QkFDZixRQUFRLEVBQUU7NEJBQ1I7Z0NBQ0UsVUFBVSxFQUFFO29DQUNWLEVBQUU7b0NBQ0Y7d0NBQ0UsTUFBTTt3Q0FDTixFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRTt3Q0FDekIsUUFBUTt3Q0FDUixFQUFFLFNBQVMsRUFBRSxpQ0FBZSxDQUFDLE1BQU0sRUFBRTtxQ0FDdEM7aUNBQ0Y7NkJBQ0Y7NEJBQ0Q7Z0NBQ0UsVUFBVSxFQUFFO29DQUNWLEVBQUU7b0NBQ0Y7d0NBQ0UsTUFBTTt3Q0FDTixFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRTt3Q0FDekIsUUFBUTt3Q0FDUixFQUFFLFNBQVMsRUFBRSxpQ0FBZSxDQUFDLE1BQU0sRUFBRTt3Q0FDckMsSUFBSTtxQ0FDTDtpQ0FDRjs2QkFDRjt5QkFDRjtxQkFDRjtvQkFDRDt3QkFDRSxNQUFNLEVBQUU7NEJBQ04sd0JBQXdCOzRCQUN4QixtQkFBbUI7NEJBQ25CLGtCQUFrQjs0QkFDbEIseUJBQXlCO3lCQUMxQjt3QkFDRCxNQUFNLEVBQUUsT0FBTzt3QkFDZixRQUFRLEVBQUU7NEJBQ1IsVUFBVSxFQUFFO2dDQUNWLEVBQUU7Z0NBQ0Y7b0NBQ0UsTUFBTTtvQ0FDTixFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRTtvQ0FDekIsT0FBTztvQ0FDUCxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7b0NBQ3RCLEdBQUc7b0NBQ0gsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUU7b0NBQ3pCLGFBQWE7b0NBQ2IsRUFBRSxHQUFHLEVBQUUseUJBQXlCLEVBQUU7aUNBQ25DOzZCQUNGO3lCQUNGO3FCQUNGO29CQUNEO3dCQUNFLE1BQU0sRUFBRTs0QkFDTixlQUFlOzRCQUNmLGVBQWU7NEJBQ2YsVUFBVTt5QkFDWDt3QkFDRCxNQUFNLEVBQUUsT0FBTzt3QkFDZixRQUFRLEVBQUU7NEJBQ1I7Z0NBQ0UsVUFBVSxFQUFFO29DQUNWLEVBQUU7b0NBQ0Y7d0NBQ0Usc0NBQXNDO3dDQUN0QyxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7cUNBQ3ZCO2lDQUNGOzZCQUNGOzRCQUNEO2dDQUNFLFVBQVUsRUFBRTtvQ0FDVixFQUFFO29DQUNGO3dDQUNFLHNDQUFzQzt3Q0FDdEMsRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFO3dDQUN0QixJQUFJO3FDQUNMO2lDQUNGOzZCQUNGO3lCQUNGO3FCQUNGO29CQUNEO3dCQUNFLE1BQU0sRUFBRTs0QkFDTixlQUFlOzRCQUNmLGVBQWU7NEJBQ2YsVUFBVTt5QkFDWDt3QkFDRCxNQUFNLEVBQUUsT0FBTzt3QkFDZixRQUFRLEVBQUU7NEJBQ1I7Z0NBQ0UsVUFBVSxFQUFFO29DQUNWLEVBQUU7b0NBQ0Y7d0NBQ0UsMENBQTBDO3dDQUMxQyxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7cUNBQ3ZCO2lDQUNGOzZCQUNGOzRCQUNEO2dDQUNFLFVBQVUsRUFBRTtvQ0FDVixFQUFFO29DQUNGO3dDQUNFLDBDQUEwQzt3Q0FDMUMsRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFO3dDQUN0QixJQUFJO3FDQUNMO2lDQUNGOzZCQUNGO3lCQUNGO3FCQUNGO2lCQUNGO2dCQUNELE9BQU8sRUFBRSxZQUFZO2FBQ3RCO1lBQ0QsVUFBVSxFQUFFLDJDQUEyQztZQUN2RCxLQUFLLEVBQUUsQ0FBRSxFQUFFLEdBQUcsRUFBRSw4QkFBOEIsRUFBRSxDQUFFO1NBQ25ELENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLElBQUksQ0FBQztRQUNSLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQztRQUNsQixDQUFDLEtBQUssRUFBRSxJQUFJLENBQUM7UUFDYixDQUFDLEVBQUUsRUFBRSxLQUFLLENBQUM7S0FDWixDQUFDLENBQUMseUdBQXlHLEVBQUUsQ0FBQyxXQUFtQixFQUFFLGtCQUE0QixFQUFFLEVBQUU7UUFDbEssUUFBUTtRQUNSLE1BQU0sSUFBSSxHQUFHLElBQUksa0JBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQzNDLFlBQVksRUFBRSxzQkFBWSxDQUFDLEVBQUUsQ0FBQyx1QkFBYSxDQUFDLEVBQUUsRUFBRSxzQkFBWSxDQUFDLEtBQUssQ0FBQztZQUNuRSxZQUFZLEVBQUUsSUFBSSwwQkFBZ0IsQ0FBQztnQkFDakMsVUFBVSxFQUFFLCtCQUFxQixDQUFDLGNBQWM7YUFDakQsQ0FBQztZQUNGLEdBQUc7U0FDSixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsSUFBSSxxQkFBZSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7WUFDekMsZ0JBQWdCO1lBQ2hCLElBQUk7WUFDSixrQkFBa0I7U0FDbkIsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO1FBQ3ZELE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxhQUFhLENBQUM7WUFDN0IsVUFBVSxFQUFFO2dCQUNWLEVBQUU7Z0JBQ0Y7b0JBQ0UseUNBQXlDLGlDQUFlLENBQUMsR0FBRyx5QkFBeUI7b0JBQ3JGLEVBQUUsU0FBUyxFQUFFLGlDQUFlLENBQUMsTUFBTSxFQUFFO29CQUNyQyxJQUFJLGlDQUFlLENBQUMsR0FBRyxjQUFjLGlDQUFlLENBQUMsR0FBRyxnQ0FBZ0MsaUNBQWUsQ0FBQyxHQUFHLGVBQWUsaUNBQWUsQ0FBQyxHQUFHLE9BQU8sV0FBVyxHQUFHO29CQUNsSyxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7b0JBQ3RCLEdBQUc7b0JBQ0gsRUFBRSxHQUFHLEVBQUUseUJBQXlCLEVBQUU7aUJBQ25DO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxJQUFJLENBQUM7UUFDUixDQUFDLEtBQUssRUFBRSxTQUFTLENBQUM7UUFDbEIsQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDO1FBQ2IsQ0FBQyxFQUFFLEVBQUUsS0FBSyxDQUFDO0tBQ1osQ0FBQyxDQUFDLDJHQUEyRyxFQUFFLENBQUMsV0FBbUIsRUFBRSxrQkFBNEIsRUFBRSxFQUFFO1FBQ3BLLFFBQVE7UUFDUixNQUFNLElBQUksR0FBRyxJQUFJLGtCQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtZQUMzQyxZQUFZLEVBQUUsc0JBQVksQ0FBQyxFQUFFLENBQUMsdUJBQWEsQ0FBQyxFQUFFLEVBQUUsc0JBQVksQ0FBQyxLQUFLLENBQUM7WUFDbkUsWUFBWSxFQUFFLElBQUksc0JBQVksQ0FBQyx3QkFBYyxDQUFDLHFDQUFxQyxDQUFDO1lBQ3BGLEdBQUc7U0FDSixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsSUFBSSxxQkFBZSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7WUFDekMsZ0JBQWdCO1lBQ2hCLElBQUk7WUFDSixrQkFBa0I7U0FDbkIsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO1FBQ3ZELE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxhQUFhLENBQUM7WUFDN0IsVUFBVSxFQUFFO2dCQUNWLEVBQUU7Z0JBQ0Y7b0JBQ0UsZ0RBQWdELG1DQUFpQixDQUFDLEdBQUcsNENBQTRDO29CQUNqSCxFQUFFLFNBQVMsRUFBRSxtQ0FBaUIsQ0FBQyxNQUFNLEVBQUU7b0JBQ3ZDLFdBQVcsbUNBQWlCLENBQUMsR0FBRyx3QkFBd0IsbUNBQWlCLENBQUMsR0FBRyxzQ0FBc0MsbUNBQWlCLENBQUMsR0FBRyxRQUFRLFdBQVcsR0FBRztvQkFDOUosRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFO29CQUN0QixHQUFHO29CQUNILEVBQUUsR0FBRyxFQUFFLHlCQUF5QixFQUFFO29CQUNsQyxrRUFBa0UsbUNBQWlCLENBQUMsR0FBRywwQ0FBMEM7aUJBQ2xJO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuaW1wb3J0IHtTdGFja30gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHtcbiAgVGVtcGxhdGUsXG59IGZyb20gJ2F3cy1jZGstbGliL2Fzc2VydGlvbnMnO1xuaW1wb3J0IHtcbiAgQW1hem9uTGludXhHZW5lcmF0aW9uLFxuICBBbWF6b25MaW51eEltYWdlLFxuICBJbnN0YW5jZSxcbiAgSW5zdGFuY2VDbGFzcyxcbiAgSW5zdGFuY2VTaXplLFxuICBJbnN0YW5jZVR5cGUsXG4gIFZwYyxcbiAgV2luZG93c0ltYWdlLFxuICBXaW5kb3dzVmVyc2lvbixcbn0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWVjMic7XG5pbXBvcnQge1xuICBDbG91ZFdhdGNoQWdlbnQsXG4gIENsb3VkV2F0Y2hDb25maWdCdWlsZGVyLFxufSBmcm9tICcuLi9saWInO1xuaW1wb3J0IHtcbiAgQ1dBX0FTU0VUX0xJTlVYLFxuICBDV0FfQVNTRVRfV0lORE9XUyxcbn0gZnJvbSAnLi9hc3NldC1jb25zdGFudHMnO1xuXG5kZXNjcmliZSgnQ2xvdWRXYXRjaEFnZW50JywgKCkgPT4ge1xuICBsZXQgc3RhY2s6IFN0YWNrO1xuICBsZXQgdnBjOiBWcGM7XG4gIGxldCBjbG91ZFdhdGNoQ29uZmlnOiBzdHJpbmc7XG5cbiAgYmVmb3JlRWFjaCgoKSA9PiB7XG4gICAgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICB2cGMgPSBuZXcgVnBjKHN0YWNrLCAnVlBDJyk7XG5cbiAgICAvLyBHZW5lcmF0ZSBDbG91ZFdhdGNoIEFnZW50IGNvbmZpZ3VyYXRpb24gSlNPTlxuICAgIGNvbnN0IGNvbmZpZ0J1aWxkZXIgPSBuZXcgQ2xvdWRXYXRjaENvbmZpZ0J1aWxkZXIoKTtcbiAgICBjbG91ZFdhdGNoQ29uZmlnID0gY29uZmlnQnVpbGRlci5nZW5lcmF0ZUNsb3VkV2F0Y2hDb25maWd1cmF0aW9uKCk7XG4gIH0pO1xuXG4gIHRlc3QoJ2NyZWF0ZXMgYW4gU1NNIHBhcmFtZXRlciBjb250YWluaW5nIHRoZSBjb25maWd1cmF0aW9uJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgaG9zdCA9IG5ldyBJbnN0YW5jZShzdGFjaywgJ0luc3RhbmNlJywge1xuICAgICAgaW5zdGFuY2VUeXBlOiBJbnN0YW5jZVR5cGUub2YoSW5zdGFuY2VDbGFzcy5UMiwgSW5zdGFuY2VTaXplLkxBUkdFKSxcbiAgICAgIG1hY2hpbmVJbWFnZTogbmV3IEFtYXpvbkxpbnV4SW1hZ2Uoe1xuICAgICAgICBnZW5lcmF0aW9uOiBBbWF6b25MaW51eEdlbmVyYXRpb24uQU1BWk9OX0xJTlVYXzIsXG4gICAgICB9KSxcbiAgICAgIHZwYyxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgQ2xvdWRXYXRjaEFnZW50KHN0YWNrLCAndGVzdFJlc291cmNlJywge1xuICAgICAgY2xvdWRXYXRjaENvbmZpZyxcbiAgICAgIGhvc3QsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6U1NNOjpQYXJhbWV0ZXInLCB7XG4gICAgICBUeXBlOiAnU3RyaW5nJyxcbiAgICAgIFZhbHVlOiBjbG91ZFdhdGNoQ29uZmlnLFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdjcmVhdGVzIGFuIElBTSBwb2xpY3kgdG8gYWNjZXNzIHRoZSBTU00gcGFyYW1ldGVyLCBDREsgYXNzZXQgYnVja2V0LCBhbmQgQ2xvdWRXYXRjaCBhZ2VudCBidWNrZXQnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBob3N0ID0gbmV3IEluc3RhbmNlKHN0YWNrLCAnSW5zdGFuY2UnLCB7XG4gICAgICBpbnN0YW5jZVR5cGU6IEluc3RhbmNlVHlwZS5vZihJbnN0YW5jZUNsYXNzLlQyLCBJbnN0YW5jZVNpemUuTEFSR0UpLFxuICAgICAgbWFjaGluZUltYWdlOiBuZXcgQW1hem9uTGludXhJbWFnZSh7XG4gICAgICAgIGdlbmVyYXRpb246IEFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVhfMixcbiAgICAgIH0pLFxuICAgICAgdnBjLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBDbG91ZFdhdGNoQWdlbnQoc3RhY2ssICd0ZXN0UmVzb3VyY2UnLCB7XG4gICAgICBjbG91ZFdhdGNoQ29uZmlnLFxuICAgICAgaG9zdCxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAnczM6R2V0T2JqZWN0KicsXG4gICAgICAgICAgICAgICdzMzpHZXRCdWNrZXQqJyxcbiAgICAgICAgICAgICAgJ3MzOkxpc3QqJyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBSZXNvdXJjZTogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICAgICAgICAgICAgeyBSZWY6ICdBV1M6OlBhcnRpdGlvbicgfSxcbiAgICAgICAgICAgICAgICAgICAgJzpzMzo6OicsXG4gICAgICAgICAgICAgICAgICAgIHsgJ0ZuOjpTdWInOiBDV0FfQVNTRVRfTElOVVguQnVja2V0IH0sXG4gICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICAgICAnJyxcbiAgICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgICAgJ2FybjonLFxuICAgICAgICAgICAgICAgICAgICB7IFJlZjogJ0FXUzo6UGFydGl0aW9uJyB9LFxuICAgICAgICAgICAgICAgICAgICAnOnMzOjo6JyxcbiAgICAgICAgICAgICAgICAgICAgeyAnRm46OlN1Yic6IENXQV9BU1NFVF9MSU5VWC5CdWNrZXQgfSxcbiAgICAgICAgICAgICAgICAgICAgJy8qJyxcbiAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgICAgJ3NzbTpEZXNjcmliZVBhcmFtZXRlcnMnLFxuICAgICAgICAgICAgICAnc3NtOkdldFBhcmFtZXRlcnMnLFxuICAgICAgICAgICAgICAnc3NtOkdldFBhcmFtZXRlcicsXG4gICAgICAgICAgICAgICdzc206R2V0UGFyYW1ldGVySGlzdG9yeScsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICAgICAgICAgIHsgUmVmOiAnQVdTOjpQYXJ0aXRpb24nIH0sXG4gICAgICAgICAgICAgICAgICAnOnNzbTonLFxuICAgICAgICAgICAgICAgICAgeyBSZWY6ICdBV1M6OlJlZ2lvbicgfSxcbiAgICAgICAgICAgICAgICAgICc6JyxcbiAgICAgICAgICAgICAgICAgIHsgUmVmOiAnQVdTOjpBY2NvdW50SWQnIH0sXG4gICAgICAgICAgICAgICAgICAnOnBhcmFtZXRlci8nLFxuICAgICAgICAgICAgICAgICAgeyBSZWY6ICdTdHJpbmdQYXJhbWV0ZXI0NzJFRUQwRScgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAnczM6R2V0T2JqZWN0KicsXG4gICAgICAgICAgICAgICdzMzpHZXRCdWNrZXQqJyxcbiAgICAgICAgICAgICAgJ3MzOkxpc3QqJyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBSZXNvdXJjZTogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICAgICdhcm46YXdzOnMzOjo6YW1hem9uY2xvdWR3YXRjaC1hZ2VudC0nLFxuICAgICAgICAgICAgICAgICAgICB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LFxuICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICAgICdhcm46YXdzOnMzOjo6YW1hem9uY2xvdWR3YXRjaC1hZ2VudC0nLFxuICAgICAgICAgICAgICAgICAgICB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LFxuICAgICAgICAgICAgICAgICAgICAnLyonLFxuICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAnczM6R2V0T2JqZWN0KicsXG4gICAgICAgICAgICAgICdzMzpHZXRCdWNrZXQqJyxcbiAgICAgICAgICAgICAgJ3MzOkxpc3QqJyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBSZXNvdXJjZTogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICAgICdhcm46YXdzOnMzOjo6cmZkay1leHRlcm5hbC1kZXBlbmRlbmNpZXMtJyxcbiAgICAgICAgICAgICAgICAgICAgeyBSZWY6ICdBV1M6OlJlZ2lvbicgfSxcbiAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgICAnYXJuOmF3czpzMzo6OnJmZGstZXh0ZXJuYWwtZGVwZW5kZW5jaWVzLScsXG4gICAgICAgICAgICAgICAgICAgIHsgUmVmOiAnQVdTOjpSZWdpb24nIH0sXG4gICAgICAgICAgICAgICAgICAgICcvKicsXG4gICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICAgIFZlcnNpb246ICcyMDEyLTEwLTE3JyxcbiAgICAgIH0sXG4gICAgICBQb2xpY3lOYW1lOiAnSW5zdGFuY2VJbnN0YW5jZVJvbGVEZWZhdWx0UG9saWN5NEFDRTkyOTAnLFxuICAgICAgUm9sZXM6IFsgeyBSZWY6ICdJbnN0YW5jZUluc3RhbmNlUm9sZUU5Nzg1REU1JyB9IF0sXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QuZWFjaChbXG4gICAgWycgLWknLCB1bmRlZmluZWRdLFxuICAgIFsnIC1pJywgdHJ1ZV0sXG4gICAgWycnLCBmYWxzZV0sXG4gIF0pKCdhZGRzIHVzZXIgZGF0YSBjb21tYW5kcyB0byBmZXRjaCBhbmQgZXhlY3V0ZSB0aGUgc2NyaXB0IChsaW51eCkuIGluc3RhbGxGbGFnOiAlcyBzaG91bGRJbnN0YWxsQWdlbnQ6ICVwJywgKGluc3RhbGxGbGFnOiBzdHJpbmcsIHNob3VsZEluc3RhbGxBZ2VudD86IGJvb2xlYW4pID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGhvc3QgPSBuZXcgSW5zdGFuY2Uoc3RhY2ssICdJbnN0YW5jZScsIHtcbiAgICAgIGluc3RhbmNlVHlwZTogSW5zdGFuY2VUeXBlLm9mKEluc3RhbmNlQ2xhc3MuVDIsIEluc3RhbmNlU2l6ZS5MQVJHRSksXG4gICAgICBtYWNoaW5lSW1hZ2U6IG5ldyBBbWF6b25MaW51eEltYWdlKHtcbiAgICAgICAgZ2VuZXJhdGlvbjogQW1hem9uTGludXhHZW5lcmF0aW9uLkFNQVpPTl9MSU5VWF8yLFxuICAgICAgfSksXG4gICAgICB2cGMsXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IENsb3VkV2F0Y2hBZ2VudChzdGFjaywgJ3Rlc3RSZXNvdXJjZScsIHtcbiAgICAgIGNsb3VkV2F0Y2hDb25maWcsXG4gICAgICBob3N0LFxuICAgICAgc2hvdWxkSW5zdGFsbEFnZW50LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IHVzZXJEYXRhID0gc3RhY2sucmVzb2x2ZShob3N0LnVzZXJEYXRhLnJlbmRlcigpKTtcbiAgICBleHBlY3QodXNlckRhdGEpLnRvU3RyaWN0RXF1YWwoe1xuICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAnJyxcbiAgICAgICAgW1xuICAgICAgICAgIGAjIS9iaW4vYmFzaFxcbm1rZGlyIC1wICQoZGlybmFtZSAnL3RtcC8ke0NXQV9BU1NFVF9MSU5VWC5LZXl9LnNoJylcXG5hd3MgczMgY3AgJ3MzOi8vYCxcbiAgICAgICAgICB7ICdGbjo6U3ViJzogQ1dBX0FTU0VUX0xJTlVYLkJ1Y2tldCB9LFxuICAgICAgICAgIGAvJHtDV0FfQVNTRVRfTElOVVguS2V5fS5zaCcgJy90bXAvJHtDV0FfQVNTRVRfTElOVVguS2V5fS5zaCdcXG5zZXQgLWVcXG5jaG1vZCAreCAnL3RtcC8ke0NXQV9BU1NFVF9MSU5VWC5LZXl9LnNoJ1xcbicvdG1wLyR7Q1dBX0FTU0VUX0xJTlVYLktleX0uc2gnJHtpbnN0YWxsRmxhZ30gYCxcbiAgICAgICAgICB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LFxuICAgICAgICAgICcgJyxcbiAgICAgICAgICB7IFJlZjogJ1N0cmluZ1BhcmFtZXRlcjQ3MkVFRDBFJyB9LFxuICAgICAgICBdLFxuICAgICAgXSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdC5lYWNoKFtcbiAgICBbJyAtaScsIHVuZGVmaW5lZF0sXG4gICAgWycgLWknLCB0cnVlXSxcbiAgICBbJycsIGZhbHNlXSxcbiAgXSkoJ2FkZHMgdXNlciBkYXRhIGNvbW1hbmRzIHRvIGZldGNoIGFuZCBleGVjdXRlIHRoZSBzY3JpcHQgKHdpbmRvd3MpLiBpbnN0YWxsRmxhZzogJXMgc2hvdWxkSW5zdGFsbEFnZW50OiAlcCcsIChpbnN0YWxsRmxhZzogc3RyaW5nLCBzaG91bGRJbnN0YWxsQWdlbnQ/OiBib29sZWFuKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBob3N0ID0gbmV3IEluc3RhbmNlKHN0YWNrLCAnSW5zdGFuY2UnLCB7XG4gICAgICBpbnN0YW5jZVR5cGU6IEluc3RhbmNlVHlwZS5vZihJbnN0YW5jZUNsYXNzLlQyLCBJbnN0YW5jZVNpemUuTEFSR0UpLFxuICAgICAgbWFjaGluZUltYWdlOiBuZXcgV2luZG93c0ltYWdlKFdpbmRvd3NWZXJzaW9uLldJTkRPV1NfU0VSVkVSXzIwMTlfRU5HTElTSF9GVUxMX0JBU0UpLFxuICAgICAgdnBjLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBDbG91ZFdhdGNoQWdlbnQoc3RhY2ssICd0ZXN0UmVzb3VyY2UnLCB7XG4gICAgICBjbG91ZFdhdGNoQ29uZmlnLFxuICAgICAgaG9zdCxcbiAgICAgIHNob3VsZEluc3RhbGxBZ2VudCxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBjb25zdCB1c2VyRGF0YSA9IHN0YWNrLnJlc29sdmUoaG9zdC51c2VyRGF0YS5yZW5kZXIoKSk7XG4gICAgZXhwZWN0KHVzZXJEYXRhKS50b1N0cmljdEVxdWFsKHtcbiAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgJycsXG4gICAgICAgIFtcbiAgICAgICAgICBgPHBvd2Vyc2hlbGw+bWtkaXIgKFNwbGl0LVBhdGggLVBhdGggJ0M6L3RlbXAvJHtDV0FfQVNTRVRfV0lORE9XUy5LZXl9LnBzMScgKSAtZWEgMFxcblJlYWQtUzNPYmplY3QgLUJ1Y2tldE5hbWUgJ2AsXG4gICAgICAgICAgeyAnRm46OlN1Yic6IENXQV9BU1NFVF9XSU5ET1dTLkJ1Y2tldCB9LFxuICAgICAgICAgIGAnIC1rZXkgJyR7Q1dBX0FTU0VUX1dJTkRPV1MuS2V5fS5wczEnIC1maWxlICdDOi90ZW1wLyR7Q1dBX0FTU0VUX1dJTkRPV1MuS2V5fS5wczEnIC1FcnJvckFjdGlvbiBTdG9wXFxuJidDOi90ZW1wLyR7Q1dBX0FTU0VUX1dJTkRPV1MuS2V5fS5wczEnJHtpbnN0YWxsRmxhZ30gYCxcbiAgICAgICAgICB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LFxuICAgICAgICAgICcgJyxcbiAgICAgICAgICB7IFJlZjogJ1N0cmluZ1BhcmFtZXRlcjQ3MkVFRDBFJyB9LFxuICAgICAgICAgIGBcXG5pZiAoISQ/KSB7IFdyaXRlLUVycm9yICdGYWlsZWQgdG8gZXhlY3V0ZSB0aGUgZmlsZSBcXFwiQzovdGVtcC8ke0NXQV9BU1NFVF9XSU5ET1dTLktleX0ucHMxXFxcIicgLUVycm9yQWN0aW9uIFN0b3AgfTwvcG93ZXJzaGVsbD5gLFxuICAgICAgICBdLFxuICAgICAgXSxcbiAgICB9KTtcbiAgfSk7XG59KTtcbiJdfQ==