"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.X509CertificatePkcs12 = exports.X509CertificatePem = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto = require("crypto");
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_dynamodb_1 = require("aws-cdk-lib/aws-dynamodb");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const constructs_1 = require("constructs");
const lambdaLayerVersionArns_1 = require("../../lambdas/lambdaLayerVersionArns");
class X509CertificateBase extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.database = new aws_dynamodb_1.Table(this, 'Table', {
            partitionKey: { name: 'PhysicalId', type: aws_dynamodb_1.AttributeType.STRING },
            sortKey: { name: 'CustomResource', type: aws_dynamodb_1.AttributeType.STRING },
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            encryption: aws_dynamodb_1.TableEncryption.AWS_MANAGED,
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
        });
        this.passphrase = new aws_secretsmanager_1.Secret(this, 'Passphrase', {
            description: `Passphrase for the private key of the X509Certificate ${aws_cdk_lib_1.Names.uniqueId(this)}`,
            encryptionKey: props.encryptionKey,
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
            },
        });
        const region = aws_cdk_lib_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        /*
         * We cannot make this a singleton function; doing so would create circular references in the lambda role (to sign
         * a cert we need a cert that this lambda generated).
         */
        this.lambdaFunc = new aws_lambda_1.Function(this, 'Generator', {
            description: `Used by a X509Certificate ${aws_cdk_lib_1.Names.uniqueId(this)} to generate certificates.`,
            code: props.lambdaCode,
            environment: {
                DATABASE: this.database.tableName,
                DEBUG: 'false',
            },
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            layers: [openSslLayer],
            handler: props.lambdaHandler,
            timeout: aws_cdk_lib_1.Duration.seconds(90),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        this.database.grantReadWriteData(this.lambdaFunc);
        this.database.grant(this.lambdaFunc, 'dynamodb:DescribeTable');
        props.encryptionKey?.grantEncrypt(this.lambdaFunc);
        this.passphrase.grantRead(this.lambdaFunc);
        const uniqueValue = crypto.createHash('md5').update(aws_cdk_lib_1.Names.uniqueId(this)).digest('hex');
        this.uniqueTag = new aws_cdk_lib_1.Tag(`X509SecretGrant-${uniqueValue.slice(0, 8).toUpperCase()}`, uniqueValue);
        const tagCondition = {};
        tagCondition[`secretsmanager:ResourceTag/${this.uniqueTag.key}`] = this.uniqueTag.value;
        this.lambdaFunc.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'secretsmanager:CreateSecret',
                'secretsmanager:DeleteSecret',
                'secretsmanager:TagResource',
                'secretsmanager:PutSecretValue',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
    }
}
/**
 * A Construct that uses a Lambda to generate an X.509 certificate and then saves the certificate's components into
 * Secrets. On an update, if any properties of the construct are changed, then a new certificate will be generated.
 * When the Stack is destroyed or the Construct is removed, the Secrets will all be deleted. An X.509 certificate is
 * comprised of the certificate, a certificate chain with the chain of signing certificates (if any), and a private key
 * that is password protected by a randomly generated passphrase.
 *
 * Cost:
 * The cost of four AWS SecretsManager Secrets in the deployed region.
 * The other resources created by this construct have negligible ongoing costs.
 *
 * ![architecture diagram](/diagrams/core/X509CertificatePem.svg)
 *
 * Resources Deployed
 * ------------------------
 * - DynamoDB Table - Used for tracking resources created by the Custom Resource.
 * - Secrets - 4 in total, for the certificate, it's private key, the passphrase to the key, and the cert chain.
 * - Lambda Function, with role - Used to create/update/delete the Custom Resource
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - Access to the AWS SecretsManager Secrets that are created by this construct should be tightly restricted
 *   to only the principal(s) that require access.
 */
class X509CertificatePem extends X509CertificateBase {
    constructor(scope, id, props) {
        super(scope, id, {
            lambdaCode: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs')),
            lambdaHandler: 'x509-certificate.generate',
            encryptionKey: props.encryptionKey,
        });
        if ((props.validFor ?? 1) < 1 && !aws_cdk_lib_1.Token.isUnresolved(props.validFor)) {
            aws_cdk_lib_1.Annotations.of(this).addError('Certificates must be valid for at least one day.');
        }
        props.signingCertificate?.cert.grantRead(this.lambdaFunc);
        props.signingCertificate?.key.grantRead(this.lambdaFunc);
        props.signingCertificate?.passphrase.grantRead(this.lambdaFunc);
        props.signingCertificate?.certChain?.grantRead(this.lambdaFunc);
        const signingCertificate = props.signingCertificate
            ? {
                Cert: props.signingCertificate.cert.secretArn,
                Key: props.signingCertificate.key.secretArn,
                Passphrase: props.signingCertificate.passphrase.secretArn,
                CertChain: props.signingCertificate.certChain ? props.signingCertificate.certChain.secretArn : '',
            }
            : undefined;
        const properties = {
            DistinguishedName: {
                CN: props.subject.cn,
                O: props.subject.o ?? 'AWS',
                OU: props.subject.ou ?? 'Thinkbox',
            },
            Passphrase: this.passphrase.secretArn,
            Secret: {
                NamePrefix: this.node.path,
                Description: this.node.path,
                EncryptionKey: props.encryptionKey?.keyArn,
                Tags: [
                    {
                        Key: this.uniqueTag.key,
                        Value: this.uniqueTag.value,
                    },
                ],
            },
            SigningCertificate: signingCertificate,
            CertificateValidFor: props.validFor?.toString(),
        };
        const resource = new aws_cdk_lib_1.CustomResource(this, 'Default', {
            serviceToken: this.lambdaFunc.functionArn,
            properties,
            resourceType: 'Custom::RFDK_X509Generator',
        });
        if (this.lambdaFunc.role) {
            // There's a race on update where this resource could execute before the role has updated.
            resource.node.addDependency(this.lambdaFunc.role);
        }
        this.cert = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Cert', {
            secretCompleteArn: aws_cdk_lib_1.Token.asString(resource.getAtt('Cert')),
            encryptionKey: props.encryptionKey,
        });
        this.key = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Key', {
            secretCompleteArn: aws_cdk_lib_1.Token.asString(resource.getAtt('Key')),
            encryptionKey: props.encryptionKey,
        });
        // We'll only have a chain if we used a ca to sign this cert. We cannot check for certChainResource being an empty
        // string because it is an unresolved token at this point.
        if (signingCertificate) {
            const certChainResource = resource.getAtt('CertChain');
            this.certChain = certChainResource
                ? aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'CertChain', {
                    secretCompleteArn: aws_cdk_lib_1.Token.asString(certChainResource),
                    encryptionKey: props.encryptionKey,
                })
                : undefined;
        }
    }
    /**
     * Grant read permissions for the certificate
     */
    grantCertRead(grantee) {
        const result = this.cert.grantRead(grantee);
        this.certChain?.grantRead(grantee);
        return result;
    }
    /**
     * Grant read permissions for the certificate, key, and passphrase
     */
    grantFullRead(grantee) {
        const result = this.cert.grantRead(grantee);
        this.certChain?.grantRead(grantee);
        this.key.grantRead(grantee);
        this.passphrase.grantRead(grantee);
        return result;
    }
}
exports.X509CertificatePem = X509CertificatePem;
_a = JSII_RTTI_SYMBOL_1;
X509CertificatePem[_a] = { fqn: "aws-rfdk.X509CertificatePem", version: "1.0.0" };
/**
 * This Construct will generate a PKCS #12 file from an X.509 certificate in PEM format. The PEM certificate must be
 * provided through an instance of the X509CertificatePem Construct. A Lambda Function is used to do the conversion and
 * the result is stored in a Secret. The PKCS #12 file is password protected with a passphrase that is randomly
 * generated and stored in a Secret.
 *
 * ![architecture diagram](/diagrams/core/X509CertificatePkcs12.svg)
 *
 * Resources Deployed
 * ------------------------
 * - DynamoDB Table - Used for tracking resources created by the CustomResource.
 * - Secrets - 2 in total, The binary of the PKCS #12 certificate and its passphrase.
 * - Lambda Function, with role - Used to create/update/delete the CustomResource.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - Access to the AWS SecretsManager Secrets that are created by this construct should be tightly restricted
 *   to only the principal(s) that require access.
 */
class X509CertificatePkcs12 extends X509CertificateBase {
    constructor(scope, id, props) {
        super(scope, id, {
            lambdaCode: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs')),
            lambdaHandler: 'x509-certificate.convert',
            encryptionKey: props.encryptionKey,
        });
        props.sourceCertificate.grantFullRead(this.lambdaFunc);
        const properties = {
            Passphrase: this.passphrase.secretArn,
            Secret: {
                NamePrefix: this.node.path,
                Description: this.node.path,
                EncryptionKey: props.encryptionKey?.keyArn,
                Tags: [
                    {
                        Key: this.uniqueTag.key,
                        Value: this.uniqueTag.value,
                    },
                ],
            },
            Certificate: {
                Cert: props.sourceCertificate.cert.secretArn,
                CertChain: props.sourceCertificate.certChain ? props.sourceCertificate.certChain.secretArn : '',
                Key: props.sourceCertificate.key.secretArn,
                Passphrase: props.sourceCertificate.passphrase.secretArn,
            },
        };
        const resource = new aws_cdk_lib_1.CustomResource(this, 'Default', {
            serviceToken: this.lambdaFunc.functionArn,
            properties,
            resourceType: 'Custom::RFDK_X509_PKCS12',
        });
        this.cert = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Cert', {
            secretCompleteArn: aws_cdk_lib_1.Token.asString(resource.getAtt('Cert')),
            encryptionKey: props.encryptionKey,
        });
    }
}
exports.X509CertificatePkcs12 = X509CertificatePkcs12;
_b = JSII_RTTI_SYMBOL_1;
X509CertificatePkcs12[_b] = { fqn: "aws-rfdk.X509CertificatePkcs12", version: "1.0.0" };
//# sourceMappingURL=data:application/json;base64,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