"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountableBlockVolume = exports.BlockVolumeFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const constructs_1 = require("constructs");
const mount_permissions_helper_1 = require("./mount-permissions-helper");
/**
 * Block format options for formatting a blank/new BlockVolume
 */
var BlockVolumeFormat;
(function (BlockVolumeFormat) {
    /**
     * See: https://en.wikipedia.org/wiki/Ext3
     */
    BlockVolumeFormat["EXT3"] = "ext3";
    /**
     * See: https://en.wikipedia.org/wiki/Ext4
     */
    BlockVolumeFormat["EXT4"] = "ext4";
    /**
     * See: https://en.wikipedia.org/wiki/XFS
     */
    BlockVolumeFormat["XFS"] = "xfs";
})(BlockVolumeFormat = exports.BlockVolumeFormat || (exports.BlockVolumeFormat = {}));
/**
 * This class encapsulates scripting that can be used by an instance to mount, format, and resize an
 * Amazon Elastic Block Store (EBS) Volume to itself when it is launched. The scripting is added to
 * the instance's UserData to be run when the instance is first launched.
 *
 * The script that is employed by this class will:
 * 1) Attach the volume to this instance if it is not already attached;
 * 2) Format the block volume to the filesystem format that's passed as an argument to this script but,
 *   **ONLY IF** the filesystem has no current format;
 * 3) Mount the volume to the given mount point with the given mount options; and
 * 4) Resize the filesystem on the volume if the volume is larger than the formatted filesystem size.
 *
 * Note: This does **NOT** support multiple partitions on the EBS Volume; the script will exit with a failure code
 * when it detects multiple partitions on the device. It is expected that the whole block device is a single partition.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *   from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *   bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *   you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *   account to assist in post-incident analysis of compromised production environments.
 *
 * @remark If using this script with an instance within an AWS Auto Scaling Group (ASG) and you resize
 * the EBS volume, then you can terminate the instance to let the ASG replace the instance and benefit
 * from the larger volume size when this script resizes the filesystem on instance launch.
 */
class MountableBlockVolume {
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
    }
    /**
     * @inheritdoc
     */
    mountToLinuxInstance(target, mount) {
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        this.grantRequiredPermissions(target);
        const mountScriptAsset = this.mountAssetSingleton(target);
        mountScriptAsset.grantRead(target.grantPrincipal);
        const mountScriptZip = target.userData.addS3DownloadCommand({
            bucket: mountScriptAsset.bucket,
            bucketKey: mountScriptAsset.s3ObjectKey,
        });
        const mountDir = path.posix.normalize(mount.location);
        const mountOptions = [mount_permissions_helper_1.MountPermissionsHelper.toLinuxMountOption(mount.permissions)];
        if (this.props.extraMountOptions) {
            mountOptions.push(...this.props.extraMountOptions);
        }
        const mountOptionsStr = mountOptions.join(',');
        const volumeFormat = this.props.volumeFormat ?? BlockVolumeFormat.XFS;
        target.userData.addCommands('TMPDIR=$(mktemp -d)', 'pushd "$TMPDIR"', `unzip ${mountScriptZip}`, `bash ./mountEbsBlockVolume.sh ${this.props.blockVolume.volumeId} ${volumeFormat} ${mountDir} ${mountOptionsStr} ""`, 'popd', `rm -f ${mountScriptZip}`);
    }
    /**
     * @inheritdoc
     */
    usesUserPosixPermissions() {
        return true;
    }
    /**
     * Grant required permissions to the target. The mounting script requires two permissions:
     * 1) Permission to describe the volume
     * 2) Permission to attach the volume
     */
    grantRequiredPermissions(target) {
        // Volume.grantAttachVolumeByResourceTag() requires that the target be a construct; it adds a tag to the construct.
        // So, we fail if we're given something that is not compatible.
        if (!constructs_1.Construct.isConstruct(target)) {
            throw new Error('Target instance must be a construct. It cannot be constructed from attributes.');
        }
        // See: https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html
        // Accessed July 2020
        // ec2:DescribeVolumes does not support resource or condition constraints.
        target.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'ec2:DescribeVolumes',
            ],
            resources: ['*'],
        }));
        this.props.blockVolume.grantAttachVolumeByResourceTag(target.grantPrincipal, [target]);
    }
    /**
     * Fetch the Asset singleton for the Volume mounting scripts, or generate it if needed.
     */
    mountAssetSingleton(scope) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const uuid = '01ca4aa6-d440-4f83-84d8-80a5a21fd0e3';
        const uniqueId = 'MountableBlockVolumeAsset' + uuid.replace(/[-]/g, '');
        return stack.node.tryFindChild(uniqueId) ?? new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'bash'),
            exclude: ['**/*', '!mountEbsBlockVolume.sh', '!metadataUtilities.sh', '!ec2-certificates.crt'],
        });
    }
}
exports.MountableBlockVolume = MountableBlockVolume;
_a = JSII_RTTI_SYMBOL_1;
MountableBlockVolume[_a] = { fqn: "aws-rfdk.MountableBlockVolume", version: "1.0.0" };
//# sourceMappingURL=data:application/json;base64,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