"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ImportedAcmCertificate = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto = require("crypto");
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_certificatemanager_1 = require("aws-cdk-lib/aws-certificatemanager");
const aws_dynamodb_1 = require("aws-cdk-lib/aws-dynamodb");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const constructs_1 = require("constructs");
const lambdaLayerVersionArns_1 = require("../../lambdas/lambdaLayerVersionArns");
/**
 * A Construct that creates an AWS CloudFormation Custom Resource that models a certificate that is imported into
 * AWS Certificate Manager (ACM). It uses an AWS Lambda Function to extract the certificate from Secrets in AWS SecretsManager
 * and then import it into ACM. The interface is intended to be used with the {@link X509CertificatePem} Construct.
 *
 * ![architecture diagram](/diagrams/core/ImportedAcmCertificate.svg)
 *
 * Resources Deployed
 * ------------------------
 * - DynamoDB Table - Used for tracking resources created by the Custom Resource.
 * - An AWS Lambda Function, with IAM Role - Used to create/update/delete the Custom Resource.
 * - AWS Certificate Manager Certificate - Created by the Custom Resource.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The AWS Lambda for this construct also has broad IAM permissions to delete any Certificate that is stored
 *   in AWS Certificate Manager. You should not grant any additional actors/principals the ability to modify or
 *   execute this Lambda.
 */
class ImportedAcmCertificate extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.stack = aws_cdk_lib_1.Stack.of(this);
        this.env = {
            account: this.stack.account,
            region: this.stack.region,
        };
        this.database = new aws_dynamodb_1.Table(this, 'Table', {
            partitionKey: { name: 'PhysicalId', type: aws_dynamodb_1.AttributeType.STRING },
            sortKey: { name: 'CustomResource', type: aws_dynamodb_1.AttributeType.STRING },
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            encryption: aws_dynamodb_1.TableEncryption.AWS_MANAGED,
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
        });
        const region = aws_cdk_lib_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        const lambda = new aws_lambda_1.SingletonFunction(this, 'AcmImporter', {
            uuid: ImportedAcmCertificate.IMPORTER_UUID,
            code: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs')),
            handler: 'x509-certificate.importCert',
            environment: {
                DATABASE: this.database.tableName,
                DEBUG: 'false',
            },
            layers: [openSslLayer],
            retryAttempts: 0,
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            timeout: aws_cdk_lib_1.Duration.minutes(5),
        });
        this.database.grantReadWriteData(lambda);
        this.database.grant(lambda, 'dynamodb:DescribeTable');
        props.cert.grantRead(lambda);
        props.key.grantRead(lambda);
        props.passphrase.grantRead(lambda);
        props.certChain?.grantRead(lambda);
        props.encryptionKey?.grantDecrypt(lambda);
        const uniqueValue = crypto.createHash('md5').update(aws_cdk_lib_1.Names.uniqueId(this)).digest('hex');
        this.uniqueTag = new aws_cdk_lib_1.Tag(`AcmCertImport-${uniqueValue.slice(0, 8).toUpperCase()}`, uniqueValue);
        const tagCondition = {};
        tagCondition[`aws:RequestTag/${this.uniqueTag.key}`] = this.uniqueTag.value;
        lambda.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'acm:AddTagsToCertificate',
                'acm:ImportCertificate',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // GetCertificate and DeleteCertificate don't currently support and conditions, so we have to give a broader policy
        // on them for now.
        // See https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanager.html#awscertificatemanager-aws_TagKeys
        // for the condition keys currently available on ACM actions.
        lambda.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'acm:DeleteCertificate',
                'acm:DescribeCertificate',
                'acm:GetCertificate',
            ],
            resources: ['*'],
        }));
        const properties = {
            X509CertificatePem: {
                Cert: props.cert.secretArn,
                Key: props.key.secretArn,
                Passphrase: props.passphrase.secretArn,
                CertChain: props.certChain ? props.certChain.secretArn : '',
            },
            Tags: [
                {
                    Key: this.uniqueTag.key,
                    Value: this.uniqueTag.value,
                },
                { Key: 'Name',
                    Value: this.uniqueTag.value,
                },
            ],
        };
        this.resource = new aws_cdk_lib_1.CustomResource(this, 'Default', {
            serviceToken: lambda.functionArn,
            properties,
            resourceType: 'Custom::RFDK_AcmImportedCertificate',
        });
        this.certificateArn = aws_cdk_lib_1.Token.asString(this.resource.getAtt('CertificateArn'));
    }
    /**
     * Apply a removal policy to the custom resource that represents the certificate imported into ACM
     */
    applyRemovalPolicy(policy) {
        this.resource.applyRemovalPolicy(policy);
    }
    /**
     * @inheritdoc
     */
    metricDaysToExpiry(props) {
        const certLookupNode = this.node.tryFindChild(ImportedAcmCertificate.CERT_LOOKUP_CONSTRUCT_ID);
        let certLookup;
        /* istanbul ignore next */
        if (certLookupNode) {
            certLookup = certLookupNode;
        }
        else {
            certLookup = aws_certificatemanager_1.Certificate.fromCertificateArn(this, ImportedAcmCertificate.CERT_LOOKUP_CONSTRUCT_ID, this.certificateArn);
        }
        return certLookup.metricDaysToExpiry(props);
    }
}
exports.ImportedAcmCertificate = ImportedAcmCertificate;
_a = JSII_RTTI_SYMBOL_1;
ImportedAcmCertificate[_a] = { fqn: "aws-rfdk.ImportedAcmCertificate", version: "1.0.0" };
ImportedAcmCertificate.IMPORTER_UUID = '2d20d8f2-7b84-444e-b738-c75b499a9eaa';
ImportedAcmCertificate.CERT_LOOKUP_CONSTRUCT_ID = 'CertificateLookup';
//# sourceMappingURL=data:application/json;base64,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