"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthMonitor = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const aws_cloudwatch_actions_1 = require("aws-cdk-lib/aws-cloudwatch-actions");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_sns_1 = require("aws-cdk-lib/aws-sns");
const aws_sns_subscriptions_1 = require("aws-cdk-lib/aws-sns-subscriptions");
const constructs_1 = require("constructs");
const load_balancer_manager_1 = require("./load-balancer-manager");
const runtime_info_1 = require("./runtime-info");
/**
 *  A new or imported Health Monitor.
 */
class HealthMonitorBase extends constructs_1.Construct {
}
/**
 * This construct is responsible for the deep health checks of compute instances.
 * It also replaces unhealthy instances and suspends unhealthy fleets.
 * Although, using this constructs adds up additional costs for monitoring,
 * it is highly recommended using this construct to help avoid / minimize runaway costs for compute instances.
 *
 * An instance is considered to be unhealthy when:
 *   1) Deadline client is not installed on it;
 *   2) Deadline client is installed but not running on it;
 *   3) RCS is not configured correctly for Deadline client;
 *   4) it is unable to connect to RCS due to any infrastructure issues;
 *   5) the health monitor is unable to reach it because of some infrastructure issues.
 *
 * A fleet is considered to be unhealthy when:
 *   1) at least 1 instance is unhealthy for the configured grace period;
 *   2) a percentage of unhealthy instances in the fleet is above a threshold at any given point of time.
 *
 * This internally creates an array of application load balancers and attaches
 * the worker-fleet (which internally is implemented as an Auto Scaling Group) to its listeners.
 * There is no load-balancing traffic on the load balancers,
 * it is only used for health checks.
 * Intention is to use the default properties of laod balancer health
 * checks which does HTTP pings at frequent intervals to all the
 * instances in the fleet and determines its health. If any of the
 * instance is found unhealthy, it is replaced. The target group
 * also publishes the unhealthy target count metric which is used
 * to identify the unhealthy fleet.
 *
 * Other than the default instance level protection, it also creates a lambda
 * which is responsible to set the fleet size to 0 in the event of a fleet
 * being sufficiently unhealthy to warrant termination.
 * This lambda is triggered by CloudWatch alarms via SNS (Simple Notification Service).
 *
 * ![architecture diagram](/diagrams/core/HealthMonitor.svg)
 *
 * Resources Deployed
 * ------------------------
 * - Application Load Balancer(s) doing frequent pings to the workers.
 * - An Amazon Simple Notification Service (SNS) topic for all unhealthy fleet notifications.
 * - An AWS Key Management Service (KMS) Key to encrypt SNS messages - If no encryption key is provided.
 * - An Amazon CloudWatch Alarm that triggers if a worker fleet is unhealthy for a long period.
 * - Another CloudWatch Alarm that triggers if the healthy host percentage of a worker fleet is lower than allowed.
 * - A single AWS Lambda function that sets fleet size to 0 when triggered in response to messages on the SNS Topic.
 * - Execution logs of the AWS Lambda function are published to a log group in Amazon CloudWatch.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The AWS Lambda that is created by this construct to terminate unhealthy worker fleets has permission to
 *   UpdateAutoScalingGroup ( https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_UpdateAutoScalingGroup.html )
 *   on all of the fleets that this construct is monitoring. You should not grant any additional actors/principals the
 *   ability to modify or execute this Lambda.
 * - Execution of the AWS Lambda for terminating unhealthy workers is triggered by messages to the Amazon Simple
 *   Notification Service (SNS) Topic that is created by this construct. Any principal that is able to publish notification
 *   to this SNS Topic can cause the Lambda to execute and reduce one of your worker fleets to zero instances. You should
 *   not grant any additional principals permissions to publish to this SNS Topic.
 */
class HealthMonitor extends HealthMonitorBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.lbFactory = new load_balancer_manager_1.LoadBalancerFactory(this, props.vpc);
        const topicEncryptKey = props.encryptionKey || new aws_kms_1.Key(this, 'SNSEncryptionKey', {
            description: `This key is used to encrypt SNS messages for ${aws_cdk_lib_1.Names.uniqueId(this)}.`,
            enableKeyRotation: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        // allow cloudwatch service to send encrypted messages
        topicEncryptKey.grant(new aws_iam_1.ServicePrincipal('cloudwatch.amazonaws.com'), 'kms:Decrypt', 'kms:GenerateDataKey');
        this.unhealthyFleetActionTopic = new aws_sns_1.Topic(this, 'UnhealthyFleetTopic', {
            masterKey: topicEncryptKey,
        });
        this.unhealthyFleetActionTopic.grantPublish(new aws_iam_1.ServicePrincipal('cloudwatch.amazonaws.com'));
        this.alarmTopicAction = new aws_cloudwatch_actions_1.SnsAction(this.unhealthyFleetActionTopic);
        this.unhealthyFleetActionLambda = new aws_lambda_1.SingletonFunction(this, 'UnhealthyFleetAction', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs', 'unhealthyFleetAction')),
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            lambdaPurpose: 'unhealthyFleetTermination',
            timeout: aws_cdk_lib_1.Duration.seconds(300),
            uuid: '28bccf6a-aa76-478c-9239-e2f5bcc0254c',
        });
        this.unhealthyFleetActionTopic.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(this.unhealthyFleetActionLambda));
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Attaches the load-balancing target to the ELB for instance-level
     * monitoring. The ELB does frequent pings to the workers and determines
     * if a worker node is unhealthy. If so, it replaces the instance.
     *
     * It also creates an Alarm for healthy host percent and suspends the
     * fleet if the given alarm is breaching. It sets the maxCapacity
     * property of the auto-scaling group to 0. This should be
     * reset manually after fixing the issue.
     *
     * @param monitorableFleet
     * @param healthCheckConfig
     */
    registerFleet(monitorableFleet, healthCheckConfig) {
        const { loadBalancer, targetGroup } = this.lbFactory.registerWorkerFleet(monitorableFleet, healthCheckConfig, this.props);
        this.createFleetAlarms(monitorableFleet, healthCheckConfig, loadBalancer, targetGroup);
    }
    createFleetAlarms(monitorableFleet, healthCheckConfig, loadBalancer, targetGroup) {
        monitorableFleet.connections.allowFrom(loadBalancer, aws_ec2_1.Port.tcp(healthCheckConfig.port || HealthMonitor.LOAD_BALANCER_LISTENING_PORT));
        const percentMetric = new aws_cloudwatch_1.MathExpression({
            label: 'UnhealthyHostPercent',
            expression: 'IF(fleetCapacity, 100*(unhealthyHostCount/fleetCapacity), 0)',
            usingMetrics: {
                unhealthyHostCount: targetGroup.metricUnhealthyHostCount({
                    statistic: 'max',
                }),
                fleetCapacity: monitorableFleet.targetCapacityMetric,
            },
            period: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_HARD,
        });
        // When unhealthy fleet is more than healthyFleetThresholdPercent or 35% at any given period of 5 minutes
        const immediateTerminationAlarm = percentMetric.createAlarm(monitorableFleet.targetScope, 'UnhealthyFleetTermination', {
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            threshold: 100 - (healthCheckConfig.healthyFleetThresholdPercent || HealthMonitor.DEFAULT_HEALTHY_FLEET_THRESHOLD_PERCENT_HARD),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD,
            datapointsToAlarm: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD,
            actionsEnabled: true,
        });
        immediateTerminationAlarm.addAlarmAction(this.alarmTopicAction);
        // When at least one node is unhealthy over a period of 2 hours
        const percentMetricGracePeriod = new aws_cloudwatch_1.MathExpression({
            label: 'UnhealthyHostPercent',
            expression: 'IF(fleetCapacity, 100*(unhealthyHostCount/fleetCapacity), 0)',
            usingMetrics: {
                unhealthyHostCount: targetGroup.metricUnhealthyHostCount({
                    statistic: 'max',
                }),
                fleetCapacity: monitorableFleet.targetCapacityMetric,
            },
            period: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_GRACE,
        });
        const gracePeriodTerminationAlarm = percentMetricGracePeriod.createAlarm(monitorableFleet.targetScope, 'UnhealthyFleetGracePeriod', {
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            threshold: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_THRESHOLD_PERCENT_GRACE,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE,
            datapointsToAlarm: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE,
            actionsEnabled: true,
        });
        gracePeriodTerminationAlarm.addAlarmAction(this.alarmTopicAction);
        monitorableFleet.targetUpdatePolicy.attachToRole(this.unhealthyFleetActionLambda.role);
    }
}
exports.HealthMonitor = HealthMonitor;
_a = JSII_RTTI_SYMBOL_1;
HealthMonitor[_a] = { fqn: "aws-rfdk.HealthMonitor", version: "1.0.0" };
/**
 * Default health check listening port
 */
HealthMonitor.DEFAULT_HEALTH_CHECK_PORT = 63415;
/**
 * Resource Tracker in Deadline currently publish health status every 5 min, hence keeping this same
 */
HealthMonitor.DEFAULT_HEALTH_CHECK_INTERVAL = aws_cdk_lib_1.Duration.minutes(5);
/**
 * Resource Tracker in Deadline currently determines host unhealthy in 15 min, hence keeping this count
 */
HealthMonitor.DEFAULT_UNHEALTHY_HOST_THRESHOLD = 3;
/**
 * This is the minimum possible value of ALB health-check config, we want to mark worker healthy ASAP
 */
HealthMonitor.DEFAULT_HEALTHY_HOST_THRESHOLD = 2;
/**
 * Since we are not doing any load balancing, this port is just an arbitrary port.
 */
HealthMonitor.LOAD_BALANCER_LISTENING_PORT = 8081;
/**
 * This number is taken from Resource Tracker implementation. If a fleet's healthy percent
 * is less than this threshold at any given point of time, it is suspended.
 */
HealthMonitor.DEFAULT_HEALTHY_FLEET_THRESHOLD_PERCENT_HARD = 65;
/**
 * This number is taken from Resource Tracker implementation. If a fleet has at least 1
 * unhealthy host for a period of 2 hours, it is suspended.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_THRESHOLD_PERCENT_GRACE = 0;
/**
 * This number is taken from Resource Tracker implementation. We monitor unhealthy fleet for immediate
 * termination for a period fo 5 minutes.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_HARD = aws_cdk_lib_1.Duration.minutes(5);
/**
 * In Resource Tracker, we evaluate the fleet's health for determining the grace period over a period
 * of 5 minutes. For the first unhealthy signal, a instance can take upto 10min (max), hence we are
 * setting this period to be 15.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_GRACE = aws_cdk_lib_1.Duration.minutes(15);
/**
 * This number is taken from Resource Tracker implementation. Fleet is terminated immediately if it
 * has unhealthy host percent above the hard limit.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD = 1;
/**
 * This number is taken from Resource Tracker implementation. The grace period duration is 2 hours,
 * since the grace period is 15 minutes, we need continuous 8 data points crossing the threshold.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE = 8;
//# sourceMappingURL=data:application/json;base64,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