# This file is part of aiootp, an asynchronous pseudo one-time pad based
# crypto and anonymity library.
#
# Licensed under the AGPLv3: https://www.gnu.org/licenses/agpl-3.0.html
# Copyright © 2019-2021 Gonzo Investigative Journalism Agency, LLC
#            <gonzo.development@protonmail.ch>
#           © 2019-2023 Richard Machado <rmlibre@riseup.net>
# All rights reserved.
#


__doc__ = (
    "A collection of conversion tables & precomputed values to standard"
    "ize, speed up, & clean up their usage in the package."
)


from . import SimpleNamespace


class Tables(SimpleNamespace):
    """
    Conversion tables that are made available throughout the package.
    """

    __slots__ = ()

    ASCII_95 = "".join(chr(val) for val in range(32, 127))
    ASCII_128 = "".join(chr(val) for val in range(128))
    BASE_36 = "0123456789abcdefghijklmnopqrstuvwxyz"
    BASE_38 = "-0123456789_abcdefghijklmnopqrstuvwxyz"
    BASE_64 = "+/0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"
    BASE_64 += "abcdefghijklmnopqrstuvwxyz"
    BYTES = b"".join(val.to_bytes(1, "big") for val in range(256))
    HEX = "0123456789abcdef"
    ONION_CHARS = "234567abcdefghijklmnopqrstuvwxyz"
    URL_SAFE = "-0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"
    URL_SAFE += "_abcdefghijklmnopqrstuvwxyz"


class PrimeGroups(SimpleNamespace):
    """
    A collection of static primes that are the largest of their bit
    length, each paired with their smallest prime primitive root >= 7
    and != 17 (we avoided 17 because it only has two bits set to `1`).

    The primes & their primitive roots are organized by bit length, with
    no other expectation placed on them.

    To reproduce this data, the following code can be run:

    from cypari import pari
    from sympy.ntheory import prevprime, nextprime

    def prime_groups_from_byte_sizes(
        byte_sizes: int = 32, *, min_primitive_root: int = 7
    ) -> list:
        results = []
        for byte_size in range(1, byte_sizes + 1):
            primitive_root = min_primitive_root
            bit_size = 8 * byte_size
            prime = prevprime(1 << bit_size)
            order = prime - 1
            factors = [  # cypari hides values under many layers, beware.
                int(factor) for factor in pari.factor(order).list()[0]
            ]
            while any(
                1 == pow(primitive_root, order // factor, prime)
                for factor in factors
            ):
                primitive_root = nextprime(primitive_root)
            results.append((bit_size, primitive_root, hex(prime)))
        return results
    """

    __slots__ = ()

    # Generators
    g8: int = 11
    g16: int = 29
    g24: int = 7
    g32: int = 19
    g40: int = 13
    g48: int = 7
    g56: int = 11
    g64: int = 7
    g72: int = 29
    g80: int = 7
    g88: int = 37
    g96: int = 11
    g104: int = 13
    g112: int = 11
    g120: int = 29
    g128: int = 7
    g136: int = 7
    g144: int = 11
    g152: int = 11
    g160: int = 13
    g168: int = 19
    g176: int = 7
    g184: int = 7
    g192: int = 7
    g200: int = 7
    g208: int = 19
    g216: int = 7
    g224: int = 13
    g232: int = 13
    g240: int = 7
    g248: int = 13
    g256: int = 7
    g264: int = 19
    g272: int = 13
    g280: int = 13
    g288: int = 23
    g296: int = 47
    g304: int = 11
    g312: int = 13
    g320: int = 13

    # Primes
    p8: int = 0xfb
    p16: int = 0xfff1
    p24: int = 0xfffffd
    p32: int = 0xfffffffb
    p40: int = 0xffffffffa9
    p48: int = 0xffffffffffc5
    p56: int = 0xfffffffffffffb
    p64: int = 0xffffffffffffffc5
    p72: int = 0xffffffffffffffffa3
    p80: int = 0xffffffffffffffffffbf
    p88: int = 0xfffffffffffffffffffed5
    p96: int = 0xffffffffffffffffffffffef
    p104: int = 0xffffffffffffffffffffffffef
    p112: int = 0xffffffffffffffffffffffffffb5
    p120: int = 0xffffffffffffffffffffffffffff89
    p128: int = 0xffffffffffffffffffffffffffffff61
    p136: int = 0xffffffffffffffffffffffffffffffff8f
    p144: int = 0xffffffffffffffffffffffffffffffffffad
    p152: int = 0xffffffffffffffffffffffffffffffffffffef
    p160: int = 0xffffffffffffffffffffffffffffffffffffffd1
    p168: int = 0xfffffffffffffffffffffffffffffffffffffffeff
    p176: int = 0xffffffffffffffffffffffffffffffffffffffffff17
    p184: int = 0xffffffffffffffffffffffffffffffffffffffffffffdf
    p192: int = 0xffffffffffffffffffffffffffffffffffffffffffffff13
    p200: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffb5
    p208: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffed5
    p216: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffffe87
    p224: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffc1
    p232: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffdc9
    p240: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffe2d
    p248: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff13
    p256: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff43
    p264: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffeed
    p272: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff13
    p280: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffd1
    p288: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff59
    p296: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffee3
    p304: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffb5
    p312: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff35
    p320: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff3b

    # Primes previous to the primes above
    o8: int = 0xf1
    o16: int = 0xffef
    o24: int = 0xffffef
    o32: int = 0xffffffef
    o40: int = 0xffffffff59
    o48: int = 0xffffffffffbf
    o56: int = 0xffffffffffffe5
    o64: int = 0xffffffffffffffad
    o72: int = 0xffffffffffffffff95
    o80: int = 0xffffffffffffffffffa3
    o88: int = 0xfffffffffffffffffffe39
    o96: int = 0xffffffffffffffffffffffa9
    o104: int = 0xffffffffffffffffffffffffe3
    o112: int = 0xffffffffffffffffffffffffff43
    o120: int = 0xfffffffffffffffffffffffffffe75
    o128: int = 0xffffffffffffffffffffffffffffff53
    o136: int = 0xffffffffffffffffffffffffffffffff0d
    o144: int = 0xffffffffffffffffffffffffffffffffff59
    o152: int = 0xffffffffffffffffffffffffffffffffffffe9
    o160: int = 0xffffffffffffffffffffffffffffffffffffffc7
    o168: int = 0xfffffffffffffffffffffffffffffffffffffffdb7
    o176: int = 0xfffffffffffffffffffffffffffffffffffffffffeb9
    o184: int = 0xffffffffffffffffffffffffffffffffffffffffffffc5
    o192: int = 0xfffffffffffffffffffffffffffffffffffffffffffffeb3
    o200: int = 0xffffffffffffffffffffffffffffffffffffffffffffffff8b
    o208: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffe59
    o216: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffffe21
    o224: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffffffe95
    o232: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffd67
    o240: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffce3
    o248: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe7d
    o256: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe9b
    o264: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffe95
    o272: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffee1
    o280: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff97
    o288: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffdf3
    o296: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffcef
    o304: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffa85
    o312: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffd4f
    o320: int = 0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffd19


PRIMES = {
    # A static table of primes for efficient and/or deterministic
    # functionality requiring prime numbers. The primes are organized
    # so there's two primes in a list in each element of the dictionary.
    # The first & last element of each list contains the first & last
    # prime of bit length ``key``.
    2: [b"\x02", b"\x03"],
    3: [b"\x05", b"\x07"],
    4: [b"\x0b", b"\r"],
    5: [b"\x11", b"\x1f"],
    6: [b"%", b"="],
    7: [b"C", b"\x7f"],
    8: [b"\x83", b"\xfb"],
    9: [2 * b"\x01", b"\x01" + b"\xfd"],
    10: [b"\x02" + b"\t", b"\x03" + b"\xfd"],
    11: [b"\x04" + b"\x07", b"\x07" + b"\xf7"],
    12: [b"\x08" + b"\x05", b"\x0f" + b"\xfd"],
    13: [b"\x10" + b"\x03", b"\x1f" + b"\xff"],
    14: [b" " + b"\x11", b"?" + b"\xfd"],
    15: [b"@" + b"\x1b", b"\x7f" + b"\xed"],
    16: [b"\x80" + b"\x03", b"\xff" + b"\xf1"],
    17: [b"\x01" + b"\x00" + b"\x01", b"\x01" + 2 * b"\xff"],
    18: [b"\x02" + b"\x00" + b"\x1d", b"\x03" + b"\xff" + b"\xfb"],
    19: [b"\x04" + b"\x00" + b"\x03", b"\x07" + 2 * b"\xff"],
    20: [b"\x08" + b"\x00" + b"\x15", b"\x0f" + b"\xff" + b"\xfd"],
    21: [b"\x10" + b"\x00" + b"\x07", b"\x1f" + b"\xff" + b"\xf7"],
    22: [b" " + b"\x00" + b"\x11", b"?" + b"\xff" + b"\xfd"],
    23: [b"@" + b"\x00" + b"\x0f", b"\x7f" + b"\xff" + b"\xf1"],
    24: [b"\x80" + b"\x00" + b"\t", 2 * b"\xff" + b"\xfd"],
    25: [b"\x01" + 2 * b"\x00" + b"+", b"\x01" + 2 * b"\xff" + b"\xd9"],
    26: [b"\x02" + 2 * b"\x00" + b"#", b"\x03" + 2 * b"\xff" + b"\xfb"],
    27: [b"\x04" + 2 * b"\x00" + b"\x0f", b"\x07" + 2 * b"\xff" + b"\xd9"],
    28: [b"\x08" + 2 * b"\x00" + b"\x1d", b"\x0f" + 2 * b"\xff" + b"\xc7"],
    29: [b"\x10" + 2 * b"\x00" + b"\x03", b"\x1f" + 2 * b"\xff" + b"\xfd"],
    30: [b" " + 2 * b"\x00" + b"\x0b", b"?" + 2 * b"\xff" + b"\xdd"],
    31: [b"@" + 2 * b"\x00" + b"\x03", b"\x7f" + 3 * b"\xff"],
    32: [b"\x80" + 2 * b"\x00" + b"\x0b", 3 * b"\xff" + b"\xfb"],
    33: [b"\x01" + 3 * b"\x00" + b"\x0f", b"\x01" + 3 * b"\xff" + b"\xf7"],
    34: [b"\x02" + 3 * b"\x00" + b"\x11", b"\x03" + 3 * b"\xff" + b"\xd7"],
    35: [b"\x04" + 3 * b"\x00" + b"\x19", b"\x07" + 3 * b"\xff" + b"\xe1"],
    36: [b"\x08" + 3 * b"\x00" + b"5", b"\x0f" + 3 * b"\xff" + b"\xfb"],
    37: [b"\x10" + 3 * b"\x00" + b"\x1f", b"\x1f" + 3 * b"\xff" + b"\xe7"],
    38: [b" " + 3 * b"\x00" + b"\t", b"?" + 3 * b"\xff" + b"\xd3"],
    39: [b"@" + 3 * b"\x00" + b"\x07", b"\x7f" + 3 * b"\xff" + b"\xf9"],
    40: [b"\x80" + 3 * b"\x00" + b"\x17", 4 * b"\xff" + b"\xa9"],
    41: [b"\x01" + 4 * b"\x00" + b"\x0f", b"\x01" + 4 * b"\xff" + b"\xeb"],
    42: [b"\x02" + 4 * b"\x00" + b"\x1b", b"\x03" + 4 * b"\xff" + b"\xf5"],
    43: [b"\x04" + 4 * b"\x00" + b"\x0f", b"\x07" + 4 * b"\xff" + b"\xc7"],
    44: [b"\x08" + 4 * b"\x00" + b"\x1d", b"\x0f" + 4 * b"\xff" + b"\xef"],
    45: [b"\x10" + 4 * b"\x00" + b"\x07", b"\x1f" + 4 * b"\xff" + b"\xc9"],
    46: [b" " + 4 * b"\x00" + b";", b"?" + 4 * b"\xff" + b"\xeb"],
    47: [b"@" + 4 * b"\x00" + b"\x0f", b"\x7f" + 4 * b"\xff" + b"\x8d"],
    48: [b"\x80" + 4 * b"\x00" + b"\x05", 5 * b"\xff" + b"\xc5"],
    49: [b"\x01" + 5 * b"\x00" + b"\x15", b"\x01" + 5 * b"\xff" + b"\xaf"],
    50: [b"\x02" + 5 * b"\x00" + b"E", b"\x03" + 5 * b"\xff" + b"\xe5"],
    51: [b"\x04" + 5 * b"\x00" + b"7", b"\x07" + 5 * b"\xff" + b"\x7f"],
    52: [b"\x08" + 5 * b"\x00" + b"\x15", b"\x0f" + 5 * b"\xff" + b"\xd1"],
    53: [b"\x10" + 5 * b"\x00" + b"\x15", b"\x1f" + 5 * b"\xff" + b"\x91"],
    54: [b" " + 5 * b"\x00" + b"\x05", b"?" + 5 * b"\xff" + b"\xdf"],
    55: [b"@" + 5 * b"\x00" + b"\x9f", b"\x7f" + 5 * b"\xff" + b"\xc9"],
    56: [b"\x80" + 5 * b"\x00" + b"\x03", 6 * b"\xff" + b"\xfb"],
    57: [b"\x01" + 6 * b"\x00" + b"Q", b"\x01" + 6 * b"\xff" + b"\xf3"],
    58: [b"\x02" + 6 * b"\x00" + b"\t", b"\x03" + 6 * b"\xff" + b"\xe5"],
    59: [b"\x04" + 6 * b"\x00" + b"E", b"\x07" + 6 * b"\xff" + b"\xc9"],
    60: [b"\x08" + 6 * b"\x00" + b"\x83", b"\x0f" + 6 * b"\xff" + b"\xa3"],
    61: [b"\x10" + 6 * b"\x00" + b"!", b"\x1f" + 7 * b"\xff"],
    62: [b" " + 6 * b"\x00" + b"\x0f", b"?" + 6 * b"\xff" + b"\xc7"],
    63: [b"@" + 6 * b"\x00" + b"\x87", b"\x7f" + 6 * b"\xff" + b"\xe7"],
    64: [b"\x80" + 6 * b"\x00" + b"\x1d", 7 * b"\xff" + b"\xc5"],
    65: [b"\x01" + 7 * b"\x00" + b"\r", b"\x01" + 7 * b"\xff" + b"\xcf"],
    66: [b"\x02" + 7 * b"\x00" + b"\x83", b"\x03" + 7 * b"\xff" + b"\xfb"],
    67: [b"\x04" + 7 * b"\x00" + b"\t", b"\x07" + 7 * b"\xff" + b"\xed"],
    68: [b"\x08" + 7 * b"\x00" + b"\x03", b"\x0f" + 7 * b"\xff" + b"\xe9"],
    69: [b"\x10" + 7 * b"\x00" + b"!", b"\x1f" + 7 * b"\xff" + b"\xed"],
    70: [b" " + 7 * b"\x00" + b"\x1d", b"?" + 7 * b"\xff" + b"\xdd"],
    71: [b"@" + 7 * b"\x00" + b"\x19", b"\x7f" + 7 * b"\xff" + b"\x19"],
    72: [b"\x80" + 7 * b"\x00" + b"\x0b", 8 * b"\xff" + b"\xa3"],
    73: [b"\x01" + 8 * b"\x00" + b"\x0f", b"\x01" + 8 * b"\xff" + b"\xbb"],
    74: [b"\x02" + 8 * b"\x00" + b"\x1d", b"\x03" + 8 * b"\xff" + b"\xdd"],
    75: [b"\x04" + 8 * b"\x00" + b"%", b"\x07" + 8 * b"\xff" + b"\x9f"],
    76: [b"\x08" + 8 * b"\x00" + b"!", b"\x0f" + 8 * b"\xff" + b"\xf1"],
    77: [b"\x10" + 8 * b"\x00" + b"\x0f", b"\x1f" + 8 * b"\xff" + b"\xdf"],
    78: [b" " + 8 * b"\x00" + b"\x0b", b"?" + 8 * b"\xff" + b"\xf5"],
    79: [b"@" + 8 * b"\x00" + b"\x07", b"\x7f" + 8 * b"\xff" + b"\xbd"],
    80: [b"\x80" + 8 * b"\x00" + b"\x17", 9 * b"\xff" + b"\xbf"],
    81: [b"\x01" + 9 * b"\x00" + b"\r", b"\x01" + 9 * b"\xff" + b"\xcd"],
    82: [b"\x02" + 9 * b"\x00" + b"\x11", b"\x03" + 9 * b"\xff" + b"\xc7"],
    83: [b"\x04" + 9 * b"\x00" + b"\t", b"\x07" + 9 * b"\xff" + b"\xc9"],
    84: [b"\x08" + 9 * b"\x00" + b"K", b"\x0f" + 9 * b"\xff" + b"\xdd"],
    85: [b"\x10" + 9 * b"\x00" + b"\x03", b"\x1f" + 9 * b"\xff" + b"\xed"],
    86: [b" " + 9 * b"\x00" + b"\xab", b"?" + 9 * b"\xff" + b"\xdd"],
    87: [b"@" + 9 * b"\x00" + b"\x1b", b"\x7f" + 9 * b"\xff" + b"\xbd"],
    88: [b"\x80" + 9 * b"\x00" + b"'", 9 * b"\xff" + b"\xfe" + b"\xd5"],
    89: [b"\x01" + 10 * b"\x00" + b"\x07", b"\x01" + 11 * b"\xff"],
    90: [
        b"\x02" + 10 * b"\x00" + b"\x1d",
        b"\x03" + 10 * b"\xff" + b"\xdf",
    ],
    91: [
        b"\x04" + 10 * b"\x00" + b"\x85",
        b"\x07" + 10 * b"\xff" + b"\xd3",
    ],
    92: [b"\x08" + 10 * b"\x00" + b";", b"\x0f" + 10 * b"\xff" + b"\xad"],
    93: [
        b"\x10" + 10 * b"\x00" + b"\x19",
        b"\x1f" + 10 * b"\xff" + b"\xe7",
    ],
    94: [b" " + 10 * b"\x00" + b"i", b"?" + 10 * b"\xff" + b"\xfd"],
    95: [b"@" + 10 * b"\x00" + b"\x81", b"\x7f" + 10 * b"\xff" + b"\xf1"],
    96: [b"\x80" + 10 * b"\x00" + b"\t", 11 * b"\xff" + b"\xef"],
    97: [b"\x01" + 11 * b"\x00" + b"=", b"\x01" + 11 * b"\xff" + b"s"],
    98: [b"\x02" + 11 * b"\x00" + b"i", b"\x03" + 11 * b"\xff" + b"\xcd"],
    99: [
        b"\x04" + 11 * b"\x00" + b"\x07",
        b"\x07" + 11 * b"\xff" + b"\x8d",
    ],
    100: [
        b"\x08" + 11 * b"\x00" + b"\xff",
        b"\x0f" + 11 * b"\xff" + b"\xf1",
    ],
    101: [
        b"\x10" + 10 * b"\x00" + b"\x01" + b"\x15",
        b"\x1f" + 11 * b"\xff" + b"\xbb",
    ],
    102: [b" " + 11 * b"\x00" + b"Q", b"?" + 11 * b"\xff" + b"\xdf"],
    103: [
        b"@" + 10 * b"\x00" + b"\x01" + b"\x0b",
        b"\x7f" + 11 * b"\xff" + b"\x9f",
    ],
    104: [b"\x80" + 11 * b"\x00" + b"Q", 12 * b"\xff" + b"\xef"],
    105: [b"\x01" + 12 * b"\x00" + b"o", b"\x01" + 12 * b"\xff" + b"\xf3"],
    106: [b"\x02" + 12 * b"\x00" + b"'", b"\x03" + 12 * b"\xff" + b"\x8b"],
    107: [b"\x04" + 12 * b"\x00" + b"c", b"\x07" + 13 * b"\xff"],
    108: [b"\x08" + 12 * b"\x00" + b"'", b"\x0f" + 12 * b"\xff" + b"\xc5"],
    109: [b"\x10" + 12 * b"\x00" + b"!", b"\x1f" + 12 * b"\xff" + b"\xe1"],
    110: [b" " + 12 * b"\x00" + b"\x93", b"?" + 12 * b"\xff" + b"\xeb"],
    111: [b"@" + 12 * b"\x00" + b"\x1b", b"\x7f" + 12 * b"\xff" + b"\xdb"],
    112: [b"\x80" + 12 * b"\x00" + b"3", 13 * b"\xff" + b"\xb5"],
    113: [b"\x01" + 13 * b"\x00" + b"\x19", b"\x01" + 13 * b"\xff" + b"{"],
    114: [
        b"\x02" + 12 * b"\x00" + b"\x01" + b"\x19",
        b"\x03" + 13 * b"\xff" + b"\xf5",
    ],
    115: [b"\x04" + 13 * b"\x00" + b"+", b"\x07" + 13 * b"\xff" + b"\xbd"],
    116: [b"\x08" + 13 * b"\x00" + b"G", b"\x0f" + 13 * b"\xff" + b"\xfd"],
    117: [
        b"\x10" + 13 * b"\x00" + b"!",
        b"\x1f" + 12 * b"\xff" + b"\xfe" + b"\xe9",
    ],
    118: [b" " + 13 * b"\x00" + b"\x1d", b"?" + 13 * b"\xff" + b"\xfb"],
    119: [b"@" + 13 * b"\x00" + b"\x19", b"\x7f" + 13 * b"\xff" + b"\xbb"],
    120: [b"\x80" + 13 * b"\x00" + b"\t", 14 * b"\xff" + b"\x89"],
    121: [
        b"\x01" + 13 * b"\x00" + b"\x01" + b"\xc3",
        b"\x01" + 14 * b"\xff" + b"\xb7",
    ],
    122: [b"\x02" + 14 * b"\x00" + b")", b"\x03" + 14 * b"\xff" + b"\xfd"],
    123: [
        b"\x04" + 13 * b"\x00" + b"\x01" + b"\x15",
        b"\x07" + 14 * b"\xff" + b"\xbd",
    ],
    124: [
        b"\x08" + 14 * b"\x00" + b"\xa5",
        b"\x0f" + 14 * b"\xff" + b"\xc5",
    ],
    125: [b"\x10" + 14 * b"\x00" + b"C", b"\x1f" + 14 * b"\xff" + b"\xf7"],
    126: [b" " + 14 * b"\x00" + b"\x1b", b"?" + 14 * b"\xff" + b"w"],
    127: [b"@" + 14 * b"\x00" + b"\x07", b"\x7f" + 15 * b"\xff"],
    128: [b"\x80" + 14 * b"\x00" + b"\x1d", 15 * b"\xff" + b"a"],
    129: [b"\x01" + 15 * b"\x00" + b"3", b"\x01" + 15 * b"\xff" + b"\xe7"],
    130: [
        b"\x02" + 15 * b"\x00" + b"\x11",
        b"\x03" + 15 * b"\xff" + b"\xfb",
    ],
    131: [
        b"\x04" + 15 * b"\x00" + b"\xa9",
        b"\x07" + 15 * b"\xff" + b"\xbb",
    ],
    132: [
        b"\x08" + 15 * b"\x00" + b"'",
        b"\x0f" + 14 * b"\xff" + b"\xfe" + b"\xa5",
    ],
    133: [b"\x10" + 15 * b"\x00" + b"C", b"\x1f" + 15 * b"\xff" + b"\x9d"],
    134: [b" " + 15 * b"\x00" + b"\x1b", b"?" + 15 * b"\xff" + b"\xd3"],
    135: [b"@" + 15 * b"\x00" + b"\x1b", b"\x7f" + 15 * b"\xff" + b"\xd3"],
    136: [b"\x80" + 15 * b"\x00" + b"!", 16 * b"\xff" + b"\x8f"],
    137: [b"\x01" + 16 * b"\x00" + b"U", b"\x01" + 16 * b"\xff" + b"\xf3"],
    138: [
        b"\x02" + 16 * b"\x00" + b"\x9b",
        b"\x03" + 16 * b"\xff" + b"\x97",
    ],
    139: [b"\x04" + 16 * b"\x00" + b"W", b"\x07" + 16 * b"\xff" + b"E"],
    140: [
        b"\x08" + 16 * b"\x00" + b"\x9b",
        b"\x0f" + 16 * b"\xff" + b"\xe5",
    ],
    141: [b"\x10" + 16 * b"\x00" + b"%", b"\x1f" + 16 * b"\xff" + b"\xf7"],
    142: [b" " + 16 * b"\x00" + b"\x05", b"?" + 16 * b"\xff" + b"\x91"],
    143: [b"@" + 16 * b"\x00" + b"\xd9", b"\x7f" + 16 * b"\xff" + b"\xbb"],
    144: [b"\x80" + 16 * b"\x00" + b"\x05", 17 * b"\xff" + b"\xad"],
    145: [b"\x01" + 17 * b"\x00" + b"\xaf", b"\x01" + 17 * b"\xff" + b"i"],
    146: [b"\x02" + 17 * b"\x00" + b"\x1b", b"\x03" + 17 * b"\xff" + b"g"],
    147: [b"\x04" + 17 * b"\x00" + b"U", b"\x07" + 17 * b"\xff" + b"o"],
    148: [b"\x08" + 17 * b"\x00" + b"3", b"\x0f" + 17 * b"\xff" + b"Y"],
    149: [b"\x10" + 17 * b"\x00" + b"[", b"\x1f" + 17 * b"\xff" + b"\xe1"],
    150: [b" " + 17 * b"\x00" + b"E", b"?" + 17 * b"\xff" + b"\xfd"],
    151: [b"@" + 17 * b"\x00" + b"\x93", b"\x7f" + 17 * b"\xff" + b"="],
    152: [b"\x80" + 17 * b"\x00" + b"-", 18 * b"\xff" + b"\xef"],
    153: [
        b"\x01" + 18 * b"\x00" + b"\xfd",
        b"\x01" + 18 * b"\xff" + b"\xbb",
    ],
    154: [b"\x02" + 18 * b"\x00" + b"_", b"\x03" + 18 * b"\xff" + b"\r"],
    155: [
        b"\x04" + 18 * b"\x00" + b"\x1b",
        b"\x07" + 18 * b"\xff" + b"\xe1",
    ],
    156: [b"\x08" + 18 * b"\x00" + b"\x0f", b"\x0f" + 18 * b"\xff" + b"q"],
    157: [b"\x10" + 18 * b"\x00" + b"-", b"\x1f" + 18 * b"\xff" + b"\xed"],
    158: [b" " + 18 * b"\x00" + b"E", b"?" + 18 * b"\xff" + b"\xf1"],
    159: [b"@" + 18 * b"\x00" + b"a", b"\x7f" + 18 * b"\xff" + b"\xa5"],
    160: [b"\x80" + 17 * b"\x00" + b"\x01" + b"+", 19 * b"\xff" + b"\xd1"],
    161: [b"\x01" + 19 * b"\x00" + b"\x07", b"\x01" + 19 * b"\xff" + b"a"],
    162: [b"\x02" + 19 * b"\x00" + b"k", b"\x03" + 19 * b"\xff" + b"\x9b"],
    163: [
        b"\x04" + 19 * b"\x00" + b"\x13",
        b"\x07" + 19 * b"\xff" + b"\xc9",
    ],
    164: [
        b"\x08" + 19 * b"\x00" + b"\x15",
        b"\x0f" + 19 * b"\xff" + b"\xc1",
    ],
    165: [b"\x10" + 19 * b"\x00" + b"u", b"\x1f" + 19 * b"\xff" + b"\xe7"],
    166: [b" " + 19 * b"\x00" + b"\x8d", b"?" + 19 * b"\xff" + b"\xfb"],
    167: [b"@" + 19 * b"\x00" + b"U", b"\x7f" + 19 * b"\xff" + b"y"],
    168: [b"\x80" + 19 * b"\x00" + b"S", 19 * b"\xff" + b"\xfe" + b"\xff"],
    169: [
        b"\x01" + 20 * b"\x00" + b"W",
        b"\x01" + 19 * b"\xff" + b"\xfd" + b"}",
    ],
    170: [b"\x02" + 20 * b"\x00" + b"\x93", b"\x03" + 20 * b"\xff" + b"q"],
    171: [b"\x04" + 20 * b"\x00" + b"1", b"\x07" + 20 * b"\xff" + b"\xed"],
    172: [
        b"\x08" + 20 * b"\x00" + b"\x81",
        b"\x0f" + 20 * b"\xff" + b"\xa1",
    ],
    173: [b"\x10" + 20 * b"\x00" + b"i", b"\x1f" + 20 * b"\xff" + b"\xc9"],
    174: [b" " + 20 * b"\x00" + b"M", b"?" + 20 * b"\xff" + b"\xfd"],
    175: [b"@" + 20 * b"\x00" + b"\x07", b"\x7f" + 20 * b"\xff" + b"\x1b"],
    176: [b"\x80" + 20 * b"\x00" + b"\t", 21 * b"\xff" + b"\x17"],
    177: [
        b"\x01" + 20 * b"\x00" + b"\x01" + b"\xab",
        b"\x01" + 20 * b"\xff" + b"\xfe" + b"\xad",
    ],
    178: [b"\x02" + 21 * b"\x00" + b"K", b"\x03" + 21 * b"\xff" + b"\xd7"],
    179: [b"\x04" + 21 * b"\x00" + b"W", b"\x07" + 21 * b"\xff" + b"\xcf"],
    180: [
        b"\x08" + 20 * b"\x00" + b"\x01" + b"5",
        b"\x0f" + 21 * b"\xff" + b"\xd1",
    ],
    181: [b"\x10" + 21 * b"\x00" + b"\x0f", b"\x1f" + 21 * b"\xff" + b"["],
    182: [b" " + 21 * b"\x00" + b"\xa5", b"?" + 21 * b"\xff" + b"_"],
    183: [b"@" + 21 * b"\x00" + b"1", b"\x7f" + 21 * b"\xff" + b"m"],
    184: [b"\x80" + 21 * b"\x00" + b"\xd7", 22 * b"\xff" + b"\xdf"],
    185: [
        b"\x01" + 22 * b"\x00" + b"\x1b",
        b"\x01" + 21 * b"\xff" + b"\xfe" + b"\xd1",
    ],
    186: [
        b"\x02" + 22 * b"\x00" + b"\x9f",
        b"\x03" + 21 * b"\xff" + b"\xfe" + b"\x8d",
    ],
    187: [
        b"\x04" + 22 * b"\x00" + b"\xcd",
        b"\x07" + 22 * b"\xff" + b"\xab",
    ],
    188: [
        b"\x08" + 21 * b"\x00" + b"\x01" + b"/",
        b"\x0f" + 22 * b"\xff" + b"\x83",
    ],
    189: [b"\x10" + 22 * b"\x00" + b"9", b"\x1f" + 22 * b"\xff" + b"\xe7"],
    190: [b" " + 22 * b"\x00" + b"#", b"?" + 22 * b"\xff" + b"\xf5"],
    191: [b"@" + 22 * b"\x00" + b"\x81", b"\x7f" + 22 * b"\xff" + b"\xed"],
    192: [b"\x80" + 22 * b"\x00" + b"\x05", 23 * b"\xff" + b"\x13"],
    193: [
        b"\x01" + 23 * b"\x00" + b"\x85",
        b"\x01" + 23 * b"\xff" + b"\xe1",
    ],
    194: [b"\x02" + 23 * b"\x00" + b"A", b"\x03" + 23 * b"\xff" + b"\xdf"],
    195: [b"\x04" + 23 * b"\x00" + b"\x1b", b"\x07" + 23 * b"\xff" + b"y"],
    196: [b"\x08" + 23 * b"\x00" + b"#", b"\x0f" + 23 * b"\xff" + b"\xf1"],
    197: [
        b"\x10" + 23 * b"\x00" + b"\x15",
        b"\x1f" + 23 * b"\xff" + b"\xb5",
    ],
    198: [b" " + 23 * b"\x00" + b"k", b"?" + 23 * b"\xff" + b"\xef"],
    199: [b"@" + 23 * b"\x00" + b"\x0f", b"\x7f" + 23 * b"\xff" + b"\xcf"],
    200: [b"\x80" + 23 * b"\x00" + b"e", 24 * b"\xff" + b"\xb5"],
    201: [
        b"\x01" + 24 * b"\x00" + b"\xeb",
        b"\x01" + 24 * b"\xff" + b"\xc9",
    ],
    202: [
        b"\x02" + 23 * b"\x00" + b"\x01" + b"_",
        b"\x03" + 24 * b"\xff" + b"I",
    ],
    203: [b"\x04" + 24 * b"\x00" + b"C", b"\x07" + 24 * b"\xff" + b"a"],
    204: [b"\x08" + 24 * b"\x00" + b"\x0f", b"\x0f" + 24 * b"\xff" + b"Y"],
    205: [
        b"\x10" + 24 * b"\x00" + b"\x07",
        b"\x1f" + 24 * b"\xff" + b"\xaf",
    ],
    206: [
        b" " + 23 * b"\x00" + b"\x02" + b"E",
        b"?" + 24 * b"\xff" + b"\xfb",
    ],
    207: [b"@" + 24 * b"\x00" + b"!", b"\x7f" + 24 * b"\xff" + b"\xa5"],
    208: [
        b"\x80" + 24 * b"\x00" + b"\xcb",
        24 * b"\xff" + b"\xfe" + b"\xd5",
    ],
    209: [
        b"\x01" + 24 * b"\x00" + b"\x01" + b"w",
        b"\x01" + 25 * b"\xff" + b"\xdf",
    ],
    210: [b"\x02" + 25 * b"\x00" + b"/", b"\x03" + 25 * b"\xff" + b"\xd1"],
    211: [b"\x04" + 25 * b"\x00" + b"!", b"\x07" + 25 * b"\xff" + b"Q"],
    212: [b"\x08" + 25 * b"\x00" + b"G", b"\x0f" + 25 * b"\xff" + b"\xe9"],
    213: [b"\x10" + 25 * b"\x00" + b"9", b"\x1f" + 25 * b"\xff" + b"\xfd"],
    214: [b" " + 25 * b"\x00" + b"K", b"?" + 25 * b"\xff" + b"G"],
    215: [b"@" + 25 * b"\x00" + b"\x07", b"\x7f" + 25 * b"\xff" + b"c"],
    216: [
        b"\x80" + 25 * b"\x00" + b"\xfb",
        25 * b"\xff" + b"\xfe" + b"\x87",
    ],
    217: [
        b"\x01" + 25 * b"\x00" + b"\x01" + b"\xa7",
        b"\x01" + 26 * b"\xff" + b"\xc3",
    ],
    218: [
        b"\x02" + 26 * b"\x00" + b"\x81",
        b"\x03" + 26 * b"\xff" + b"\xdf",
    ],
    219: [
        b"\x04" + 26 * b"\x00" + b"\xa3",
        b"\x07" + 26 * b"\xff" + b"\x87",
    ],
    220: [
        b"\x08" + 26 * b"\x00" + b"\xb9",
        b"\x0f" + 26 * b"\xff" + b"\xb3",
    ],
    221: [
        b"\x10" + 26 * b"\x00" + b"\xd9",
        b"\x1f" + 26 * b"\xff" + b"\xfd",
    ],
    222: [b" " + 26 * b"\x00" + b"Q", b"?" + 26 * b"\xff" + b"\x8b"],
    223: [b"@" + 26 * b"\x00" + b"1", b"\x7f" + 26 * b"\xff" + b"\x15"],
    224: [b"\x80" + 26 * b"\x00" + b"\xbd", 27 * b"\xff" + b"\xc1"],
    225: [
        b"\x01" + 26 * b"\x00" + b"\x02" + b"\xdf",
        b"\x01" + 27 * b"\xff" + b"\xcf",
    ],
    226: [b"\x02" + 27 * b"\x00" + b"w", b"\x03" + 27 * b"\xff" + b"\xfb"],
    227: [
        b"\x04" + 26 * b"\x00" + b"\x02" + b"\xdf",
        b"\x07" + 26 * b"\xff" + b"\xfe" + b"k",
    ],
    228: [
        b"\x08" + 26 * b"\x00" + b"\x01" + b"\xe3",
        b"\x0f" + 27 * b"\xff" + b"\xa3",
    ],
    229: [
        b"\x10" + 27 * b"\x00" + b"\x03",
        b"\x1f" + 27 * b"\xff" + b"\xa5",
    ],
    230: [b" " + 27 * b"\x00" + b"\xf9", b"?" + 27 * b"\xff" + b"\xe5"],
    231: [b"@" + 27 * b"\x00" + b"C", b"\x7f" + 27 * b"\xff" + b"["],
    232: [b"\x80" + 27 * b"\x00" + b"i", 27 * b"\xff" + b"\xfd" + b"\xc9"],
    233: [
        b"\x01" + 27 * b"\x00" + b"\x01" + b"e",
        b"\x01" + 28 * b"\xff" + b"\xfd",
    ],
    234: [
        b"\x02" + 27 * b"\x00" + b"\x01" + b"\xaf",
        b"\x03" + 28 * b"\xff" + b"\xad",
    ],
    235: [b"\x04" + 28 * b"\x00" + b"+", b"\x07" + 28 * b"\xff" + b"\xf1"],
    236: [b"\x08" + 28 * b"\x00" + b"Q", b"\x0f" + 28 * b"\xff" + b"/"],
    237: [b"\x10" + 28 * b"\x00" + b"\x19", b"\x1f" + 28 * b"\xff" + b"K"],
    238: [b" " + 28 * b"\x00" + b"\xf9", b"?" + 28 * b"\xff" + b"_"],
    239: [b"@" + 28 * b"\x00" + b"C", b"\x7f" + 28 * b"\xff" + b"\xa9"],
    240: [b"\x80" + 28 * b"\x00" + b"\x1d", 28 * b"\xff" + b"\xfe" + b"-"],
    241: [b"\x01" + 29 * b"\x00" + b"s", b"\x01" + 29 * b"\xff" + b"\xd9"],
    242: [
        b"\x02" + 28 * b"\x00" + b"\x01" + b"\x05",
        b"\x03" + 29 * b"\xff" + b"\xc1",
    ],
    243: [b"\x04" + 29 * b"\x00" + b"E", b"\x07" + 29 * b"\xff" + b"\xf7"],
    244: [b"\x08" + 29 * b"\x00" + b";", b"\x0f" + 29 * b"\xff" + b"C"],
    245: [b"\x10" + 29 * b"\x00" + b"\x85", b"\x1f" + 29 * b"\xff" + b"]"],
    246: [
        b" " + 28 * b"\x00" + b"\x01" + b";",
        b"?" + 29 * b"\xff" + b"\x95",
    ],
    247: [
        b"@" + 28 * b"\x00" + b"\x01" + b"Q",
        b"\x7f" + 29 * b"\xff" + b"\xaf",
    ],
    248: [b"\x80" + 29 * b"\x00" + b"?", 30 * b"\xff" + b"\x13"],
    249: [b"\x01" + 30 * b"\x00" + b"Q", b"\x01" + 30 * b"\xff" + b"\xb5"],
    250: [b"\x02" + 30 * b"\x00" + b"w", b"\x03" + 30 * b"\xff" + b"1"],
    251: [
        b"\x04" + 30 * b"\x00" + b"\x19",
        b"\x07" + 30 * b"\xff" + b"\xf7",
    ],
    252: [b"\x08" + 30 * b"\x00" + b"A", b"\x0f" + 30 * b"\xff" + b"\x7f"],
    253: [
        b"\x10" + 29 * b"\x00" + b"\x01" + b"\xa5",
        b"\x1f" + 29 * b"\xff" + b"\xfe" + b"\xef",
    ],
    254: [b" " + 30 * b"\x00" + b"'", b"?" + 30 * b"\xff" + b"\x0b"],
    255: [b"@" + 30 * b"\x00" + b"O", b"\x7f" + 30 * b"\xff" + b"\xed"],
    256: [b"\x80" + 30 * b"\x00" + b"_", 31 * b"\xff" + b"C"],
    257: [
        b"\x01" + 30 * b"\x00" + b"\x01" + b")",
        b"\x01" + 31 * b"\xff" + b"\xa3",
    ],
    383: [b"@" + 46 * b"\x00" + b"\xff", b"\x7f" + 46 * b"\xff" + b"\xe1"],
    384: [
        b"\x80" + 45 * b"\x00" + b"\x01" + b"q",
        46 * b"\xff" + b"\xfe" + b"\xc3",
    ],
    385: [
        b"\x01" + 47 * b"\x00" + b"\xe7",
        b"\x01" + 46 * b"\xff" + b"\xfe" + b"\xf7",
    ],
    511: [b"@" + 62 * b"\x00" + b"\x0f", b"\x7f" + 62 * b"\xff" + b"E"],
    512: [b"\x80" + 62 * b"\x00" + b"o", 62 * b"\xff" + b"\xfd" + b"\xc7"],
    513: [
        b"\x01" + 63 * b"\x00" + b"K",
        b"\x01" + 62 * b"\xff" + b"\xfe" + b"C",
    ],
    1023: [
        b"@" + 125 * b"\x00" + b"\x05" + b"\xa3",
        b"\x7f" + 125 * b"\xff" + b"\xfe" + b"\x97",
    ],
    1024: [
        b"\x80" + 125 * b"\x00" + b"\x04" + b"\x83",
        127 * b"\xff" + b"\x97",
    ],
    1025: [
        b"\x01" + 126 * b"\x00" + b"\x02" + b"\x83",
        b"\x01" + 126 * b"\xff" + b"\xf5" + b"\x8f",
    ],
    1535: [
        b"@" + 189 * b"\x00" + b"\x01" + b"\t",
        b"\x7f" + 189 * b"\xff" + b"\xfa" + b"\xdb",
    ],
    1536: [
        b"\x80" + 189 * b"\x00" + b"\x02" + b"\xbb",
        190 * b"\xff" + b"\xf2" + b"\x83",
    ],
    1537: [b"\x01" + 191 * b"\x00" + b"K", b"\x01" + 191 * b"\xff" + b"Q"],
    2047: [
        b"@" + 253 * b"\x00" + b"\x10" + b"3",
        b"\x7f" + 254 * b"\xff" + b"\xab",
    ],
    2048: [
        b"\x80" + 253 * b"\x00" + b"\x07" + b"\x7f",
        254 * b"\xff" + b"\xf9" + b"\xeb",
    ],
    2049: [
        b"\x01" + 254 * b"\x00" + b"\x03" + b"\xd5",
        b"\x01" + 254 * b"\xff" + b"\xfa" + b"W",
    ],
    2559: [
        b"@" + 317 * b"\x00" + b"\x03" + b"\x87",
        b"\x7f" + 317 * b"\xff" + b"\xfd" + b";",
    ],
    2560: [
        b"\x80" + 317 * b"\x00" + b"\x08" + b"\xeb",
        319 * b"\xff" + b"\xb5",
    ],
    2561: [
        b"\x01" + 318 * b"\x00" + b"\x03" + b"\x87",
        b"\x01" + 318 * b"\xff" + b"\xfd" + b"\x99",
    ],
    3071: [b"@" + 382 * b"\x00" + b"\xc3", b"\x7f" + 382 * b"\xff" + b"E"],
    3072: [
        b"\x80" + 381 * b"\x00" + b"\x08" + b"\xf3",
        383 * b"\xff" + b"\xd1",
    ],
    3073: [
        b"\x01" + 382 * b"\x00" + b"\x03" + b"-",
        b"\x01" + 382 * b"\xff" + b"\xfc" + b"\xb7",
    ],
    3583: [
        b"@" + 445 * b"\x00" + b"\t" + b"'",
        b"\x7f" + 445 * b"\xff" + b"\xf3" + b"\xe7",
    ],
    3584: [
        b"\x80" + 445 * b"\x00" + b"\x10" + b"\xa3",
        446 * b"\xff" + b"\xfe" + b"S",
    ],
    3585: [
        b"\x01" + 447 * b"\x00" + b"\x15",
        b"\x01" + 446 * b"\xff" + b"\xfe" + b"e",
    ],
    4095: [
        b"@" + 509 * b"\x00" + b"\x01" + b"\xe9",
        b"\x7f" + 509 * b"\xff" + b"\xf9" + b"\xb1",
    ],
    4096: [
        b"\x80" + 509 * b"\x00" + b"\x02" + b"C",
        510 * b"\xff" + b"\xf6" + b"\x0b",
    ],
    4097: [
        b"\x01" + 510 * b"\x00" + b"\x06" + b"\xe1",
        b"\x01" + 510 * b"\xff" + b"\xfc" + b"\xf1",
    ],
}


for level in PRIMES.values():
    level[0] = int.from_bytes(level[0], "big")
    level[1] = int.from_bytes(level[1], "big")


WORD_LIST = [
    b"abandon",
    b"ability",
    b"able",
    b"about",
    b"above",
    b"absent",
    b"absorb",
    b"abstract",
    b"absurd",
    b"abuse",
    b"access",
    b"accident",
    b"account",
    b"accuse",
    b"achieve",
    b"acid",
    b"acoustic",
    b"acquire",
    b"across",
    b"act",
    b"action",
    b"actor",
    b"actress",
    b"actual",
    b"adapt",
    b"add",
    b"addict",
    b"address",
    b"adjust",
    b"admit",
    b"adult",
    b"advance",
    b"advice",
    b"aerobic",
    b"affair",
    b"afford",
    b"afraid",
    b"again",
    b"age",
    b"agent",
    b"agree",
    b"ahead",
    b"aim",
    b"air",
    b"airport",
    b"aisle",
    b"alarm",
    b"album",
    b"alcohol",
    b"alert",
    b"alien",
    b"all",
    b"alley",
    b"allow",
    b"almost",
    b"alone",
    b"alpha",
    b"already",
    b"also",
    b"alter",
    b"always",
    b"amateur",
    b"amazing",
    b"among",
    b"amount",
    b"amused",
    b"analyst",
    b"anchor",
    b"ancient",
    b"anger",
    b"angle",
    b"angry",
    b"animal",
    b"ankle",
    b"announce",
    b"annual",
    b"another",
    b"answer",
    b"antenna",
    b"antique",
    b"anxiety",
    b"any",
    b"apart",
    b"apology",
    b"appear",
    b"apple",
    b"approve",
    b"april",
    b"arch",
    b"arctic",
    b"area",
    b"arena",
    b"argue",
    b"arm",
    b"armed",
    b"armor",
    b"army",
    b"around",
    b"arrange",
    b"arrest",
    b"arrive",
    b"arrow",
    b"art",
    b"artefact",
    b"artist",
    b"artwork",
    b"ask",
    b"aspect",
    b"assault",
    b"asset",
    b"assist",
    b"assume",
    b"asthma",
    b"athlete",
    b"atom",
    b"attack",
    b"attend",
    b"attitude",
    b"attract",
    b"auction",
    b"audit",
    b"august",
    b"aunt",
    b"author",
    b"auto",
    b"autumn",
    b"average",
    b"avocado",
    b"avoid",
    b"awake",
    b"aware",
    b"away",
    b"awesome",
    b"awful",
    b"awkward",
    b"axis",
    b"baby",
    b"bachelor",
    b"bacon",
    b"badge",
    b"bag",
    b"balance",
    b"balcony",
    b"ball",
    b"bamboo",
    b"banana",
    b"banner",
    b"bar",
    b"barely",
    b"bargain",
    b"barrel",
    b"base",
    b"basic",
    b"basket",
    b"battle",
    b"beach",
    b"bean",
    b"beauty",
    b"because",
    b"become",
    b"beef",
    b"before",
    b"begin",
    b"behave",
    b"behind",
    b"believe",
    b"below",
    b"belt",
    b"bench",
    b"benefit",
    b"best",
    b"betray",
    b"better",
    b"between",
    b"beyond",
    b"bicycle",
    b"bid",
    b"bike",
    b"bind",
    b"biology",
    b"bird",
    b"birth",
    b"bitter",
    b"black",
    b"blade",
    b"blame",
    b"blanket",
    b"blast",
    b"bleak",
    b"bless",
    b"blind",
    b"blood",
    b"blossom",
    b"blouse",
    b"blue",
    b"blur",
    b"blush",
    b"board",
    b"boat",
    b"body",
    b"boil",
    b"bomb",
    b"bone",
    b"bonus",
    b"book",
    b"boost",
    b"border",
    b"boring",
    b"borrow",
    b"boss",
    b"bottom",
    b"bounce",
    b"box",
    b"boy",
    b"bracket",
    b"brain",
    b"brand",
    b"brass",
    b"brave",
    b"bread",
    b"breeze",
    b"brick",
    b"bridge",
    b"brief",
    b"bright",
    b"bring",
    b"brisk",
    b"broccoli",
    b"broken",
    b"bronze",
    b"broom",
    b"brother",
    b"brown",
    b"brush",
    b"bubble",
    b"buddy",
    b"budget",
    b"buffalo",
    b"build",
    b"bulb",
    b"bulk",
    b"bullet",
    b"bundle",
    b"bunker",
    b"burden",
    b"burger",
    b"burst",
    b"bus",
    b"business",
    b"busy",
    b"butter",
    b"buyer",
    b"buzz",
    b"cabbage",
    b"cabin",
    b"cable",
    b"cactus",
    b"cage",
    b"cake",
    b"call",
    b"calm",
    b"camera",
    b"camp",
    b"can",
    b"canal",
    b"cancel",
    b"candy",
    b"cannon",
    b"canoe",
    b"canvas",
    b"canyon",
    b"capable",
    b"capital",
    b"captain",
    b"car",
    b"carbon",
    b"card",
    b"cargo",
    b"carpet",
    b"carry",
    b"cart",
    b"case",
    b"cash",
    b"casino",
    b"castle",
    b"casual",
    b"cat",
    b"catalog",
    b"catch",
    b"category",
    b"cattle",
    b"caught",
    b"cause",
    b"caution",
    b"cave",
    b"ceiling",
    b"celery",
    b"cement",
    b"census",
    b"century",
    b"cereal",
    b"certain",
    b"chair",
    b"chalk",
    b"champion",
    b"change",
    b"chaos",
    b"chapter",
    b"charge",
    b"chase",
    b"chat",
    b"cheap",
    b"check",
    b"cheese",
    b"chef",
    b"cherry",
    b"chest",
    b"chicken",
    b"chief",
    b"child",
    b"chimney",
    b"choice",
    b"choose",
    b"chronic",
    b"chuckle",
    b"chunk",
    b"churn",
    b"cigar",
    b"cinnamon",
    b"circle",
    b"citizen",
    b"city",
    b"civil",
    b"claim",
    b"clap",
    b"clarify",
    b"claw",
    b"clay",
    b"clean",
    b"clerk",
    b"clever",
    b"click",
    b"client",
    b"cliff",
    b"climb",
    b"clinic",
    b"clip",
    b"clock",
    b"clog",
    b"close",
    b"cloth",
    b"cloud",
    b"clown",
    b"club",
    b"clump",
    b"cluster",
    b"clutch",
    b"coach",
    b"coast",
    b"coconut",
    b"code",
    b"coffee",
    b"coil",
    b"coin",
    b"collect",
    b"color",
    b"column",
    b"combine",
    b"come",
    b"comfort",
    b"comic",
    b"common",
    b"company",
    b"concert",
    b"conduct",
    b"confirm",
    b"congress",
    b"connect",
    b"consider",
    b"control",
    b"convince",
    b"cook",
    b"cool",
    b"copper",
    b"copy",
    b"coral",
    b"core",
    b"corn",
    b"correct",
    b"cost",
    b"cotton",
    b"couch",
    b"country",
    b"couple",
    b"course",
    b"cousin",
    b"cover",
    b"coyote",
    b"crack",
    b"cradle",
    b"craft",
    b"cram",
    b"crane",
    b"crash",
    b"crater",
    b"crawl",
    b"crazy",
    b"cream",
    b"credit",
    b"creek",
    b"crew",
    b"cricket",
    b"crime",
    b"crisp",
    b"critic",
    b"crop",
    b"cross",
    b"crouch",
    b"crowd",
    b"crucial",
    b"cruel",
    b"cruise",
    b"crumble",
    b"crunch",
    b"crush",
    b"cry",
    b"crystal",
    b"cube",
    b"culture",
    b"cup",
    b"cupboard",
    b"curious",
    b"current",
    b"curtain",
    b"curve",
    b"cushion",
    b"custom",
    b"cute",
    b"cycle",
    b"dad",
    b"damage",
    b"damp",
    b"dance",
    b"danger",
    b"daring",
    b"dash",
    b"daughter",
    b"dawn",
    b"day",
    b"deal",
    b"debate",
    b"debris",
    b"decade",
    b"december",
    b"decide",
    b"decline",
    b"decorate",
    b"decrease",
    b"deer",
    b"defense",
    b"define",
    b"defy",
    b"degree",
    b"delay",
    b"deliver",
    b"demand",
    b"demise",
    b"denial",
    b"dentist",
    b"deny",
    b"depart",
    b"depend",
    b"deposit",
    b"depth",
    b"deputy",
    b"derive",
    b"describe",
    b"desert",
    b"design",
    b"desk",
    b"despair",
    b"destroy",
    b"detail",
    b"detect",
    b"develop",
    b"device",
    b"devote",
    b"diagram",
    b"dial",
    b"diamond",
    b"diary",
    b"dice",
    b"diesel",
    b"diet",
    b"differ",
    b"digital",
    b"dignity",
    b"dilemma",
    b"dinner",
    b"dinosaur",
    b"direct",
    b"dirt",
    b"disagree",
    b"discover",
    b"disease",
    b"dish",
    b"dismiss",
    b"disorder",
    b"display",
    b"distance",
    b"divert",
    b"divide",
    b"divorce",
    b"dizzy",
    b"doctor",
    b"document",
    b"dog",
    b"doll",
    b"dolphin",
    b"domain",
    b"donate",
    b"donkey",
    b"donor",
    b"door",
    b"dose",
    b"double",
    b"dove",
    b"draft",
    b"dragon",
    b"drama",
    b"drastic",
    b"draw",
    b"dream",
    b"dress",
    b"drift",
    b"drill",
    b"drink",
    b"drip",
    b"drive",
    b"drop",
    b"drum",
    b"dry",
    b"duck",
    b"dumb",
    b"dune",
    b"during",
    b"dust",
    b"dutch",
    b"duty",
    b"dwarf",
    b"dynamic",
    b"eager",
    b"eagle",
    b"early",
    b"earn",
    b"earth",
    b"easily",
    b"east",
    b"easy",
    b"echo",
    b"ecology",
    b"economy",
    b"edge",
    b"edit",
    b"educate",
    b"effort",
    b"egg",
    b"eight",
    b"either",
    b"elbow",
    b"elder",
    b"electric",
    b"elegant",
    b"element",
    b"elephant",
    b"elevator",
    b"elite",
    b"else",
    b"embark",
    b"embody",
    b"embrace",
    b"emerge",
    b"emotion",
    b"employ",
    b"empower",
    b"empty",
    b"enable",
    b"enact",
    b"end",
    b"endless",
    b"endorse",
    b"enemy",
    b"energy",
    b"enforce",
    b"engage",
    b"engine",
    b"enhance",
    b"enjoy",
    b"enlist",
    b"enough",
    b"enrich",
    b"enroll",
    b"ensure",
    b"enter",
    b"entire",
    b"entry",
    b"envelope",
    b"episode",
    b"equal",
    b"equip",
    b"era",
    b"erase",
    b"erode",
    b"erosion",
    b"error",
    b"erupt",
    b"escape",
    b"essay",
    b"essence",
    b"estate",
    b"eternal",
    b"ethics",
    b"evidence",
    b"evil",
    b"evoke",
    b"evolve",
    b"exact",
    b"example",
    b"excess",
    b"exchange",
    b"excite",
    b"exclude",
    b"excuse",
    b"execute",
    b"exercise",
    b"exhaust",
    b"exhibit",
    b"exile",
    b"exist",
    b"exit",
    b"exotic",
    b"expand",
    b"expect",
    b"expire",
    b"explain",
    b"expose",
    b"express",
    b"extend",
    b"extra",
    b"eye",
    b"eyebrow",
    b"fabric",
    b"face",
    b"faculty",
    b"fade",
    b"faint",
    b"faith",
    b"fall",
    b"false",
    b"fame",
    b"family",
    b"famous",
    b"fan",
    b"fancy",
    b"fantasy",
    b"farm",
    b"fashion",
    b"fat",
    b"fatal",
    b"father",
    b"fatigue",
    b"fault",
    b"favorite",
    b"feature",
    b"february",
    b"federal",
    b"fee",
    b"feed",
    b"feel",
    b"female",
    b"fence",
    b"festival",
    b"fetch",
    b"fever",
    b"few",
    b"fiber",
    b"fiction",
    b"field",
    b"figure",
    b"file",
    b"film",
    b"filter",
    b"final",
    b"find",
    b"fine",
    b"finger",
    b"finish",
    b"fire",
    b"firm",
    b"first",
    b"fiscal",
    b"fish",
    b"fit",
    b"fitness",
    b"fix",
    b"flag",
    b"flame",
    b"flash",
    b"flat",
    b"flavor",
    b"flee",
    b"flight",
    b"flip",
    b"float",
    b"flock",
    b"floor",
    b"flower",
    b"fluid",
    b"flush",
    b"fly",
    b"foam",
    b"focus",
    b"fog",
    b"foil",
    b"fold",
    b"follow",
    b"food",
    b"foot",
    b"force",
    b"forest",
    b"forget",
    b"fork",
    b"fortune",
    b"forum",
    b"forward",
    b"fossil",
    b"foster",
    b"found",
    b"fox",
    b"fragile",
    b"frame",
    b"frequent",
    b"fresh",
    b"friend",
    b"fringe",
    b"frog",
    b"front",
    b"frost",
    b"frown",
    b"frozen",
    b"fruit",
    b"fuel",
    b"fun",
    b"funny",
    b"furnace",
    b"fury",
    b"future",
    b"gadget",
    b"gain",
    b"galaxy",
    b"gallery",
    b"game",
    b"gap",
    b"garage",
    b"garbage",
    b"garden",
    b"garlic",
    b"garment",
    b"gas",
    b"gasp",
    b"gate",
    b"gather",
    b"gauge",
    b"gaze",
    b"general",
    b"genius",
    b"genre",
    b"gentle",
    b"genuine",
    b"gesture",
    b"ghost",
    b"giant",
    b"gift",
    b"giggle",
    b"ginger",
    b"giraffe",
    b"girl",
    b"give",
    b"glad",
    b"glance",
    b"glare",
    b"glass",
    b"glide",
    b"glimpse",
    b"globe",
    b"gloom",
    b"glory",
    b"glove",
    b"glow",
    b"glue",
    b"goat",
    b"goddess",
    b"gold",
    b"good",
    b"goose",
    b"gorilla",
    b"gospel",
    b"gossip",
    b"govern",
    b"gown",
    b"grab",
    b"grace",
    b"grain",
    b"grant",
    b"grape",
    b"grass",
    b"gravity",
    b"great",
    b"green",
    b"grid",
    b"grief",
    b"grit",
    b"grocery",
    b"group",
    b"grow",
    b"grunt",
    b"guard",
    b"guess",
    b"guide",
    b"guilt",
    b"guitar",
    b"gun",
    b"gym",
    b"habit",
    b"hair",
    b"half",
    b"hammer",
    b"hamster",
    b"hand",
    b"happy",
    b"harbor",
    b"hard",
    b"harsh",
    b"harvest",
    b"hat",
    b"have",
    b"hawk",
    b"hazard",
    b"head",
    b"health",
    b"heart",
    b"heavy",
    b"hedgehog",
    b"height",
    b"hello",
    b"helmet",
    b"help",
    b"hen",
    b"hero",
    b"hidden",
    b"high",
    b"hill",
    b"hint",
    b"hip",
    b"hire",
    b"history",
    b"hobby",
    b"hockey",
    b"hold",
    b"hole",
    b"holiday",
    b"hollow",
    b"home",
    b"honey",
    b"hood",
    b"hope",
    b"horn",
    b"horror",
    b"horse",
    b"hospital",
    b"host",
    b"hotel",
    b"hour",
    b"hover",
    b"hub",
    b"huge",
    b"human",
    b"humble",
    b"humor",
    b"hundred",
    b"hungry",
    b"hunt",
    b"hurdle",
    b"hurry",
    b"hurt",
    b"husband",
    b"hybrid",
    b"ice",
    b"icon",
    b"idea",
    b"identify",
    b"idle",
    b"ignore",
    b"ill",
    b"illegal",
    b"illness",
    b"image",
    b"imitate",
    b"immense",
    b"immune",
    b"impact",
    b"impose",
    b"improve",
    b"impulse",
    b"inch",
    b"include",
    b"income",
    b"increase",
    b"index",
    b"indicate",
    b"indoor",
    b"industry",
    b"infant",
    b"inflict",
    b"inform",
    b"inhale",
    b"inherit",
    b"initial",
    b"inject",
    b"injury",
    b"inmate",
    b"inner",
    b"innocent",
    b"input",
    b"inquiry",
    b"insane",
    b"insect",
    b"inside",
    b"inspire",
    b"install",
    b"intact",
    b"interest",
    b"into",
    b"invest",
    b"invite",
    b"involve",
    b"iron",
    b"island",
    b"isolate",
    b"issue",
    b"item",
    b"ivory",
    b"jacket",
    b"jaguar",
    b"jar",
    b"jazz",
    b"jealous",
    b"jeans",
    b"jelly",
    b"jewel",
    b"job",
    b"join",
    b"joke",
    b"journey",
    b"joy",
    b"judge",
    b"juice",
    b"jump",
    b"jungle",
    b"junior",
    b"junk",
    b"just",
    b"kangaroo",
    b"keen",
    b"keep",
    b"ketchup",
    b"key",
    b"kick",
    b"kid",
    b"kidney",
    b"kind",
    b"kingdom",
    b"kiss",
    b"kit",
    b"kitchen",
    b"kite",
    b"kitten",
    b"kiwi",
    b"knee",
    b"knife",
    b"knock",
    b"know",
    b"lab",
    b"label",
    b"labor",
    b"ladder",
    b"lady",
    b"lake",
    b"lamp",
    b"language",
    b"laptop",
    b"large",
    b"later",
    b"latin",
    b"laugh",
    b"laundry",
    b"lava",
    b"law",
    b"lawn",
    b"lawsuit",
    b"layer",
    b"lazy",
    b"leader",
    b"leaf",
    b"learn",
    b"leave",
    b"lecture",
    b"left",
    b"leg",
    b"legal",
    b"legend",
    b"leisure",
    b"lemon",
    b"lend",
    b"length",
    b"lens",
    b"leopard",
    b"lesson",
    b"letter",
    b"level",
    b"liar",
    b"liberty",
    b"library",
    b"license",
    b"life",
    b"lift",
    b"light",
    b"like",
    b"limb",
    b"limit",
    b"link",
    b"lion",
    b"liquid",
    b"list",
    b"little",
    b"live",
    b"lizard",
    b"load",
    b"loan",
    b"lobster",
    b"local",
    b"lock",
    b"logic",
    b"lonely",
    b"long",
    b"loop",
    b"lottery",
    b"loud",
    b"lounge",
    b"love",
    b"loyal",
    b"lucky",
    b"luggage",
    b"lumber",
    b"lunar",
    b"lunch",
    b"luxury",
    b"lyrics",
    b"machine",
    b"mad",
    b"magic",
    b"magnet",
    b"maid",
    b"mail",
    b"main",
    b"major",
    b"make",
    b"mammal",
    b"man",
    b"manage",
    b"mandate",
    b"mango",
    b"mansion",
    b"manual",
    b"maple",
    b"marble",
    b"march",
    b"margin",
    b"marine",
    b"market",
    b"marriage",
    b"mask",
    b"mass",
    b"master",
    b"match",
    b"material",
    b"math",
    b"matrix",
    b"matter",
    b"maximum",
    b"maze",
    b"meadow",
    b"mean",
    b"measure",
    b"meat",
    b"mechanic",
    b"medal",
    b"media",
    b"melody",
    b"melt",
    b"member",
    b"memory",
    b"mention",
    b"menu",
    b"mercy",
    b"merge",
    b"merit",
    b"merry",
    b"mesh",
    b"message",
    b"metal",
    b"method",
    b"middle",
    b"midnight",
    b"milk",
    b"million",
    b"mimic",
    b"mind",
    b"minimum",
    b"minor",
    b"minute",
    b"miracle",
    b"mirror",
    b"misery",
    b"miss",
    b"mistake",
    b"mix",
    b"mixed",
    b"mixture",
    b"mobile",
    b"model",
    b"modify",
    b"mom",
    b"moment",
    b"monitor",
    b"monkey",
    b"monster",
    b"month",
    b"moon",
    b"moral",
    b"more",
    b"morning",
    b"mosquito",
    b"mother",
    b"motion",
    b"motor",
    b"mountain",
    b"mouse",
    b"move",
    b"movie",
    b"much",
    b"muffin",
    b"mule",
    b"multiply",
    b"muscle",
    b"museum",
    b"mushroom",
    b"music",
    b"must",
    b"mutual",
    b"myself",
    b"mystery",
    b"myth",
    b"naive",
    b"name",
    b"napkin",
    b"narrow",
    b"nasty",
    b"nation",
    b"nature",
    b"near",
    b"neck",
    b"need",
    b"negative",
    b"neglect",
    b"neither",
    b"nephew",
    b"nerve",
    b"nest",
    b"net",
    b"network",
    b"neutral",
    b"never",
    b"news",
    b"next",
    b"nice",
    b"night",
    b"noble",
    b"noise",
    b"nominee",
    b"noodle",
    b"normal",
    b"north",
    b"nose",
    b"notable",
    b"note",
    b"nothing",
    b"notice",
    b"novel",
    b"now",
    b"nuclear",
    b"number",
    b"nurse",
    b"nut",
    b"oak",
    b"obey",
    b"object",
    b"oblige",
    b"obscure",
    b"observe",
    b"obtain",
    b"obvious",
    b"occur",
    b"ocean",
    b"october",
    b"odor",
    b"off",
    b"offer",
    b"office",
    b"often",
    b"oil",
    b"okay",
    b"old",
    b"olive",
    b"olympic",
    b"omit",
    b"once",
    b"one",
    b"onion",
    b"online",
    b"only",
    b"open",
    b"opera",
    b"opinion",
    b"oppose",
    b"option",
    b"orange",
    b"orbit",
    b"orchard",
    b"order",
    b"ordinary",
    b"organ",
    b"orient",
    b"original",
    b"orphan",
    b"ostrich",
    b"other",
    b"outdoor",
    b"outer",
    b"output",
    b"outside",
    b"oval",
    b"oven",
    b"over",
    b"own",
    b"owner",
    b"oxygen",
    b"oyster",
    b"ozone",
    b"pact",
    b"paddle",
    b"page",
    b"pair",
    b"palace",
    b"palm",
    b"panda",
    b"panel",
    b"panic",
    b"panther",
    b"paper",
    b"parade",
    b"parent",
    b"park",
    b"parrot",
    b"party",
    b"pass",
    b"patch",
    b"path",
    b"patient",
    b"patrol",
    b"pattern",
    b"pause",
    b"pave",
    b"payment",
    b"peace",
    b"peanut",
    b"pear",
    b"peasant",
    b"pelican",
    b"pen",
    b"penalty",
    b"pencil",
    b"people",
    b"pepper",
    b"perfect",
    b"permit",
    b"person",
    b"pet",
    b"phone",
    b"photo",
    b"phrase",
    b"physical",
    b"piano",
    b"picnic",
    b"picture",
    b"piece",
    b"pig",
    b"pigeon",
    b"pill",
    b"pilot",
    b"pink",
    b"pioneer",
    b"pipe",
    b"pistol",
    b"pitch",
    b"pizza",
    b"place",
    b"planet",
    b"plastic",
    b"plate",
    b"play",
    b"please",
    b"pledge",
    b"pluck",
    b"plug",
    b"plunge",
    b"poem",
    b"poet",
    b"point",
    b"polar",
    b"pole",
    b"police",
    b"pond",
    b"pony",
    b"pool",
    b"popular",
    b"portion",
    b"position",
    b"possible",
    b"post",
    b"potato",
    b"pottery",
    b"poverty",
    b"powder",
    b"power",
    b"practice",
    b"praise",
    b"predict",
    b"prefer",
    b"prepare",
    b"present",
    b"pretty",
    b"prevent",
    b"price",
    b"pride",
    b"primary",
    b"print",
    b"priority",
    b"prison",
    b"private",
    b"prize",
    b"problem",
    b"process",
    b"produce",
    b"profit",
    b"program",
    b"project",
    b"promote",
    b"proof",
    b"property",
    b"prosper",
    b"protect",
    b"proud",
    b"provide",
    b"public",
    b"pudding",
    b"pull",
    b"pulp",
    b"pulse",
    b"pumpkin",
    b"punch",
    b"pupil",
    b"puppy",
    b"purchase",
    b"purity",
    b"purpose",
    b"purse",
    b"push",
    b"put",
    b"puzzle",
    b"pyramid",
    b"quality",
    b"quantum",
    b"quarter",
    b"question",
    b"quick",
    b"quit",
    b"quiz",
    b"quote",
    b"rabbit",
    b"raccoon",
    b"race",
    b"rack",
    b"radar",
    b"radio",
    b"rail",
    b"rain",
    b"raise",
    b"rally",
    b"ramp",
    b"ranch",
    b"random",
    b"range",
    b"raaad",
    b"rare",
    b"rate",
    b"rather",
    b"raven",
    b"raw",
    b"razor",
    b"ready",
    b"real",
    b"reason",
    b"rebel",
    b"rebuild",
    b"recall",
    b"receive",
    b"recipe",
    b"record",
    b"recycle",
    b"reduce",
    b"reflect",
    b"reform",
    b"refuse",
    b"region",
    b"regret",
    b"regular",
    b"reject",
    b"relax",
    b"release",
    b"relief",
    b"rely",
    b"remain",
    b"remember",
    b"remind",
    b"remove",
    b"render",
    b"renew",
    b"rent",
    b"reopen",
    b"repair",
    b"repeat",
    b"replace",
    b"report",
    b"require",
    b"rescue",
    b"resemble",
    b"resist",
    b"resource",
    b"response",
    b"result",
    b"retire",
    b"retreat",
    b"return",
    b"reunion",
    b"reveal",
    b"review",
    b"reward",
    b"rhythm",
    b"rib",
    b"ribbon",
    b"rice",
    b"rich",
    b"ride",
    b"ridge",
    b"rifle",
    b"right",
    b"rigid",
    b"ring",
    b"riot",
    b"ripple",
    b"risk",
    b"ritual",
    b"rival",
    b"river",
    b"road",
    b"roast",
    b"robot",
    b"robust",
    b"rocket",
    b"romance",
    b"roof",
    b"rookie",
    b"room",
    b"rose",
    b"rotate",
    b"rough",
    b"round",
    b"route",
    b"royal",
    b"rubber",
    b"rude",
    b"rug",
    b"rule",
    b"run",
    b"runway",
    b"rural",
    b"sad",
    b"saddle",
    b"sadness",
    b"safe",
    b"sail",
    b"salad",
    b"salmon",
    b"salon",
    b"salt",
    b"salute",
    b"same",
    b"sample",
    b"sand",
    b"satisfy",
    b"satoshi",
    b"sauce",
    b"sausage",
    b"save",
    b"say",
    b"scale",
    b"scan",
    b"scare",
    b"scatter",
    b"scene",
    b"scheme",
    b"school",
    b"science",
    b"scissors",
    b"scorpion",
    b"scout",
    b"scrap",
    b"screen",
    b"script",
    b"scrub",
    b"sea",
    b"search",
    b"season",
    b"seat",
    b"second",
    b"secret",
    b"section",
    b"security",
    b"seed",
    b"seek",
    b"segment",
    b"select",
    b"sell",
    b"seminar",
    b"senior",
    b"sense",
    b"sentence",
    b"series",
    b"service",
    b"session",
    b"settle",
    b"setup",
    b"seven",
    b"shadow",
    b"shaft",
    b"shallow",
    b"share",
    b"shed",
    b"shell",
    b"sheriff",
    b"shield",
    b"shift",
    b"shine",
    b"ship",
    b"shiver",
    b"shock",
    b"shoe",
    b"shoot",
    b"shop",
    b"short",
    b"shoulder",
    b"shove",
    b"shrimp",
    b"shrug",
    b"shuffle",
    b"shy",
    b"sibling",
    b"sick",
    b"side",
    b"siege",
    b"sight",
    b"sign",
    b"silent",
    b"silk",
    b"silly",
    b"silver",
    b"similar",
    b"simple",
    b"since",
    b"sing",
    b"siren",
    b"sister",
    b"situate",
    b"six",
    b"size",
    b"skate",
    b"sketch",
    b"ski",
    b"skill",
    b"skin",
    b"skirt",
    b"skull",
    b"slab",
    b"slam",
    b"sleep",
    b"slender",
    b"slice",
    b"slide",
    b"slight",
    b"slim",
    b"slogan",
    b"slot",
    b"slow",
    b"slush",
    b"small",
    b"smart",
    b"smile",
    b"smoke",
    b"smooth",
    b"snack",
    b"snake",
    b"snap",
    b"sniff",
    b"snow",
    b"soap",
    b"soccer",
    b"social",
    b"sock",
    b"soda",
    b"soft",
    b"solar",
    b"soldier",
    b"solid",
    b"solution",
    b"solve",
    b"someone",
    b"song",
    b"soon",
    b"sorry",
    b"sort",
    b"soul",
    b"sound",
    b"soup",
    b"source",
    b"south",
    b"space",
    b"spare",
    b"spatial",
    b"spawn",
    b"speak",
    b"special",
    b"speed",
    b"spell",
    b"spend",
    b"sphere",
    b"spice",
    b"saader",
    b"spike",
    b"spin",
    b"spirit",
    b"split",
    b"spoil",
    b"sponsor",
    b"spoon",
    b"sport",
    b"spot",
    b"spray",
    b"spread",
    b"spring",
    b"spy",
    b"square",
    b"squeeze",
    b"squirrel",
    b"stable",
    b"stadium",
    b"staff",
    b"stage",
    b"stairs",
    b"stamp",
    b"stand",
    b"start",
    b"state",
    b"stay",
    b"steak",
    b"steel",
    b"stem",
    b"step",
    b"stereo",
    b"stick",
    b"still",
    b"sting",
    b"stock",
    b"stomach",
    b"stone",
    b"stool",
    b"story",
    b"stove",
    b"strategy",
    b"street",
    b"strike",
    b"strong",
    b"struggle",
    b"student",
    b"stuff",
    b"stumble",
    b"style",
    b"subject",
    b"submit",
    b"subway",
    b"success",
    b"such",
    b"sudden",
    b"suffer",
    b"sugar",
    b"suggest",
    b"suit",
    b"summer",
    b"sun",
    b"sunny",
    b"sunset",
    b"super",
    b"supply",
    b"supreme",
    b"sure",
    b"surface",
    b"surge",
    b"surprise",
    b"surround",
    b"survey",
    b"suspect",
    b"sustain",
    b"swallow",
    b"swamp",
    b"swap",
    b"swarm",
    b"swear",
    b"sweet",
    b"swift",
    b"swim",
    b"swing",
    b"switch",
    b"sword",
    b"symbol",
    b"symptom",
    b"syrup",
    b"system",
    b"table",
    b"tackle",
    b"tag",
    b"tail",
    b"talent",
    b"talk",
    b"tank",
    b"tape",
    b"target",
    b"task",
    b"taste",
    b"tattoo",
    b"taxi",
    b"teach",
    b"team",
    b"tell",
    b"ten",
    b"tenant",
    b"tennis",
    b"tent",
    b"term",
    b"test",
    b"text",
    b"thank",
    b"that",
    b"theme",
    b"then",
    b"theory",
    b"there",
    b"they",
    b"thing",
    b"this",
    b"thought",
    b"three",
    b"thrive",
    b"throw",
    b"thumb",
    b"thunder",
    b"ticket",
    b"tide",
    b"tiger",
    b"tilt",
    b"timber",
    b"time",
    b"tiny",
    b"tip",
    b"tired",
    b"tissue",
    b"title",
    b"toast",
    b"tobacco",
    b"today",
    b"toddler",
    b"toe",
    b"together",
    b"toilet",
    b"token",
    b"tomato",
    b"tomorrow",
    b"tone",
    b"tongue",
    b"tonight",
    b"tool",
    b"tooth",
    b"top",
    b"topic",
    b"topple",
    b"torch",
    b"tornado",
    b"tortoise",
    b"toss",
    b"total",
    b"tourist",
    b"toward",
    b"tower",
    b"town",
    b"toy",
    b"track",
    b"trade",
    b"traffic",
    b"tragic",
    b"train",
    b"transfer",
    b"trap",
    b"trash",
    b"travel",
    b"tray",
    b"treat",
    b"tree",
    b"trend",
    b"trial",
    b"tribe",
    b"trick",
    b"trigger",
    b"trim",
    b"trip",
    b"trophy",
    b"trouble",
    b"truck",
    b"true",
    b"truly",
    b"trumpet",
    b"trust",
    b"truth",
    b"try",
    b"tube",
    b"tuition",
    b"tumble",
    b"tuna",
    b"tunnel",
    b"turkey",
    b"turn",
    b"turtle",
    b"twelve",
    b"twenty",
    b"twice",
    b"twin",
    b"twist",
    b"two",
    b"type",
    b"typical",
    b"ugly",
    b"umbrella",
    b"unable",
    b"unaware",
    b"uncle",
    b"uncover",
    b"under",
    b"undo",
    b"unfair",
    b"unfold",
    b"unhappy",
    b"uniform",
    b"unique",
    b"unit",
    b"universe",
    b"unknown",
    b"unlock",
    b"until",
    b"unusual",
    b"unveil",
    b"update",
    b"upgrade",
    b"uphold",
    b"upon",
    b"upper",
    b"upset",
    b"urban",
    b"urge",
    b"usage",
    b"use",
    b"used",
    b"useful",
    b"useless",
    b"usual",
    b"utility",
    b"vacant",
    b"vacuum",
    b"vague",
    b"valid",
    b"valley",
    b"valve",
    b"van",
    b"vanish",
    b"vapor",
    b"various",
    b"vast",
    b"vault",
    b"vehicle",
    b"velvet",
    b"vendor",
    b"venture",
    b"venue",
    b"verb",
    b"verify",
    b"version",
    b"very",
    b"vessel",
    b"veteran",
    b"viable",
    b"vibrant",
    b"vicious",
    b"victory",
    b"video",
    b"view",
    b"village",
    b"vintage",
    b"violin",
    b"virtual",
    b"virus",
    b"visa",
    b"visit",
    b"visual",
    b"vital",
    b"vivid",
    b"vocal",
    b"voice",
    b"void",
    b"volcano",
    b"volume",
    b"vote",
    b"voyage",
    b"wage",
    b"wagon",
    b"wait",
    b"walk",
    b"wall",
    b"walnut",
    b"want",
    b"warfare",
    b"warm",
    b"warrior",
    b"wash",
    b"wasp",
    b"waste",
    b"water",
    b"wave",
    b"way",
    b"wealth",
    b"weapon",
    b"wear",
    b"weasel",
    b"weather",
    b"web",
    b"wedding",
    b"weekend",
    b"weird",
    b"welcome",
    b"west",
    b"wet",
    b"whale",
    b"what",
    b"wheat",
    b"wheel",
    b"when",
    b"where",
    b"whip",
    b"whisper",
    b"wide",
    b"width",
    b"wife",
    b"wild",
    b"will",
    b"win",
    b"window",
    b"wine",
    b"wing",
    b"wink",
    b"winner",
    b"winter",
    b"wire",
    b"wisdom",
    b"wise",
    b"wish",
    b"witness",
    b"wolf",
    b"woman",
    b"wonder",
    b"wood",
    b"wool",
    b"word",
    b"work",
    b"world",
    b"worry",
    b"worth",
    b"wrap",
    b"wreck",
    b"wrestle",
    b"wrist",
    b"write",
    b"wrong",
    b"yard",
    b"year",
    b"yellow",
    b"you",
    b"young",
    b"youth",
    b"zebra",
    b"zero",
    b"zone",
    b"zoo",
]


datasets = {n: v for n, v in globals().items() if n[0].isupper()}
datasets["__all__"] = [*datasets]

