"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsRunTask = exports.EcsEc2LaunchTarget = exports.EcsFargateLaunchTarget = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Configuration for running an ECS task on Fargate
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/userguide/launch_types.html#launch-type-fargate
 */
class EcsFargateLaunchTarget {
    constructor(options) {
        this.options = options;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EcsFargateLaunchTargetOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, EcsFargateLaunchTarget);
            }
            throw error;
        }
    }
    /**
     * Called when the Fargate launch type configured on RunTask
     */
    bind(_task, launchTargetOptions) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EcsRunTask(_task);
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_LaunchTargetBindOptions(launchTargetOptions);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        if (!launchTargetOptions.taskDefinition.isFargateCompatible) {
            throw new Error('Supplied TaskDefinition is not compatible with Fargate');
        }
        return {
            parameters: {
                LaunchType: 'FARGATE',
                PlatformVersion: this.options?.platformVersion,
            },
        };
    }
}
exports.EcsFargateLaunchTarget = EcsFargateLaunchTarget;
_a = JSII_RTTI_SYMBOL_1;
EcsFargateLaunchTarget[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EcsFargateLaunchTarget", version: "1.168.0" };
/**
 * Configuration for running an ECS task on EC2
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/userguide/launch_types.html#launch-type-ec2
 */
class EcsEc2LaunchTarget {
    constructor(options) {
        this.options = options;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EcsEc2LaunchTargetOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, EcsEc2LaunchTarget);
            }
            throw error;
        }
    }
    /**
     * Called when the EC2 launch type is configured on RunTask
     */
    bind(_task, launchTargetOptions) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EcsRunTask(_task);
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_LaunchTargetBindOptions(launchTargetOptions);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        if (!launchTargetOptions.taskDefinition.isEc2Compatible) {
            throw new Error('Supplied TaskDefinition is not compatible with EC2');
        }
        if (!launchTargetOptions.cluster?.hasEc2Capacity) {
            throw new Error('Cluster for this service needs Ec2 capacity. Call addCapacity() on the cluster.');
        }
        return {
            parameters: {
                LaunchType: 'EC2',
                // takes an array of placement constraints each of which contain a single item array of constraints, flattens it
                // and renders the Json to be passed as a parameter in the state machine.
                // input: [ecs.PlacementConstraint.distinctInstances()] - distinctInstances() returns [{ type: 'distinctInstance' }]
                // output: {Type: 'distinctInstance'}
                PlacementConstraints: noEmpty(flatten((this.options?.placementConstraints ?? []).map((c) => c.toJson().map(uppercaseKeys)))),
                PlacementStrategy: noEmpty(flatten((this.options?.placementStrategies ?? []).map((c) => c.toJson().map(uppercaseKeys)))),
            },
        };
        function uppercaseKeys(obj) {
            const ret = {};
            for (const key of Object.keys(obj)) {
                ret[key.slice(0, 1).toUpperCase() + key.slice(1)] = obj[key];
            }
            return ret;
        }
        function flatten(xs) {
            return Array.prototype.concat([], ...xs);
        }
        function noEmpty(xs) {
            if (xs.length === 0) {
                return undefined;
            }
            return xs;
        }
    }
}
exports.EcsEc2LaunchTarget = EcsEc2LaunchTarget;
_b = JSII_RTTI_SYMBOL_1;
EcsEc2LaunchTarget[_b] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EcsEc2LaunchTarget", version: "1.168.0" };
/**
 * Run a Task on ECS or Fargate
 */
class EcsRunTask extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        /**
         * Manage allowed network traffic for this service
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EcsRunTaskProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, EcsRunTask);
            }
            throw error;
        }
        this.integrationPattern = props.integrationPattern ?? sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, EcsRunTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.containerOverrides?.map(override => override.environment))) {
            throw new Error('Task Token is required in at least one `containerOverrides.environment` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (!this.props.taskDefinition.defaultContainer) {
            throw new Error('A TaskDefinition must have at least one essential container');
        }
        if (this.props.taskDefinition.networkMode === ecs.NetworkMode.AWS_VPC) {
            this.configureAwsVpcNetworking();
        }
        else {
            // Either None, Bridge or Host networking. Copy SecurityGroup from ASG.
            this.validateNoNetworkingProps();
            this.connections.addSecurityGroup(...this.props.cluster.connections.securityGroups);
        }
        for (const override of this.props.containerOverrides ?? []) {
            const name = override.containerDefinition.containerName;
            if (!cdk.Token.isUnresolved(name)) {
                const cont = this.props.taskDefinition.findContainer(name);
                if (!cont) {
                    throw new Error(`Overrides mention container with name '${name}', but no such container in task definition`);
                }
            }
        }
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('ecs', 'runTask', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                Cluster: this.props.cluster.clusterArn,
                TaskDefinition: this.props.taskDefinition.family,
                NetworkConfiguration: this.networkConfiguration,
                Overrides: renderOverrides(this.props.containerOverrides),
                ...this.props.launchTarget.bind(this, { taskDefinition: this.props.taskDefinition, cluster: this.props.cluster }).parameters,
            }),
        };
    }
    configureAwsVpcNetworking() {
        const subnetSelection = this.props.subnets ?? { subnetType: this.props.assignPublicIp ? ec2.SubnetType.PUBLIC : ec2.SubnetType.PRIVATE_WITH_NAT };
        this.networkConfiguration = {
            AwsvpcConfiguration: {
                AssignPublicIp: this.props.assignPublicIp ? (this.props.assignPublicIp ? 'ENABLED' : 'DISABLED') : undefined,
                Subnets: this.props.cluster.vpc.selectSubnets(subnetSelection).subnetIds,
                SecurityGroups: cdk.Lazy.list({ produce: () => this.securityGroups?.map(sg => sg.securityGroupId) }),
            },
        };
        // Make sure we have a security group if we're using AWSVPC networking
        this.securityGroups = this.props.securityGroups ?? [new ec2.SecurityGroup(this, 'SecurityGroup', { vpc: this.props.cluster.vpc })];
        this.connections.addSecurityGroup(...this.securityGroups);
    }
    validateNoNetworkingProps() {
        if (this.props.subnets !== undefined || this.props.securityGroups !== undefined) {
            throw new Error(`Supplied TaskDefinition must have 'networkMode' of 'AWS_VPC' to use 'vpcSubnets' and 'securityGroup'. Received: ${this.props.taskDefinition.networkMode}`);
        }
    }
    makePolicyStatements() {
        const stack = cdk.Stack.of(this);
        // https://docs.aws.amazon.com/step-functions/latest/dg/ecs-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.getTaskDefinitionFamilyArn()],
            }),
            new iam.PolicyStatement({
                actions: ['ecs:StopTask', 'ecs:DescribeTasks'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: this.taskExecutionRoles().map((r) => r.roleArn),
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForECSTaskRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
    /**
     * Returns the ARN of the task definition family by removing the
     * revision from the task definition ARN
     * Before - arn:aws:ecs:us-west-2:123456789012:task-definition/hello_world:8
     * After - arn:aws:ecs:us-west-2:123456789012:task-definition/hello_world
     */
    getTaskDefinitionFamilyArn() {
        const arnComponents = cdk.Stack.of(this).splitArn(this.props.taskDefinition.taskDefinitionArn, cdk.ArnFormat.SLASH_RESOURCE_NAME);
        let { resourceName } = arnComponents;
        if (resourceName) {
            resourceName = resourceName.split(':')[0];
        }
        return cdk.Stack.of(this).formatArn({
            partition: arnComponents.partition,
            service: arnComponents.service,
            account: arnComponents.account,
            region: arnComponents.region,
            resource: arnComponents.resource,
            arnFormat: arnComponents.arnFormat,
            resourceName,
        });
    }
    taskExecutionRoles() {
        // Need to be able to pass both Task and Execution role, apparently
        const ret = new Array();
        ret.push(this.props.taskDefinition.taskRole);
        if (this.props.taskDefinition.executionRole) {
            ret.push(this.props.taskDefinition.executionRole);
        }
        return ret;
    }
}
exports.EcsRunTask = EcsRunTask;
_c = JSII_RTTI_SYMBOL_1;
EcsRunTask[_c] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EcsRunTask", version: "1.168.0" };
EcsRunTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
function renderOverrides(containerOverrides) {
    if (!containerOverrides || containerOverrides.length === 0) {
        return undefined;
    }
    const ret = new Array();
    for (const override of containerOverrides) {
        ret.push({
            Name: override.containerDefinition.containerName,
            Command: override.command,
            Cpu: override.cpu,
            Memory: override.memoryLimit,
            MemoryReservation: override.memoryReservation,
            Environment: override.environment?.map((e) => ({
                Name: e.name,
                Value: e.value,
            })),
        });
    }
    return { ContainerOverrides: ret };
}
//# sourceMappingURL=data:application/json;base64,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