from argparse import ArgumentParser

from . import git, terraform, plans
from .storage import StorageObjectNotFoundError
from .exceptions import LoggedError


def main() -> None:
    try:
        git.ensure_repo()
        terraform.ensure_tf_dir()

        parser = ArgumentParser()
        parser.add_argument("--workspace", "-w", required=True)

        subparsers = parser.add_subparsers()

        parser_plan = subparsers.add_parser("plan")
        parser_plan.set_defaults(func=plan)

        parser_apply = subparsers.add_parser("apply")
        parser_apply.set_defaults(func=apply)

        args = parser.parse_args()
        args.func(args.workspace)
    except LoggedError as e:
        print(e)


def plan(workspace: str):
    git.ensure_clean_working_tree()
    head = git.get_head()
    terraform.init(workspace)
    terraform.plan(head.sha)
    plans.upload(workspace, head.sha)


def apply(workspace: str) -> None:
    head = git.get_head()
    try:
        plans.download(workspace, head.sha)
        planfile = head.sha
    except StorageObjectNotFoundError:
        if head.second_parent and head.is_empty_merge:
            plans.download(workspace, head.second_parent)
            planfile = head.second_parent
        else:
            raise
    terraform.init(workspace)
    terraform.apply(planfile)
