"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.QueueProcessingServiceBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * The base class for QueueProcessingEc2Service and QueueProcessingFargateService services.
 *
 * @stability stable
 */
class QueueProcessingServiceBase extends core_2.Construct {
    /**
     * Constructs a new instance of the QueueProcessingServiceBase class.
     *
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e;
        super(scope, id);
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        // Create the SQS queue and it's corresponding DLQ if one is not provided
        if (props.queue) {
            this.sqsQueue = props.queue;
        }
        else {
            this.deadLetterQueue = new aws_sqs_1.Queue(this, 'EcsProcessingDeadLetterQueue', {
                retentionPeriod: props.retentionPeriod || core_1.Duration.days(14),
            });
            this.sqsQueue = new aws_sqs_1.Queue(this, 'EcsProcessingQueue', {
                deadLetterQueue: {
                    queue: this.deadLetterQueue,
                    maxReceiveCount: props.maxReceiveCount || 3,
                },
            });
            new core_1.CfnOutput(this, 'SQSDeadLetterQueue', { value: this.deadLetterQueue.queueName });
            new core_1.CfnOutput(this, 'SQSDeadLetterQueueArn', { value: this.deadLetterQueue.queueArn });
        }
        // Setup autoscaling scaling intervals
        const defaultScalingSteps = [{ upper: 0, change: -1 }, { lower: 100, change: +1 }, { lower: 500, change: +5 }];
        this.scalingSteps = (_b = props.scalingSteps) !== null && _b !== void 0 ? _b : defaultScalingSteps;
        // Create log driver if logging is enabled
        const enableLogging = (_c = props.enableLogging) !== null && _c !== void 0 ? _c : true;
        this.logDriver = (_d = props.logDriver) !== null && _d !== void 0 ? _d : (enableLogging ? this.createAWSLogDriver(this.node.id) : undefined);
        // Add the queue name to environment variables
        this.environment = { ...(props.environment || {}), QUEUE_NAME: this.sqsQueue.queueName };
        this.secrets = props.secrets;
        // Determine the desired task count (minimum) and maximum scaling capacity
        this.desiredCount = (_e = props.desiredTaskCount) !== null && _e !== void 0 ? _e : 1;
        this.maxCapacity = props.maxScalingCapacity || (2 * this.desiredCount);
        if (!this.desiredCount && !this.maxCapacity) {
            throw new Error('maxScalingCapacity must be set and greater than 0 if desiredCount is 0');
        }
        new core_1.CfnOutput(this, 'SQSQueue', { value: this.sqsQueue.queueName });
        new core_1.CfnOutput(this, 'SQSQueueArn', { value: this.sqsQueue.queueArn });
    }
    /**
     * Configure autoscaling based off of CPU utilization as well as the number of messages visible in the SQS queue.
     *
     * @param service the ECS/Fargate service for which to apply the autoscaling rules to.
     * @stability stable
     */
    configureAutoscalingForService(service) {
        const scalingTarget = service.autoScaleTaskCount({ maxCapacity: this.maxCapacity, minCapacity: this.desiredCount });
        scalingTarget.scaleOnCpuUtilization('CpuScaling', {
            targetUtilizationPercent: 50,
        });
        scalingTarget.scaleOnMetric('QueueMessagesVisibleScaling', {
            metric: this.sqsQueue.metricApproximateNumberOfMessagesVisible(),
            scalingSteps: this.scalingSteps,
        });
    }
    /**
     * Grant SQS permissions to an ECS service.
     *
     * @param service the ECS/Fargate service to which to grant SQS permissions.
     * @stability stable
     */
    grantPermissionsToService(service) {
        this.sqsQueue.grantConsumeMessages(service.taskDefinition.taskRole);
    }
    /**
     * Returns the default cluster.
     *
     * @stability stable
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = core_1.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    /**
     * Create an AWS Log Driver with the provided streamPrefix
     *
     * @param prefix the Cloudwatch logging prefix
     */
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
}
exports.QueueProcessingServiceBase = QueueProcessingServiceBase;
_a = JSII_RTTI_SYMBOL_1;
QueueProcessingServiceBase[_a] = { fqn: "@aws-cdk/aws-ecs-patterns.QueueProcessingServiceBase", version: "1.91.0" };
//# sourceMappingURL=data:application/json;base64,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