"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkTransformer = void 0;
const aws_appsync_1 = require("@aws-cdk/aws-appsync");
const graphql_transformer_core_1 = require("graphql-transformer-core");
const graphqlTypeStatements = ['Query', 'Mutation', 'Subscription'];
class CdkTransformer extends graphql_transformer_core_1.Transformer {
    constructor() {
        super('CdkTransformer', 'directive @nullable on FIELD_DEFINITION');
        this.after = (ctx) => {
            this.buildResources(ctx);
            // TODO: Improve this iteration
            Object.keys(this.tables).forEach(tableName => {
                let table = this.tables[tableName];
                Object.keys(this.resolverTableMap).forEach(resolverName => {
                    if (this.resolverTableMap[resolverName] === tableName)
                        table.resolvers.push(resolverName);
                });
                Object.keys(this.gsiResolverTableMap).forEach(resolverName => {
                    if (this.gsiResolverTableMap[resolverName] === tableName)
                        table.gsiResolvers.push(resolverName);
                });
            });
            // @ts-ignore - we are overloading the use of outputs here...
            ctx.setOutput('cdkTables', this.tables);
            // @ts-ignore - we are overloading the use of outputs here...
            ctx.setOutput('noneResolvers', this.noneDataSources);
            // @ts-ignore - we are overloading the use of outputs here...
            ctx.setOutput('functionResolvers', this.functionResolvers);
            // @ts-ignore - we are overloading the use of outputs here...
            ctx.setOutput('httpResolvers', this.httpResolvers);
            const query = ctx.getQuery();
            if (query) {
                const queryFields = graphql_transformer_core_1.getFieldArguments(query);
                ctx.setOutput('queries', queryFields);
            }
            const mutation = ctx.getMutation();
            if (mutation) {
                const mutationFields = graphql_transformer_core_1.getFieldArguments(mutation);
                ctx.setOutput('mutations', mutationFields);
            }
            const subscription = ctx.getSubscription();
            if (subscription) {
                const subscriptionFields = graphql_transformer_core_1.getFieldArguments(subscription);
                ctx.setOutput('subscriptions', subscriptionFields);
            }
        };
        this.tables = {};
        this.noneDataSources = {};
        this.functionResolvers = {};
        this.httpResolvers = {};
        this.resolverTableMap = {};
        this.gsiResolverTableMap = {};
    }
    buildResources(ctx) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r;
        const templateResources = ctx.template.Resources;
        if (!templateResources)
            return;
        for (const [resourceName, resource] of Object.entries(templateResources)) {
            if (resource.Type === 'AWS::DynamoDB::Table') {
                this.buildTablesFromResource(resourceName, ctx);
            }
            else if (resource.Type === 'AWS::AppSync::Resolver') {
                if (((_a = resource.Properties) === null || _a === void 0 ? void 0 : _a.DataSourceName) === 'NONE') {
                    this.noneDataSources[resource.Properties.FieldName] = {
                        typeName: resource.Properties.TypeName,
                        fieldName: resource.Properties.FieldName,
                    };
                }
                else if (((_b = resource.Properties) === null || _b === void 0 ? void 0 : _b.Kind) === 'PIPELINE') {
                    // Inspired by:
                    // https://github.com/aws-amplify/amplify-cli/blob/master/packages/graphql-function-transformer/src/__tests__/FunctionTransformer.test.ts#L20
                    const dependsOn = (_c = resource.DependsOn) !== null && _c !== void 0 ? _c : '';
                    const functionConfiguration = templateResources[dependsOn];
                    const functionDependsOn = (_d = functionConfiguration.DependsOn) !== null && _d !== void 0 ? _d : '';
                    const functionDataSource = templateResources[functionDependsOn];
                    const functionArn = (_g = (_f = (_e = functionDataSource.Properties) === null || _e === void 0 ? void 0 : _e.LambdaConfig) === null || _f === void 0 ? void 0 : _f.LambdaFunctionArn) === null || _g === void 0 ? void 0 : _g.payload[1].payload[0];
                    const functionName = functionArn.split(':').slice(-1)[0];
                    const fieldName = resource.Properties.FieldName;
                    const typeName = resource.Properties.TypeName;
                    if (!this.functionResolvers[functionName])
                        this.functionResolvers[functionName] = [];
                    this.functionResolvers[functionName].push({
                        typeName: typeName,
                        fieldName: fieldName,
                        defaultRequestMappingTemplate: aws_appsync_1.MappingTemplate.lambdaRequest().renderTemplate(),
                        defaultResponseMappingTemplate: (_h = functionConfiguration.Properties) === null || _h === void 0 ? void 0 : _h.ResponseMappingTemplate,
                    });
                }
                else { // Should be a table/model resolver -> Maybe not true when we add in @searchable, etc
                    const dataSourceName = (_k = (_j = resource.Properties) === null || _j === void 0 ? void 0 : _j.DataSourceName) === null || _k === void 0 ? void 0 : _k.payload[0];
                    const dataSource = templateResources[dataSourceName];
                    const dataSourceType = (_l = dataSource.Properties) === null || _l === void 0 ? void 0 : _l.Type;
                    let typeName = (_m = resource.Properties) === null || _m === void 0 ? void 0 : _m.TypeName;
                    let fieldName = (_o = resource.Properties) === null || _o === void 0 ? void 0 : _o.FieldName;
                    switch (dataSourceType) {
                        case 'AMAZON_DYNAMODB':
                            let tableName = dataSourceName.replace('DataSource', 'Table');
                            if (graphqlTypeStatements.indexOf(typeName) >= 0) {
                                this.resolverTableMap[fieldName] = tableName;
                            }
                            else { // this is a GSI
                                this.gsiResolverTableMap[`${typeName}${fieldName}`] = tableName;
                            }
                            break;
                        case 'HTTP':
                            const httpConfig = (_p = dataSource.Properties) === null || _p === void 0 ? void 0 : _p.HttpConfig;
                            const endpoint = httpConfig.Endpoint;
                            if (!this.httpResolvers[endpoint])
                                this.httpResolvers[endpoint] = [];
                            this.httpResolvers[endpoint].push({
                                typeName,
                                fieldName,
                                httpConfig,
                                defaultRequestMappingTemplate: (_q = resource.Properties) === null || _q === void 0 ? void 0 : _q.RequestMappingTemplate,
                                defaultResponseMappingTemplate: (_r = resource.Properties) === null || _r === void 0 ? void 0 : _r.ResponseMappingTemplate,
                            });
                            break;
                        default:
                            throw new Error(`Unsupported Data Source Type: ${dataSourceType}`);
                    }
                }
            }
        }
    }
    buildTablesFromResource(resourceName, ctx) {
        var _a, _b, _c, _d;
        const tableResource = ctx.template.Resources ? ctx.template.Resources[resourceName] : undefined;
        const attributeDefinitions = (_a = tableResource === null || tableResource === void 0 ? void 0 : tableResource.Properties) === null || _a === void 0 ? void 0 : _a.AttributeDefinitions;
        const keySchema = (_b = tableResource === null || tableResource === void 0 ? void 0 : tableResource.Properties) === null || _b === void 0 ? void 0 : _b.KeySchema;
        const keys = this.parseKeySchema(keySchema, attributeDefinitions);
        let ttl = (_c = tableResource === null || tableResource === void 0 ? void 0 : tableResource.Properties) === null || _c === void 0 ? void 0 : _c.TimeToLiveSpecification;
        if (ttl) {
            ttl = {
                attributeName: ttl.AttributeName,
                enabled: ttl.Enabled,
            };
        }
        let table = {
            tableName: resourceName,
            partitionKey: keys.partitionKey,
            sortKey: keys.sortKey,
            ttl: ttl,
            globalSecondaryIndexes: [],
            resolvers: [],
            gsiResolvers: [],
        };
        const gsis = (_d = tableResource === null || tableResource === void 0 ? void 0 : tableResource.Properties) === null || _d === void 0 ? void 0 : _d.GlobalSecondaryIndexes;
        if (gsis) {
            gsis.forEach((gsi) => {
                const gsiKeys = this.parseKeySchema(gsi.KeySchema, attributeDefinitions);
                const gsiDefinition = {
                    indexName: gsi.IndexName,
                    projection: gsi.Projection,
                    partitionKey: gsiKeys.partitionKey,
                    sortKey: gsiKeys.sortKey,
                };
                table.globalSecondaryIndexes.push(gsiDefinition);
            });
        }
        this.tables[resourceName] = table;
    }
    parseKeySchema(keySchema, attributeDefinitions) {
        let partitionKey = {};
        let sortKey = {};
        keySchema.forEach((key) => {
            const keyType = key.KeyType;
            const attributeName = key.AttributeName;
            const attribute = attributeDefinitions.find((attr) => attr.AttributeName === attributeName);
            if (keyType === 'HASH') {
                partitionKey = {
                    name: attribute.AttributeName,
                    type: attribute.AttributeType,
                };
            }
            else if (keyType === 'RANGE') {
                sortKey = {
                    name: attribute.AttributeName,
                    type: attribute.AttributeType,
                };
            }
        });
        return { partitionKey, sortKey };
    }
}
exports.CdkTransformer = CdkTransformer;
//# sourceMappingURL=data:application/json;base64,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