import os
import re
from setuptools.config import read_configuration
import sys
import subprocess
from neocli.exceptions import (
	InvalidRemoteException,
	InvalidBranchException,
	CommandFailedError,
)
from neocli.app import get_repo_dir


def is_version_upgrade(app="neo", neocli_path=".", branch=None):
	upstream_version = get_upstream_version(app=app, branch=branch, neocli_path=neocli_path)

	if not upstream_version:
		raise InvalidBranchException(
			f"Specified branch of app {app} is not in upstream remote"
		)

	local_version = get_major_version(get_current_version(app, neocli_path=neocli_path))
	upstream_version = get_major_version(upstream_version)

	if upstream_version > local_version:
		return (True, local_version, upstream_version)

	return (False, local_version, upstream_version)


def switch_branch(branch, apps=None, neocli_path=".", upgrade=False, check_upgrade=True):
	import git
	from neocli.neocli import NeoCLI
	from neocli.utils import log, exec_cmd
	from neocli.utils.neocli import (
		build_assets,
		patch_sites,
		post_upgrade,
	)
	from neocli.utils.system import backup_all_sites

	apps_dir = os.path.join(neocli_path, "apps")
	version_upgrade = (False,)
	switched_apps = []

	if not apps:
		apps = [
			name for name in os.listdir(apps_dir) if os.path.isdir(os.path.join(apps_dir, name))
		]

	for app in apps:
		app_dir = os.path.join(apps_dir, app)

		if not os.path.exists(app_dir):
			log(f"{app} does not exist!", level=2)
			continue

		repo = git.Repo(app_dir)
		unshallow_flag = os.path.exists(os.path.join(app_dir, ".git", "shallow"))
		log(f"Fetching upstream {'unshallow ' if unshallow_flag else ''}for {app}")

		exec_cmd("git remote set-branches upstream  '*'", cwd=app_dir)
		exec_cmd(
			f"git fetch --all{' --unshallow' if unshallow_flag else ''} --quiet", cwd=app_dir
		)

		if check_upgrade:
			version_upgrade = is_version_upgrade(app=app, neocli_path=neocli_path, branch=branch)
			if version_upgrade[0] and not upgrade:
				log(
					f"Switching to {branch} will cause upgrade from"
					f" {version_upgrade[1]} to {version_upgrade[2]}. Pass --upgrade to"
					" confirm",
					level=2,
				)
				sys.exit(1)

		print("Switching for " + app)
		exec_cmd(f"git checkout -f {branch}", cwd=app_dir)

		if str(repo.active_branch) == branch:
			switched_apps.append(app)
		else:
			log(f"Switching branches failed for: {app}", level=2)

	if switched_apps:
		log(f"Successfully switched branches for: {', '.join(switched_apps)}", level=1)
		print(
			"Please run `neocli update --patch` to be safe from any differences in"
			" database schema"
		)

	if version_upgrade[0] and upgrade:
		NeoCLI(neocli_path).setup.requirements()
		backup_all_sites()
		patch_sites()
		build_assets()
		post_upgrade(version_upgrade[1], version_upgrade[2])


def switch_to_branch(branch=None, apps=None, neocli_path=".", upgrade=False):
	switch_branch(branch, apps=apps, neocli_path=neocli_path, upgrade=upgrade)


def switch_to_develop(apps=None, neocli_path=".", upgrade=True):
	switch_branch("develop", apps=apps, neocli_path=neocli_path, upgrade=upgrade)


def get_version_from_string(contents, field="__version__"):
	match = re.search(r"^(\s*%s\s*=\s*['\\\"])(.+?)(['\"])(?sm)" % field, contents)
	return match.group(2)


def get_major_version(version):
	import semantic_version

	return semantic_version.Version(version).major


def get_develop_version(app, neocli_path="."):
	repo_dir = get_repo_dir(app, neocli_path=neocli_path)
	with open(os.path.join(repo_dir, os.path.basename(repo_dir), "hooks.py")) as f:
		return get_version_from_string(f.read(), field="develop_version")


def get_upstream_version(app, branch=None, neocli_path="."):
	repo_dir = get_repo_dir(app, neocli_path=neocli_path)
	if not branch:
		branch = get_current_branch(app, neocli_path=neocli_path)

	try:
		subprocess.call(
			f"git fetch --depth=1 --no-tags upstream {branch}", shell=True, cwd=repo_dir
		)
	except CommandFailedError:
		raise InvalidRemoteException(f"Failed to fetch from remote named upstream for {app}")

	try:
		contents = subprocess.check_output(
			f"git show upstream/{branch}:{app}/__init__.py",
			shell=True,
			cwd=repo_dir,
			stderr=subprocess.STDOUT,
		)
		contents = contents.decode("utf-8")
	except subprocess.CalledProcessError as e:
		if b"Invalid object" in e.output:
			return None
		else:
			raise
	return get_version_from_string(contents)


def get_current_neo_version(neocli_path="."):
	try:
		return get_major_version(get_current_version("neo", neocli_path=neocli_path))
	except IOError:
		return 0


def get_current_branch(app, neocli_path="."):
	from neocli.utils import get_cmd_output

	repo_dir = get_repo_dir(app, neocli_path=neocli_path)
	return get_cmd_output("basename $(git symbolic-ref -q HEAD)", cwd=repo_dir)


def get_remote(app, neocli_path="."):
	repo_dir = get_repo_dir(app, neocli_path=neocli_path)
	contents = subprocess.check_output(
		["git", "remote", "-v"], cwd=repo_dir, stderr=subprocess.STDOUT
	)
	contents = contents.decode("utf-8")
	if re.findall(r"upstream[\s]+", contents):
		return "upstream"
	elif not contents:
		# if contents is an empty string => remote doesn't exist
		return False
	else:
		# get the first remote
		return contents.splitlines()[0].split()[0]


def get_app_name(neocli_path, repo_name):
	app_name = None
	apps_path = os.path.join(os.path.abspath(neocli_path), "apps")
	config_path = os.path.join(apps_path, repo_name, "setup.cfg")
	if os.path.exists(config_path):
		config = read_configuration(config_path)
		app_name = config.get("metadata", {}).get("name")

	if not app_name:
		# retrieve app name from setup.py as fallback
		app_path = os.path.join(apps_path, repo_name, "setup.py")
		with open(app_path, "rb") as f:
			app_name = re.search(r'name\s*=\s*[\'"](.*)[\'"]', f.read().decode("utf-8")).group(1)

	if app_name and repo_name != app_name:
		os.rename(os.path.join(apps_path, repo_name), os.path.join(apps_path, app_name))
		return app_name

	return repo_name


def get_current_version(app, neocli_path="."):
	current_version = None
	repo_dir = get_repo_dir(app, neocli_path=neocli_path)
	config_path = os.path.join(repo_dir, "setup.cfg")
	init_path = os.path.join(repo_dir, os.path.basename(repo_dir), "__init__.py")
	setup_path = os.path.join(repo_dir, "setup.py")

	try:
		if os.path.exists(config_path):
			config = read_configuration(config_path)
			current_version = config.get("metadata", {}).get("version")
		if not current_version:
			with open(init_path) as f:
				current_version = get_version_from_string(f.read())

	except AttributeError:
		# backward compatibility
		with open(setup_path) as f:
			current_version = get_version_from_string(f.read(), field="version")

	return current_version
