# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/01_utils.ipynb (unless otherwise specified).

__all__ = ['logger', 'nasa_date_format', 'nasa_dt_format', 'nasa_dt_format_with_ms', 'standard_date_format',
           'standard_dt_format', 'standard_dt_format_with_ms', 'nasa_date_to_iso', 'iso_to_nasa_date',
           'nasa_datetime_to_iso', 'iso_to_nasa_datetime', 'replace_all_nasa_times', 'ProgressBar', 'parse_http_date',
           'get_remote_timestamp', 'download', 'url_retrieve', 'have_internet', 'height_from_shadow',
           'get_gdal_center_coords', 'file_variations']

# Cell
import datetime as dt
import email.utils as eut
import http.client as httplib
import logging
from math import radians, tan
from pathlib import Path
from urllib.request import urlopen, urlretrieve

import requests
from tqdm.auto import tqdm

import pandas as pd

logger = logging.getLogger(__name__)
try:
    from osgeo import gdal
except ImportError:
    GDAL_INSTALLED = False
    logger.warning(
        "No GDAL found. Some planetary.utils functions not working, but okay."
    )
else:
    GDAL_INSTALLED = True

# Cell
nasa_date_format = "%Y-%j"
nasa_dt_format = nasa_date_format + "T%H:%M:%S"
nasa_dt_format_with_ms = nasa_dt_format + ".%f"
standard_date_format = "%Y-%m-%d"
standard_dt_format = standard_date_format + "T%H:%M:%S"
standard_dt_format_with_ms = standard_dt_format + ".%f"

# Cell
def nasa_date_to_iso(datestr, with_hours=False):
    """Convert the day-number based NASA date format to ISO.

    Parameters
    ----------
    datestr : str
        Date string in the form Y-j

    Returns
    -------
    Datestring in ISO standard yyyy-mm-ddTHH:MM:SS.MMMMMM
    """
    date = dt.datetime.strptime(datestr, nasa_date_format)
    if with_hours:
        return date.isoformat()
    else:
        return date.strftime(standard_date_format)

# Cell
def iso_to_nasa_date(datestr):
    """Convert iso date to day-number based NASA date.

    Parameters
    ----------
    datestr : str
        Date string in the form Y-m-d

    Returns
    -------
    Datestring in NASA standard yyyy-jjj
    """
    date = dt.datetime.strptime(datestr, standard_date_format)
    return date.strftime(nasa_date_format)

# Cell
def nasa_datetime_to_iso(dtimestr):
    """Convert the day-number based NASA datetime format to ISO.

    Note: This is dateTIME vs `nasa_date_to_iso` which is just DATE.

    Parameters
    ----------
    dtimestr : str
        Datetime string in the form Y-jTH:M:S

    Returns
    -------
    Datestring in ISO standard yyyy-mm-ddTHH:MM:SS.MMMMMM
    """
    try:
        dtimestr.split(".")[1]
    except IndexError:
        source_format = nasa_dt_format
    else:
        source_format = nasa_dt_format_with_ms
    time = dt.datetime.strptime(dtimestr, source_format)
    return time.isoformat()

# Cell
def iso_to_nasa_datetime(dtimestr):
    """Convert iso datetime to day-number based NASA datetime.

    Parameters
    ----------
    dtimestr : str
        Datetime string in the form yyyy-mm-ddTHH-MM-SS

    Returns
    -------
    Datestring in NASA standard yyyy-jjjTHH-MM-SS
    """

    try:
        dtimestr.split(".")[1]
    except IndexError:
        source_format = standard_dt_format
        target_format = nasa_dt_format
    else:
        source_format = standard_dt_format_with_ms
        target_format = nasa_dt_format_with_ms
    date = dt.datetime.strptime(dtimestr, source_format)
    return date.strftime(target_format)

# Cell
def replace_all_nasa_times(df):
    """Find all NASA times in dataframe and replace with ISO.

    Changes will be implemented on incoming dataframe!

    This will be done for all columns with the word TIME in the column name.

    Parameters
    ----------
    df: pandas.DataFrame
        DataFrame with NASA time columns.

    Returns
    -------
    Nothing (Changes implemented in-place)
    """
    for col in [col for col in df.columns if "TIME" in col]:
        if "T" in df[col].iloc[0]:
            df[col] = pd.to_datetime(df[col].map(nasa_datetime_to_iso))

# Cell


class ProgressBar(tqdm):
    """Provides `update_to(n)` which uses `tqdm.update(delta_n)`."""

    def update_to(self, b=1, bsize=1, tsize=None):
        """
        b  : int, optional
            Number of blocks transferred so far [default: 1].
        bsize  : int, optional
            Size of each block (in tqdm units) [default: 1].
        tsize  : int, optional
            Total size (in tqdm units). If [default: None] remains unchanged.
        """
        if tsize is not None:
            self.total = tsize
        self.update(b * bsize - self.n)  # will also set self.n = b * bsize


def parse_http_date(text):
    "Parse date string retrieved via urllib.request."
    return dt.datetime(*eut.parsedate(text)[:6])


def get_remote_timestamp(url):
    conn = urlopen(url, timeout=10)
    t = parse_http_date(conn.headers["last-modified"])
    conn.close()
    return t


def download(url, local_dir=".", use_tqdm=True, **kwargs):
    """Simple wrapper of urlretrieve

    Adding a default path to urlretrieve

    Parameters:
    ----------
    url : str
        HTTP(S) URL to download
    local_dir : str,pathlib.Path
        Local directory where to store the download.
    **kwargs : {dict}
        Keyword args to be handed to urlretrieve.
    Returns
    -------
    Tuple
        Tuple returned by urlretrieve
    """
    name = url.split("/")[-1]
    local = Path(local_dir)
    savepath = local / name if local.is_dir() else local
    logger.debug("Downloading %s into %s", url, savepath)
    if use_tqdm:
        with ProgressBar(
            unit="B", unit_scale=True, miniters=1, desc=url.split("/")[-1]
        ) as t:  # all optional kwargs
            return urlretrieve(url, savepath, reporthook=t.update_to)
    else:
        return urlretrieve(url, savepath, **kwargs)


def url_retrieve(url: str, outfile: str, chunk_size: int = 128):
    """Improved urlretrieve with progressbar, timeout and chunker.

    This downloader has built-in progress bar using tqdm and using the `requests`
    package it improves standard `urllib` behavior by adding time-out capability.

    I tested different chunk_sizes and most of the time 128 was actually fastest, YMMV.

    Parameters
    ----------
    url : str, urlpath.URL
        The URL to download
    outfile: str, pathlib.Path
        The path where to store the downloaded file.
    chunk_size : int, optional
        The size of the chunk for the request.iter_content call. Default: 128

    See also
    --------
    Inspired by https://stackoverflow.com/a/61575758/680232
    """
    R = requests.get(url, stream=True, allow_redirects=True)
    if R.status_code != 200:
        raise ConnectionError(f"Could not download {url}\nError code: {R.status_code}")
    with tqdm.wrapattr(
        open(outfile, "wb"),
        "write",
        miniters=1,
        total=int(R.headers.get("content-length", 0)),
        desc=str(Path(outfile).name) + "\n",
    ) as fd:
        for chunk in R.iter_content(chunk_size=chunk_size):
            fd.write(chunk)


def have_internet():
    """Fastest way to check for active internet connection.

    From https://stackoverflow.com/a/29854274/680232
    """
    conn = httplib.HTTPConnection("www.google.com", timeout=5)
    try:
        conn.request("HEAD", "/")
        conn.close()
        return True
    except:
        conn.close()
        return False

# Cell

def height_from_shadow(shadow_in_pixels, sun_elev):
    """Calculate height of an object from its shadow length.

    Note, that your image might have been binned.
    You need to correct `shadow_in_pixels` for that.

    Parameters
    ----------
    shadow_in_pixels : float
        Measured length of shadow in pixels
    sun_elev : angle(float)
        Angle of sun over horizon

    Returns
    -------
    height [meter]
    """
    return tan(radians(sun_elev)) * shadow_in_pixels

# export


def get_gdal_center_coords(imgpath):
    """Get center rows/cols pixel coordinate for GDAL-readable dataset.

    Check CLI `gdalinfo --formats` to see all formats that GDAL can open.

    Parameters
    ----------
    imgpath: str, pathlib.Path
        Path to raster image that is readable by GDLA
    """
    if not GDAL_INSTALLED:
        logger.error("GDAL not installed. Returning")
        return
    ds = gdal.Open(str(imgpath))
    xmean = ds.RasterXSize // 2
    ymean = ds.RasterYSize // 2
    return xmean, ymean


def file_variations(filename, extensions):
    """Create a variation of file names.

    Generate a list of variations on a filename by replacing the extension with
    the provided list.

    Parameters
    ----------
    filename: str, pathlib.Path
        The original file name to use as a base.
    extensions: list-like
        A list of file extensions to generate new filenames.

    Returns
    -------
    list of pathlib.Paths

    Notes
    -----
    Adapted from T. Olsens `file_variations of the pysis module for using pathlib.
    """
    return [Path(filename).with_suffix(extension) for extension in extensions]