import networkx as nx
import numpy as np
import pandas as pd
import numdifftools
import itertools #this is a base package and does not need to go in the requirements



def find_loops(jacobian,max_num_loops=100000):
	"""
	Function to determine all feedback loops in an ordinary differential equation (ODE)
	model given by its Jacobian matrix. Strongly connected components are determined 
	before cycle detection.

	:Parameters:

		- jacobian (numpy array): matrix indicating the Jacobian of the ODE, :math:`J_ij = df_i/dx_j`.
		- max_num_loops (int, optional): positive integer (default: 1e5) giving 
		  the maximal number of reported feedback loops. Note that only up to max_num_loops feedback loops are returned (without warning).

	:Returns:

		A pandas dataframe: dataframe with the three columns "loop", "length" and "sign" that contain the 
		information on one feedback loop in each row. "loop" gives the order of the variables (indices)
		forming the loop as tuple, "length" indicates the number of participating species and "sign" is +1 or -1 indicating
		a positive or a negative loop, respectively.

	:Details:

		A feedback loop is a circular regulation without visiting any node twice. This function relies on
		the Python modules networkx to detect strongly connected components and simple cycles in the graph
		generated by the Jacobian matrix.

	:Example:

	Define the Jacobian matrix of a four-variable system (jac) and compute all feedback loops::

		#import the relevant packages 
		import loopdetect.core
		import numpy as np
		#define the Jacobian matrix as numpy array
		jac = np.array([[-1,0,0,-1],[1,-1,0,1],[0,1,-1,0],[0,0,1,-1]])
		#compute the loop list
		loop_list = loopdetect.core.find_loops(jac)
	
	"""
	#only use sign of Jacobian
	jacobian_tmp=np.sign(jacobian);
	#get selfloops 
	list_selfloops=[(tuple([i,i]),1,int(jacobian_tmp[i,i])) for i in range(len(jacobian_tmp)) if not(jacobian_tmp[i,i]==0)]
	#erase self-loops from the matrix in order not to count them twice (if necessary)
	if len(list_selfloops)>0:
		np.fill_diagonal(jacobian_tmp,0)
	#generate graph from matrix
	g = nx.DiGraph(np.transpose(jacobian_tmp))
	#determine strongly connected components
	comps = nx.strongly_connected_components(g)
	#initialize loop list
	loop_list=[]
	loop_count=0 # count loops along
	#go through each component
	for comp_temp in comps: 
		if len(comp_temp)>1: #only iterate if the component has more than 1 node
			g_subnetwork = g.subgraph(comp_temp)
			if max_num_loops==None: #reduce number of searched cycles, if necessary
				max_loops=None
			else:
				max_loops=max_num_loops-loop_count
			cycles = itertools.islice(nx.simple_cycles(g_subnetwork), max_loops)
			loop_list = loop_list + [(tuple(cycle+[cycle[0]]), #the cycle
				len(cycle), #the length of the cycle
				int(np.sign(np.prod([jacobian_tmp[cycle[i],cycle[i-1]] 
				for i in range(1,len(cycle))])*jacobian_tmp[cycle[0],cycle[len(cycle)-1]]))) #the sign of the cycle (positive or negative)
				for cycle in cycles]
			loop_count=len(loop_list)
			if loop_count == max_num_loops: #stop if we have enough loops
				break
	#add self-loops
	if max_num_loops == None:
		loop_list=loop_list+list_selfloops
	else:
		loop_list=loop_list+list_selfloops[0:(max_num_loops-loop_count)]
	#cast list into pandas dataframe
	df = pd.DataFrame(loop_list,
		columns=["loop","length","sign"])
	return df



def find_loops_noscc(jacobian,max_num_loops=100000):
	"""
	Function to determine all feedback loops in an ordinary differential equation (ODE)
	model given by its Jacobian matrix. No strongly connected components are determined 
	before cycle detection.

	:Parameters:

		- jacobian (numpy array): matrix indicating the Jacobian of the ODE, :math:`J_ij = df_i/dx_j`.
		- max_num_loops (int, optional): positive integer (default: 1e5) giving the maximal number 
		  of reported feedback loops. Note that only up to max_num_loops feedback loops are returned (without warning).

	:Returns:

		A pandas dataframe: dataframe with the three columns "loop", "length" and "sign" that contain the 
		information on one feedback loop in each row. "loop" gives the order of the variables (indices)
		forming the loop as tuple, "length" indicates the number of participating species and "sign" is +1 or -1 indicating
		a positive or a negative loop, respectively.

	:Details:

		A feedback loop is a circular regulation without visiting any node twice. This function relies on
		the Python modules networkx to detect simple cycles in the graph
		generated by the Jacobian matrix.

	:See Also:

		find_loops

	:Example:

	Define the Jacobian matrix of a four-variable system (jac) and compute all feedback loops::

		#import the relevant packages 
		import loopdetect.core
		import numpy as np
		#define the Jacobian matrix as numpy array
		jac = np.array([[-1,0,0,-1],[1,-1,0,1],[0,1,-1,0],[0,0,1,-1]])
		#compute the loop list
		loop_list = loopdetect.core.find_loops_noscc(jac)
	
	"""
	# wrapper for simple_cycles without detecting strongly connected components.
	# restrict default maximal number of detected loops to max_num_loops, default 1e5.
	# runs just fine on smaller matrices, to be determined for larger matrices
	#only use sign of the Jacobian matrix
	jacobian_tmp=np.sign(jacobian);
	#generate graph from matrix
	g = nx.DiGraph(np.transpose(jacobian_tmp))
	cycles = itertools.islice(nx.simple_cycles(g), max_num_loops) #this cuts the loops such that only a portion is given
	# simple_cycles uses Johnson's algorithm
	#to determine whether this is the best or whether determining strongly connected components is worth
	#go through all cycles and determine a list; generate a pandas dataframe from the list
	df = pd.DataFrame(
		[(tuple(cycle+[cycle[0]]), #define the cycle as list
		len(cycle), #save the length of the cycle
		int(np.sign(np.prod([jacobian_tmp[cycle[i],cycle[i-1]] 
		for i in range(1,len(cycle))])*jacobian_tmp[cycle[0],cycle[len(cycle)-1]])))#the sign of the cycle (positive or negative)
		for cycle in cycles],
		columns=["loop","length","sign"])
	return df




def find_loops_vset(fun,vset,*args,numdiff_method='central',max_num_loops=100000,compute_full_list=True,**kwargs):
	"""
	Determines loop lists for an ODE system given by a function
	and at multiple sets of variables. Loop lists are reported if signs of
	Jacobian matrix have changed.


	:Parameters:

		- fun (function): function defining the ODE system, returns the vector :math:`dx/dt` of derivatives of the variables
		  as a numpy ndarray. The derivative is taken in direction of the first parameter of the function (i.e. :math:`x` if the function returns 
		  :math:`dx/dt`). It may depend on further parameters (defined in \*args, \*\*kwargs).
		- vset (list): list of lists or list of tuples of variable values at which the loops are determined.	
		- \*args  Further parameters except variable values to the function fun.
		- numdiff_method (string, optional): central finite difference approach for derivative calculation ('central',default)
		  or 'complex' for complex-step approach.
		- max_num_loops (int, optional): Positive numeric value indicating the maximal number
		  of loops that are reported in a loop list. Defaults to :math:`10^5`.
		- compute_full_list (bool, optional): Logical value indicating whether for each
		  Jacobian matrix with any different sign the loop list is computed (True, default),
		  or whether further checks are performed to ensure that loops may be altered (see Details).
		- \*\*kwargs: any number of further (named) parameters that are all used as input to the function fun.

	:Details:

		The Jacobian matrices are computed for each of the variable
		values defined in vset with the package numdifftools using
		the standard central finite difference approach or the complex-step approach, depending
		on the input numdiff_method. The complex-step will only deliver correct results
		if the function works with complex numbers and does not contain non-analytic parts 
		such as min, max or abs. Please be aware that functions have to return numpy arrays!
		If compute_full_list is set to False, loop lists are not re-computed
		for Jacobians that clearly do not allow for altered loop lists. These two criteria 
		are checked: (a) no new regulations appear, and (b) only signs of regulations are
		altered that are not member of any loop. Loop lists can still be
		identical for different Jacobians, e.g. if two sign switches occur that
		are both affecting the same loops.

	:Returns:

		dictionary: containing four entries:

		- loop_rep: List of loop lists (pandas dataframes).
		- loop_rep_index: Vector of integer numbers returning the index of the
		  loop list in loop_rep belonging to each entry in vset.
		- jac_rep: List of signed Jacobian matrices.
		- jac_rep_index: Vector of integer numbers returning the index of the
		  Jacobian matrix in jac_rep belonging to each entry in vset.

	:Details:
		
		If there is only one class of Jacobian matrix (i.e. the signs of the
		Jacobian matrix are the same for all entries in vset), loop_rep and
		jac_rep will have only one entry each. The number of entries for
		loop_rep_index and jac_rep_index corresponds to the length of vset.
		Only if compute_full_list is set to False, loop_rep can contain
		fewer elements than jac_rep, otherwise both have the same number of
		elements.

	:See Also:

		find_loops
		numdifftools.Jacobian

	:Examples:

	Perform the loop analysis for a bacterial cell cycle system::

		#import the relevant packages 
		import loopdetect.core as ld
		import loopdetect.examples as lde
		import numpy as np
		#Note: the example function from the script func_li08.py is loaded together
		#with the other functions from loopdetect
		#read in solutions from the data accompanying the package, file li08_solution.tsv
		sols=lde.load_li08_sol()
		sols_as_tuples=[tuple(sols.iloc[i,1:20]) for i in range(len(sols))] #removing the time column
		#we reduce the loop lists as much as possible
		#attention: this might take a minute to run
		loop_results = ld.find_loops_vset(lde.func_li08,vset=sols_as_tuples,numdiff_method='central',
			max_num_loops=100000,compute_full_list=False,t=0)
		loop_results['loop_rep']

	Perform the loop analysis for a simpler system with a function with more input parameters and defined in complex numbers::
		
		#The example function func_POSm4_comp() is that of a 4-variable system and is supplied
		#together with the other functions in loopdetect.
		#define 5 possible variables (4-variable-system) in as list, as tuples
		vset_def = [f for f in zip([1,1,2,2,3],[0,1,2,3,4],[2,2,2,2,2],[3,0,1,2,3])]
		#we compute the loop lists at parameter values klin = [1,2,3,4,5,6,7,8] (in *args) 
		#and knonlin=[2,2] (as **kwargs) using complex step derivatives
		loop_results = ld.find_loops_vset(lde.func_POSm4_comp,vset_def,[1,2,3,4,5,6,7,8],
			numdiff_method='complex',compute_full_list=True,knonlin=[2,2])
		#please be aware that the function func_POSm4_comp defines complex values 
		# (otherwise the Jacobian would be always zeros everywhere for numdiff_method='complex')
	
	"""
	#determine the signs of the Jacobian matrices using numdifftools
	jac_list=[np.sign(numdifftools.Jacobian(fun,method=numdiff_method)(solvec,*args,**kwargs)).real
		for solvec in vset]
	#only keep the unique entries: generate a tuple from all jacobians, keep only uniques (set)
	#and then cast the tuples back into ndarrays
	jac_rep = [np.reshape(x,jac_list[0].shape) for x in list(set([tuple(v.flatten()) for v in jac_list]))]
	#the order of filling the array coincides with the order of flattening!
	#indices of the Jacobian for each vset entry
	jac_rep_index = np.array([[np.all(jac_rep_temp-jac_temp==0) for jac_rep_temp in jac_rep].index(True) for jac_temp in jac_list])
	#compute the loops
	if compute_full_list: #compute Jacobian for every entry
		loop_rep = [find_loops(jac_temp,max_num_loops) for jac_temp in jac_rep]
		loop_rep_index = jac_rep_index
	else:
		#here we save the indices of the loop_list belonging to a set in vset
		loop_rep_index = np.array([0 if jac_rep_index[i]==0 else -1 for i in range(len(jac_rep_index))])
		loop_rep = [find_loops(jac_rep[0],max_num_loops)]
		if len(jac_rep)>1: #if we have more than one different sign structure in the Jacobians
			for i in range(1,len(jac_rep)): #start from 1, this is on purpose (2nd Jacobian matrix), find appropriate loop list for Jacobian i
				j_temp = jac_rep[i]
				#check whether there was a switch from zero to nonzero in any entry to a Jacobian already examined
				switch_to_nz_vec = [np.any(np.abs(j_temp[jac_rep_temp==0])>0) 
					for jac_rep_temp in jac_rep[0:i]]
				#save for each entry whether a sign switch affects instead an existing loop
				existing_loop_changed = [False]*i
				#check the Jacobians in that we did not have any switch further
				for jtempswitch_ind in np.where(np.logical_not(switch_to_nz_vec))[0]: 
					#obtain those indices in which there is a switch in sign instead
					tn_sn_ind = np.where(np.abs(j_temp-jac_rep[jtempswitch_ind])>0)
					#find loops that contain these edges
					for edge_ind in range(len(tn_sn_ind[0])): #loop over all edges with switched sign
						#detect all loops with such an edge
						num_edges = sum(find_edge(loop_rep[loop_rep_index[jac_rep_index==jtempswitch_ind][0]],
							source_node=tn_sn_ind[1][edge_ind],target_node=tn_sn_ind[0][edge_ind]))
						if num_edges>0: #if there is at least one loop affected, 
							existing_loop_changed[jtempswitch_ind] = True
							break
				#if there is a switch from non-existing to existing or a switch of an edge 
				#in an existing loop - the new jacobian could give rise to a new loop list
				if np.all([sw | elc for sw, elc in zip(switch_to_nz_vec, existing_loop_changed)]):
					#add the correct loop list index to the list for each element in vset
					loop_rep_index[jac_rep_index==i] = len(loop_rep)
					#add the correct loop list
					loop_rep.append(find_loops(jac_rep[i],max_num_loops))
					
				else: #this mean the new jacobian does not change any edge 
				#from non-existing to existing for at least one jacobian and also does not alter
				#any edge within an existing loop from this jacobian
				#find out which loop result corresponds to the jacobian at hand
					loop_rep_index[jac_rep_index==i] = loop_rep_index[
						jac_rep_index==np.where(
						[sw | elc for sw, elc in zip(switch_to_nz_vec, existing_loop_changed)])[0][0]][0]
	#return all quantities
	return {'jac_rep':jac_rep,'jac_rep_index':jac_rep_index,'loop_rep':loop_rep,'loop_rep_index':loop_rep_index}	



def compare_loops(loop_list_a,loop_list_b,list_b_inds='True'):
	"""Compares two loop lists to find identical loops. Same node indices are considered 
to encode the same nodes in the compared loops. Sets of indices are returned.

:Parameters:

	- loop_list_a (pandas dataframe): loop list, i.e. pandas dataframe with columns loop, length, sign
	- loop_list_b (pandas dataframe): loop list, i.e. pandas dataframe with columns loop, length, sign
	- list_b_inds (boolean, optional): whether also the indices for the second list are computed (might take long)

:Details:

	Loop lists could be generated from functions find_loops or find_loops_noscc. 
	Indices of the loops are sorted internally with the function sort_loop_index()
	to start with the lowest node index and make comparison possible.

:Returns:

	A dictionary containing five entries

	- ind_a_id: indices of the loops in the first loop list that occur 
	  identically in the second loop list
	- ind_a_switch: indices of the loops in the first loop list that occur
	  in the second loop list with a different sign
	- ind_a_notin: indices of the loops in the first loop list that do not
	  occur in the second loop list
	- ind_b_id: indices of loops in the second loop list corresponding to
	  the loops reported in ind_a_id
	- ind_b_switch: indices of loops in the second loop list corresponding
	  to loops reported in ind_a_switch.

:See Also:
	
	find_loops, find_loops_noscc

:Example:

Comparing loop lists from two systems of size 4 with coinciding nodes::

	#import the relevant packages 
	import loopdetect.core
	import numpy as np
	#define the Jacobian matrix as numpy array
	jac = np.array([[-1,0,0,-1],[1,-1,0,1],[0,1,-1,0],[0,0,1,-1]])
	#compute the loop list
	loop_list = loopdetect.core.find_loops(jac)
	#define a slightly different Jacobian matrix
	jac2 = np.array([[-1,0,0,0],[1,-1,0,-1],[0,1,1,0],[0,0,1,-1]])
	#compute the loop list
	loop_list2 = loopdetect.core.find_loops(jac2)
	#compare the loop lists
	loop_compare = loopdetect.core.compare_loops(loop_list,loop_list2)
	#get sublist of all loops that match
	loop_list.loc[loop_compare['ind_a_id']]
	#get only loops in first list that match exactly a loop in the second list
	loop_list.loop[loop_compare['ind_a_id']]



"""
	#sort the indices of both lists
	loop_list_a_sort=sort_loop_index(loop_list_a)	
	loop_list_b_sort=sort_loop_index(loop_list_b)
	#
	#compare loops via merging pandas dataframes by the loops in list a
	merge_ab=loop_list_a_sort.merge(loop_list_b_sort,on='loop',how='left')
	#make sure that integer arrays are reported, this enables the use as indices
	ind_a_id=np.where(merge_ab.sign_x.eq(merge_ab.sign_y))[0].astype('int64')
	#find NaNs - these loops have not been found in the list
	ind_a_notin=np.where(np.isnan(merge_ab.sign_y))[0].astype('int64')
	#mapped loops with same length (i.e. not NA for second list) but different sign
	ind_a_switch=np.where(merge_ab.length_x.eq(merge_ab.length_y) & 
		merge_ab.sign_x.ne(merge_ab.sign_y))[0].astype('int64')
	#if also the indices for the second list should be determined
	if list_b_inds:
		ind_b_id=np.array([np.where(loop_list_b_sort.loop==id_loop)[0][0] for 
			id_loop in loop_list_a_sort.loop[ind_a_id]]).astype('int64')
		ind_b_switch=np.array([np.where(loop_list_b_sort.loop==id_loop)[0][0] for 
			id_loop in loop_list_a_sort.loop[ind_a_switch]]).astype('int64')
	else:
		ind_b_id=np.nan
		ind_b_switch=np.nan
	#
	return {'ind_a_id':ind_a_id, 'ind_a_switch':ind_a_switch, 'ind_a_notin':ind_a_notin, 
		'ind_b_id':ind_b_id, 'ind_b_switch':ind_b_switch}




def find_edge(loop_list,source_node,target_node):
	"""
	Reports which loops in a loop list contain a certain edge.

	:Parameters:
		
		- loop_list (pandas dataframe): loop list as generated from find_loops, has to contain a column named 'loop'
		- source_node (int): index of the source node of the edge
		- target_node (int): index of the target node of the edge


	:Returns:

		A list with Boolean values of the same length as the number of rows in the 
		loop list. The values are True if the rows contain the edge, False if not.

	:See Also:

		find_loops, find_loops_noscc

	:Example:
	
	Finding all loops with an edge between species 0 and 1::

		#import the relevant packages 
		import loopdetect.core
		import numpy as np
		#define the Jacobian matrix as numpy array
		jac = np.array([[-1,0,0,-1],[1,-1,0,1],[0,1,-1,0],[0,0,1,-1]])
		#compute the loop list
		loop_list = loopdetect.core.find_loops_noscc(jac)
		#find loops containing the edge [0,1]
		first_edge = loopdetect.core.find_edge(loop_list,0,1)
		#return the loops containing the edge
		loop_list.iloc[first_edge]
	"""
	def fedge(listin):
		#this function checks a list for the entry source_node followed by target_node
		inds = -1
		try: 
			inds = listin.index(source_node)
		except (ValueError):
			pass
		if inds>-1:
			if listin[inds+1]==target_node:
				return True
			else:
				return False
		else:
			return False
	#
	return [fedge(list(l)) for l in loop_list['loop']]






def sort_loop_index(loop_list):
	"""
	Sort all loops in a loop list such that they start with the node 
	with the smallest index. This is a helper function for comparing loop lists.


	:Parameters:

		- loop_list (pandas dataframe): loop list as generated from find_loops, has to 
		  contain a column named 'loop'

    :Returns:

    	- pandas dataframe containing the sorted list of loops 

    :See Also:

		find_loops, find_loops_noscc

    :Example:

    Create a new loop list with sorted entries (starting from the smalles node index).::

    	#import the relevant packages 
		import loopdetect.core
		import numpy as np
		#define the Jacobian matrix as numpy array
		jac = np.array([[-1,0,0,-1],[1,-1,0,1],[0,1,-1,0],[0,0,1,-1]])
		#compute the loop list
		loop_list = loopdetect.core.find_loops(jac)
		#manipulate the second loop to start with a different entry than the smallest
		loop_list.loop[1]=tuple([2,3,1,2])
		#sort loop order
		sorted_loop_list = loopdetect.core.sort_loop_index(loop_list)
    """ 
	sorted_loops = pd.DataFrame(zip([listin[list(listin).index(min(listin)):len(listin)]+
		listin[1:list(listin).index(min(listin))+1] if len(listin)>2 
		else listin for listin in loop_list['loop']],
		loop_list['length'],loop_list['sign']),columns=["loop","length","sign"])
	return(sorted_loops)


def loop_summary(loop_list,column_val='length'):
	"""
	Reports a summary of the loops in a loop list: numbers of positive (pos)
	loops or negative (neg) loops of a certain length

	:Parameters:

		- loop_list (pandas dataframe): loop list as generated from find_loops, has to 
		  contain a column named 'loop'
		- column_val (string, optional): if this is 'length' (default), the column names are the length, 
		  otherwise the sign of the loop are the columns

	:Returns:

		A pandas dataframe containing the counts of negative (neg), positive (pos) or all loops
		(total) subdivided by their length. Lengths that do not occur are omitted.

	:See Also:

		find_loops, find_loops_noscc

	:Example:

	Summarize a short loop list.::

		#import the relevant packages 
		import loopdetect.core
		import numpy as np
		#define the Jacobian matrix as numpy array
		jac = np.array([[-1,0,0,-1],[1,-1,0,1],[0,1,-1,0],[0,0,1,-1]])
		#compute the loop list
		loop_list = loopdetect.core.find_loops_noscc(jac)
		#determine summary
		sum_tab = loopdetect.core.loop_summary(loop_list)
	"""
	def iseqm1(x):
		#helper function, count how many negative loops (entry -1)
		return sum(x==-1)
	#
	def iseqp1(x):
		#helper function, count how many positive loops (entry 1)
		return sum(x==1)
	#
	sum_tab = loop_list.groupby('length').agg(total=pd.NamedAgg(column='sign',aggfunc=len),
		neg=pd.NamedAgg(column='sign',aggfunc=iseqm1),
		pos=pd.NamedAgg(column='sign',aggfunc=iseqp1)
		)
	if column_val=='length':
		sum_tab = sum_tab.transpose()
	#
	return(sum_tab)





