# Copyright (c) Facebook, Inc. and its affiliates. (http://www.facebook.com)
# -*- coding: utf-8 -*-

"""
tests.conftest.py

This module loads variables needed to run the test

:copyright: (c) 2021 Facebook
:license: MIT LICENSE
"""

# Package imports
import os
import re
import pytest
import logging
import requests
import mercantile
import mapillary as mly
from vt2geojson.tools import vt_bytes_to_geojson

logger = logging.getLogger(__name__)


@pytest.fixture(autouse=True)
def testing_envs():
    """Set the environment variables needed for testing"""

    envs = {}

    # Data directory, where the data to be used in tests as input is to be used
    envs["DATA_DIRECTORY"] = "tests/data"

    # Dump directory, for data that is generated by tests
    envs["DUMP_DIRECTORY"] = "tests/dump"

    ## Monaco Dump - tests.test_interface.test_traffic_signs_in_bbox
    envs["MONACO_DUMP"] = f"{envs['DUMP_DIRECTORY']}/"

    # List of tests potentially able to break code. To test against
    envs[
        "BAD_TILES_US_WEST_COAST"
    ] = f"{envs['DATA_DIRECTORY']}/bad_tiles_west_coast_run.txt"

    # List of traffic signs offered at
    # https://www.mapillary.com/developer/api-documentation/traffic-signs
    envs[
        "TRAFFIC_SIGNS_FILTER_VALUES"
    ] = f"{envs['DATA_DIRECTORY']}/traffic_signs_filter_values.csv"

    # TEST_INTERFACE

    ## test_traffic_signs_in_bbox
    envs[
        "TRAFFIC_SIGNS_IN_BBOX_DIR"
    ] = f"{envs['DUMP_DIRECTORY']}/test_interface.test_traffic_signs_in_bbox"

    # If needed, take back the OS Environment Variable Dictionary
    return envs


@pytest.fixture(autouse=True)
def mly_access_token():
    """Specify the access token here"""

    # The default token is the "Testing" token specially created for testing services
    # Ref:: 'PyTest Workflow for Mapillary API v4 Python SDK'
    token = "MLY|3976749565769581|3022bf582da20b0ce0bb44373289ebb2"

    # Check if the token is specified correctly
    if "MLY|YYY" in token or "MLY|XXX" in token:
        raise ValueError("[tests.conftest]: MLY Access Token is not specified")

    # Return token
    return token


@pytest.fixture(autouse=True)
def data(mly_access_token):
    """For retrieving the data from the API to check against the SDK's result"""

    # Initializes the default values
    zoom, longitude, latitude = 14, 31, 30

    # Retrieve the tile to test
    tile = mercantile.tile(lng=longitude, lat=latitude, zoom=zoom)

    # Get the GeoJSON from the API end point
    data = vt_bytes_to_geojson(
        b_content=requests.get(
            f"https://tiles.mapillary.com/maps/vtp/mly1_public/2/14/{tile[0]}/{tile[1]}/"
            f"?access_token={mly_access_token}"
        ).content,
        x=tile.x,
        y=tile.y,
        z=tile.z,
        layer="image",
    )

    # Return results as a dictionary
    return {"data": data, "tile": tile, "longitude": longitude, "latitude": latitude}


@pytest.fixture(autouse=True)
def directories(testing_envs):
    """Creates the necessary needed directories"""

    # A simpler variable to read
    testing_envs = testing_envs

    # Check for DUMP_DIRECTORY
    if not os.path.exists(testing_envs["DUMP_DIRECTORY"]):
        try:
            os.makedirs(testing_envs["TRAFFIC_SIGNS_IN_BBOX_DIR"])
        except OSError as error:
            logger.error(
                f"\n[tests.conftest.directories] Creation of needed directories for testing failed, raising OSError: {error}"
            )
            raise OSError

    # Check for DATA_DIRECTORY
    if not os.path.exists(testing_envs["DATA_DIRECTORY"]):
        logger.error(
            f'\n[tests.conftest.directories] Data directory needed for tests not found, raising OSError: "Data directory not found"'
        )
        raise OSError

    # Directories successfully operated against
    return True


@pytest.fixture(autouse=True)
def test_initialize(mly_access_token: str, directories: bool, testing_envs: dict):

    # Assert directories were successfully created
    assert directories == True

    # Assert that the environment variable set is not empty
    assert bool(testing_envs) == True

    # Assert that the token is not empty
    assert mly_access_token != ""

    ## ... and then set the token
    mly.interface.set_access_token(mly_access_token)

    # For storing tile coordinates
    testing_tile_data = []

    ## Regex string
    re_str = r"^\(.*\(x\=(-?\d+.?\d+?), y\=(-?\d+.?\d+?), z=(\d+)\)\)$"

    # Opening bad tiles file ...
    with open(testing_envs["BAD_TILES_US_WEST_COAST"]) as bad_tile_fp:

        ## ... while file input has not finished ...
        for line in bad_tile_fp:

            ## Append the extracted Tile parameters
            testing_tile_data.append(re.search(re_str, line).groups())

    # Return dictionary
    return {"testing_envs": testing_envs, "testing_tile_data": testing_tile_data}
