"""Provides functions to manipulate projects using the Project Manager API
The module contains project related functions, allowing for creation, update and deletion of
projects.
"""

# imports
from typing import TypedDict
from .case import Case
from .study import Study
from .geometry import Geometry, getGeometry
from .base import *

# project dictionary types
class CreateProjectData(TypedDict):
    name: str
    params: list[str]

class Authorization(TypedDict):
    """Provides a user with a set of rights to a project.
    
    Fields:
        `by`: The ID of the user that has created this authorization. This user must have the right
         to authorize other users. The owner of the project implicitly can authorize other users.
         `user`: The ID of the user authorized to access the project.
         `rights`: The list of rights granted to the project.
    """
    by: str
    """The ID of the user that has created this authorization.
    Type:
        `str`: The ID is a string representation of the ObjectID autogenerated user record ID.
    """

    user: str
    """The ID of the user authorized to access the project
    
    Type:
        `str`: The ID is a string representation of the ObjectID autogenerated user record ID.
    """

    rights: list[str]
    """The list of rights granted to the project
    
    Type:
        `list[str]`: The name of the rights are restricted to the following (case sensitive):
            `create`: Allows the user to add new elements to the project (i.e. create a new case,
            or add a new geometry).
            `read`: Allows the user to read the content of the project.
            `update`: Allows the user to change the data for the project elements (i.e. change
            parameter values for a case).
            `delete`: Allows the user to delete elements of a project.
            `assign`: Allows the user to give rights to other users to access the project.
            `download`: Allows the user to download files associated with the project (i.e.
            download a geometry, or results of a run).
    """

class Project(TypedDict):
    """Contains the data for a project.
    
    Fields:
        `_id`: The string representation of ObjectID autogenerated record ID.
        `name`: The project name
        `owner`: The string representation of ObjectID autogenerated record ID for the user that
        owns the project. This typically is the user that initially created the project.
        `children`: A list of ObjectID, containing the IDs of the studies and cases that are
        direct children of the project.
        `params`: The list of parameter group IDs for the parameter groups that are supported by
        this project. Currently only `EDP$DefaultParameters` is supported.
        `geometries`: The list of geometry IDs for the geometries that are used in this project.
        `authorized`: A list of `Authorization` objects providing other users with rights to this
        project.
    """

    _id: str
    """The string representation of ObjectID autogenerated record ID.
    Type:
        `str`: The ID is a string representation of the ObjectID autogenerated record ID.
    """
    
    name: str
    """The project name as given by the user
    
    Type:
        `str`: The project name
    """
    
    owner: str
    """The ID of the user that owns this project, typically the user that created the project.
    Type:
        `str`: The ID is a string representation of the ObjectID autogenerated user record ID.
    """

    children: list[str]
    """The list of the studies and cases that are direct children of the project.
    
    Type:
        `list[str]`: A list of ObjectIDs in their string representation.
    """

    params: list[str]
    """The list of parameter groups that can be used in the project.
    Type:
        `list[str]`: A list of IDs for the parameter groups that can be used in the project.
        Currently the only ID supported is `EDP$DefaultParameters`.
    """

    geometries: list[str]
    """The list of geometries that are used in this project.
    
    Type:
        `list[str]`: A list of IDs for the geometries that are used in this project.
    """

    authorized: list[Authorization]
    """The list of authorizations issued to users other than the owner that allow them access to
    this project.
    Type:
        `list[Authorization]`: A list of `Authorization` instances that describe the rights
        assigned to users other than the owner to provide them access to this project.
    """

def createProject(accessToken: str, projectName: str) -> Project | None:
    """Creates a new project.
    Args:
        accessToken: A string containing a valid access token returned by a call to the `login`
        function.
        projectName: A string containing the name of the project to be created.
    
    Returns:
        `Project | None`: the newly created project if successful or `None` otherwise
    Remarks:
        The function assigns `EDP$DefaultParameters` to the list of supported parameter groups.
    """
    
    return post(
        url = "/project",
        data =  { 'name': projectName, 'params': [ 'EDP$DefaultParameters' ] },
        accessToken = accessToken,
        success = 201,
        knownErrors =  {
            400: "Failed to create the project.",
            "unknown": "Unknown error. Failed to create the project."
        },
    )

def enumAllProjects(accessToken: str) ->list[Project] | None:
    """Enumerates all user's projects.
    
    Args:
        accessToken: A string containing a valid access token returned by a call to the `login`
        function.
    
    Returns:
        `list[Project] | None`: the list of user's projects if successful or `None` otherwise.
    """

    projects = get(
        url = "/project",
        accessToken = accessToken,
        knownErrors =  {
            404: "Failed to retrieve the projects.",
            "unknown": "Unknown error. Failed to retrieve the projects."
        },
    )

    if projects != None:
        return projects
    
    print('Failed to get the list of projects')

def getProject(accessToken: str, projectId: str) -> Project | None:
    """Retrieves a project based on its ID.

    Args:
        accessToken: A string containing a valid access token returned by a call to the `login`
        function.
        projectId: A string containing the ID of the project to be retrieved.

    Returns:
        `Project | None`: The project data if a project with the requested ID exists and belongs
        to the owner of the `accessToken`, or `None` otherwise.
    """

    return get(
        url = "/project/{projectId}".format(projectId = projectId),
        accessToken = accessToken,
        knownErrors =  {
            404: "The project does not exist.",
            "unknown": "Unknown error. Failed to retrieve the project."
        },
    )

def getProjectCases(accessToken: str, projectId: str) -> list[Case] | None:
    """Retrieves all cases that are in the root of the project identified by its ID.

    Args:
        accessToken: A string containing a valid access token returned by a call to the `login`
        function.
        projectId: A string containing the ID of the project from where the cases are retrieved.

    Returns:
        `list[Case] | None`: A list containing the cases in the root of the project if successfull,
        or `None` otherwise.
    """

    return get(
        url = "/project/{projectId}/case".format(projectId = projectId),
        accessToken = accessToken,
        knownErrors =  {
            404: "Failed to retrieve the project's cases.",
            "unknown": "Unknown error: Failed to retrieve the project's cases."
        },
    )

def getProjectStudies(accessToken: str, projectId: str) -> list[Study] | None:
    """Retrieves all studies in the project identified by its ID.

    Args:
        accessToken: A string containing a valid access token returned by a call to the `login`
        function.
        projectId: A string containing the ID of the project from where the studies are retrieved.

    Returns:
        `list[Study] | None`: A list containing the studies in the project if successfull, or
        `None` otherwise.
    """

    return get(
        url = "/project/{projectId}/study".format(projectId = projectId),
        accessToken = accessToken,
        knownErrors =  {
            "unknown": "Failed to retrieve project's studies."
        },
    )

def importXlsx(accessToken: str, projectId: str, excelFile: str) -> bool:
    """Imports data from an Excel file as studies and cases in the project.

    Args:
        accessToken: A string containing a valid access token returned by a call to the `login`
        function.
        projectId: A string containing the ID of the project where the data is to be imported.
    Returns:
        `bool`: Returns `true` if the upload is successfull and the import process has started,
        `false` otherwise.  
    """
    return uploadFiles(
        url = "/project/{project}/xlsx".format(project = projectId),
        filePaths = [excelFile],
        accessToken = accessToken,
        mimeType = "application/vnd.ms-excel",
        knownErrors =  {
            400: "Failed to initiate Excel import.",
            "unknown": "Unknown error. Failed to initiate Excel import."
        },
    ) != None

def getProjectGeometryByName(accessToken: str, projectId: str, geometryName: str) -> Geometry | None:
    """Retrieves a geometry from a project selected by its ID.

    Args:
        accessToken: A string containing a valid access token returned by a call to the `login`
        function.
        projectId: A string containing the ID of the project from where the geometry is to be
        retrieved.
        geometryName: A string containing the name of the geometry to be retrieved.
    Returns:
        `Geometry | None`: Returns the geometry if successfull, `None` otherwise.

    If there are multiple geometries with the same name in the project, one of them will be
    returned. Which one is picked up is random. It is recommended that only uniquely named
    geometries are included in a project. 
    """

    project = getProject(accessToken, projectId)
    if project == None:
        return

    geometries: list[Geometry | None] = []
    for geometryId in project['geometries']:
        geometries.append(getGeometry(accessToken, geometryId))
    
    ret = list(filter(lambda g: g != None and g['file'] == geometryName, geometries))
    if len(ret) == 0:
        return None
    return ret[0]

def getStudyByName(accessToken: str, projectId: str, studyName: str) -> Study | None:
    """Retrieves a study from a project identified by the study name.

    Args:
        accessToken: A string containing a valid access token returned by a call to the `login`
        function.
        projectId: A string containing the ID of the project from where the study is to be
        retrieved.
        studyName: A string containing the name of the study to be retrieved.
    Returns:
        `Study | None`: Returns the study if successfull, `None` otherwise.

    If there are multiple studies with the same name in the project, one of them will be
    returned. Which one is picked up is random. It is recommended that only uniquely named
    studies are included in a project. 
    """
    studies = getProjectStudies(accessToken, projectId)
    if studies == None:
        print("Project has no studies")
        return
    
    studyAsList = list(filter(lambda s: s["name"] == studyName, studies))
    if len(studyAsList) == 0:
        print("Project does not contain a study with name {}".format(studyName))
        return
    
    if len(studyAsList) == 0:
        return None
    return studyAsList[0]

def getProjectCaseByName(accessToken: str, projectId: str, caseName: str) -> Study | None:
    """Retrieves a case from a project's root identified by the case name.

    Args:
        accessToken: A string containing a valid access token returned by a call to the `login`
        function.
        projectId: A string containing the ID of the project from where the case is to be
        retrieved.
        caseName: A string containing the name of the case to be retrieved.
    Returns:
        `Case | None`: Returns the case if successfull, `None` otherwise.

    If there are multiple cases with the same name in the project, one of them will be
    returned. Which one is picked up is random. It is recommended that only uniquely named
    cases are included in a project. 
    """
    cases = getProjectCases(accessToken, projectId)
    if cases == None:
        print("Project has no cases")
        return
    
    casesAsList = list(filter(lambda s: s["name"] == caseName, cases))
    if len(casesAsList) == 0:
        print("Project does not contain a case with name {}".format(caseName))
        return
    if len(casesAsList) == 0:
        return None
    return casesAsList[0]

__all__ = [
    "CreateProjectData",
    "Authorization",
    "Project",
    "createProject",
    "enumAllProjects",
    "getProject",
    "getProjectCases",
    "getProjectStudies",
    "importXlsx",
    "getProjectGeometryByName",
    "getStudyByName",
    "getProjectCaseByName",
]