"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const defaults = require("@aws-solutions-konstruk/core");
const core_1 = require("@aws-cdk/core");
/**
 * @summary The LambdaToS3 class.
 */
class LambdaToS3 extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToS3 class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToS3Props} props - user provided props for the construct.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the Lambda function
        this.fn = defaults.buildLambdaFunction(scope, {
            deployLambda: props.deployLambda,
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        // Setup the S3 bucket
        this.bucket = defaults.buildS3Bucket(this, {
            deployBucket: props.deployBucket,
            existingBucketObj: props.existingBucketObj,
            bucketProps: props.bucketProps
        });
        // Configure environment variables
        this.fn.addEnvironment('S3_BUCKET_NAME', this.bucket.bucketName);
        // Add the requested or default bucket permissions
        if (props.hasOwnProperty('bucketPermissions') && props.bucketPermissions) {
            if (props.bucketPermissions.includes('Delete')) {
                this.bucket.grantDelete(this.fn.grantPrincipal);
            }
            if (props.bucketPermissions.includes('Put')) {
                this.bucket.grantPut(this.fn.grantPrincipal);
            }
            if (props.bucketPermissions.includes('Read')) {
                this.bucket.grantRead(this.fn.grantPrincipal);
            }
            if (props.bucketPermissions.includes('ReadWrite')) {
                this.bucket.grantReadWrite(this.fn.grantPrincipal);
            }
            if (props.bucketPermissions.includes('Write')) {
                this.bucket.grantWrite(this.fn.grantPrincipal);
            }
        }
        else {
            this.bucket.grantReadWrite(this.fn.grantPrincipal);
        }
        // Add appropriate metadata
        const s3BucketResource = this.bucket.node.findChild('Resource');
        s3BucketResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W51',
                        reason: `This S3 bucket Bucket does not need a bucket policy`
                    }]
            }
        };
    }
    /**
     * @summary Returns an instance of the lambda.Function created by the construct.
     * @returns {lambda.Function} Instance of the Function created by the construct.
     * @since 0.8.0
     * @access public
     */
    lambdaFunction() {
        return this.fn;
    }
    /**
     * @summary Returns an instance of the s3.Bucket created by the construct.
     * @returns {s3.Bucket} Instance of the Bucket created by the construct.
     * @since 0.8.0
     * @access public
     */
    s3Bucket() {
        return this.bucket;
    }
}
exports.LambdaToS3 = LambdaToS3;
//# sourceMappingURL=data:application/json;base64,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