"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TextractA2ISfnTask = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` + (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
/**
 * Calls and A2I endpoint arn with a task_token and waits for the A2I job to finish in order to continue the workflow.
 *
 * Basic implementation
 *
 * Input: "Payload"."a2iInputPath"
 * Output:
 * ```json
 * {
        'humanLoopStatus': human_loop_status,
        'humanLoopResultPath': human_loop_result,
        'humanLoopCreationTime': human_loop_creation_time,
    }
    ```
 *
   ```python
   textract_a2i_task = tcdk.TextractA2ISfnTask(
        self,
        "TextractA2I",
        a2i_flow_definition_arn=
        "arn:aws:sagemaker:us-east-1:913165245630:flow-definition/textract-classifiction",
        integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
        lambda_log_level="DEBUG",
        timeout=Duration.hours(24),
        input=sfn.TaskInput.from_object({
            "Token":
            sfn.JsonPath.task_token,
            "ExecutionId":
            sfn.JsonPath.string_at('$$.Execution.Id'),
            "Payload":
            sfn.JsonPath.entire_payload,
        }),
        result_path="$.a2i_result")
  ```
 */
class TextractA2ISfnTask extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.version = '0.0.1';
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, TextractA2ISfnTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        var lambdaLogLevel = props.lambdaLogLevel === undefined ? 'DEBUG' : props.lambdaLogLevel;
        const taskTokenTable = new dynamodb.Table(this, 'A2ITaskTokenTable', {
            partitionKey: {
                name: 'ID',
                type: dynamodb.AttributeType.STRING,
            },
            billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            timeToLiveAttribute: 'ttltimestamp',
        });
        this.taskTokenTableName = taskTokenTable.tableName;
        const startA2IFunction = new lambda.DockerImageFunction(this, 'startA2ICall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/a2i_preprocess/')),
            memorySize: 128,
            architecture: lambda.Architecture.X86_64,
            environment: {
                TOKEN_STORE_DDB: this.taskTokenTableName,
                A2I_FLOW_DEFINITION_ARN: props.a2iFlowDefinitionARN,
                LOG_LEVEL: lambdaLogLevel,
            },
        });
        const startA2ITask = new tasks.LambdaInvoke(this, 'StartA2ITask', {
            lambdaFunction: startA2IFunction,
        });
        const postA2IFunction = new lambda.DockerImageFunction(this, 'endA2ICall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/a2i_postprocess/')),
            memorySize: 128,
            architecture: lambda.Architecture.X86_64,
            environment: {
                TOKEN_STORE_DDB: this.taskTokenTableName,
                LOG_LEVEL: lambdaLogLevel,
            },
        });
        const rule = new events.Rule(this, 'EventBridgeRule', {
            enabled: true,
            eventPattern: {
                source: ['aws.sagemaker'],
                detailType: ['SageMaker A2I HumanLoop Status Change'],
                detail: {
                    humanLoopStatus: ['Completed', 'Failed'],
                    flowDefinitionArn: [props.a2iFlowDefinitionARN],
                },
            },
        });
        rule.addTarget(new targets.LambdaFunction(postA2IFunction));
        startA2IFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'sagemaker:StartHumanLoop',
            ],
            resources: [props.a2iFlowDefinitionARN],
        }));
        startA2IFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskSuccess', 'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        startA2IFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['dynamodb:PutItem', 'dynamodb:GetItem'], resources: [taskTokenTable.tableArn] }));
        startA2IFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['s3:PutObject', 's3:Get*', 's3:List*'], resources: ['*'] }));
        postA2IFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskSuccess', 'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        postA2IFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['dynamodb:PutItem', 'dynamodb:GetItem'], resources: [taskTokenTable.tableArn] }));
        const workflow_chain = sfn.Chain.start(startA2ITask);
        this.stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: workflow_chain,
            timeout: aws_cdk_lib_1.Duration.hours(2),
            tracingEnabled: true,
        });
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
         * @internal
         */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? {
                ...this.props.input.value,
                ...associateWithParentEntry,
            } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
         * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
         *
         * the scoped access policy should be generated accordingly.
         *
         * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
         */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'states:DescribeExecution', 'states:StopExecution',
                ],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'events:PutTargets', 'events:PutRule', 'events:DescribeRule',
                ],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.TextractA2ISfnTask = TextractA2ISfnTask;
_a = JSII_RTTI_SYMBOL_1;
TextractA2ISfnTask[_a] = { fqn: "amazon-textract-idp-cdk-constructs.TextractA2ISfnTask", version: "0.0.15" };
TextractA2ISfnTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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