"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TextractGenericAsyncSfnTask = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const sns = require("aws-cdk-lib/aws-sns");
const aws_sns_subscriptions_1 = require("aws-cdk-lib/aws-sns-subscriptions");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
var textractSync_1 = require("./textractSync");
Object.defineProperty(exports, "TextractGenericSyncSfnTask", { enumerable: true, get: function () { return textractSync_1.TextractGenericSyncSfnTask; } });
var textractDecider_1 = require("./textractDecider");
Object.defineProperty(exports, "TextractPOCDecider", { enumerable: true, get: function () { return textractDecider_1.TextractPOCDecider; } });
var textractClassificationConfigurator_1 = require("./textractClassificationConfigurator");
Object.defineProperty(exports, "TextractClassificationConfigurator", { enumerable: true, get: function () { return textractClassificationConfigurator_1.TextractClassificationConfigurator; } });
var textractOutputConfigToJSON_1 = require("./textractOutputConfigToJSON");
Object.defineProperty(exports, "TextractAsyncToJSON", { enumerable: true, get: function () { return textractOutputConfigToJSON_1.TextractAsyncToJSON; } });
var textractGenerateCSV_1 = require("./textractGenerateCSV");
Object.defineProperty(exports, "TextractGenerateCSV", { enumerable: true, get: function () { return textractGenerateCSV_1.TextractGenerateCSV; } });
var textractA2I_1 = require("./textractA2I");
Object.defineProperty(exports, "TextractA2ISfnTask", { enumerable: true, get: function () { return textractA2I_1.TextractA2ISfnTask; } });
var rdsCSVToAurora_1 = require("./rdsCSVToAurora");
Object.defineProperty(exports, "CSVToAuroraTask", { enumerable: true, get: function () { return rdsCSVToAurora_1.CSVToAuroraTask; } });
var comprehendClassification_1 = require("./comprehendClassification");
Object.defineProperty(exports, "ComprehendGenericSyncSfnTask", { enumerable: true, get: function () { return comprehendClassification_1.ComprehendGenericSyncSfnTask; } });
var spacyClassification_1 = require("./spacyClassification");
Object.defineProperty(exports, "SpacySfnTask", { enumerable: true, get: function () { return spacyClassification_1.SpacySfnTask; } });
var documentSplitter_1 = require("./documentSplitter");
Object.defineProperty(exports, "DocumentSplitter", { enumerable: true, get: function () { return documentSplitter_1.DocumentSplitter; } });
var rdsAuroraServerless_1 = require("./rdsAuroraServerless");
Object.defineProperty(exports, "RDSAuroraServerless", { enumerable: true, get: function () { return rdsAuroraServerless_1.RDSAuroraServerless; } });
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` + (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
/**
 * This Task calls the Textract through the asynchronous API.
 *
 * Which API to call is defined in
 *
 * When GENERIC is called with features in the manifest definition, will call the AnalzyeDocument API.
 *
 * Takes the configuration from "Payload"."manifest"
 *
 * Will retry on recoverable errors based on textractAsyncCallMaxRetries
 * errors for retry: ['ThrottlingException', 'LimitExceededException', 'InternalServerError', 'ProvisionedThroughputExceededException'],
 *
 * Internally calls Start* calls with OutputConfig and SNSNotification.
 * Another Lambda functions waits for SNS Notification event and notifies the Step Function flow with the task token.
 *
 * Step Function JSON input requirements
 *
 * **Input**: "Payload"."manifest"
 *
 * **Output**: "TextractTempOutputJsonPath" points to potentially paginated Textract JSON Schema output at "TextractTempOutputJsonPath" (using the example code it will be at: "textract_result"."TextractTempOutputJsonPath")
 *
 * Works together with TextractAsyncToJSON, which takes the s3_output_bucket/s3_temp_output_prefix location as input
 *
 * Example (Python)
 * ```python
    textract_async_task = tcdk.TextractGenericAsyncSfnTask(
        self,
        "TextractAsync",
        s3_output_bucket=s3_output_bucket,
        s3_temp_output_prefix=s3_temp_output_prefix,
        integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
        lambda_log_level="DEBUG",
        timeout=Duration.hours(24),
        input=sfn.TaskInput.from_object({
            "Token":
            sfn.JsonPath.task_token,
            "ExecutionId":
            sfn.JsonPath.string_at('$$.Execution.Id'),
            "Payload":
            sfn.JsonPath.entire_payload,
        }),
        result_path="$.textract_result")
    ```

 */
class TextractGenericAsyncSfnTask extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        /**resources
         * DynamoDB table
         * textractAsyncSNSRole
         */
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, TextractGenericAsyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        var textractStateMachineTimeoutMinutes = props.textractStateMachineTimeoutMinutes === undefined ? 2880 : props.textractStateMachineTimeoutMinutes;
        var lambdaLogLevel = props.lambdaLogLevel === undefined ? 'DEBUG' : props.lambdaLogLevel;
        var textractAPI = props.textractAPI === undefined ? 'GENERIC' : props.textractAPI;
        var textractAsyncCallMaxRetries = props.textractAsyncCallMaxRetries === undefined ? 100 : props.textractAsyncCallMaxRetries;
        var textractAsyncCallBackoffRate = props.textractAsyncCallBackoffRate === undefined ? 1.1 : props.textractAsyncCallBackoffRate;
        var textractAsyncCallInterval = props.textractAsyncCallInterval === undefined ? 1 : props.textractAsyncCallInterval;
        var s3TempOutputPrefix = props.s3TempOutputPrefix === undefined ? '' : props.s3TempOutputPrefix;
        var s3InputPrefix = props.s3InputPrefix === undefined ? '' : props.s3InputPrefix;
        var enableCloudWatchMetricsAndDashboard = props.enableCloudWatchMetricsAndDashboard === undefined ? false :
            props.enableCloudWatchMetricsAndDashboard;
        /** RESOURCE DYNAMODB TABLE for TASK TOKEN */
        if (props.taskTokenTable === undefined) {
            this.taskTokenTable = new dynamodb.Table(this, 'TextractTaskTokenTable', {
                partitionKey: {
                    name: 'ID',
                    type: dynamodb.AttributeType.STRING,
                },
                billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                timeToLiveAttribute: 'ttltimestamp',
            });
        }
        else {
            this.taskTokenTable = props.taskTokenTable;
        }
        this.taskTokenTableName = this.taskTokenTable.tableName;
        /** RESOURCE: SNS Role for Textract to use*/
        if (props.snsRoleTextract === undefined) {
            this.textractAsyncSNSRole = new iam.Role(this, 'TextractAsyncSNSRole', {
                assumedBy: new iam.ServicePrincipal('textract.amazonaws.com'),
                managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSQSFullAccess'),
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSNSFullAccess'),
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonS3ReadOnlyAccess'),
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonTextractFullAccess')],
            });
        }
        else {
            this.textractAsyncSNSRole = props.snsRoleTextract;
        }
        this.textractAsyncSNS = new sns.Topic(this, 'TextractAsyncSNS');
        this.textractAsyncCallFunction = new lambda.DockerImageFunction(this, 'TextractAsyncCall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/textract_async/')),
            memorySize: 128,
            architecture: lambda.Architecture.X86_64,
            environment: {
                NOTIFICATION_SNS: this.textractAsyncSNS.topicArn,
                NOTIFICATION_ROLE_ARN: this.textractAsyncSNSRole.roleArn,
                TOKEN_STORE_DDB: this.taskTokenTableName,
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_TEMP_OUTPUT_PREFIX: props.s3TempOutputPrefix,
                LOG_LEVEL: lambdaLogLevel,
                TEXTRACT_API: textractAPI,
            },
        });
        const textractAsyncCallTask = new tasks.LambdaInvoke(this, 'TextractAsyncCallTask', { lambdaFunction: this.textractAsyncCallFunction });
        textractAsyncCallTask.addRetry({
            maxAttempts: textractAsyncCallMaxRetries,
            backoffRate: textractAsyncCallBackoffRate,
            interval: aws_cdk_lib_1.Duration.seconds(textractAsyncCallInterval),
            errors: ['ThrottlingException', 'LimitExceededException', 'InternalServerError', 'ProvisionedThroughputExceededException'],
        });
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'textract:Start*',
                'textract:Get*',
            ],
            resources: ['*'],
        }));
        /** ################ INPUT BUCKET POLICIES */
        if (props.inputPolicyStatements === undefined) {
            if (props.s3InputBucket === undefined) {
                this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:GetObject', 's3:ListBucket'],
                    resources: ['*'],
                }));
            }
            else {
                this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:GetObject'],
                    resources: [
                        path.join(`arn:aws:s3:::${props.s3InputBucket}`, '/*'),
                        path.join(`arn:aws:s3:::${props.s3InputBucket}`, s3InputPrefix, '/*'),
                    ],
                }));
                this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:ListBucket'],
                    resources: [
                        path.join(`arn:aws:s3:::${props.s3InputBucket}`),
                    ],
                }));
            }
        }
        else {
            for (var policyStatement of props.inputPolicyStatements) {
                this.textractAsyncCallFunction.addToRolePolicy(policyStatement);
            }
        }
        /** ##################### OUTPUT BUCKET POLICIES */
        if (props.outputPolicyStatements === undefined) {
            if (props.s3OutputBucket === undefined) {
                this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:PutObject'],
                    resources: ['*'],
                }));
            }
            else {
                this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
                    actions: ['s3:PutObject'],
                    resources: [
                        path.join(`arn:aws:s3:::${props.s3OutputBucket}`, s3TempOutputPrefix, '/'),
                        path.join(`arn:aws:s3:::${props.s3OutputBucket}`, s3TempOutputPrefix, '/*'),
                    ],
                }));
            }
        }
        else {
            for (var policyStatement of props.outputPolicyStatements) {
                this.textractAsyncCallFunction.addToRolePolicy(policyStatement);
            }
        }
        // this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['sns:*'], resources: [this.textractAsyncSNS.topicArn] }));
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['dynamodb:PutItem', 'dynamodb:GetItem'], resources: [this.taskTokenTable.tableArn] }));
        this.startTextractLambdaLogGroup = this.textractAsyncCallFunction.logGroup;
        this.textractAsyncReceiveSNSFunction = new lambda.DockerImageFunction(this, 'TextractAsyncSNSFunction', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/textract_async_sns_listener/')),
            memorySize: 128,
            architecture: lambda.Architecture.X86_64,
            timeout: aws_cdk_lib_1.Duration.seconds(900),
            environment: {
                TOKEN_STORE_DDB: this.taskTokenTableName,
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_TEMP_OUTPUT_PREFIX: props.s3TempOutputPrefix,
                LOG_LEVEL: lambdaLogLevel,
            },
        });
        this.textractAsyncSNS.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(this.textractAsyncReceiveSNSFunction));
        this.textractAsyncReceiveSNSFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['dynamodb:GetItem'], resources: [this.taskTokenTable.tableArn] }));
        this.receiveStartSNSLambdaLogGroup = this.textractAsyncReceiveSNSFunction.logGroup;
        const workflow_chain = sfn.Chain.start(textractAsyncCallTask);
        this.stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: workflow_chain,
            timeout: aws_cdk_lib_1.Duration.hours(textractStateMachineTimeoutMinutes),
            tracingEnabled: true,
        });
        this.textractAsyncReceiveSNSFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskSuccess', 'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        // =========
        // DASHBOARD
        // =========
        if (enableCloudWatchMetricsAndDashboard) {
            const appName = this.node.tryGetContext('appName');
            const customMetricNamespace = 'TextractConstructGenericAsync';
            // OPERATIONAL
            const asyncDurationMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-DurationFilter`, {
                logGroup: this.textractAsyncReceiveSNSFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'Duration',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'durationMs')
                    .whereString('message', '=', `textract_async_${textractAPI}_job_duration_in_ms:`),
                metricValue: '$durationMs',
            });
            this.asyncDurationMetric = asyncDurationMetricFilter.metric({ statistic: 'avg' });
            const asyncNumberPagesMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberPagesFilter`, {
                logGroup: this.textractAsyncReceiveSNSFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'NumberPages',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                    .whereString('message', '=', `textract_async_${textractAPI}_number_of_pages_processed:`),
                metricValue: '$pages',
            });
            this.asyncNumberPagesMetric = asyncNumberPagesMetricFilter.metric({ statistic: 'sum' });
            // TODO: expose all filters as properties of Construct, so users can build their own metrics
            const asyncJobFinshedMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-JobsFinishedFilter`, {
                logGroup: this.textractAsyncReceiveSNSFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'JobsFinished',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                    .whereString('message', '=', `textract_async_${textractAPI}_number_of_pages_processed:`),
                metricValue: '1',
            });
            this.asyncJobFinshedMetric = asyncJobFinshedMetricFilter.metric({ statistic: 'sum' });
            const asyncJobStartedMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-JobsStartedFilter`, {
                logGroup: this.textractAsyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'JobsStarted',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message')
                    .whereString('message', '=', `textract_async_${textractAPI}_job_started`),
                metricValue: '1',
            });
            this.asyncJobStartedMetric = asyncJobStartedMetricFilter.metric({ statistic: 'sum' });
            const asyncNumberPagesSendMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberPagesSendFilter`, {
                logGroup: this.textractAsyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'NumberPages',
                filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                    .whereString('message', '=', `textract_async_${textractAPI}_number_of_pages_send_to_process:`),
                metricValue: '$pages',
            });
            this.asyncNumberPagesSendMetric = asyncNumberPagesSendMetricFilter.metric({ statistic: 'sum' });
            // OPERATIONAL STOP
            // CALCUATED OPERATIONAL METRICS
            const pagesPerSecond = new cloudwatch.MathExpression({
                expression: 'pages / (duration / 1000)',
                usingMetrics: {
                    pages: this.asyncNumberPagesMetric,
                    duration: this.asyncDurationMetric,
                },
            });
            const openJobs = new cloudwatch.MathExpression({
                expression: 'startedJobs - finishedJobs',
                usingMetrics: {
                    startedJobs: this.asyncJobStartedMetric,
                    finishedJobs: this.asyncJobFinshedMetric,
                },
            });
            // CALCUATED OPERATIONAL METRICS STOP
            const errorFilterMetric = new aws_logs_1.MetricFilter(this, `${appName}-ErrorFilter`, {
                logGroup: this.textractAsyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'Errors',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('ERROR', 'Error', 'error'),
                metricValue: '1',
            });
            const limitExceededExceptionFilterMetric = new aws_logs_1.MetricFilter(this, `${appName}-limitExceededExceptionFilter`, {
                logGroup: this.textractAsyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'LimitExceededException',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.LimitExceededException'),
                metricValue: '1',
            });
            const throttlingException = new aws_logs_1.MetricFilter(this, `${appName}-throttlingExceptionFilter`, {
                logGroup: this.textractAsyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'ThrottlingException',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.ThrottlingException'),
                metricValue: '1',
            });
            const provisionedThroughputExceededException = new aws_logs_1.MetricFilter(this, `${appName}-provisionedThroughputFilter`, {
                logGroup: this.textractAsyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'ProvisionedThroughputExceededException',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.ProvisionedThroughputExceededException'),
                metricValue: '1',
            });
            const internalServerError = new aws_logs_1.MetricFilter(this, `${appName}-internalServerErrorFilter`, {
                logGroup: this.textractAsyncCallFunction.logGroup,
                metricNamespace: customMetricNamespace,
                metricName: 'InternalServerError',
                filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.InternalServerError'),
                metricValue: '1',
            });
            const errorMetric = errorFilterMetric.metric({ statistic: 'sum' });
            const limitExceededMetric = limitExceededExceptionFilterMetric.metric({ statistic: 'sum' });
            const throttlingMetric = throttlingException.metric({ statistic: 'sum' });
            const provisionedThroughputMetric = provisionedThroughputExceededException.metric({ statistic: 'sum' });
            const internalServerErrorMetric = internalServerError.metric({ statistic: 'sum' });
            const textractStartDocumentTextThrottle = new cloudwatch.Metric({
                namespace: 'AWS/Textract',
                metricName: 'ThrottledCount',
                dimensionsMap: {
                    Operation: 'StartDocumentTextDetection',
                },
                statistic: 'sum',
            });
            // DASHBOARD LAMBDA
            const dashboardWidth = 24;
            // const widgetStandardHeight=9;
            // DASHBOARD
            new cloudwatch.Dashboard(this, '${appName}-TestDashboard', {
                end: 'end',
                periodOverride: cloudwatch.PeriodOverride.AUTO,
                start: 'start',
                widgets: [
                    [
                        new cloudwatch.Column(new cloudwatch.TextWidget({ markdown: '# Operational Data Row widgets', width: dashboardWidth })),
                    ],
                    [
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'PagesPerSecond', left: [pagesPerSecond], width: Math.floor(dashboardWidth / 2) })),
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'OpenJobs', left: [openJobs], width: Math.floor(dashboardWidth / 2) })),
                    ],
                    [
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'Duration', left: [this.asyncDurationMetric], width: Math.floor(dashboardWidth / 4) })),
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'NumberPages', left: [this.asyncNumberPagesMetric], width: Math.floor(dashboardWidth / 4) })),
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'NumberPages', left: [this.asyncNumberPagesSendMetric], width: Math.floor(dashboardWidth / 4) })),
                        new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'JobsStartedAndFinished', left: [this.asyncJobFinshedMetric], right: [this.asyncJobStartedMetric], width: Math.floor(dashboardWidth / 4) })),
                    ],
                    [
                        new cloudwatch.Column(new cloudwatch.TextWidget({ markdown: '# Async Textract Exceptions Row', width: dashboardWidth })),
                    ],
                    [
                        new cloudwatch.GraphWidget({ title: 'Errors', left: [errorMetric], width: Math.floor(dashboardWidth / 5) }),
                        new cloudwatch.GraphWidget({ title: 'LimitExceeded', left: [limitExceededMetric], width: Math.floor(dashboardWidth / 5) }),
                        new cloudwatch.GraphWidget({ title: 'Throttling', left: [throttlingMetric], width: Math.floor(dashboardWidth / 5) }),
                        new cloudwatch.GraphWidget({ title: 'ProvisionedThrougput', left: [provisionedThroughputMetric], width: Math.floor(dashboardWidth / 5) }),
                        new cloudwatch.GraphWidget({ title: 'InternalServerError', left: [internalServerErrorMetric], width: Math.floor(dashboardWidth / 5) }),
                    ],
                    [
                        new cloudwatch.TextWidget({ markdown: '# Textract', width: dashboardWidth }),
                    ],
                    [
                        new cloudwatch.GraphWidget({ title: 'Textract-StartDetectText-ThrottledCount', left: [textractStartDocumentTextThrottle] }),
                    ],
                    [
                        new cloudwatch.TextWidget({ markdown: '# textractAsyncCallFunction', width: dashboardWidth }),
                    ],
                    [
                        new cloudwatch.GraphWidget({ title: 'Async-Function-Errors', left: [this.textractAsyncCallFunction.metricErrors()], width: Math.floor(dashboardWidth / 3) }),
                        new cloudwatch.GraphWidget({ title: 'Async-Function-Invocations', left: [this.textractAsyncCallFunction.metricInvocations()], width: Math.floor(dashboardWidth / 3) }),
                        new cloudwatch.GraphWidget({ title: 'Async-Function-Throttles', left: [this.textractAsyncCallFunction.metricThrottles()], width: Math.floor(dashboardWidth / 3) }),
                    ],
                    [
                        new cloudwatch.TextWidget({ markdown: '# textractAsyncReceiveSNSFunction', width: dashboardWidth }),
                    ],
                    [
                        new cloudwatch.GraphWidget({ title: 'SNS-Function-Errors', left: [this.textractAsyncReceiveSNSFunction.metricErrors()], width: Math.floor(dashboardWidth / 3) }),
                        new cloudwatch.GraphWidget({ title: 'SNS-Function-Invocations', left: [this.textractAsyncReceiveSNSFunction.metricInvocations()], width: Math.floor(dashboardWidth / 3) }),
                        new cloudwatch.GraphWidget({ title: 'SNS-Function-Throttles', left: [this.textractAsyncReceiveSNSFunction.metricThrottles()], width: Math.floor(dashboardWidth / 3) }),
                    ],
                ],
            });
        }
        // END DASHBOARD
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
         * @internal
         */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? {
                ...this.props.input.value,
                ...associateWithParentEntry,
            } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
         * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
         *
         * the scoped access policy should be generated accordingly.
         *
         * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
         */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'states:DescribeExecution', 'states:StopExecution',
                ],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'events:PutTargets', 'events:PutRule', 'events:DescribeRule',
                ],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.TextractGenericAsyncSfnTask = TextractGenericAsyncSfnTask;
_a = JSII_RTTI_SYMBOL_1;
TextractGenericAsyncSfnTask[_a] = { fqn: "amazon-textract-idp-cdk-constructs.TextractGenericAsyncSfnTask", version: "0.0.15" };
TextractGenericAsyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSw2QkFBNkI7QUFDN0IsNkNBQTZFO0FBQzdFLHlEQUF5RDtBQUN6RCxxREFBcUQ7QUFDckQsMkNBQTJDO0FBQzNDLGlEQUEyRDtBQUMzRCxpREFBaUQ7QUFDakQsbURBQThFO0FBQzlFLDJDQUEyQztBQUMzQyw2RUFBdUU7QUFDdkUscURBQXFEO0FBQ3JELDZEQUE2RDtBQUU3RCwrQ0FBNkY7QUFBcEYsMEhBQUEsMEJBQTBCLE9BQUE7QUFDbkMscURBQWtGO0FBQXpFLHFIQUFBLGtCQUFrQixPQUFBO0FBQzNCLDJGQUFtSTtBQUExSCx3SkFBQSxrQ0FBa0MsT0FBQTtBQUMzQywyRUFBNkY7QUFBcEYsaUlBQUEsbUJBQW1CLE9BQUE7QUFDNUIsNkRBQXNGO0FBQTdFLDBIQUFBLG1CQUFtQixPQUFBO0FBQzVCLDZDQUE0RTtBQUFuRSxpSEFBQSxrQkFBa0IsT0FBQTtBQUMzQixtREFBeUU7QUFBaEUsaUhBQUEsZUFBZSxPQUFBO0FBQ3hCLHVFQUE2RztBQUFwRyx3SUFBQSw0QkFBNEIsT0FBQTtBQUNyQyw2REFBd0U7QUFBL0QsbUhBQUEsWUFBWSxPQUFBO0FBQ3JCLHVEQUE2RTtBQUFwRSxvSEFBQSxnQkFBZ0IsT0FBQTtBQUN6Qiw2REFBc0Y7QUFBN0UsMEhBQUEsbUJBQW1CLE9BQUE7QUFHNUIsU0FBZ0Isd0JBQXdCLENBQUMsa0JBQTBDLEVBQUUsaUJBQTJDO0lBQzlILElBQUksQ0FBRSxpQkFBaUIsQ0FBQyxRQUFRLENBQUMsa0JBQWtCLENBQUMsRUFBRTtRQUNwRCxNQUFNLElBQUksS0FBSyxDQUFDLGdFQUFnRSxpQkFBaUIsZUFBZSxrQkFBa0IsRUFBRSxDQUFDLENBQUM7S0FDdkk7QUFDSCxDQUFDO0FBSkQsNERBSUM7QUFDRDs7Ozs7O0dBTUc7QUFDSCxNQUFNLGlCQUFpQixHQUEyQztJQUNoRSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLEVBQUU7SUFDN0MsQ0FBQyxHQUFHLENBQUMsa0JBQWtCLENBQUMsT0FBTyxDQUFDLEVBQUUsT0FBTztJQUN6QyxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFLG1CQUFtQjtDQUNsRSxDQUFDO0FBRUYsU0FBZ0Isc0JBQXNCLENBQUMsT0FBZSxFQUFFLEdBQVcsRUFBRSxrQkFBMkM7SUFDOUcsSUFBSSxDQUFFLE9BQU8sSUFBSSxDQUFFLEdBQUcsRUFBRTtRQUN0QixNQUFNLElBQUksS0FBSyxDQUFDLHNFQUFzRSxDQUFDLENBQUM7S0FDekY7SUFDRCxPQUFPLE9BQ0wsaUJBQUcsQ0FBQyxTQUNOLGFBQWEsT0FBTyxJQUFJLEdBQUcsRUFBRSxHQUFHLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDLGlCQUFpQixDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO0FBQ3BHLENBQUM7QUFQRCx3REFPQztBQXVGRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0E0Q0c7QUFDSCxNQUFhLDJCQUE0QixTQUFRLEdBQUcsQ0FBQyxhQUFhO0lBMkJoRSxZQUFZLEtBQWlCLEVBQUUsRUFBVyxFQUFtQixLQUF3QztRQUNuRzs7O1dBR0c7UUFDSCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUxtQyxVQUFLLEdBQUwsS0FBSyxDQUFtQztRQU9uRyxJQUFJLENBQUMsa0JBQWtCLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixJQUFJLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxnQkFBZ0IsQ0FBQztRQUM5Rix3QkFBd0IsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLEVBQUUsMkJBQTJCLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUU5RyxJQUFJLElBQUksQ0FBQyxrQkFBa0IsS0FBSyxHQUFHLENBQUMsa0JBQWtCLENBQUMsbUJBQW1CLElBQUksQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsRUFBRTtZQUM1SCxNQUFNLElBQUksS0FBSyxDQUFDLDBGQUEwRixDQUFDLENBQUM7U0FDN0c7UUFFRCxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsbUJBQW1CLElBQUksS0FBSyxDQUFDLEtBQUssSUFBSSxLQUFLLENBQUMsS0FBSyxDQUFDLElBQUksS0FBSyxHQUFHLENBQUMsU0FBUyxDQUFDLE1BQU0sRUFBRTtZQUM5RixNQUFNLElBQUksS0FBSyxDQUFDLG9JQUFvSSxDQUFDLENBQUM7U0FDdko7UUFFRCxJQUFJLGtDQUFrQyxHQUFHLEtBQUssQ0FBQyxrQ0FBa0MsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGtDQUFrQyxDQUFDO1FBQ2xKLElBQUksY0FBYyxHQUFHLEtBQUssQ0FBQyxjQUFjLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUM7UUFDekYsSUFBSSxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQztRQUNsRixJQUFJLDJCQUEyQixHQUFHLEtBQUssQ0FBQywyQkFBMkIsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLDJCQUEyQixDQUFDO1FBQzVILElBQUksNEJBQTRCLEdBQUcsS0FBSyxDQUFDLDRCQUE0QixLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsNEJBQTRCLENBQUM7UUFDL0gsSUFBSSx5QkFBeUIsR0FBRyxLQUFLLENBQUMseUJBQXlCLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyx5QkFBeUIsQ0FBQztRQUNwSCxJQUFJLGtCQUFrQixHQUNwQixLQUFLLENBQUMsa0JBQWtCLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQztRQUN6RSxJQUFJLGFBQWEsR0FDZixLQUFLLENBQUMsYUFBYSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDO1FBQy9ELElBQUksbUNBQW1DLEdBQUcsS0FBSyxDQUFDLG1DQUFtQyxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDekcsS0FBSyxDQUFDLG1DQUFtQyxDQUFDO1FBRTVDLDZDQUE2QztRQUM3QyxJQUFJLEtBQUssQ0FBQyxjQUFjLEtBQUssU0FBUyxFQUFFO1lBQ3RDLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxRQUFRLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSx3QkFBd0IsRUFBRTtnQkFDdkUsWUFBWSxFQUFFO29CQUNaLElBQUksRUFBRSxJQUFJO29CQUNWLElBQUksRUFBRSxRQUFRLENBQUMsYUFBYSxDQUFDLE1BQU07aUJBQ3BDO2dCQUNELFdBQVcsRUFBRSxRQUFRLENBQUMsV0FBVyxDQUFDLGVBQWU7Z0JBQ2pELGFBQWEsRUFBRSwyQkFBYSxDQUFDLE9BQU87Z0JBQ3BDLG1CQUFtQixFQUFFLGNBQWM7YUFDcEMsQ0FBQyxDQUFDO1NBQ0o7YUFBTTtZQUNMLElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsQ0FBQztTQUM1QztRQUNELElBQUksQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLFNBQVMsQ0FBQztRQUV4RCw0Q0FBNEM7UUFDNUMsSUFBSSxLQUFLLENBQUMsZUFBZSxLQUFLLFNBQVMsRUFBRTtZQUN2QyxJQUFJLENBQUMsb0JBQW9CLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxzQkFBc0IsRUFBRTtnQkFDckUsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHdCQUF3QixDQUFDO2dCQUM3RCxlQUFlLEVBQUUsQ0FBQyx1QkFBYSxDQUFDLHdCQUF3QixDQUFDLHFCQUFxQixDQUFDO29CQUM3RSx1QkFBYSxDQUFDLHdCQUF3QixDQUFDLHFCQUFxQixDQUFDO29CQUM3RCx1QkFBYSxDQUFDLHdCQUF3QixDQUFDLHdCQUF3QixDQUFDO29CQUNoRSx1QkFBYSxDQUFDLHdCQUF3QixDQUFDLDBCQUEwQixDQUFDLENBQUM7YUFDdEUsQ0FBQyxDQUFDO1NBQ0o7YUFBTTtZQUNMLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxLQUFLLENBQUMsZUFBZSxDQUFDO1NBRW5EO1FBRUQsSUFBSSxDQUFDLGdCQUFnQixHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztRQUNoRSxJQUFJLENBQUMseUJBQXlCLEdBQUcsSUFBSSxNQUFNLENBQUMsbUJBQW1CLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFO1lBQ3pGLElBQUksRUFBRSxNQUFNLENBQUMsZUFBZSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSwyQkFBMkIsQ0FBQyxDQUFDO1lBQzlGLFVBQVUsRUFBRSxHQUFHO1lBQ2YsWUFBWSxFQUFFLE1BQU0sQ0FBQyxZQUFZLENBQUMsTUFBTTtZQUN4QyxXQUFXLEVBQUU7Z0JBQ1gsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFFBQVE7Z0JBQ2hELHFCQUFxQixFQUFFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxPQUFPO2dCQUN4RCxlQUFlLEVBQUUsSUFBSSxDQUFDLGtCQUFrQjtnQkFDeEMsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGNBQWM7Z0JBQ3RDLHFCQUFxQixFQUFFLEtBQUssQ0FBQyxrQkFBa0I7Z0JBQy9DLFNBQVMsRUFBRSxjQUFjO2dCQUN6QixZQUFZLEVBQUUsV0FBVzthQUMxQjtTQUNGLENBQUMsQ0FBQztRQUNILE1BQU0scUJBQXFCLEdBQUcsSUFBSSxLQUFLLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSx1QkFBdUIsRUFBRSxFQUFFLGNBQWMsRUFBRSxJQUFJLENBQUMseUJBQXlCLEVBQUUsQ0FBQyxDQUFDO1FBQ3hJLHFCQUFxQixDQUFDLFFBQVEsQ0FBQztZQUM3QixXQUFXLEVBQUUsMkJBQTJCO1lBQ3hDLFdBQVcsRUFBRSw0QkFBNEI7WUFDekMsUUFBUSxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLHlCQUF5QixDQUFDO1lBQ3JELE1BQU0sRUFBRSxDQUFDLHFCQUFxQixFQUFFLHdCQUF3QixFQUFFLHFCQUFxQixFQUFFLHdDQUF3QyxDQUFDO1NBQzNILENBQUMsQ0FBQztRQUdILElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxlQUFlLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3JFLE9BQU8sRUFBRTtnQkFDUCxpQkFBaUI7Z0JBQ2pCLGVBQWU7YUFDaEI7WUFDRCxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7U0FDakIsQ0FBQyxDQUFDLENBQUM7UUFDSiw2Q0FBNkM7UUFDN0MsSUFBSSxLQUFLLENBQUMscUJBQXFCLEtBQUssU0FBUyxFQUFFO1lBQzdDLElBQUksS0FBSyxDQUFDLGFBQWEsS0FBSyxTQUFTLEVBQUU7Z0JBQ3JDLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxlQUFlLENBQzVDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsT0FBTyxFQUFFLENBQUMsY0FBYyxFQUFFLGVBQWUsQ0FBQztvQkFDMUMsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO2lCQUNqQixDQUFDLENBQ0gsQ0FBQzthQUNIO2lCQUFNO2dCQUNMLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxlQUFlLENBQzVDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDO29CQUN6QixTQUFTLEVBQUU7d0JBQ1QsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsS0FBSyxDQUFDLGFBQWEsRUFBRSxFQUFFLElBQUksQ0FBQzt3QkFDdEQsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsS0FBSyxDQUFDLGFBQWEsRUFBRSxFQUFFLGFBQWEsRUFBRSxJQUFJLENBQUM7cUJBQ3RFO2lCQUNGLENBQUMsQ0FDSCxDQUFDO2dCQUNGLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxlQUFlLENBQzVDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsT0FBTyxFQUFFLENBQUMsZUFBZSxDQUFDO29CQUMxQixTQUFTLEVBQUU7d0JBQ1QsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsS0FBSyxDQUFDLGFBQWEsRUFBRSxDQUFDO3FCQUNqRDtpQkFDRixDQUFDLENBQ0gsQ0FBQzthQUNIO1NBQ0Y7YUFBTTtZQUNMLEtBQUssSUFBSSxlQUFlLElBQUksS0FBSyxDQUFDLHFCQUFxQixFQUFFO2dCQUN2RCxJQUFJLENBQUMseUJBQXlCLENBQUMsZUFBZSxDQUFDLGVBQWUsQ0FBQyxDQUFDO2FBQ2pFO1NBQ0Y7UUFDRCxtREFBbUQ7UUFDbkQsSUFBSSxLQUFLLENBQUMsc0JBQXNCLEtBQUssU0FBUyxFQUFFO1lBQzlDLElBQUksS0FBSyxDQUFDLGNBQWMsS0FBSyxTQUFTLEVBQUU7Z0JBQ3RDLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxlQUFlLENBQzVDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDO29CQUN6QixTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7aUJBQ2pCLENBQUMsQ0FDSCxDQUFDO2FBQ0g7aUJBQU07Z0JBQ0wsSUFBSSxDQUFDLHlCQUF5QixDQUFDLGVBQWUsQ0FDNUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO29CQUN0QixPQUFPLEVBQUUsQ0FBQyxjQUFjLENBQUM7b0JBQ3pCLFNBQVMsRUFBRTt3QkFDVCxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixLQUFLLENBQUMsY0FBYyxFQUFFLEVBQUUsa0JBQWtCLEVBQUUsR0FBRyxDQUFDO3dCQUMxRSxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixLQUFLLENBQUMsY0FBYyxFQUFFLEVBQUUsa0JBQWtCLEVBQUUsSUFBSSxDQUFDO3FCQUM1RTtpQkFDRixDQUFDLENBQ0gsQ0FBQzthQUNIO1NBQ0Y7YUFBTTtZQUNMLEtBQUssSUFBSSxlQUFlLElBQUksS0FBSyxDQUFDLHNCQUFzQixFQUFFO2dCQUN4RCxJQUFJLENBQUMseUJBQXlCLENBQUMsZUFBZSxDQUFDLGVBQWUsQ0FBQyxDQUFDO2FBQ2pFO1NBQ0Y7UUFFRCxnSkFBZ0o7UUFDaEosSUFBSSxDQUFDLHlCQUF5QixDQUFDLGVBQWUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUMsRUFBRSxPQUFPLEVBQUUsQ0FBQyxrQkFBa0IsRUFBRSxrQkFBa0IsQ0FBQyxFQUFFLFNBQVMsRUFBRSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDMUssSUFBSSxDQUFDLDJCQUEyQixHQUFvQixJQUFJLENBQUMseUJBQTBCLENBQUMsUUFBUSxDQUFDO1FBRTdGLElBQUksQ0FBQywrQkFBK0IsR0FBRyxJQUFJLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsMEJBQTBCLEVBQUU7WUFDdEcsSUFBSSxFQUFFLE1BQU0sQ0FBQyxlQUFlLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLHdDQUF3QyxDQUFDLENBQUM7WUFDM0csVUFBVSxFQUFFLEdBQUc7WUFDZixZQUFZLEVBQUUsTUFBTSxDQUFDLFlBQVksQ0FBQyxNQUFNO1lBQ3hDLE9BQU8sRUFBRSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUM7WUFDOUIsV0FBVyxFQUFFO2dCQUNYLGVBQWUsRUFBRSxJQUFJLENBQUMsa0JBQWtCO2dCQUN4QyxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsY0FBYztnQkFDdEMscUJBQXFCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQjtnQkFDL0MsU0FBUyxFQUFFLGNBQWM7YUFDMUI7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsZ0JBQWdCLENBQUMsZUFBZSxDQUFDLElBQUksMENBQWtCLENBQUMsSUFBSSxDQUFDLCtCQUErQixDQUFDLENBQUMsQ0FBQztRQUNwRyxJQUFJLENBQUMsK0JBQStCLENBQUMsZUFBZSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQyxFQUFFLE9BQU8sRUFBRSxDQUFDLGtCQUFrQixDQUFDLEVBQUUsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUM1SixJQUFJLENBQUMsNkJBQTZCLEdBQW9CLElBQUksQ0FBQywrQkFBZ0MsQ0FBQyxRQUFRLENBQUM7UUFFckcsTUFBTSxjQUFjLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUU5RCxJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQzdELFVBQVUsRUFBRSxjQUFjO1lBQzFCLE9BQU8sRUFBRSxzQkFBUSxDQUFDLEtBQUssQ0FBQyxrQ0FBa0MsQ0FBQztZQUMzRCxjQUFjLEVBQUUsSUFBSTtTQUNyQixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsK0JBQStCLENBQUMsZUFBZSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUMzRSxPQUFPLEVBQUU7Z0JBQ1Asd0JBQXdCLEVBQUUsd0JBQXdCO2FBQ25EO1lBQ0QsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO1NBQ2pCLENBQUMsQ0FBQyxDQUFDO1FBQ0osSUFBSSxDQUFDLHlCQUF5QixDQUFDLGVBQWUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDckUsT0FBTyxFQUFFO2dCQUNQLHdCQUF3QjthQUN6QjtZQUNELFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztTQUNqQixDQUFDLENBQUMsQ0FBQztRQUNKLFlBQVk7UUFDWixZQUFZO1FBQ1osWUFBWTtRQUNaLElBQUksbUNBQW1DLEVBQUU7WUFDdkMsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDLENBQUM7WUFFbkQsTUFBTSxxQkFBcUIsR0FBRywrQkFBK0IsQ0FBQztZQUU5RCxjQUFjO1lBQ2QsTUFBTSx5QkFBeUIsR0FBRyxJQUFJLHVCQUFZLENBQUMsSUFBSSxFQUFFLEdBQUcsT0FBTyxpQkFBaUIsRUFBRTtnQkFDcEYsUUFBUSxFQUFxQixJQUFJLENBQUMsK0JBQWdDLENBQUMsUUFBUTtnQkFDM0UsZUFBZSxFQUFFLHFCQUFxQjtnQkFDdEMsVUFBVSxFQUFFLFVBQVU7Z0JBQ3RCLGFBQWEsRUFBRSx3QkFBYSxDQUFDLGNBQWMsQ0FBQyxNQUFNLEVBQUUsV0FBVyxFQUFFLElBQUksRUFBRSxTQUFTLEVBQUUsWUFBWSxDQUFDO3FCQUM1RixXQUFXLENBQUMsU0FBUyxFQUFFLEdBQUcsRUFBRSxrQkFBa0IsV0FBVyxzQkFBc0IsQ0FBQztnQkFDbkYsV0FBVyxFQUFFLGFBQWE7YUFDM0IsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxDQUFDLG1CQUFtQixHQUFHLHlCQUF5QixDQUFDLE1BQU0sQ0FBQyxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO1lBRWxGLE1BQU0sNEJBQTRCLEdBQUcsSUFBSSx1QkFBWSxDQUFDLElBQUksRUFBRSxHQUFHLE9BQU8sb0JBQW9CLEVBQUU7Z0JBQzFGLFFBQVEsRUFBcUIsSUFBSSxDQUFDLCtCQUFnQyxDQUFDLFFBQVE7Z0JBQzNFLGVBQWUsRUFBRSxxQkFBcUI7Z0JBQ3RDLFVBQVUsRUFBRSxhQUFhO2dCQUN6QixhQUFhLEVBQUUsd0JBQWEsQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLFdBQVcsRUFBRSxJQUFJLEVBQUUsU0FBUyxFQUFFLE9BQU8sQ0FBQztxQkFDdkYsV0FBVyxDQUFDLFNBQVMsRUFBRSxHQUFHLEVBQUUsa0JBQWtCLFdBQVcsNkJBQTZCLENBQUM7Z0JBQzFGLFdBQVcsRUFBRSxRQUFRO2FBQ3RCLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxzQkFBc0IsR0FBRyw0QkFBNEIsQ0FBQyxNQUFNLENBQUMsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztZQUV4Riw0RkFBNEY7WUFDNUYsTUFBTSwyQkFBMkIsR0FBRyxJQUFJLHVCQUFZLENBQUMsSUFBSSxFQUFFLEdBQUcsT0FBTyxxQkFBcUIsRUFBRTtnQkFDMUYsUUFBUSxFQUFxQixJQUFJLENBQUMsK0JBQWdDLENBQUMsUUFBUTtnQkFDM0UsZUFBZSxFQUFFLHFCQUFxQjtnQkFDdEMsVUFBVSxFQUFFLGNBQWM7Z0JBQzFCLGFBQWEsRUFBRSx3QkFBYSxDQUFDLGNBQWMsQ0FBQyxNQUFNLEVBQUUsV0FBVyxFQUFFLElBQUksRUFBRSxTQUFTLEVBQUUsT0FBTyxDQUFDO3FCQUN2RixXQUFXLENBQUMsU0FBUyxFQUFFLEdBQUcsRUFBRSxrQkFBa0IsV0FBVyw2QkFBNkIsQ0FBQztnQkFDMUYsV0FBVyxFQUFFLEdBQUc7YUFDakIsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxDQUFDLHFCQUFxQixHQUFHLDJCQUEyQixDQUFDLE1BQU0sQ0FBQyxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO1lBRXRGLE1BQU0sMkJBQTJCLEdBQUcsSUFBSSx1QkFBWSxDQUFDLElBQUksRUFBRSxHQUFHLE9BQU8sb0JBQW9CLEVBQUU7Z0JBQ3pGLFFBQVEsRUFBcUIsSUFBSSxDQUFDLHlCQUEwQixDQUFDLFFBQVE7Z0JBQ3JFLGVBQWUsRUFBRSxxQkFBcUI7Z0JBQ3RDLFVBQVUsRUFBRSxhQUFhO2dCQUN6QixhQUFhLEVBQUUsd0JBQWEsQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLFdBQVcsRUFBRSxJQUFJLEVBQUUsU0FBUyxDQUFDO3FCQUM5RSxXQUFXLENBQUMsU0FBUyxFQUFFLEdBQUcsRUFBRSxrQkFBa0IsV0FBVyxjQUFjLENBQUM7Z0JBQzNFLFdBQVcsRUFBRSxHQUFHO2FBQ2pCLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxxQkFBcUIsR0FBRywyQkFBMkIsQ0FBQyxNQUFNLENBQUMsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztZQUV0RixNQUFNLGdDQUFnQyxHQUFHLElBQUksdUJBQVksQ0FBQyxJQUFJLEVBQUUsR0FBRyxPQUFPLHdCQUF3QixFQUFFO2dCQUNsRyxRQUFRLEVBQXFCLElBQUksQ0FBQyx5QkFBMEIsQ0FBQyxRQUFRO2dCQUNyRSxlQUFlLEVBQUUscUJBQXFCO2dCQUN0QyxVQUFVLEVBQUUsYUFBYTtnQkFDekIsYUFBYSxFQUFFLHdCQUFhLENBQUMsY0FBYyxDQUFDLE1BQU0sRUFBRSxXQUFXLEVBQUUsSUFBSSxFQUFFLFNBQVMsRUFBRSxPQUFPLENBQUM7cUJBQ3ZGLFdBQVcsQ0FBQyxTQUFTLEVBQUUsR0FBRyxFQUFFLGtCQUFrQixXQUFXLG1DQUFtQyxDQUFDO2dCQUNoRyxXQUFXLEVBQUUsUUFBUTthQUN0QixDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsMEJBQTBCLEdBQUcsZ0NBQWdDLENBQUMsTUFBTSxDQUFDLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7WUFDaEcsbUJBQW1CO1lBRW5CLGdDQUFnQztZQUVoQyxNQUFNLGNBQWMsR0FBRyxJQUFJLFVBQVUsQ0FBQyxjQUFjLENBQUM7Z0JBQ25ELFVBQVUsRUFBRSwyQkFBMkI7Z0JBQ3ZDLFlBQVksRUFBRTtvQkFDWixLQUFLLEVBQUUsSUFBSSxDQUFDLHNCQUFzQjtvQkFDbEMsUUFBUSxFQUFFLElBQUksQ0FBQyxtQkFBbUI7aUJBQ25DO2FBQ0YsQ0FBQyxDQUFDO1lBQ0gsTUFBTSxRQUFRLEdBQUcsSUFBSSxVQUFVLENBQUMsY0FBYyxDQUFDO2dCQUM3QyxVQUFVLEVBQUUsNEJBQTRCO2dCQUN4QyxZQUFZLEVBQUU7b0JBQ1osV0FBVyxFQUFFLElBQUksQ0FBQyxxQkFBcUI7b0JBQ3ZDLFlBQVksRUFBRSxJQUFJLENBQUMscUJBQXFCO2lCQUN6QzthQUNGLENBQUMsQ0FBQztZQUNILHFDQUFxQztZQUVyQyxNQUFNLGlCQUFpQixHQUFHLElBQUksdUJBQVksQ0FBQyxJQUFJLEVBQUUsR0FBRyxPQUFPLGNBQWMsRUFBRTtnQkFDekUsUUFBUSxFQUFxQixJQUFJLENBQUMseUJBQTBCLENBQUMsUUFBUTtnQkFDckUsZUFBZSxFQUFFLHFCQUFxQjtnQkFDdEMsVUFBVSxFQUFFLFFBQVE7Z0JBQ3BCLGFBQWEsRUFBRSx3QkFBYSxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsT0FBTyxFQUFFLE9BQU8sQ0FBQztnQkFDL0QsV0FBVyxFQUFFLEdBQUc7YUFDakIsQ0FBQyxDQUFDO1lBRUgsTUFBTSxrQ0FBa0MsR0FBRyxJQUFJLHVCQUFZLENBQUMsSUFBSSxFQUFFLEdBQUcsT0FBTywrQkFBK0IsRUFBRTtnQkFDM0csUUFBUSxFQUFxQixJQUFJLENBQUMseUJBQTBCLENBQUMsUUFBUTtnQkFDckUsZUFBZSxFQUFFLHFCQUFxQjtnQkFDdEMsVUFBVSxFQUFFLHdCQUF3QjtnQkFDcEMsYUFBYSxFQUFFLHdCQUFhLENBQUMsT0FBTyxDQUFDLDRDQUE0QyxDQUFDO2dCQUNsRixXQUFXLEVBQUUsR0FBRzthQUNqQixDQUFDLENBQUM7WUFDSCxNQUFNLG1CQUFtQixHQUFHLElBQUksdUJBQVksQ0FBQyxJQUFJLEVBQUUsR0FBRyxPQUFPLDRCQUE0QixFQUFFO2dCQUN6RixRQUFRLEVBQXFCLElBQUksQ0FBQyx5QkFBMEIsQ0FBQyxRQUFRO2dCQUNyRSxlQUFlLEVBQUUscUJBQXFCO2dCQUN0QyxVQUFVLEVBQUUscUJBQXFCO2dCQUNqQyxhQUFhLEVBQUUsd0JBQWEsQ0FBQyxPQUFPLENBQUMseUNBQXlDLENBQUM7Z0JBQy9FLFdBQVcsRUFBRSxHQUFHO2FBQ2pCLENBQUMsQ0FBQztZQUVILE1BQU0sc0NBQXNDLEdBQUcsSUFBSSx1QkFBWSxDQUFDLElBQUksRUFBRSxHQUFHLE9BQU8sOEJBQThCLEVBQUU7Z0JBQzlHLFFBQVEsRUFBcUIsSUFBSSxDQUFDLHlCQUEwQixDQUFDLFFBQVE7Z0JBQ3JFLGVBQWUsRUFBRSxxQkFBcUI7Z0JBQ3RDLFVBQVUsRUFBRSx3Q0FBd0M7Z0JBQ3BELGFBQWEsRUFBRSx3QkFBYSxDQUFDLE9BQU8sQ0FBQyw0REFBNEQsQ0FBQztnQkFDbEcsV0FBVyxFQUFFLEdBQUc7YUFDakIsQ0FBQyxDQUFDO1lBQ0gsTUFBTSxtQkFBbUIsR0FBRyxJQUFJLHVCQUFZLENBQUMsSUFBSSxFQUFFLEdBQUcsT0FBTyw0QkFBNEIsRUFBRTtnQkFDekYsUUFBUSxFQUFxQixJQUFJLENBQUMseUJBQTBCLENBQUMsUUFBUTtnQkFDckUsZUFBZSxFQUFFLHFCQUFxQjtnQkFDdEMsVUFBVSxFQUFFLHFCQUFxQjtnQkFDakMsYUFBYSxFQUFFLHdCQUFhLENBQUMsT0FBTyxDQUFDLHlDQUF5QyxDQUFDO2dCQUMvRSxXQUFXLEVBQUUsR0FBRzthQUNqQixDQUFDLENBQUM7WUFFSCxNQUFNLFdBQVcsR0FBc0IsaUJBQWlCLENBQUMsTUFBTSxDQUFDLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7WUFDdEYsTUFBTSxtQkFBbUIsR0FBc0Isa0NBQWtDLENBQUMsTUFBTSxDQUFDLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7WUFDL0csTUFBTSxnQkFBZ0IsR0FBc0IsbUJBQW1CLENBQUMsTUFBTSxDQUFDLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7WUFDN0YsTUFBTSwyQkFBMkIsR0FBc0Isc0NBQXNDLENBQUMsTUFBTSxDQUFDLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7WUFDM0gsTUFBTSx5QkFBeUIsR0FBc0IsbUJBQW1CLENBQUMsTUFBTSxDQUFDLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7WUFFdEcsTUFBTSxpQ0FBaUMsR0FBcUIsSUFBSSxVQUFVLENBQUMsTUFBTSxDQUFDO2dCQUNoRixTQUFTLEVBQUUsY0FBYztnQkFDekIsVUFBVSxFQUFFLGdCQUFnQjtnQkFDNUIsYUFBYSxFQUFFO29CQUNiLFNBQVMsRUFBRSw0QkFBNEI7aUJBQ3hDO2dCQUNELFNBQVMsRUFBRSxLQUFLO2FBQ2pCLENBQUMsQ0FBQztZQUNILG1CQUFtQjtZQUVuQixNQUFNLGNBQWMsR0FBQyxFQUFFLENBQUM7WUFDeEIsZ0NBQWdDO1lBRWhDLFlBQVk7WUFDWixJQUFJLFVBQVUsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLDBCQUEwQixFQUFFO2dCQUN6RCxHQUFHLEVBQUUsS0FBSztnQkFDVixjQUFjLEVBQUUsVUFBVSxDQUFDLGNBQWMsQ0FBQyxJQUFJO2dCQUM5QyxLQUFLLEVBQUUsT0FBTztnQkFDZCxPQUFPLEVBQUU7b0JBQ1A7d0JBQ0UsSUFBSSxVQUFVLENBQUMsTUFBTSxDQUFDLElBQUksVUFBVSxDQUFDLFVBQVUsQ0FBQyxFQUFFLFFBQVEsRUFBRSxnQ0FBZ0MsRUFBRSxLQUFLLEVBQUUsY0FBYyxFQUFFLENBQUMsQ0FBQztxQkFDeEg7b0JBQ0Q7d0JBQ0UsSUFBSSxVQUFVLENBQUMsTUFBTSxDQUFDLElBQUksVUFBVSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEtBQUssRUFBRSxnQkFBZ0IsRUFBRSxJQUFJLEVBQUUsQ0FBQyxjQUFjLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEdBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO3dCQUMzSSxJQUFJLFVBQVUsQ0FBQyxNQUFNLENBQUMsSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDLEVBQUUsS0FBSyxFQUFFLFVBQVUsRUFBRSxJQUFJLEVBQUUsQ0FBQyxRQUFRLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEdBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO3FCQUNoSTtvQkFDRDt3QkFDRSxJQUFJLFVBQVUsQ0FBQyxNQUFNLENBQUMsSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDLEVBQUUsS0FBSyxFQUFFLFVBQVUsRUFBRSxJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEdBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO3dCQUMvSSxJQUFJLFVBQVUsQ0FBQyxNQUFNLENBQUMsSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDLEVBQUUsS0FBSyxFQUFFLGFBQWEsRUFBRSxJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMsc0JBQXNCLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEdBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO3dCQUNySixJQUFJLFVBQVUsQ0FBQyxNQUFNLENBQUMsSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDLEVBQUUsS0FBSyxFQUFFLGFBQWEsRUFBRSxJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMsMEJBQTBCLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEdBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO3dCQUN6SixJQUFJLFVBQVUsQ0FBQyxNQUFNLENBQUMsSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDLEVBQUUsS0FBSyxFQUFFLHdCQUF3QixFQUFFLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxFQUFFLEtBQUssRUFBRSxDQUFDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsR0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUM7cUJBQ3JNO29CQUNEO3dCQUNFLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQyxJQUFJLFVBQVUsQ0FBQyxVQUFVLENBQUMsRUFBRSxRQUFRLEVBQUUsaUNBQWlDLEVBQUUsS0FBSyxFQUFFLGNBQWMsRUFBRSxDQUFDLENBQUM7cUJBQ3pIO29CQUNEO3dCQUNFLElBQUksVUFBVSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEtBQUssRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsV0FBVyxDQUFDLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxHQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7d0JBQ3pHLElBQUksVUFBVSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEtBQUssRUFBRSxlQUFlLEVBQUUsSUFBSSxFQUFFLENBQUMsbUJBQW1CLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEdBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQzt3QkFDeEgsSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDLEVBQUUsS0FBSyxFQUFFLFlBQVksRUFBRSxJQUFJLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsR0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO3dCQUNsSCxJQUFJLFVBQVUsQ0FBQyxXQUFXLENBQUMsRUFBRSxLQUFLLEVBQUUsc0JBQXNCLEVBQUUsSUFBSSxFQUFFLENBQUMsMkJBQTJCLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEdBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQzt3QkFDdkksSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDLEVBQUUsS0FBSyxFQUFFLHFCQUFxQixFQUFFLElBQUksRUFBRSxDQUFDLHlCQUF5QixDQUFDLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxHQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7cUJBQ3JJO29CQUNEO3dCQUNFLElBQUksVUFBVSxDQUFDLFVBQVUsQ0FBQyxFQUFFLFFBQVEsRUFBRSxZQUFZLEVBQUUsS0FBSyxFQUFFLGNBQWMsRUFBRSxDQUFDO3FCQUM3RTtvQkFDRDt3QkFDRSxJQUFJLFVBQVUsQ0FBQyxXQUFXLENBQUMsRUFBRSxLQUFLLEVBQUUseUNBQXlDLEVBQUUsSUFBSSxFQUFFLENBQUMsaUNBQWlDLENBQUMsRUFBRSxDQUFDO3FCQUM1SDtvQkFDRDt3QkFDRSxJQUFJLFVBQVUsQ0FBQyxVQUFVLENBQUMsRUFBRSxRQUFRLEVBQUUsNkJBQTZCLEVBQUUsS0FBSyxFQUFFLGNBQWMsRUFBRSxDQUFDO3FCQUM5RjtvQkFDRDt3QkFDRSxJQUFJLFVBQVUsQ0FBQyxXQUFXLENBQUMsRUFBRSxLQUFLLEVBQUUsdUJBQXVCLEVBQUUsSUFBSSxFQUFFLENBQUMsSUFBSSxDQUFDLHlCQUF5QixDQUFDLFlBQVksRUFBRSxDQUFDLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxHQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7d0JBQzFKLElBQUksVUFBVSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEtBQUssRUFBRSw0QkFBNEIsRUFBRSxJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMseUJBQXlCLENBQUMsaUJBQWlCLEVBQUUsQ0FBQyxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsR0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO3dCQUNwSyxJQUFJLFVBQVUsQ0FBQyxXQUFXLENBQUMsRUFBRSxLQUFLLEVBQUUsMEJBQTBCLEVBQUUsSUFBSSxFQUFFLENBQUMsSUFBSSxDQUFDLHlCQUF5QixDQUFDLGVBQWUsRUFBRSxDQUFDLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxHQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7cUJBQ2pLO29CQUNEO3dCQUNFLElBQUksVUFBVSxDQUFDLFVBQVUsQ0FBQyxFQUFFLFFBQVEsRUFBRSxtQ0FBbUMsRUFBRSxLQUFLLEVBQUUsY0FBYyxFQUFFLENBQUM7cUJBQ3BHO29CQUNEO3dCQUNFLElBQUksVUFBVSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEtBQUssRUFBRSxxQkFBcUIsRUFBRSxJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMsK0JBQStCLENBQUMsWUFBWSxFQUFFLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEdBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQzt3QkFDOUosSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDLEVBQUUsS0FBSyxFQUFFLDBCQUEwQixFQUFFLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQywrQkFBK0IsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxHQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7d0JBQ3hLLElBQUksVUFBVSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEtBQUssRUFBRSx3QkFBd0IsRUFBRSxJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMsK0JBQStCLENBQUMsZUFBZSxFQUFFLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEdBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztxQkFDcks7aUJBQ0Y7YUFDRixDQUFDLENBQUM7U0FDSjtRQUNELGdCQUFnQjtRQUNoQixJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksQ0FBQyx3QkFBd0IsRUFBRSxDQUFDO0lBQ3RELENBQUM7SUFDRDs7V0FFTztJQUNHLFdBQVc7UUFDbkIsc0ZBQXNGO1FBQ3RGLDBGQUEwRjtRQUMxRixrRkFBa0Y7UUFDbEYsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixLQUFLLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO1FBQ3RGLElBQUksS0FBVSxDQUFDO1FBQ2YsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLG1CQUFtQixFQUFFO1lBQ2xDLE1BQU0sd0JBQXdCLEdBQUc7Z0JBQy9CLDBDQUEwQyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLGlCQUFpQixDQUFDO2FBQ3JGLENBQUM7WUFDRixLQUFLLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO2dCQUN6QixHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLEtBQUs7Z0JBQ3pCLEdBQUksd0JBQXdCO2FBQzdCLENBQUMsQ0FBQyxDQUFDLHdCQUF3QixDQUFDO1NBQzlCO2FBQU07WUFDTCxLQUFLLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLENBQUMsS0FBSyxDQUFDO1NBQzdGO1FBR0QsT0FBTztZQUNMLFFBQVEsRUFBRSxHQUNSLHNCQUFzQixDQUFDLFFBQVEsRUFBRSxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQzVFLEdBQUcsTUFBTSxFQUFFO1lBQ1gsVUFBVSxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsWUFBWSxDQUNyQztnQkFDRSxLQUFLLEVBQUUsS0FBSztnQkFDWixlQUFlLEVBQUUsSUFBSSxDQUFDLFlBQVksQ0FBQyxlQUFlO2dCQUNsRCxJQUFJLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJO2FBQ3RCLENBQ0Y7U0FDRixDQUFDO0lBQ0osQ0FBQztJQUNEOzs7Ozs7V0FNTztJQUNDLHdCQUF3QjtRQUM5QixNQUFNLEtBQUssR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUU3QixNQUFNLGdCQUFnQixHQUFHO1lBQ3ZCLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FDckI7Z0JBQ0UsT0FBTyxFQUFFLENBQUMsdUJBQXVCLENBQUM7Z0JBQ2xDLFNBQVMsRUFBRSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsZUFBZSxDQUFDO2FBQy9DLENBQ0Y7U0FDRixDQUFDO1FBRUYsK0VBQStFO1FBQy9FLElBQUksSUFBSSxDQUFDLGtCQUFrQixLQUFLLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxPQUFPLEVBQUU7WUFDOUQsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDNUMsT0FBTyxFQUFFO29CQUNQLDBCQUEwQixFQUFFLHNCQUFzQjtpQkFDbkQ7Z0JBQ0QsOEhBQThIO2dCQUM5SCxTQUFTLEVBQUU7b0JBQ1QsS0FBSyxDQUFDLFNBQVMsQ0FDYjt3QkFDRSxPQUFPLEVBQUUsUUFBUTt3QkFDakIsUUFBUSxFQUFFLFdBQVc7d0JBQ3JCLFNBQVMsRUFBRSx1QkFBUyxDQUFDLG1CQUFtQjt3QkFDeEMsWUFBWSxFQUFFLEdBQ1osS0FBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLGVBQWUsRUFBRSx1QkFBUyxDQUFDLG1CQUFtQixDQUFDLENBQUMsWUFDbkYsR0FBRztxQkFDSixDQUNGO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUM1QyxPQUFPLEVBQUU7b0JBQ1AsbUJBQW1CLEVBQUUsZ0JBQWdCLEVBQUUscUJBQXFCO2lCQUM3RDtnQkFDRCxTQUFTLEVBQUU7b0JBQ1QsS0FBSyxDQUFDLFNBQVMsQ0FDYjt3QkFDRSxPQUFPLEVBQUUsUUFBUTt3QkFDakIsUUFBUSxFQUFFLE1BQU07d0JBQ2hCLFlBQVksRUFBRSxxREFBcUQ7cUJBQ3BFLENBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztTQUNMO1FBRUQsT0FBTyxnQkFBZ0IsQ0FBQztJQUMxQixDQUFDOztBQXZmSCxrRUF3ZkM7OztBQXZmeUIsMERBQThCLEdBQUc7SUFDdkQsR0FBRyxDQUFDLGtCQUFrQixDQUFDLGdCQUFnQjtJQUN2QyxHQUFHLENBQUMsa0JBQWtCLENBQUMsT0FBTztJQUM5QixHQUFHLENBQUMsa0JBQWtCLENBQUMsbUJBQW1CO0NBQzNDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0IHsgRHVyYXRpb24sIEF3cywgQXJuRm9ybWF0LCBTdGFjaywgUmVtb3ZhbFBvbGljeSB9IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCAqIGFzIGNsb3Vkd2F0Y2ggZnJvbSAnYXdzLWNkay1saWIvYXdzLWNsb3Vkd2F0Y2gnO1xuaW1wb3J0ICogYXMgZHluYW1vZGIgZnJvbSAnYXdzLWNkay1saWIvYXdzLWR5bmFtb2RiJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCB7IElSb2xlLCBNYW5hZ2VkUG9saWN5IH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBGaWx0ZXJQYXR0ZXJuLCBJTG9nR3JvdXAsIE1ldHJpY0ZpbHRlciB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1sb2dzJztcbmltcG9ydCAqIGFzIHNucyBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc25zJztcbmltcG9ydCB7IExhbWJkYVN1YnNjcmlwdGlvbiB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1zbnMtc3Vic2NyaXB0aW9ucyc7XG5pbXBvcnQgKiBhcyBzZm4gZnJvbSAnYXdzLWNkay1saWIvYXdzLXN0ZXBmdW5jdGlvbnMnO1xuaW1wb3J0ICogYXMgdGFza3MgZnJvbSAnYXdzLWNkay1saWIvYXdzLXN0ZXBmdW5jdGlvbnMtdGFza3MnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5leHBvcnQgeyBUZXh0cmFjdEdlbmVyaWNTeW5jU2ZuVGFzaywgVGV4dHJhY3RHZW5lcmljU3luY1NmblRhc2tQcm9wcyB9IGZyb20gJy4vdGV4dHJhY3RTeW5jJztcbmV4cG9ydCB7IFRleHRyYWN0UE9DRGVjaWRlciwgVGV4dHJhY3REUFBPQ0RlY2lkZXJQcm9wcyB9IGZyb20gJy4vdGV4dHJhY3REZWNpZGVyJztcbmV4cG9ydCB7IFRleHRyYWN0Q2xhc3NpZmljYXRpb25Db25maWd1cmF0b3IsIFRleHRyYWN0Q2xhc3NpZmljYXRpb25Db25maWd1cmF0b3JQcm9wcyB9IGZyb20gJy4vdGV4dHJhY3RDbGFzc2lmaWNhdGlvbkNvbmZpZ3VyYXRvcic7XG5leHBvcnQgeyBUZXh0cmFjdEFzeW5jVG9KU09OLCBUZXh0cmFjdEFzeW5jVG9KU09OUHJvcHMgfSBmcm9tICcuL3RleHRyYWN0T3V0cHV0Q29uZmlnVG9KU09OJztcbmV4cG9ydCB7IFRleHRyYWN0R2VuZXJhdGVDU1YsIFRleHRyYWN0R2VuZXJhdGVDU1ZQcm9wcyB9IGZyb20gJy4vdGV4dHJhY3RHZW5lcmF0ZUNTVic7XG5leHBvcnQgeyBUZXh0cmFjdEEySVNmblRhc2ssIFRleHRyYWN0QTJJU2ZuVGFza1Byb3BzIH0gZnJvbSAnLi90ZXh0cmFjdEEySSc7XG5leHBvcnQgeyBDU1ZUb0F1cm9yYVRhc2ssIENTVlRvQXVyb3JhVGFza1Byb3BzIH0gZnJvbSAnLi9yZHNDU1ZUb0F1cm9yYSc7XG5leHBvcnQgeyBDb21wcmVoZW5kR2VuZXJpY1N5bmNTZm5UYXNrLCBDb21wcmVoZW5kR2VuZXJpY1N5bmNTZm5UYXNrUHJvcHMgfSBmcm9tICcuL2NvbXByZWhlbmRDbGFzc2lmaWNhdGlvbic7XG5leHBvcnQgeyBTcGFjeVNmblRhc2ssIFNwYWN5U2ZuVGFza1Byb3BzIH0gZnJvbSAnLi9zcGFjeUNsYXNzaWZpY2F0aW9uJztcbmV4cG9ydCB7IERvY3VtZW50U3BsaXR0ZXIsIERvY3VtZW50U3BsaXR0ZXJQcm9wcyB9IGZyb20gJy4vZG9jdW1lbnRTcGxpdHRlcic7XG5leHBvcnQgeyBSRFNBdXJvcmFTZXJ2ZXJsZXNzLCBSRFNBdXJvcmFTZXJ2ZXJsZXNzUHJvcHMgfSBmcm9tICcuL3Jkc0F1cm9yYVNlcnZlcmxlc3MnO1xuXG5cbmV4cG9ydCBmdW5jdGlvbiB2YWxpZGF0ZVBhdHRlcm5TdXBwb3J0ZWQoaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLCBzdXBwb3J0ZWRQYXR0ZXJuczogc2ZuLkludGVncmF0aW9uUGF0dGVybltdKSB7XG4gIGlmICghIHN1cHBvcnRlZFBhdHRlcm5zLmluY2x1ZGVzKGludGVncmF0aW9uUGF0dGVybikpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoYFVuc3VwcG9ydGVkIHNlcnZpY2UgaW50ZWdyYXRpb24gcGF0dGVybi4gU3VwcG9ydGVkIFBhdHRlcm5zOiAke3N1cHBvcnRlZFBhdHRlcm5zfS4gUmVjZWl2ZWQ6ICR7aW50ZWdyYXRpb25QYXR0ZXJufWApO1xuICB9XG59XG4vKipcbiAqIFN1ZmZpeGVzIGNvcnJlc3BvbmRpbmcgdG8gZGlmZmVyZW50IHNlcnZpY2UgaW50ZWdyYXRpb24gcGF0dGVybnNcbiAqXG4gKiBLZXkgaXMgdGhlIHNlcnZpY2UgaW50ZWdyYXRpb24gcGF0dGVybiwgdmFsdWUgaXMgdGhlIHJlc291cmNlIEFSTiBzdWZmaXguXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc3RlcC1mdW5jdGlvbnMvbGF0ZXN0L2RnL2Nvbm5lY3QtdG8tcmVzb3VyY2UuaHRtbFxuICovXG5jb25zdCByZXNvdXJjZUFyblN1ZmZpeDogUmVjb3JkPHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4sIHN0cmluZz4gPSB7XG4gIFtzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLlJFUVVFU1RfUkVTUE9OU0VdOiAnJyxcbiAgW3Nmbi5JbnRlZ3JhdGlvblBhdHRlcm4uUlVOX0pPQl06ICcuc3luYycsXG4gIFtzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLldBSVRfRk9SX1RBU0tfVE9LRU5dOiAnLndhaXRGb3JUYXNrVG9rZW4nLFxufTtcblxuZXhwb3J0IGZ1bmN0aW9uIGludGVncmF0aW9uUmVzb3VyY2VBcm4oc2VydmljZTogc3RyaW5nLCBhcGk6IHN0cmluZywgaW50ZWdyYXRpb25QYXR0ZXJuID86c2ZuLkludGVncmF0aW9uUGF0dGVybik6IHN0cmluZyB7XG4gIGlmICghIHNlcnZpY2UgfHwgISBhcGkpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoXCJCb3RoICdzZXJ2aWNlJyBhbmQgJ2FwaScgbXVzdCBiZSBwcm92aWRlZCB0byBidWlsZCB0aGUgcmVzb3VyY2UgQVJOLlwiKTtcbiAgfVxuICByZXR1cm4gYGFybjoke1xuICAgIEF3cy5QQVJUSVRJT05cbiAgfTpzdGF0ZXM6Ojoke3NlcnZpY2V9OiR7YXBpfWAgKyAoaW50ZWdyYXRpb25QYXR0ZXJuID8gcmVzb3VyY2VBcm5TdWZmaXhbaW50ZWdyYXRpb25QYXR0ZXJuXSA6ICcnKTtcbn1cbmV4cG9ydCBpbnRlcmZhY2UgVGV4dHJhY3RHZW5lcmljQXN5bmNTZm5UYXNrUHJvcHMgZXh0ZW5kcyBzZm4uVGFza1N0YXRlQmFzZVByb3BzIHtcbiAgLyoqIEJ1Y2tldG5hbWUgdG8gb3V0cHV0IGRhdGEgdG8gKi9cbiAgcmVhZG9ubHkgczNPdXRwdXRCdWNrZXQgOiBzdHJpbmc7XG4gIC8qKiBUaGUgcHJlZml4IHRvIHVzZSBmb3IgdGhlIHRlbXBvcmFyeSBvdXRwdXQgZmlsZXMgKGUuIGcuIG91dHB1dCBmcm9tIGFzeW5jIHByb2Nlc3MgYmVmb3JlIHN0aWNoaW5nIHRvZ2V0aGVyKSAqL1xuICByZWFkb25seSBzM1RlbXBPdXRwdXRQcmVmaXggOiBzdHJpbmc7XG4gIC8qKiBCdWNrZXRuYW1lIGFuZCBwcmVmaXggdG8gcmVhZCBkb2N1bWVudCBmcm9tXG4gIC8qKiBsb2NhdGlvbiBvZiBpbnB1dCBTMyBvYmplY3RzIC0gaWYgbGVmdCBlbXB0eSB3aWxsIGdlbmVyYXRlIHJ1bGUgZm9yIHMzIGFjY2VzcyB0byBhbGwgWypdICovXG4gIHJlYWRvbmx5IHMzSW5wdXRCdWNrZXQ/OiBzdHJpbmc7XG4gIC8qKiBwcmVmaXggZm9yIGlucHV0IFMzIG9iamVjdHMgLSBpZiBsZWZ0IGVtcHR5IHdpbGwgZ2VuZXJhdGUgcnVsZSBmb3IgczMgYWNjZXNzIHRvIGFsbCBpbiBidWNrZXQgKi9cbiAgcmVhZG9ubHkgczNJbnB1dFByZWZpeD86IHN0cmluZztcbiAgLyoqIFdoaWNoIFRleHRyYWN0IEFQSSB0byBjYWxsXG4gICAqIEFMTCBhc3luY2hyb25vdXMgVGV4dHJhY3QgQVBJIGNhbGxzIGFyZSBzdXBwb3J0ZWQuIFZhbGlkIHZhbHVlcyBhcmUgR0VORVJJQyB8IEVYUEVOU0UgfCBMRU5ESU5HLlxuICAgKlxuICAgKiBGb3IgR0VORVJJQywgd2hlbiBjYWxsZWQgd2l0aG91dCBmZWF0dXJlcyAoZS4gZy4gRk9STVMsIFRBQkxFUywgUVVFUklFUywgU0lHTkFUVVJFKSwgU3RhcnREZXRlY3RUZXh0IGlzIGNhbGxlZCBhbmQgb25seSBPQ1IgaXMgcmV0dXJuZWQuXG4gICAqIEZvciBHRU5FUklDLCB3aGVuIGNhbGxlZCB3aXRoIGEgZmVhdHVyZSAoZS4gZy4gRk9STVMsIFRBQkxFUywgUVVFUklFUywgU0lHTkFUVVJFKSwgIFN0YXJ0QW5hbHl6ZURvY3VtZW50IGlzIGNhbGxlZC5cbiAgICogQGRlZmF1bHQgLSBHRU5FUklDICovXG4gIHJlYWRvbmx5IHRleHRyYWN0QVBJPzogJ0dFTkVSSUMnIHwgJ0VYUEVOU0UnIHwgJ0xFTkRJTkcnO1xuICAvKiogbnVtYmVyIG9mIHJldHJpZXMgaW4gU3RlcCBGdW5jdGlvbiBmbG93XG4gICAqIEBkZWZhdWx0IGlzIDEwMCAqL1xuICByZWFkb25seSB0ZXh0cmFjdEFzeW5jQ2FsbE1heFJldHJpZXM/OiBudW1iZXI7XG4gIC8qKnJldHlyIGJhY2tvZmYgcmF0ZVxuICAgKiBAZGVmYXVsdCBpcyAxLjEgKi9cbiAgcmVhZG9ubHkgdGV4dHJhY3RBc3luY0NhbGxCYWNrb2ZmUmF0ZT86IG51bWJlcjtcbiAgLyoqIGxvZyBsZXZlbCBmb3IgTGFtYmRhIGZ1bmN0aW9uLCBzdXBwb3J0cyBERUJVR3xJTkZPfFdBUk5JTkd8RVJST1J8RkFUQUxcbiAgICogQGRlZmF1bHQgPSBERUJVR1xuICAqL1xuICByZWFkb25seSBsYW1iZGFMb2dMZXZlbD8gOiAnREVCVUcnfCdJTkZPJ3wnV0FSTklORyd8J0VSUk9SJ3wnRkFUQUwnO1xuICAvKip0aW1lIGluIHNlY29uZHMgdG8gd2FpdCBiZWZvcmUgbmV4dCByZXRyeVxuICAgKiBAZGVmYXVsdCBpcyAxICovXG4gIHJlYWRvbmx5IHRleHRyYWN0QXN5bmNDYWxsSW50ZXJ2YWw/OiBudW1iZXI7XG4gIC8qKiBob3cgbG9uZyBjYW4gd2Ugd2FpdCBmb3IgdGhlIHByb2Nlc3NcbiAgICogQGRlZmF1bHQgLSAyODgwICg0OCBob3VycyAoNjAgbWluICogNDggaG91cnMgPSAyODgwKSkgKi9cbiAgcmVhZG9ubHkgdGV4dHJhY3RTdGF0ZU1hY2hpbmVUaW1lb3V0TWludXRlcz8gOiBudW1iZXI7XG4gIC8qKiBlbmFibGUgQ2xvdWRXYXRjaCBNZXRyaWNzIGFuZCBEYXNoYm9hcmRcbiAgICogQGRlZmF1bHQgLSBmYWxzZVxuICAgKi9cbiAgLyoqIExpc3Qgb2YgUG9saWN5U3RhdGVtZW50cyB0byBhdHRhY2ggdG8gdGhlIExhbWJkYSBmdW5jdGlvbi4gICovXG4gIHJlYWRvbmx5IGlucHV0UG9saWN5U3RhdGVtZW50cz86IFtpYW0uUG9saWN5U3RhdGVtZW50XTtcbiAgLyoqIExpc3Qgb2YgUG9saWN5U3RhdGVtZW50cyB0byBhdHRhY2ggdG8gdGhlIExhbWJkYSBmdW5jdGlvbi4gICovXG4gIHJlYWRvbmx5IG91dHB1dFBvbGljeVN0YXRlbWVudHM/OiBbaWFtLlBvbGljeVN0YXRlbWVudF07XG4gIHJlYWRvbmx5IGVuYWJsZUNsb3VkV2F0Y2hNZXRyaWNzQW5kRGFzaGJvYXJkPyA6IGJvb2xlYW47XG4gIC8qKiB0YXNrIHRva2VuIHRhYmxlIHRvIHVzZSBmb3IgbWFwcGluZyBvZiBUZXh0cmFjdCBbSm9iVGFnXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdGV4dHJhY3QvbGF0ZXN0L2RnL0FQSV9TdGFydERvY3VtZW50VGV4dERldGVjdGlvbi5odG1sI1RleHRyYWN0LVN0YXJ0RG9jdW1lbnRUZXh0RGV0ZWN0aW9uLXJlcXVlc3QtSm9iVGFnKVxuICAgKiB0byB0aGUgW1Rhc2tUb2tlbl0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N0ZXAtZnVuY3Rpb25zL2xhdGVzdC9kZy9jb25uZWN0LXRvLXJlc291cmNlLmh0bWwpXG4gICovXG4gIHJlYWRvbmx5IHRhc2tUb2tlblRhYmxlPzogZHluYW1vZGIuSVRhYmxlO1xuICAvKiogSUFNIFJvbGUgdG8gYXNzaWduIHRvIFRleHRyYWN0LCBieSBkZWZhdWx0XG4gICogIG5ldyBpYW0uUm9sZSh0aGlzLCAnVGV4dHJhY3RBc3luY1NOU1JvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgndGV4dHJhY3QuYW1hem9uYXdzLmNvbScpLFxuICAgICAgbWFuYWdlZFBvbGljaWVzOiBbTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0FtYXpvblNRU0Z1bGxBY2Nlc3MnKSxcbiAgICAgICAgTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0FtYXpvblNOU0Z1bGxBY2Nlc3MnKSxcbiAgICAgICAgTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0FtYXpvblMzUmVhZE9ubHlBY2Nlc3MnKSxcbiAgICAgICAgTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0FtYXpvblRleHRyYWN0RnVsbEFjY2VzcycpXSxcbiAgICB9KTsqL1xuICByZWFkb25seSBzbnNSb2xlVGV4dHJhY3Q/OiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAgICAgKiBUaGUgSlNPTiBpbnB1dCBmb3IgdGhlIGV4ZWN1dGlvbiwgc2FtZSBhcyB0aGF0IG9mIFN0YXJ0RXhlY3V0aW9uLlxuICAgICAgICpcbiAgICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N0ZXAtZnVuY3Rpb25zL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX1N0YXJ0RXhlY3V0aW9uLmh0bWxcbiAgICAgICAqXG4gICAgICAgKiBAZGVmYXVsdCAtIFRoZSBzdGF0ZSBpbnB1dCAoSlNPTiBwYXRoICckJylcbiAgICAgICAqL1xuICByZWFkb25seSBpbnB1dD8gOiBzZm4uVGFza0lucHV0O1xuXG4gIC8qKlxuICAgICAgICAgICogVGhlIG5hbWUgb2YgdGhlIGV4ZWN1dGlvbiwgc2FtZSBhcyB0aGF0IG9mIFN0YXJ0RXhlY3V0aW9uLlxuICAgICAgICAgICpcbiAgICAgICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N0ZXAtZnVuY3Rpb25zL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX1N0YXJ0RXhlY3V0aW9uLmh0bWxcbiAgICAgICAgICAqXG4gICAgICAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgICAgICAqL1xuICByZWFkb25seSBuYW1lPyA6IHN0cmluZztcblxuICAvKipcbiAgICAgICAgICAqIFBhc3MgdGhlIGV4ZWN1dGlvbiBJRCBmcm9tIHRoZSBjb250ZXh0IG9iamVjdCB0byB0aGUgZXhlY3V0aW9uIGlucHV0LlxuICAgICAgICAgICogVGhpcyBhbGxvd3MgdGhlIFN0ZXAgRnVuY3Rpb25zIFVJIHRvIGxpbmsgY2hpbGQgZXhlY3V0aW9ucyBmcm9tIHBhcmVudCBleGVjdXRpb25zLCBtYWtpbmcgaXQgZWFzaWVyIHRvIHRyYWNlIGV4ZWN1dGlvbiBmbG93IGFjcm9zcyBzdGF0ZSBtYWNoaW5lcy5cbiAgICAgICAgICAqXG4gICAgICAgICAgKiBJZiB5b3Ugc2V0IHRoaXMgcHJvcGVydHkgdG8gYHRydWVgLCB0aGUgYGlucHV0YCBwcm9wZXJ0eSBtdXN0IGJlIGFuIG9iamVjdCAocHJvdmlkZWQgYnkgYHNmbi5UYXNrSW5wdXQuZnJvbU9iamVjdGApIG9yIG9taXR0ZWQgZW50aXJlbHkuXG4gICAgICAgICAgKlxuICAgICAgICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc3RlcC1mdW5jdGlvbnMvbGF0ZXN0L2RnL2NvbmNlcHRzLW5lc3RlZC13b3JrZmxvd3MuaHRtbCNuZXN0ZWQtZXhlY3V0aW9uLXN0YXJ0aWRcbiAgICAgICAgICAqXG4gICAgICAgICAgKiBAZGVmYXVsdCAtIGZhbHNlXG4gICAgICAgICAgKi9cbiAgcmVhZG9ubHkgYXNzb2NpYXRlV2l0aFBhcmVudD8gOiBib29sZWFuO1xufVxuXG4vKipcbiAqIFRoaXMgVGFzayBjYWxscyB0aGUgVGV4dHJhY3QgdGhyb3VnaCB0aGUgYXN5bmNocm9ub3VzIEFQSS5cbiAqXG4gKiBXaGljaCBBUEkgdG8gY2FsbCBpcyBkZWZpbmVkIGluXG4gKlxuICogV2hlbiBHRU5FUklDIGlzIGNhbGxlZCB3aXRoIGZlYXR1cmVzIGluIHRoZSBtYW5pZmVzdCBkZWZpbml0aW9uLCB3aWxsIGNhbGwgdGhlIEFuYWx6eWVEb2N1bWVudCBBUEkuXG4gKlxuICogVGFrZXMgdGhlIGNvbmZpZ3VyYXRpb24gZnJvbSBcIlBheWxvYWRcIi5cIm1hbmlmZXN0XCJcbiAqXG4gKiBXaWxsIHJldHJ5IG9uIHJlY292ZXJhYmxlIGVycm9ycyBiYXNlZCBvbiB0ZXh0cmFjdEFzeW5jQ2FsbE1heFJldHJpZXNcbiAqIGVycm9ycyBmb3IgcmV0cnk6IFsnVGhyb3R0bGluZ0V4Y2VwdGlvbicsICdMaW1pdEV4Y2VlZGVkRXhjZXB0aW9uJywgJ0ludGVybmFsU2VydmVyRXJyb3InLCAnUHJvdmlzaW9uZWRUaHJvdWdocHV0RXhjZWVkZWRFeGNlcHRpb24nXSxcbiAqXG4gKiBJbnRlcm5hbGx5IGNhbGxzIFN0YXJ0KiBjYWxscyB3aXRoIE91dHB1dENvbmZpZyBhbmQgU05TTm90aWZpY2F0aW9uLlxuICogQW5vdGhlciBMYW1iZGEgZnVuY3Rpb25zIHdhaXRzIGZvciBTTlMgTm90aWZpY2F0aW9uIGV2ZW50IGFuZCBub3RpZmllcyB0aGUgU3RlcCBGdW5jdGlvbiBmbG93IHdpdGggdGhlIHRhc2sgdG9rZW4uXG4gKlxuICogU3RlcCBGdW5jdGlvbiBKU09OIGlucHV0IHJlcXVpcmVtZW50c1xuICpcbiAqICoqSW5wdXQqKjogXCJQYXlsb2FkXCIuXCJtYW5pZmVzdFwiXG4gKlxuICogKipPdXRwdXQqKjogXCJUZXh0cmFjdFRlbXBPdXRwdXRKc29uUGF0aFwiIHBvaW50cyB0byBwb3RlbnRpYWxseSBwYWdpbmF0ZWQgVGV4dHJhY3QgSlNPTiBTY2hlbWEgb3V0cHV0IGF0IFwiVGV4dHJhY3RUZW1wT3V0cHV0SnNvblBhdGhcIiAodXNpbmcgdGhlIGV4YW1wbGUgY29kZSBpdCB3aWxsIGJlIGF0OiBcInRleHRyYWN0X3Jlc3VsdFwiLlwiVGV4dHJhY3RUZW1wT3V0cHV0SnNvblBhdGhcIilcbiAqXG4gKiBXb3JrcyB0b2dldGhlciB3aXRoIFRleHRyYWN0QXN5bmNUb0pTT04sIHdoaWNoIHRha2VzIHRoZSBzM19vdXRwdXRfYnVja2V0L3MzX3RlbXBfb3V0cHV0X3ByZWZpeCBsb2NhdGlvbiBhcyBpbnB1dFxuICpcbiAqIEV4YW1wbGUgKFB5dGhvbilcbiAqIGBgYHB5dGhvblxuICAgIHRleHRyYWN0X2FzeW5jX3Rhc2sgPSB0Y2RrLlRleHRyYWN0R2VuZXJpY0FzeW5jU2ZuVGFzayhcbiAgICAgICAgc2VsZixcbiAgICAgICAgXCJUZXh0cmFjdEFzeW5jXCIsXG4gICAgICAgIHMzX291dHB1dF9idWNrZXQ9czNfb3V0cHV0X2J1Y2tldCxcbiAgICAgICAgczNfdGVtcF9vdXRwdXRfcHJlZml4PXMzX3RlbXBfb3V0cHV0X3ByZWZpeCxcbiAgICAgICAgaW50ZWdyYXRpb25fcGF0dGVybj1zZm4uSW50ZWdyYXRpb25QYXR0ZXJuLldBSVRfRk9SX1RBU0tfVE9LRU4sXG4gICAgICAgIGxhbWJkYV9sb2dfbGV2ZWw9XCJERUJVR1wiLFxuICAgICAgICB0aW1lb3V0PUR1cmF0aW9uLmhvdXJzKDI0KSxcbiAgICAgICAgaW5wdXQ9c2ZuLlRhc2tJbnB1dC5mcm9tX29iamVjdCh7XG4gICAgICAgICAgICBcIlRva2VuXCI6XG4gICAgICAgICAgICBzZm4uSnNvblBhdGgudGFza190b2tlbixcbiAgICAgICAgICAgIFwiRXhlY3V0aW9uSWRcIjpcbiAgICAgICAgICAgIHNmbi5Kc29uUGF0aC5zdHJpbmdfYXQoJyQkLkV4ZWN1dGlvbi5JZCcpLFxuICAgICAgICAgICAgXCJQYXlsb2FkXCI6XG4gICAgICAgICAgICBzZm4uSnNvblBhdGguZW50aXJlX3BheWxvYWQsXG4gICAgICAgIH0pLFxuICAgICAgICByZXN1bHRfcGF0aD1cIiQudGV4dHJhY3RfcmVzdWx0XCIpXG4gICAgYGBgXG5cbiAqL1xuZXhwb3J0IGNsYXNzIFRleHRyYWN0R2VuZXJpY0FzeW5jU2ZuVGFzayBleHRlbmRzIHNmbi5UYXNrU3RhdGVCYXNlIHtcbiAgcHJpdmF0ZSBzdGF0aWMgcmVhZG9ubHkgU1VQUE9SVEVEX0lOVEVHUkFUSU9OX1BBVFRFUk5TID0gW1xuICAgIHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uUkVRVUVTVF9SRVNQT05TRSxcbiAgICBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLlJVTl9KT0IsXG4gICAgc2ZuLkludGVncmF0aW9uUGF0dGVybi5XQUlUX0ZPUl9UQVNLX1RPS0VOLFxuICBdO1xuXG4gIHByb3RlY3RlZCByZWFkb25seSB0YXNrTWV0cmljcz86IHNmbi5UYXNrTWV0cmljc0NvbmZpZztcbiAgcHJvdGVjdGVkIHJlYWRvbmx5IHRhc2tQb2xpY2llcz86IGlhbS5Qb2xpY3lTdGF0ZW1lbnRbXTtcblxuICBwcml2YXRlIHJlYWRvbmx5IGludGVncmF0aW9uUGF0dGVybiA6IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm47XG4gIHB1YmxpYyBzdGF0ZU1hY2hpbmUgOiBzZm4uSVN0YXRlTWFjaGluZTtcbiAgcHVibGljIHN0YXJ0VGV4dHJhY3RMYW1iZGFMb2dHcm91cDpJTG9nR3JvdXA7XG4gIHB1YmxpYyByZWNlaXZlU3RhcnRTTlNMYW1iZGFMb2dHcm91cDpJTG9nR3JvdXA7XG4gIHB1YmxpYyB0YXNrVG9rZW5UYWJsZTpkeW5hbW9kYi5JVGFibGU7XG4gIHB1YmxpYyB0YXNrVG9rZW5UYWJsZU5hbWU6c3RyaW5nO1xuICBwdWJsaWMgdGV4dHJhY3RBc3luY1NOU1JvbGU6SVJvbGU7XG4gIHB1YmxpYyB0ZXh0cmFjdEFzeW5jU05TOnNucy5JVG9waWM7XG4gIHB1YmxpYyB0ZXh0cmFjdEFzeW5jQ2FsbEZ1bmN0aW9uOmxhbWJkYS5JRnVuY3Rpb247XG4gIHB1YmxpYyB0ZXh0cmFjdEFzeW5jUmVjZWl2ZVNOU0Z1bmN0aW9uOmxhbWJkYS5JRnVuY3Rpb247XG4gIHB1YmxpYyBhc3luY0R1cmF0aW9uTWV0cmljPzpjbG91ZHdhdGNoLklNZXRyaWM7XG4gIHB1YmxpYyBhc3luY051bWJlclBhZ2VzTWV0cmljPzpjbG91ZHdhdGNoLklNZXRyaWM7XG4gIHB1YmxpYyBhc3luY0pvYkZpbnNoZWRNZXRyaWM/OmNsb3Vkd2F0Y2guSU1ldHJpYztcbiAgcHVibGljIGFzeW5jSm9iU3RhcnRlZE1ldHJpYz86Y2xvdWR3YXRjaC5JTWV0cmljO1xuICBwdWJsaWMgYXN5bmNOdW1iZXJQYWdlc1NlbmRNZXRyaWM/OmNsb3Vkd2F0Y2guSU1ldHJpYztcblxuXG4gIGNvbnN0cnVjdG9yKHNjb3BlIDogQ29uc3RydWN0LCBpZCA6IHN0cmluZywgcHJpdmF0ZSByZWFkb25seSBwcm9wcyA6IFRleHRyYWN0R2VuZXJpY0FzeW5jU2ZuVGFza1Byb3BzKSB7XG4gICAgLyoqcmVzb3VyY2VzXG4gICAgICogRHluYW1vREIgdGFibGVcbiAgICAgKiB0ZXh0cmFjdEFzeW5jU05TUm9sZVxuICAgICAqL1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuXG4gICAgdGhpcy5pbnRlZ3JhdGlvblBhdHRlcm4gPSBwcm9wcy5pbnRlZ3JhdGlvblBhdHRlcm4gfHwgc2ZuLkludGVncmF0aW9uUGF0dGVybi5SRVFVRVNUX1JFU1BPTlNFO1xuICAgIHZhbGlkYXRlUGF0dGVyblN1cHBvcnRlZCh0aGlzLmludGVncmF0aW9uUGF0dGVybiwgVGV4dHJhY3RHZW5lcmljQXN5bmNTZm5UYXNrLlNVUFBPUlRFRF9JTlRFR1JBVElPTl9QQVRURVJOUyk7XG5cbiAgICBpZiAodGhpcy5pbnRlZ3JhdGlvblBhdHRlcm4gPT09IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uV0FJVF9GT1JfVEFTS19UT0tFTiAmJiAhc2ZuLkZpZWxkVXRpbHMuY29udGFpbnNUYXNrVG9rZW4ocHJvcHMuaW5wdXQpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Rhc2sgVG9rZW4gaXMgcmVxdWlyZWQgaW4gYGlucHV0YCBmb3IgY2FsbGJhY2suIFVzZSBKc29uUGF0aC50YXNrVG9rZW4gdG8gc2V0IHRoZSB0b2tlbi4nKTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5wcm9wcy5hc3NvY2lhdGVXaXRoUGFyZW50ICYmIHByb3BzLmlucHV0ICYmIHByb3BzLmlucHV0LnR5cGUgIT09IHNmbi5JbnB1dFR5cGUuT0JKRUNUKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0NvdWxkIG5vdCBlbmFibGUgYGFzc29jaWF0ZVdpdGhQYXJlbnRgIGJlY2F1c2UgYGlucHV0YCBpcyB0YWtlbiBkaXJlY3RseSBmcm9tIGEgSlNPTiBwYXRoLiBVc2UgYHNmbi5UYXNrSW5wdXQuZnJvbU9iamVjdGAgaW5zdGVhZC4nKTtcbiAgICB9XG5cbiAgICB2YXIgdGV4dHJhY3RTdGF0ZU1hY2hpbmVUaW1lb3V0TWludXRlcyA9IHByb3BzLnRleHRyYWN0U3RhdGVNYWNoaW5lVGltZW91dE1pbnV0ZXMgPT09IHVuZGVmaW5lZCA/IDI4ODAgOiBwcm9wcy50ZXh0cmFjdFN0YXRlTWFjaGluZVRpbWVvdXRNaW51dGVzO1xuICAgIHZhciBsYW1iZGFMb2dMZXZlbCA9IHByb3BzLmxhbWJkYUxvZ0xldmVsID09PSB1bmRlZmluZWQgPyAnREVCVUcnIDogcHJvcHMubGFtYmRhTG9nTGV2ZWw7XG4gICAgdmFyIHRleHRyYWN0QVBJID0gcHJvcHMudGV4dHJhY3RBUEkgPT09IHVuZGVmaW5lZCA/ICdHRU5FUklDJyA6IHByb3BzLnRleHRyYWN0QVBJO1xuICAgIHZhciB0ZXh0cmFjdEFzeW5jQ2FsbE1heFJldHJpZXMgPSBwcm9wcy50ZXh0cmFjdEFzeW5jQ2FsbE1heFJldHJpZXMgPT09IHVuZGVmaW5lZCA/IDEwMCA6IHByb3BzLnRleHRyYWN0QXN5bmNDYWxsTWF4UmV0cmllcztcbiAgICB2YXIgdGV4dHJhY3RBc3luY0NhbGxCYWNrb2ZmUmF0ZSA9IHByb3BzLnRleHRyYWN0QXN5bmNDYWxsQmFja29mZlJhdGUgPT09IHVuZGVmaW5lZCA/IDEuMSA6IHByb3BzLnRleHRyYWN0QXN5bmNDYWxsQmFja29mZlJhdGU7XG4gICAgdmFyIHRleHRyYWN0QXN5bmNDYWxsSW50ZXJ2YWwgPSBwcm9wcy50ZXh0cmFjdEFzeW5jQ2FsbEludGVydmFsID09PSB1bmRlZmluZWQgPyAxIDogcHJvcHMudGV4dHJhY3RBc3luY0NhbGxJbnRlcnZhbDtcbiAgICB2YXIgczNUZW1wT3V0cHV0UHJlZml4ID1cbiAgICAgIHByb3BzLnMzVGVtcE91dHB1dFByZWZpeCA9PT0gdW5kZWZpbmVkID8gJycgOiBwcm9wcy5zM1RlbXBPdXRwdXRQcmVmaXg7XG4gICAgdmFyIHMzSW5wdXRQcmVmaXggPVxuICAgICAgcHJvcHMuczNJbnB1dFByZWZpeCA9PT0gdW5kZWZpbmVkID8gJycgOiBwcm9wcy5zM0lucHV0UHJlZml4O1xuICAgIHZhciBlbmFibGVDbG91ZFdhdGNoTWV0cmljc0FuZERhc2hib2FyZCA9IHByb3BzLmVuYWJsZUNsb3VkV2F0Y2hNZXRyaWNzQW5kRGFzaGJvYXJkID09PSB1bmRlZmluZWQgPyBmYWxzZSA6XG4gICAgICBwcm9wcy5lbmFibGVDbG91ZFdhdGNoTWV0cmljc0FuZERhc2hib2FyZDtcblxuICAgIC8qKiBSRVNPVVJDRSBEWU5BTU9EQiBUQUJMRSBmb3IgVEFTSyBUT0tFTiAqL1xuICAgIGlmIChwcm9wcy50YXNrVG9rZW5UYWJsZSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aGlzLnRhc2tUb2tlblRhYmxlID0gbmV3IGR5bmFtb2RiLlRhYmxlKHRoaXMsICdUZXh0cmFjdFRhc2tUb2tlblRhYmxlJywge1xuICAgICAgICBwYXJ0aXRpb25LZXk6IHtcbiAgICAgICAgICBuYW1lOiAnSUQnLFxuICAgICAgICAgIHR5cGU6IGR5bmFtb2RiLkF0dHJpYnV0ZVR5cGUuU1RSSU5HLFxuICAgICAgICB9LFxuICAgICAgICBiaWxsaW5nTW9kZTogZHluYW1vZGIuQmlsbGluZ01vZGUuUEFZX1BFUl9SRVFVRVNULFxuICAgICAgICByZW1vdmFsUG9saWN5OiBSZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgICAgIHRpbWVUb0xpdmVBdHRyaWJ1dGU6ICd0dGx0aW1lc3RhbXAnLFxuICAgICAgfSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMudGFza1Rva2VuVGFibGUgPSBwcm9wcy50YXNrVG9rZW5UYWJsZTtcbiAgICB9XG4gICAgdGhpcy50YXNrVG9rZW5UYWJsZU5hbWUgPSB0aGlzLnRhc2tUb2tlblRhYmxlLnRhYmxlTmFtZTtcblxuICAgIC8qKiBSRVNPVVJDRTogU05TIFJvbGUgZm9yIFRleHRyYWN0IHRvIHVzZSovXG4gICAgaWYgKHByb3BzLnNuc1JvbGVUZXh0cmFjdCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aGlzLnRleHRyYWN0QXN5bmNTTlNSb2xlID0gbmV3IGlhbS5Sb2xlKHRoaXMsICdUZXh0cmFjdEFzeW5jU05TUm9sZScsIHtcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ3RleHRyYWN0LmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgbWFuYWdlZFBvbGljaWVzOiBbTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0FtYXpvblNRU0Z1bGxBY2Nlc3MnKSxcbiAgICAgICAgICBNYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnQW1hem9uU05TRnVsbEFjY2VzcycpLFxuICAgICAgICAgIE1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdBbWF6b25TM1JlYWRPbmx5QWNjZXNzJyksXG4gICAgICAgICAgTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ0FtYXpvblRleHRyYWN0RnVsbEFjY2VzcycpXSxcbiAgICAgIH0pO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLnRleHRyYWN0QXN5bmNTTlNSb2xlID0gcHJvcHMuc25zUm9sZVRleHRyYWN0O1xuXG4gICAgfVxuXG4gICAgdGhpcy50ZXh0cmFjdEFzeW5jU05TID0gbmV3IHNucy5Ub3BpYyh0aGlzLCAnVGV4dHJhY3RBc3luY1NOUycpO1xuICAgIHRoaXMudGV4dHJhY3RBc3luY0NhbGxGdW5jdGlvbiA9IG5ldyBsYW1iZGEuRG9ja2VySW1hZ2VGdW5jdGlvbih0aGlzLCAnVGV4dHJhY3RBc3luY0NhbGwnLCB7XG4gICAgICBjb2RlOiBsYW1iZGEuRG9ja2VySW1hZ2VDb2RlLmZyb21JbWFnZUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9sYW1iZGEvdGV4dHJhY3RfYXN5bmMvJykpLFxuICAgICAgbWVtb3J5U2l6ZTogMTI4LFxuICAgICAgYXJjaGl0ZWN0dXJlOiBsYW1iZGEuQXJjaGl0ZWN0dXJlLlg4Nl82NCxcbiAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgIE5PVElGSUNBVElPTl9TTlM6IHRoaXMudGV4dHJhY3RBc3luY1NOUy50b3BpY0FybixcbiAgICAgICAgTk9USUZJQ0FUSU9OX1JPTEVfQVJOOiB0aGlzLnRleHRyYWN0QXN5bmNTTlNSb2xlLnJvbGVBcm4sXG4gICAgICAgIFRPS0VOX1NUT1JFX0REQjogdGhpcy50YXNrVG9rZW5UYWJsZU5hbWUsXG4gICAgICAgIFMzX09VVFBVVF9CVUNLRVQ6IHByb3BzLnMzT3V0cHV0QnVja2V0LFxuICAgICAgICBTM19URU1QX09VVFBVVF9QUkVGSVg6IHByb3BzLnMzVGVtcE91dHB1dFByZWZpeCxcbiAgICAgICAgTE9HX0xFVkVMOiBsYW1iZGFMb2dMZXZlbCxcbiAgICAgICAgVEVYVFJBQ1RfQVBJOiB0ZXh0cmFjdEFQSSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgY29uc3QgdGV4dHJhY3RBc3luY0NhbGxUYXNrID0gbmV3IHRhc2tzLkxhbWJkYUludm9rZSh0aGlzLCAnVGV4dHJhY3RBc3luY0NhbGxUYXNrJywgeyBsYW1iZGFGdW5jdGlvbjogdGhpcy50ZXh0cmFjdEFzeW5jQ2FsbEZ1bmN0aW9uIH0pO1xuICAgIHRleHRyYWN0QXN5bmNDYWxsVGFzay5hZGRSZXRyeSh7XG4gICAgICBtYXhBdHRlbXB0czogdGV4dHJhY3RBc3luY0NhbGxNYXhSZXRyaWVzLFxuICAgICAgYmFja29mZlJhdGU6IHRleHRyYWN0QXN5bmNDYWxsQmFja29mZlJhdGUsXG4gICAgICBpbnRlcnZhbDogRHVyYXRpb24uc2Vjb25kcyh0ZXh0cmFjdEFzeW5jQ2FsbEludGVydmFsKSxcbiAgICAgIGVycm9yczogWydUaHJvdHRsaW5nRXhjZXB0aW9uJywgJ0xpbWl0RXhjZWVkZWRFeGNlcHRpb24nLCAnSW50ZXJuYWxTZXJ2ZXJFcnJvcicsICdQcm92aXNpb25lZFRocm91Z2hwdXRFeGNlZWRlZEV4Y2VwdGlvbiddLFxuICAgIH0pO1xuXG5cbiAgICB0aGlzLnRleHRyYWN0QXN5bmNDYWxsRnVuY3Rpb24uYWRkVG9Sb2xlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ3RleHRyYWN0OlN0YXJ0KicsXG4gICAgICAgICd0ZXh0cmFjdDpHZXQqJyxcbiAgICAgIF0sXG4gICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgIH0pKTtcbiAgICAvKiogIyMjIyMjIyMjIyMjIyMjIyBJTlBVVCBCVUNLRVQgUE9MSUNJRVMgKi9cbiAgICBpZiAocHJvcHMuaW5wdXRQb2xpY3lTdGF0ZW1lbnRzID09PSB1bmRlZmluZWQpIHtcbiAgICAgIGlmIChwcm9wcy5zM0lucHV0QnVja2V0ID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgdGhpcy50ZXh0cmFjdEFzeW5jQ2FsbEZ1bmN0aW9uLmFkZFRvUm9sZVBvbGljeShcbiAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICBhY3Rpb25zOiBbJ3MzOkdldE9iamVjdCcsICdzMzpMaXN0QnVja2V0J10sXG4gICAgICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICAgIH0pLFxuICAgICAgICApO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgdGhpcy50ZXh0cmFjdEFzeW5jQ2FsbEZ1bmN0aW9uLmFkZFRvUm9sZVBvbGljeShcbiAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICBhY3Rpb25zOiBbJ3MzOkdldE9iamVjdCddLFxuICAgICAgICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICAgICAgIHBhdGguam9pbihgYXJuOmF3czpzMzo6OiR7cHJvcHMuczNJbnB1dEJ1Y2tldH1gLCAnLyonKSxcbiAgICAgICAgICAgICAgcGF0aC5qb2luKGBhcm46YXdzOnMzOjo6JHtwcm9wcy5zM0lucHV0QnVja2V0fWAsIHMzSW5wdXRQcmVmaXgsICcvKicpLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgKTtcbiAgICAgICAgdGhpcy50ZXh0cmFjdEFzeW5jQ2FsbEZ1bmN0aW9uLmFkZFRvUm9sZVBvbGljeShcbiAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICBhY3Rpb25zOiBbJ3MzOkxpc3RCdWNrZXQnXSxcbiAgICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgICBwYXRoLmpvaW4oYGFybjphd3M6czM6Ojoke3Byb3BzLnMzSW5wdXRCdWNrZXR9YCksXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0pLFxuICAgICAgICApO1xuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICBmb3IgKHZhciBwb2xpY3lTdGF0ZW1lbnQgb2YgcHJvcHMuaW5wdXRQb2xpY3lTdGF0ZW1lbnRzKSB7XG4gICAgICAgIHRoaXMudGV4dHJhY3RBc3luY0NhbGxGdW5jdGlvbi5hZGRUb1JvbGVQb2xpY3kocG9saWN5U3RhdGVtZW50KTtcbiAgICAgIH1cbiAgICB9XG4gICAgLyoqICMjIyMjIyMjIyMjIyMjIyMjIyMjIyBPVVRQVVQgQlVDS0VUIFBPTElDSUVTICovXG4gICAgaWYgKHByb3BzLm91dHB1dFBvbGljeVN0YXRlbWVudHMgPT09IHVuZGVmaW5lZCkge1xuICAgICAgaWYgKHByb3BzLnMzT3V0cHV0QnVja2V0ID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgdGhpcy50ZXh0cmFjdEFzeW5jQ2FsbEZ1bmN0aW9uLmFkZFRvUm9sZVBvbGljeShcbiAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICBhY3Rpb25zOiBbJ3MzOlB1dE9iamVjdCddLFxuICAgICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgKTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHRoaXMudGV4dHJhY3RBc3luY0NhbGxGdW5jdGlvbi5hZGRUb1JvbGVQb2xpY3koXG4gICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgYWN0aW9uczogWydzMzpQdXRPYmplY3QnXSxcbiAgICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgICBwYXRoLmpvaW4oYGFybjphd3M6czM6Ojoke3Byb3BzLnMzT3V0cHV0QnVja2V0fWAsIHMzVGVtcE91dHB1dFByZWZpeCwgJy8nKSxcbiAgICAgICAgICAgICAgcGF0aC5qb2luKGBhcm46YXdzOnMzOjo6JHtwcm9wcy5zM091dHB1dEJ1Y2tldH1gLCBzM1RlbXBPdXRwdXRQcmVmaXgsICcvKicpLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICB9IGVsc2Uge1xuICAgICAgZm9yICh2YXIgcG9saWN5U3RhdGVtZW50IG9mIHByb3BzLm91dHB1dFBvbGljeVN0YXRlbWVudHMpIHtcbiAgICAgICAgdGhpcy50ZXh0cmFjdEFzeW5jQ2FsbEZ1bmN0aW9uLmFkZFRvUm9sZVBvbGljeShwb2xpY3lTdGF0ZW1lbnQpO1xuICAgICAgfVxuICAgIH1cblxuICAgIC8vIHRoaXMudGV4dHJhY3RBc3luY0NhbGxGdW5jdGlvbi5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoeyBhY3Rpb25zOiBbJ3NuczoqJ10sIHJlc291cmNlczogW3RoaXMudGV4dHJhY3RBc3luY1NOUy50b3BpY0Fybl0gfSkpO1xuICAgIHRoaXMudGV4dHJhY3RBc3luY0NhbGxGdW5jdGlvbi5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoeyBhY3Rpb25zOiBbJ2R5bmFtb2RiOlB1dEl0ZW0nLCAnZHluYW1vZGI6R2V0SXRlbSddLCByZXNvdXJjZXM6IFt0aGlzLnRhc2tUb2tlblRhYmxlLnRhYmxlQXJuXSB9KSk7XG4gICAgdGhpcy5zdGFydFRleHRyYWN0TGFtYmRhTG9nR3JvdXA9KDxsYW1iZGEuRnVuY3Rpb24+IHRoaXMudGV4dHJhY3RBc3luY0NhbGxGdW5jdGlvbikubG9nR3JvdXA7XG5cbiAgICB0aGlzLnRleHRyYWN0QXN5bmNSZWNlaXZlU05TRnVuY3Rpb24gPSBuZXcgbGFtYmRhLkRvY2tlckltYWdlRnVuY3Rpb24odGhpcywgJ1RleHRyYWN0QXN5bmNTTlNGdW5jdGlvbicsIHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Eb2NrZXJJbWFnZUNvZGUuZnJvbUltYWdlQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJy4uL2xhbWJkYS90ZXh0cmFjdF9hc3luY19zbnNfbGlzdGVuZXIvJykpLFxuICAgICAgbWVtb3J5U2l6ZTogMTI4LFxuICAgICAgYXJjaGl0ZWN0dXJlOiBsYW1iZGEuQXJjaGl0ZWN0dXJlLlg4Nl82NCxcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLnNlY29uZHMoOTAwKSxcbiAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgIFRPS0VOX1NUT1JFX0REQjogdGhpcy50YXNrVG9rZW5UYWJsZU5hbWUsXG4gICAgICAgIFMzX09VVFBVVF9CVUNLRVQ6IHByb3BzLnMzT3V0cHV0QnVja2V0LFxuICAgICAgICBTM19URU1QX09VVFBVVF9QUkVGSVg6IHByb3BzLnMzVGVtcE91dHB1dFByZWZpeCxcbiAgICAgICAgTE9HX0xFVkVMOiBsYW1iZGFMb2dMZXZlbCxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgdGhpcy50ZXh0cmFjdEFzeW5jU05TLmFkZFN1YnNjcmlwdGlvbihuZXcgTGFtYmRhU3Vic2NyaXB0aW9uKHRoaXMudGV4dHJhY3RBc3luY1JlY2VpdmVTTlNGdW5jdGlvbikpO1xuICAgIHRoaXMudGV4dHJhY3RBc3luY1JlY2VpdmVTTlNGdW5jdGlvbi5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoeyBhY3Rpb25zOiBbJ2R5bmFtb2RiOkdldEl0ZW0nXSwgcmVzb3VyY2VzOiBbdGhpcy50YXNrVG9rZW5UYWJsZS50YWJsZUFybl0gfSkpO1xuICAgIHRoaXMucmVjZWl2ZVN0YXJ0U05TTGFtYmRhTG9nR3JvdXA9KDxsYW1iZGEuRnVuY3Rpb24+IHRoaXMudGV4dHJhY3RBc3luY1JlY2VpdmVTTlNGdW5jdGlvbikubG9nR3JvdXA7XG5cbiAgICBjb25zdCB3b3JrZmxvd19jaGFpbiA9IHNmbi5DaGFpbi5zdGFydCh0ZXh0cmFjdEFzeW5jQ2FsbFRhc2spO1xuXG4gICAgdGhpcy5zdGF0ZU1hY2hpbmUgPSBuZXcgc2ZuLlN0YXRlTWFjaGluZSh0aGlzLCAnU3RhdGVNYWNoaW5lJywge1xuICAgICAgZGVmaW5pdGlvbjogd29ya2Zsb3dfY2hhaW4sXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5ob3Vycyh0ZXh0cmFjdFN0YXRlTWFjaGluZVRpbWVvdXRNaW51dGVzKSxcbiAgICAgIHRyYWNpbmdFbmFibGVkOiB0cnVlLFxuICAgIH0pO1xuXG4gICAgdGhpcy50ZXh0cmFjdEFzeW5jUmVjZWl2ZVNOU0Z1bmN0aW9uLmFkZFRvUm9sZVBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdzdGF0ZXM6U2VuZFRhc2tTdWNjZXNzJywgJ3N0YXRlczpTZW5kVGFza0ZhaWx1cmUnLFxuICAgICAgXSxcbiAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgfSkpO1xuICAgIHRoaXMudGV4dHJhY3RBc3luY0NhbGxGdW5jdGlvbi5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgYWN0aW9uczogW1xuICAgICAgICAnc3RhdGVzOlNlbmRUYXNrRmFpbHVyZScsXG4gICAgICBdLFxuICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICB9KSk7XG4gICAgLy8gPT09PT09PT09XG4gICAgLy8gREFTSEJPQVJEXG4gICAgLy8gPT09PT09PT09XG4gICAgaWYgKGVuYWJsZUNsb3VkV2F0Y2hNZXRyaWNzQW5kRGFzaGJvYXJkKSB7XG4gICAgICBjb25zdCBhcHBOYW1lID0gdGhpcy5ub2RlLnRyeUdldENvbnRleHQoJ2FwcE5hbWUnKTtcblxuICAgICAgY29uc3QgY3VzdG9tTWV0cmljTmFtZXNwYWNlID0gJ1RleHRyYWN0Q29uc3RydWN0R2VuZXJpY0FzeW5jJztcblxuICAgICAgLy8gT1BFUkFUSU9OQUxcbiAgICAgIGNvbnN0IGFzeW5jRHVyYXRpb25NZXRyaWNGaWx0ZXIgPSBuZXcgTWV0cmljRmlsdGVyKHRoaXMsIGAke2FwcE5hbWV9LUR1cmF0aW9uRmlsdGVyYCwge1xuICAgICAgICBsb2dHcm91cDogKDxsYW1iZGEuRnVuY3Rpb24+IHRoaXMudGV4dHJhY3RBc3luY1JlY2VpdmVTTlNGdW5jdGlvbikubG9nR3JvdXAsXG4gICAgICAgIG1ldHJpY05hbWVzcGFjZTogY3VzdG9tTWV0cmljTmFtZXNwYWNlLFxuICAgICAgICBtZXRyaWNOYW1lOiAnRHVyYXRpb24nLFxuICAgICAgICBmaWx0ZXJQYXR0ZXJuOiBGaWx0ZXJQYXR0ZXJuLnNwYWNlRGVsaW1pdGVkKCdJTkZPJywgJ3RpbWVzdGFtcCcsICdpZCcsICdtZXNzYWdlJywgJ2R1cmF0aW9uTXMnKVxuICAgICAgICAgIC53aGVyZVN0cmluZygnbWVzc2FnZScsICc9JywgYHRleHRyYWN0X2FzeW5jXyR7dGV4dHJhY3RBUEl9X2pvYl9kdXJhdGlvbl9pbl9tczpgKSxcbiAgICAgICAgbWV0cmljVmFsdWU6ICckZHVyYXRpb25NcycsXG4gICAgICB9KTtcbiAgICAgIHRoaXMuYXN5bmNEdXJhdGlvbk1ldHJpYyA9IGFzeW5jRHVyYXRpb25NZXRyaWNGaWx0ZXIubWV0cmljKHsgc3RhdGlzdGljOiAnYXZnJyB9KTtcblxuICAgICAgY29uc3QgYXN5bmNOdW1iZXJQYWdlc01ldHJpY0ZpbHRlciA9IG5ldyBNZXRyaWNGaWx0ZXIodGhpcywgYCR7YXBwTmFtZX0tTnVtYmVyUGFnZXNGaWx0ZXJgLCB7XG4gICAgICAgIGxvZ0dyb3VwOiAoPGxhbWJkYS5GdW5jdGlvbj4gdGhpcy50ZXh0cmFjdEFzeW5jUmVjZWl2ZVNOU0Z1bmN0aW9uKS5sb2dHcm91cCxcbiAgICAgICAgbWV0cmljTmFtZXNwYWNlOiBjdXN0b21NZXRyaWNOYW1lc3BhY2UsXG4gICAgICAgIG1ldHJpY05hbWU6ICdOdW1iZXJQYWdlcycsXG4gICAgICAgIGZpbHRlclBhdHRlcm46IEZpbHRlclBhdHRlcm4uc3BhY2VEZWxpbWl0ZWQoJ0lORk8nLCAndGltZXN0YW1wJywgJ2lkJywgJ21lc3NhZ2UnLCAncGFnZXMnKVxuICAgICAgICAgIC53aGVyZVN0cmluZygnbWVzc2FnZScsICc9JywgYHRleHRyYWN0X2FzeW5jXyR7dGV4dHJhY3RBUEl9X251bWJlcl9vZl9wYWdlc19wcm9jZXNzZWQ6YCksXG4gICAgICAgIG1ldHJpY1ZhbHVlOiAnJHBhZ2VzJyxcbiAgICAgIH0pO1xuICAgICAgdGhpcy5hc3luY051bWJlclBhZ2VzTWV0cmljID0gYXN5bmNOdW1iZXJQYWdlc01ldHJpY0ZpbHRlci5tZXRyaWMoeyBzdGF0aXN0aWM6ICdzdW0nIH0pO1xuXG4gICAgICAvLyBUT0RPOiBleHBvc2UgYWxsIGZpbHRlcnMgYXMgcHJvcGVydGllcyBvZiBDb25zdHJ1Y3QsIHNvIHVzZXJzIGNhbiBidWlsZCB0aGVpciBvd24gbWV0cmljc1xuICAgICAgY29uc3QgYXN5bmNKb2JGaW5zaGVkTWV0cmljRmlsdGVyID0gbmV3IE1ldHJpY0ZpbHRlcih0aGlzLCBgJHthcHBOYW1lfS1Kb2JzRmluaXNoZWRGaWx0ZXJgLCB7XG4gICAgICAgIGxvZ0dyb3VwOiAoPGxhbWJkYS5GdW5jdGlvbj4gdGhpcy50ZXh0cmFjdEFzeW5jUmVjZWl2ZVNOU0Z1bmN0aW9uKS5sb2dHcm91cCxcbiAgICAgICAgbWV0cmljTmFtZXNwYWNlOiBjdXN0b21NZXRyaWNOYW1lc3BhY2UsXG4gICAgICAgIG1ldHJpY05hbWU6ICdKb2JzRmluaXNoZWQnLFxuICAgICAgICBmaWx0ZXJQYXR0ZXJuOiBGaWx0ZXJQYXR0ZXJuLnNwYWNlRGVsaW1pdGVkKCdJTkZPJywgJ3RpbWVzdGFtcCcsICdpZCcsICdtZXNzYWdlJywgJ3BhZ2VzJylcbiAgICAgICAgICAud2hlcmVTdHJpbmcoJ21lc3NhZ2UnLCAnPScsIGB0ZXh0cmFjdF9hc3luY18ke3RleHRyYWN0QVBJfV9udW1iZXJfb2ZfcGFnZXNfcHJvY2Vzc2VkOmApLFxuICAgICAgICBtZXRyaWNWYWx1ZTogJzEnLFxuICAgICAgfSk7XG4gICAgICB0aGlzLmFzeW5jSm9iRmluc2hlZE1ldHJpYyA9IGFzeW5jSm9iRmluc2hlZE1ldHJpY0ZpbHRlci5tZXRyaWMoeyBzdGF0aXN0aWM6ICdzdW0nIH0pO1xuXG4gICAgICBjb25zdCBhc3luY0pvYlN0YXJ0ZWRNZXRyaWNGaWx0ZXIgPSBuZXcgTWV0cmljRmlsdGVyKHRoaXMsIGAke2FwcE5hbWV9LUpvYnNTdGFydGVkRmlsdGVyYCwge1xuICAgICAgICBsb2dHcm91cDogKDxsYW1iZGEuRnVuY3Rpb24+IHRoaXMudGV4dHJhY3RBc3luY0NhbGxGdW5jdGlvbikubG9nR3JvdXAsXG4gICAgICAgIG1ldHJpY05hbWVzcGFjZTogY3VzdG9tTWV0cmljTmFtZXNwYWNlLFxuICAgICAgICBtZXRyaWNOYW1lOiAnSm9ic1N0YXJ0ZWQnLFxuICAgICAgICBmaWx0ZXJQYXR0ZXJuOiBGaWx0ZXJQYXR0ZXJuLnNwYWNlRGVsaW1pdGVkKCdJTkZPJywgJ3RpbWVzdGFtcCcsICdpZCcsICdtZXNzYWdlJylcbiAgICAgICAgICAud2hlcmVTdHJpbmcoJ21lc3NhZ2UnLCAnPScsIGB0ZXh0cmFjdF9hc3luY18ke3RleHRyYWN0QVBJfV9qb2Jfc3RhcnRlZGApLFxuICAgICAgICBtZXRyaWNWYWx1ZTogJzEnLFxuICAgICAgfSk7XG4gICAgICB0aGlzLmFzeW5jSm9iU3RhcnRlZE1ldHJpYyA9IGFzeW5jSm9iU3RhcnRlZE1ldHJpY0ZpbHRlci5tZXRyaWMoeyBzdGF0aXN0aWM6ICdzdW0nIH0pO1xuXG4gICAgICBjb25zdCBhc3luY051bWJlclBhZ2VzU2VuZE1ldHJpY0ZpbHRlciA9IG5ldyBNZXRyaWNGaWx0ZXIodGhpcywgYCR7YXBwTmFtZX0tTnVtYmVyUGFnZXNTZW5kRmlsdGVyYCwge1xuICAgICAgICBsb2dHcm91cDogKDxsYW1iZGEuRnVuY3Rpb24+IHRoaXMudGV4dHJhY3RBc3luY0NhbGxGdW5jdGlvbikubG9nR3JvdXAsXG4gICAgICAgIG1ldHJpY05hbWVzcGFjZTogY3VzdG9tTWV0cmljTmFtZXNwYWNlLFxuICAgICAgICBtZXRyaWNOYW1lOiAnTnVtYmVyUGFnZXMnLFxuICAgICAgICBmaWx0ZXJQYXR0ZXJuOiBGaWx0ZXJQYXR0ZXJuLnNwYWNlRGVsaW1pdGVkKCdJTkZPJywgJ3RpbWVzdGFtcCcsICdpZCcsICdtZXNzYWdlJywgJ3BhZ2VzJylcbiAgICAgICAgICAud2hlcmVTdHJpbmcoJ21lc3NhZ2UnLCAnPScsIGB0ZXh0cmFjdF9hc3luY18ke3RleHRyYWN0QVBJfV9udW1iZXJfb2ZfcGFnZXNfc2VuZF90b19wcm9jZXNzOmApLFxuICAgICAgICBtZXRyaWNWYWx1ZTogJyRwYWdlcycsXG4gICAgICB9KTtcbiAgICAgIHRoaXMuYXN5bmNOdW1iZXJQYWdlc1NlbmRNZXRyaWMgPSBhc3luY051bWJlclBhZ2VzU2VuZE1ldHJpY0ZpbHRlci5tZXRyaWMoeyBzdGF0aXN0aWM6ICdzdW0nIH0pO1xuICAgICAgLy8gT1BFUkFUSU9OQUwgU1RPUFxuXG4gICAgICAvLyBDQUxDVUFURUQgT1BFUkFUSU9OQUwgTUVUUklDU1xuXG4gICAgICBjb25zdCBwYWdlc1BlclNlY29uZCA9IG5ldyBjbG91ZHdhdGNoLk1hdGhFeHByZXNzaW9uKHtcbiAgICAgICAgZXhwcmVzc2lvbjogJ3BhZ2VzIC8gKGR1cmF0aW9uIC8gMTAwMCknLFxuICAgICAgICB1c2luZ01ldHJpY3M6IHtcbiAgICAgICAgICBwYWdlczogdGhpcy5hc3luY051bWJlclBhZ2VzTWV0cmljLFxuICAgICAgICAgIGR1cmF0aW9uOiB0aGlzLmFzeW5jRHVyYXRpb25NZXRyaWMsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICAgIGNvbnN0IG9wZW5Kb2JzID0gbmV3IGNsb3Vkd2F0Y2guTWF0aEV4cHJlc3Npb24oe1xuICAgICAgICBleHByZXNzaW9uOiAnc3RhcnRlZEpvYnMgLSBmaW5pc2hlZEpvYnMnLFxuICAgICAgICB1c2luZ01ldHJpY3M6IHtcbiAgICAgICAgICBzdGFydGVkSm9iczogdGhpcy5hc3luY0pvYlN0YXJ0ZWRNZXRyaWMsXG4gICAgICAgICAgZmluaXNoZWRKb2JzOiB0aGlzLmFzeW5jSm9iRmluc2hlZE1ldHJpYyxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgICAgLy8gQ0FMQ1VBVEVEIE9QRVJBVElPTkFMIE1FVFJJQ1MgU1RPUFxuXG4gICAgICBjb25zdCBlcnJvckZpbHRlck1ldHJpYyA9IG5ldyBNZXRyaWNGaWx0ZXIodGhpcywgYCR7YXBwTmFtZX0tRXJyb3JGaWx0ZXJgLCB7XG4gICAgICAgIGxvZ0dyb3VwOiAoPGxhbWJkYS5GdW5jdGlvbj4gdGhpcy50ZXh0cmFjdEFzeW5jQ2FsbEZ1bmN0aW9uKS5sb2dHcm91cCxcbiAgICAgICAgbWV0cmljTmFtZXNwYWNlOiBjdXN0b21NZXRyaWNOYW1lc3BhY2UsXG4gICAgICAgIG1ldHJpY05hbWU6ICdFcnJvcnMnLFxuICAgICAgICBmaWx0ZXJQYXR0ZXJuOiBGaWx0ZXJQYXR0ZXJuLmFueVRlcm0oJ0VSUk9SJywgJ0Vycm9yJywgJ2Vycm9yJyksXG4gICAgICAgIG1ldHJpY1ZhbHVlOiAnMScsXG4gICAgICB9KTtcblxuICAgICAgY29uc3QgbGltaXRFeGNlZWRlZEV4Y2VwdGlvbkZpbHRlck1ldHJpYyA9IG5ldyBNZXRyaWNGaWx0ZXIodGhpcywgYCR7YXBwTmFtZX0tbGltaXRFeGNlZWRlZEV4Y2VwdGlvbkZpbHRlcmAsIHtcbiAgICAgICAgbG9nR3JvdXA6ICg8bGFtYmRhLkZ1bmN0aW9uPiB0aGlzLnRleHRyYWN0QXN5bmNDYWxsRnVuY3Rpb24pLmxvZ0dyb3VwLFxuICAgICAgICBtZXRyaWNOYW1lc3BhY2U6IGN1c3RvbU1ldHJpY05hbWVzcGFjZSxcbiAgICAgICAgbWV0cmljTmFtZTogJ0xpbWl0RXhjZWVkZWRFeGNlcHRpb24nLFxuICAgICAgICBmaWx0ZXJQYXR0ZXJuOiBGaWx0ZXJQYXR0ZXJuLmFueVRlcm0oJ3RleHRyYWN0LmV4Y2VwdGlvbnMuTGltaXRFeGNlZWRlZEV4Y2VwdGlvbicpLFxuICAgICAgICBtZXRyaWNWYWx1ZTogJzEnLFxuICAgICAgfSk7XG4gICAgICBjb25zdCB0aHJvdHRsaW5nRXhjZXB0aW9uID0gbmV3IE1ldHJpY0ZpbHRlcih0aGlzLCBgJHthcHBOYW1lfS10aHJvdHRsaW5nRXhjZXB0aW9uRmlsdGVyYCwge1xuICAgICAgICBsb2dHcm91cDogKDxsYW1iZGEuRnVuY3Rpb24+IHRoaXMudGV4dHJhY3RBc3luY0NhbGxGdW5jdGlvbikubG9nR3JvdXAsXG4gICAgICAgIG1ldHJpY05hbWVzcGFjZTogY3VzdG9tTWV0cmljTmFtZXNwYWNlLFxuICAgICAgICBtZXRyaWNOYW1lOiAnVGhyb3R0bGluZ0V4Y2VwdGlvbicsXG4gICAgICAgIGZpbHRlclBhdHRlcm46IEZpbHRlclBhdHRlcm4uYW55VGVybSgndGV4dHJhY3QuZXhjZXB0aW9ucy5UaHJvdHRsaW5nRXhjZXB0aW9uJyksXG4gICAgICAgIG1ldHJpY1ZhbHVlOiAnMScsXG4gICAgICB9KTtcblxuICAgICAgY29uc3QgcHJvdmlzaW9uZWRUaHJvdWdocHV0RXhjZWVkZWRFeGNlcHRpb24gPSBuZXcgTWV0cmljRmlsdGVyKHRoaXMsIGAke2FwcE5hbWV9LXByb3Zpc2lvbmVkVGhyb3VnaHB1dEZpbHRlcmAsIHtcbiAgICAgICAgbG9nR3JvdXA6ICg8bGFtYmRhLkZ1bmN0aW9uPiB0aGlzLnRleHRyYWN0QXN5bmNDYWxsRnVuY3Rpb24pLmxvZ0dyb3VwLFxuICAgICAgICBtZXRyaWNOYW1lc3BhY2U6IGN1c3RvbU1ldHJpY05hbWVzcGFjZSxcbiAgICAgICAgbWV0cmljTmFtZTogJ1Byb3Zpc2lvbmVkVGhyb3VnaHB1dEV4Y2VlZGVkRXhjZXB0aW9uJyxcbiAgICAgICAgZmlsdGVyUGF0dGVybjogRmlsdGVyUGF0dGVybi5hbnlUZXJtKCd0ZXh0cmFjdC5leGNlcHRpb25zLlByb3Zpc2lvbmVkVGhyb3VnaHB1dEV4Y2VlZGVkRXhjZXB0aW9uJyksXG4gICAgICAgIG1ldHJpY1ZhbHVlOiAnMScsXG4gICAgICB9KTtcbiAgICAgIGNvbnN0IGludGVybmFsU2VydmVyRXJyb3IgPSBuZXcgTWV0cmljRmlsdGVyKHRoaXMsIGAke2FwcE5hbWV9LWludGVybmFsU2VydmVyRXJyb3JGaWx0ZXJgLCB7XG4gICAgICAgIGxvZ0dyb3VwOiAoPGxhbWJkYS5GdW5jdGlvbj4gdGhpcy50ZXh0cmFjdEFzeW5jQ2FsbEZ1bmN0aW9uKS5sb2dHcm91cCxcbiAgICAgICAgbWV0cmljTmFtZXNwYWNlOiBjdXN0b21NZXRyaWNOYW1lc3BhY2UsXG4gICAgICAgIG1ldHJpY05hbWU6ICdJbnRlcm5hbFNlcnZlckVycm9yJyxcbiAgICAgICAgZmlsdGVyUGF0dGVybjogRmlsdGVyUGF0dGVybi5hbnlUZXJtKCd0ZXh0cmFjdC5leGNlcHRpb25zLkludGVybmFsU2VydmVyRXJyb3InKSxcbiAgICAgICAgbWV0cmljVmFsdWU6ICcxJyxcbiAgICAgIH0pO1xuXG4gICAgICBjb25zdCBlcnJvck1ldHJpYzpjbG91ZHdhdGNoLklNZXRyaWMgPSBlcnJvckZpbHRlck1ldHJpYy5tZXRyaWMoeyBzdGF0aXN0aWM6ICdzdW0nIH0pO1xuICAgICAgY29uc3QgbGltaXRFeGNlZWRlZE1ldHJpYzpjbG91ZHdhdGNoLklNZXRyaWMgPSBsaW1pdEV4Y2VlZGVkRXhjZXB0aW9uRmlsdGVyTWV0cmljLm1ldHJpYyh7IHN0YXRpc3RpYzogJ3N1bScgfSk7XG4gICAgICBjb25zdCB0aHJvdHRsaW5nTWV0cmljOmNsb3Vkd2F0Y2guSU1ldHJpYyA9IHRocm90dGxpbmdFeGNlcHRpb24ubWV0cmljKHsgc3RhdGlzdGljOiAnc3VtJyB9KTtcbiAgICAgIGNvbnN0IHByb3Zpc2lvbmVkVGhyb3VnaHB1dE1ldHJpYzpjbG91ZHdhdGNoLklNZXRyaWMgPSBwcm92aXNpb25lZFRocm91Z2hwdXRFeGNlZWRlZEV4Y2VwdGlvbi5tZXRyaWMoeyBzdGF0aXN0aWM6ICdzdW0nIH0pO1xuICAgICAgY29uc3QgaW50ZXJuYWxTZXJ2ZXJFcnJvck1ldHJpYzpjbG91ZHdhdGNoLklNZXRyaWMgPSBpbnRlcm5hbFNlcnZlckVycm9yLm1ldHJpYyh7IHN0YXRpc3RpYzogJ3N1bScgfSk7XG5cbiAgICAgIGNvbnN0IHRleHRyYWN0U3RhcnREb2N1bWVudFRleHRUaHJvdHRsZTpjbG91ZHdhdGNoLklNZXRyaWM9IG5ldyBjbG91ZHdhdGNoLk1ldHJpYyh7XG4gICAgICAgIG5hbWVzcGFjZTogJ0FXUy9UZXh0cmFjdCcsXG4gICAgICAgIG1ldHJpY05hbWU6ICdUaHJvdHRsZWRDb3VudCcsXG4gICAgICAgIGRpbWVuc2lvbnNNYXA6IHtcbiAgICAgICAgICBPcGVyYXRpb246ICdTdGFydERvY3VtZW50VGV4dERldGVjdGlvbicsXG4gICAgICAgIH0sXG4gICAgICAgIHN0YXRpc3RpYzogJ3N1bScsXG4gICAgICB9KTtcbiAgICAgIC8vIERBU0hCT0FSRCBMQU1CREFcblxuICAgICAgY29uc3QgZGFzaGJvYXJkV2lkdGg9MjQ7XG4gICAgICAvLyBjb25zdCB3aWRnZXRTdGFuZGFyZEhlaWdodD05O1xuXG4gICAgICAvLyBEQVNIQk9BUkRcbiAgICAgIG5ldyBjbG91ZHdhdGNoLkRhc2hib2FyZCh0aGlzLCAnJHthcHBOYW1lfS1UZXN0RGFzaGJvYXJkJywge1xuICAgICAgICBlbmQ6ICdlbmQnLFxuICAgICAgICBwZXJpb2RPdmVycmlkZTogY2xvdWR3YXRjaC5QZXJpb2RPdmVycmlkZS5BVVRPLFxuICAgICAgICBzdGFydDogJ3N0YXJ0JyxcbiAgICAgICAgd2lkZ2V0czogW1xuICAgICAgICAgIFtcbiAgICAgICAgICAgIG5ldyBjbG91ZHdhdGNoLkNvbHVtbihuZXcgY2xvdWR3YXRjaC5UZXh0V2lkZ2V0KHsgbWFya2Rvd246ICcjIE9wZXJhdGlvbmFsIERhdGEgUm93IHdpZGdldHMnLCB3aWR0aDogZGFzaGJvYXJkV2lkdGggfSkpLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgW1xuICAgICAgICAgICAgbmV3IGNsb3Vkd2F0Y2guQ29sdW1uKG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHsgdGl0bGU6ICdQYWdlc1BlclNlY29uZCcsIGxlZnQ6IFtwYWdlc1BlclNlY29uZF0sIHdpZHRoOiBNYXRoLmZsb29yKGRhc2hib2FyZFdpZHRoLzIpIH0pKSxcbiAgICAgICAgICAgIG5ldyBjbG91ZHdhdGNoLkNvbHVtbihuZXcgY2xvdWR3YXRjaC5HcmFwaFdpZGdldCh7IHRpdGxlOiAnT3BlbkpvYnMnLCBsZWZ0OiBbb3BlbkpvYnNdLCB3aWR0aDogTWF0aC5mbG9vcihkYXNoYm9hcmRXaWR0aC8yKSB9KSksXG4gICAgICAgICAgXSxcbiAgICAgICAgICBbXG4gICAgICAgICAgICBuZXcgY2xvdWR3YXRjaC5Db2x1bW4obmV3IGNsb3Vkd2F0Y2guR3JhcGhXaWRnZXQoeyB0aXRsZTogJ0R1cmF0aW9uJywgbGVmdDogW3RoaXMuYXN5bmNEdXJhdGlvbk1ldHJpY10sIHdpZHRoOiBNYXRoLmZsb29yKGRhc2hib2FyZFdpZHRoLzQpIH0pKSxcbiAgICAgICAgICAgIG5ldyBjbG91ZHdhdGNoLkNvbHVtbihuZXcgY2xvdWR3YXRjaC5HcmFwaFdpZGdldCh7IHRpdGxlOiAnTnVtYmVyUGFnZXMnLCBsZWZ0OiBbdGhpcy5hc3luY051bWJlclBhZ2VzTWV0cmljXSwgd2lkdGg6IE1hdGguZmxvb3IoZGFzaGJvYXJkV2lkdGgvNCkgfSkpLFxuICAgICAgICAgICAgbmV3IGNsb3Vkd2F0Y2guQ29sdW1uKG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHsgdGl0bGU6ICdOdW1iZXJQYWdlcycsIGxlZnQ6IFt0aGlzLmFzeW5jTnVtYmVyUGFnZXNTZW5kTWV0cmljXSwgd2lkdGg6IE1hdGguZmxvb3IoZGFzaGJvYXJkV2lkdGgvNCkgfSkpLFxuICAgICAgICAgICAgbmV3IGNsb3Vkd2F0Y2guQ29sdW1uKG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHsgdGl0bGU6ICdKb2JzU3RhcnRlZEFuZEZpbmlzaGVkJywgbGVmdDogW3RoaXMuYXN5bmNKb2JGaW5zaGVkTWV0cmljXSwgcmlnaHQ6IFt0aGlzLmFzeW5jSm9iU3RhcnRlZE1ldHJpY10sIHdpZHRoOiBNYXRoLmZsb29yKGRhc2hib2FyZFdpZHRoLzQpIH0pKSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIFtcbiAgICAgICAgICAgIG5ldyBjbG91ZHdhdGNoLkNvbHVtbihuZXcgY2xvdWR3YXRjaC5UZXh0V2lkZ2V0KHsgbWFya2Rvd246ICcjIEFzeW5jIFRleHRyYWN0IEV4Y2VwdGlvbnMgUm93Jywgd2lkdGg6IGRhc2hib2FyZFdpZHRoIH0pKSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIFtcbiAgICAgICAgICAgIG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHsgdGl0bGU6ICdFcnJvcnMnLCBsZWZ0OiBbZXJyb3JNZXRyaWNdLCB3aWR0aDogTWF0aC5mbG9vcihkYXNoYm9hcmRXaWR0aC81KSB9KSxcbiAgICAgICAgICAgIG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHsgdGl0bGU6ICdMaW1pdEV4Y2VlZGVkJywgbGVmdDogW2xpbWl0RXhjZWVkZWRNZXRyaWNdLCB3aWR0aDogTWF0aC5mbG9vcihkYXNoYm9hcmRXaWR0aC81KSB9KSxcbiAgICAgICAgICAgIG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHsgdGl0bGU6ICdUaHJvdHRsaW5nJywgbGVmdDogW3Rocm90dGxpbmdNZXRyaWNdLCB3aWR0aDogTWF0aC5mbG9vcihkYXNoYm9hcmRXaWR0aC81KSB9KSxcbiAgICAgICAgICAgIG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHsgdGl0bGU6ICdQcm92aXNpb25lZFRocm91Z3B1dCcsIGxlZnQ6IFtwcm92aXNpb25lZFRocm91Z2hwdXRNZXRyaWNdLCB3aWR0aDogTWF0aC5mbG9vcihkYXNoYm9hcmRXaWR0aC81KSB9KSxcbiAgICAgICAgICAgIG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHsgdGl0bGU6ICdJbnRlcm5hbFNlcnZlckVycm9yJywgbGVmdDogW2ludGVybmFsU2VydmVyRXJyb3JNZXRyaWNdLCB3aWR0aDogTWF0aC5mbG9vcihkYXNoYm9hcmRXaWR0aC81KSB9KSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIFtcbiAgICAgICAgICAgIG5ldyBjbG91ZHdhdGNoLlRleHRXaWRnZXQoeyBtYXJrZG93bjogJyMgVGV4dHJhY3QnLCB3aWR0aDogZGFzaGJvYXJkV2lkdGggfSksXG4gICAgICAgICAgXSxcbiAgICAgICAgICBbXG4gICAgICAgICAgICBuZXcgY2xvdWR3YXRjaC5HcmFwaFdpZGdldCh7IHRpdGxlOiAnVGV4dHJhY3QtU3RhcnREZXRlY3RUZXh0LVRocm90dGxlZENvdW50JywgbGVmdDogW3RleHRyYWN0U3RhcnREb2N1bWVudFRleHRUaHJvdHRsZV0gfSksXG4gICAgICAgICAgXSxcbiAgICAgICAgICBbXG4gICAgICAgICAgICBuZXcgY2xvdWR3YXRjaC5UZXh0V2lkZ2V0KHsgbWFya2Rvd246ICcjIHRleHRyYWN0QXN5bmNDYWxsRnVuY3Rpb24nLCB3aWR0aDogZGFzaGJvYXJkV2lkdGggfSksXG4gICAgICAgICAgXSxcbiAgICAgICAgICBbXG4gICAgICAgICAgICBuZXcgY2xvdWR3YXRjaC5HcmFwaFdpZGdldCh7IHRpdGxlOiAnQXN5bmMtRnVuY3Rpb24tRXJyb3JzJywgbGVmdDogW3RoaXMudGV4dHJhY3RBc3luY0NhbGxGdW5jdGlvbi5tZXRyaWNFcnJvcnMoKV0sIHdpZHRoOiBNYXRoLmZsb29yKGRhc2hib2FyZFdpZHRoLzMpIH0pLFxuICAgICAgICAgICAgbmV3IGNsb3Vkd2F0Y2guR3JhcGhXaWRnZXQoeyB0aXRsZTogJ0FzeW5jLUZ1bmN0aW9uLUludm9jYXRpb25zJywgbGVmdDogW3RoaXMudGV4dHJhY3RBc3luY0NhbGxGdW5jdGlvbi5tZXRyaWNJbnZvY2F0aW9ucygpXSwgd2lkdGg6IE1hdGguZmxvb3IoZGFzaGJvYXJkV2lkdGgvMykgfSksXG4gICAgICAgICAgICBuZXcgY2xvdWR3YXRjaC5HcmFwaFdpZGdldCh7IHRpdGxlOiAnQXN5bmMtRnVuY3Rpb24tVGhyb3R0bGVzJywgbGVmdDogW3RoaXMudGV4dHJhY3RBc3luY0NhbGxGdW5jdGlvbi5tZXRyaWNUaHJvdHRsZXMoKV0sIHdpZHRoOiBNYXRoLmZsb29yKGRhc2hib2FyZFdpZHRoLzMpIH0pLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgW1xuICAgICAgICAgICAgbmV3IGNsb3Vkd2F0Y2guVGV4dFdpZGdldCh7IG1hcmtkb3duOiAnIyB0ZXh0cmFjdEFzeW5jUmVjZWl2ZVNOU0Z1bmN0aW9uJywgd2lkdGg6IGRhc2hib2FyZFdpZHRoIH0pLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgW1xuICAgICAgICAgICAgbmV3IGNsb3Vkd2F0Y2guR3JhcGhXaWRnZXQoeyB0aXRsZTogJ1NOUy1GdW5jdGlvbi1FcnJvcnMnLCBsZWZ0OiBbdGhpcy50ZXh0cmFjdEFzeW5jUmVjZWl2ZVNOU0Z1bmN0aW9uLm1ldHJpY0Vycm9ycygpXSwgd2lkdGg6IE1hdGguZmxvb3IoZGFzaGJvYXJkV2lkdGgvMykgfSksXG4gICAgICAgICAgICBuZXcgY2xvdWR3YXRjaC5HcmFwaFdpZGdldCh7IHRpdGxlOiAnU05TLUZ1bmN0aW9uLUludm9jYXRpb25zJywgbGVmdDogW3RoaXMudGV4dHJhY3RBc3luY1JlY2VpdmVTTlNGdW5jdGlvbi5tZXRyaWNJbnZvY2F0aW9ucygpXSwgd2lkdGg6IE1hdGguZmxvb3IoZGFzaGJvYXJkV2lkdGgvMykgfSksXG4gICAgICAgICAgICBuZXcgY2xvdWR3YXRjaC5HcmFwaFdpZGdldCh7IHRpdGxlOiAnU05TLUZ1bmN0aW9uLVRocm90dGxlcycsIGxlZnQ6IFt0aGlzLnRleHRyYWN0QXN5bmNSZWNlaXZlU05TRnVuY3Rpb24ubWV0cmljVGhyb3R0bGVzKCldLCB3aWR0aDogTWF0aC5mbG9vcihkYXNoYm9hcmRXaWR0aC8zKSB9KSxcbiAgICAgICAgICBdLFxuICAgICAgICBdLFxuICAgICAgfSk7XG4gICAgfVxuICAgIC8vIEVORCBEQVNIQk9BUkRcbiAgICB0aGlzLnRhc2tQb2xpY2llcyA9IHRoaXMuY3JlYXRlU2NvcGVkQWNjZXNzUG9saWN5KCk7XG4gIH1cbiAgLyoqXG4gICAgICAgKiBAaW50ZXJuYWxcbiAgICAgICAqL1xuICBwcm90ZWN0ZWQgX3JlbmRlclRhc2soKTogYW55IHtcbiAgICAvLyBzdWZmaXggb2YgJzoyJyBpbmRpY2F0ZXMgdGhhdCB0aGUgb3V0cHV0IG9mIHRoZSBuZXN0ZWQgc3RhdGUgbWFjaGluZSBzaG91bGQgYmUgSlNPTlxuICAgIC8vIHN1ZmZpeCBpcyBvbmx5IGFwcGxpY2FibGUgd2hlbiB3YWl0aW5nIGZvciBhIG5lc3RlZCBzdGF0ZSBtYWNoaW5lIHRvIGNvbXBsZXRlIChSVU5fSk9CKVxuICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zdGVwLWZ1bmN0aW9ucy9sYXRlc3QvZGcvY29ubmVjdC1zdGVwZnVuY3Rpb25zLmh0bWxcbiAgICBjb25zdCBzdWZmaXggPSB0aGlzLmludGVncmF0aW9uUGF0dGVybiA9PT0gc2ZuLkludGVncmF0aW9uUGF0dGVybi5SVU5fSk9CID8gJzoyJyA6ICcnO1xuICAgIGxldCBpbnB1dDogYW55O1xuICAgIGlmICh0aGlzLnByb3BzLmFzc29jaWF0ZVdpdGhQYXJlbnQpIHtcbiAgICAgIGNvbnN0IGFzc29jaWF0ZVdpdGhQYXJlbnRFbnRyeSA9IHtcbiAgICAgICAgQVdTX1NURVBfRlVOQ1RJT05TX1NUQVJURURfQllfRVhFQ1VUSU9OX0lEOiBzZm4uSnNvblBhdGguc3RyaW5nQXQoJyQkLkV4ZWN1dGlvbi5JZCcpLFxuICAgICAgfTtcbiAgICAgIGlucHV0ID0gdGhpcy5wcm9wcy5pbnB1dCA/IHtcbiAgICAgICAgLi4udGhpcy5wcm9wcy5pbnB1dC52YWx1ZSxcbiAgICAgICAgLi4uIGFzc29jaWF0ZVdpdGhQYXJlbnRFbnRyeSxcbiAgICAgIH0gOiBhc3NvY2lhdGVXaXRoUGFyZW50RW50cnk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGlucHV0ID0gdGhpcy5wcm9wcy5pbnB1dCA/IHRoaXMucHJvcHMuaW5wdXQudmFsdWUgOiBzZm4uVGFza0lucHV0LmZyb21Kc29uUGF0aEF0KCckJykudmFsdWU7XG4gICAgfVxuXG5cbiAgICByZXR1cm4ge1xuICAgICAgUmVzb3VyY2U6IGAke1xuICAgICAgICBpbnRlZ3JhdGlvblJlc291cmNlQXJuKCdzdGF0ZXMnLCAnc3RhcnRFeGVjdXRpb24nLCB0aGlzLmludGVncmF0aW9uUGF0dGVybilcbiAgICAgIH0ke3N1ZmZpeH1gLFxuICAgICAgUGFyYW1ldGVyczogc2ZuLkZpZWxkVXRpbHMucmVuZGVyT2JqZWN0KFxuICAgICAgICB7XG4gICAgICAgICAgSW5wdXQ6IGlucHV0LFxuICAgICAgICAgIFN0YXRlTWFjaGluZUFybjogdGhpcy5zdGF0ZU1hY2hpbmUuc3RhdGVNYWNoaW5lQXJuLFxuICAgICAgICAgIE5hbWU6IHRoaXMucHJvcHMubmFtZSxcbiAgICAgICAgfSxcbiAgICAgICksXG4gICAgfTtcbiAgfVxuICAvKipcbiAgICAgICAqIEFzIFN0YXRlTWFjaGluZUFybiBpcyBleHRyYWN0ZWQgYXV0b21hdGljYWxseSBmcm9tIHRoZSBzdGF0ZSBtYWNoaW5lIG9iamVjdCBpbmNsdWRlZCBpbiB0aGUgY29uc3RydWN0b3IsXG4gICAgICAgKlxuICAgICAgICogdGhlIHNjb3BlZCBhY2Nlc3MgcG9saWN5IHNob3VsZCBiZSBnZW5lcmF0ZWQgYWNjb3JkaW5nbHkuXG4gICAgICAgKlxuICAgICAgICogVGhpcyBtZWFucyB0aGUgYWN0aW9uIG9mIFN0YXJ0RXhlY3V0aW9uIHNob3VsZCBiZSByZXN0cmljdGVkIG9uIHRoZSBnaXZlbiBzdGF0ZSBtYWNoaW5lLCBpbnN0ZWFkIG9mIGJlaW5nIGdyYW50ZWQgdG8gYWxsIHRoZSByZXNvdXJjZXMgKCopLlxuICAgICAgICovXG4gIHByaXZhdGUgY3JlYXRlU2NvcGVkQWNjZXNzUG9saWN5KCk6IGlhbS5Qb2xpY3lTdGF0ZW1lbnRbXSB7XG4gICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZih0aGlzKTtcblxuICAgIGNvbnN0IHBvbGljeVN0YXRlbWVudHMgPSBbXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudChcbiAgICAgICAge1xuICAgICAgICAgIGFjdGlvbnM6IFsnc3RhdGVzOlN0YXJ0RXhlY3V0aW9uJ10sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbdGhpcy5zdGF0ZU1hY2hpbmUuc3RhdGVNYWNoaW5lQXJuXSxcbiAgICAgICAgfSxcbiAgICAgICksXG4gICAgXTtcblxuICAgIC8vIFN0ZXAgRnVuY3Rpb25zIHVzZSBDbG91ZCBXYXRjaCBtYW5hZ2VkIHJ1bGVzIHRvIGRlYWwgd2l0aCBzeW5jaHJvbm91cyB0YXNrcy5cbiAgICBpZiAodGhpcy5pbnRlZ3JhdGlvblBhdHRlcm4gPT09IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uUlVOX0pPQikge1xuICAgICAgcG9saWN5U3RhdGVtZW50cy5wdXNoKG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICdzdGF0ZXM6RGVzY3JpYmVFeGVjdXRpb24nLCAnc3RhdGVzOlN0b3BFeGVjdXRpb24nLFxuICAgICAgICBdLFxuICAgICAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc3RlcC1mdW5jdGlvbnMvbGF0ZXN0L2RnL2NvbmNlcHQtY3JlYXRlLWlhbS1hZHZhbmNlZC5odG1sI2NvbmNlcHQtY3JlYXRlLWlhbS1hZHZhbmNlZC1leGVjdXRpb25cbiAgICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICAgc3RhY2suZm9ybWF0QXJuKFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBzZXJ2aWNlOiAnc3RhdGVzJyxcbiAgICAgICAgICAgICAgcmVzb3VyY2U6ICdleGVjdXRpb24nLFxuICAgICAgICAgICAgICBhcm5Gb3JtYXQ6IEFybkZvcm1hdC5DT0xPTl9SRVNPVVJDRV9OQU1FLFxuICAgICAgICAgICAgICByZXNvdXJjZU5hbWU6IGAke1xuICAgICAgICAgICAgICAgIHN0YWNrLnNwbGl0QXJuKHRoaXMuc3RhdGVNYWNoaW5lLnN0YXRlTWFjaGluZUFybiwgQXJuRm9ybWF0LkNPTE9OX1JFU09VUkNFX05BTUUpLnJlc291cmNlTmFtZVxuICAgICAgICAgICAgICB9KmAsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICksXG4gICAgICAgIF0sXG4gICAgICB9KSk7XG5cbiAgICAgIHBvbGljeVN0YXRlbWVudHMucHVzaChuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAnZXZlbnRzOlB1dFRhcmdldHMnLCAnZXZlbnRzOlB1dFJ1bGUnLCAnZXZlbnRzOkRlc2NyaWJlUnVsZScsXG4gICAgICAgIF0sXG4gICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgIHN0YWNrLmZvcm1hdEFybihcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgc2VydmljZTogJ2V2ZW50cycsXG4gICAgICAgICAgICAgIHJlc291cmNlOiAncnVsZScsXG4gICAgICAgICAgICAgIHJlc291cmNlTmFtZTogJ1N0ZXBGdW5jdGlvbnNHZXRFdmVudHNGb3JTdGVwRnVuY3Rpb25zRXhlY3V0aW9uUnVsZScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICksXG4gICAgICAgIF0sXG4gICAgICB9KSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHBvbGljeVN0YXRlbWVudHM7XG4gIH1cbn1cbiJdfQ==