"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alps = exports.FormatType = void 0;
;
/**
 * Format type to convert the ALPS spec into.
 *
 * @stability stable
 */
var FormatType;
(function (FormatType) {
    FormatType[FormatType["S"] = 0] = "S";
    FormatType[FormatType["SDL"] = 1] = "SDL";
    FormatType[FormatType["A"] = 2] = "A";
    FormatType[FormatType["ASYNC"] = 3] = "ASYNC";
    FormatType[FormatType["ASYNCAPI"] = 4] = "ASYNCAPI";
    FormatType[FormatType["O"] = 5] = "O";
    FormatType[FormatType["OAS"] = 6] = "OAS";
    FormatType[FormatType["OPEN"] = 7] = "OPEN";
    FormatType[FormatType["OPENAPI"] = 8] = "OPENAPI";
    FormatType[FormatType["OPENAPI_JSON"] = 9] = "OPENAPI_JSON";
    FormatType[FormatType["P"] = 10] = "P";
    FormatType[FormatType["PROTO"] = 11] = "PROTO";
    FormatType[FormatType["J"] = 12] = "J";
    FormatType[FormatType["JSON"] = 13] = "JSON";
    FormatType[FormatType["W"] = 14] = "W";
    FormatType[FormatType["WSDL"] = 15] = "WSDL";
    FormatType[FormatType["SOAP"] = 16] = "SOAP";
})(FormatType = exports.FormatType || (exports.FormatType = {}));
;
// cleanup regex
const rxHash = /#/g;
const rxQ = /\?\?/g;
/**
 * @stability stable
 */
class Alps {
    /**
     * Converts an ALPS spec JSON object into a specified API like openApi or graph ql schema.
     *
     * @param alpsDocument The ALPS document.
     * @param options Options for the convertion.
     * @returns the requested api as a string
     * @stability stable
     */
    static unified(alpsDocument, options = { formatType: FormatType.OPENAPI }) {
        let rtn = '';
        // process requested translation
        switch (options === null || options === void 0 ? void 0 : options.formatType) {
            case FormatType.S:
            case FormatType.SDL:
                rtn = toSDL(alpsDocument);
                break;
            case FormatType.A:
            case FormatType.ASYNC:
            case FormatType.ASYNCAPI:
                rtn = toAsync(alpsDocument);
                break;
            case FormatType.O:
            case FormatType.OAS:
            case FormatType.OPEN:
            case FormatType.OPENAPI:
                rtn = toOAS(alpsDocument);
                break;
            case FormatType.OPENAPI_JSON:
                rtn = toOAS(alpsDocument);
                // convert yaml string to json
                // eslint-disable-next-line @typescript-eslint/no-require-imports
                const yaml = require('js-yaml');
                rtn = JSON.stringify(yaml.safeLoad(rtn));
                break;
            case FormatType.P:
            case FormatType.PROTO:
                rtn = toProto(alpsDocument);
                break;
            case FormatType.J:
            case FormatType.JSON:
                rtn = toJSON(alpsDocument);
                break;
            case FormatType.W:
            case FormatType.WSDL:
            case FormatType.SOAP:
                rtn = toWSDL(alpsDocument);
                break;
            default:
                console.log(`ERROR: unknown format: ${options === null || options === void 0 ? void 0 : options.formatType}`);
        }
        //console.log(`rtn: ${rtn}`);
        return rtn;
    }
    /**
     * loads the ALPS document.
     *
     * @param path ALPS spec file path.
     * @stability stable
     */
    static loadYaml(path) {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const fs = require('fs');
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const yaml = require('js-yaml');
        try {
            let fileContents = fs.readFileSync(path, 'utf8');
            let data = yaml.safeLoad(fileContents);
            // console.log(JSON.stringify(data));
            return data;
        }
        catch (e) {
            console.log(e);
        }
    }
    /**
     * @stability stable
     */
    static spec(spec) {
        return Object.assign({}, spec);
    }
}
exports.Alps = Alps;
;
;
// *******************************************
// translators
// *******************************************
// ****************************************************
// to WSDL
// ****************************************************
function toWSDL(_doc) {
    let rtn = '';
    rtn += '<?xml version = \'1.0\' encoding = \'UTF-8\'?>\n';
    rtn += '<!-- generated by "unified" -->\n';
    rtn += `<!-- created: ${new Date()} -->\n`;
    rtn += '<!-- source: http://github.com/mamund/2020-11-unified -->\n';
    rtn += '\n';
    rtn += '<definitions>\n';
    rtn += '  <todo />\n';
    rtn += '</definitions>\n';
    return rtn;
}
// ****************************************************
// to ALPS JSON
// ****************************************************
function toJSON(doc) {
    var rtn = '';
    try {
        rtn = JSON.stringify(doc, null, 2);
    }
    catch (err) {
        console.log(`ERROR: ${err}`);
    }
    return rtn;
}
// ****************************************************
// to proto file
// passes https://protogen.marcgravell.com/ validator
// ****************************************************
function toProto(doc) {
    var rtn = '';
    var coll;
    // preamble
    rtn += 'syntax = "proto3";\n';
    rtn += `package ${doc.alps.ext.filter(metadata_title)[0].value.replace(/ /g, '_') || 'ALPS_API'};\n`;
    rtn += '\n';
    // signature
    rtn += '// *******************************************************************\n';
    rtn += '// generated by "unified"\n';
    rtn += `// date: ${new Date()}`;
    rtn += '\n';
    rtn += '// http://github.com/mamund/2020-11-unified\n';
    rtn += '// *******************************************************************\n';
    rtn += '\n';
    // params
    coll = doc.alps.descriptor.filter(semantic);
    coll.forEach(function (msg) {
        rtn += `message ${msg.id}Params {\n`;
        var c = 0;
        c++;
        rtn += `  string ${msg.id} = ${c};\n`;
        rtn += '}\n';
    });
    rtn += '\n';
    // objects
    coll = doc.alps.descriptor.filter(groups);
    coll.forEach(function (msg) {
        rtn += `message ${msg.id} {\n`;
        var c = 0;
        msg.descriptor.forEach(function (prop) {
            c++;
            rtn += `  string ${prop.href} = ${c};\n`;
        });
        rtn += '}\n';
        rtn += `message ${msg.id}Response {\n`;
        rtn += `  repeated ${msg.id} ${msg.id}Collection = 1;\n`;
        rtn += '}\n';
        rtn += `message ${msg.id}Empty {}\n`;
    });
    rtn += '\n';
    // procedures
    rtn += `service ${doc.alps.ext.filter(metadata_title)[0].value.replace(/ /g, '_') || 'ALPS_API'}_Service {\n`;
    coll = doc.alps.descriptor.filter(safe);
    coll.forEach(function (item) {
        rtn += `  rpc ${item.id}(`;
        if (item.descriptor) {
            rtn += item.descriptor[0].href;
        }
        else {
            rtn += `${item.rt}Empty`;
        }
        rtn += `) returns (${item.rt}Response) {};\n`;
    });
    coll = doc.alps.descriptor.filter(unsafe);
    coll.forEach(function (item) {
        rtn += `  rpc ${item.id}(`;
        if (item.descriptor) {
            rtn += item.descriptor[0].href;
        }
        rtn += `) returns (${item.rt}Response) {};\n`;
    });
    coll = doc.alps.descriptor.filter(idempotent);
    coll.forEach(function (item) {
        rtn += `  rpc ${item.id}(`;
        if (item.descriptor) {
            rtn += item.descriptor[0].href;
            if (item.descriptor[0].href === '#id') {
                rtn += 'Params';
            }
        }
        rtn += `) returns (${item.rt}Response) {};\n`;
    });
    rtn += '}\n';
    // clean up
    rtn = rtn.replace(rxHash, '');
    rtn = rtn.replace(rxQ, '#');
    return rtn;
}
// *******************************************
// to graphql sdl
// passes https://app.graphqleditor.com/
// *******************************************
function toSDL(doc) {
    var rtn = '';
    var coll;
    // signature
    rtn += '?? *******************************************************************\n';
    rtn += '?? generated by "unified"\n';
    rtn += `?? date: ${new Date()}`;
    rtn += '\n';
    rtn += '?? http://github.com/mamund/2020-11-unified\n';
    rtn += '?? *******************************************************************\n';
    rtn += '\n';
    // types
    coll = doc.alps.descriptor.filter(groups);
    coll.forEach(function (item) {
        rtn += `type ${item.id} {\n`;
        item.descriptor.forEach(function (prop) {
            rtn += `  ${prop.href}: String!\n`;
        });
        rtn += '}\n';
    });
    rtn += '\n';
    // query
    coll = doc.alps.descriptor.filter(safe);
    coll.forEach(function (item) {
        rtn += 'type Query {\n';
        rtn += `  ${item.id}: [${item.rt}]\n`;
        rtn += '}\n';
    });
    rtn += '\n';
    // mutations
    rtn += 'type Mutation {\n';
    coll = doc.alps.descriptor.filter(unsafe);
    coll.forEach(function (item) {
        rtn += `  ${item.id}(`;
        if (item.descriptor) {
            rtn += `${item.descriptor[0].href}: String!`;
        }
        rtn += `): ${item.rt}\n`;
    });
    coll = doc.alps.descriptor.filter(idempotent);
    coll.forEach(function (item) {
        rtn += `  ${item.id}(`;
        if (item.descriptor) {
            rtn += `${item.descriptor[0].href}: String!`;
        }
        rtn += `): ${item.rt}\n`;
    });
    rtn += '}\n';
    // final schema declaration
    rtn += '\n';
    rtn += 'schema {\n';
    rtn += '  query: Query,\n';
    rtn += '  mutation: Mutation\n';
    rtn += '}\n';
    rtn = rtn.replace(rxHash, '');
    rtn = rtn.replace(rxQ, '#');
    return rtn;
}
// ***************************************************
// to OpenAPI document
// passes https://apitools.dev/swagger-parser/online/
// ***************************************************
function toOAS(doc) {
    var rtn = '';
    var coll;
    // preamble
    rtn += 'openapi: 3.0.1\n';
    rtn += '\n';
    // signature
    rtn += '?? *******************************************************************\n';
    rtn += '?? generated by "unified" from\n';
    rtn += `?? date: ${new Date()}`;
    rtn += '\n';
    rtn += '?? http://github.com/mamund/2020-11-unified\n';
    rtn += '?? *******************************************************************\n';
    rtn += '\n';
    // info section
    rtn += 'info:\n';
    rtn += `  title: ${doc.alps.ext.filter(metadata_title)[0].value || 'ALPS API'}\n`;
    rtn += `  description: ${doc.alps.doc.value || 'Generated from ALPS file'}\n`;
    rtn += '  version: 1.0.0\n';
    rtn += '\n';
    if (doc.alps.ext.filter(metadata_root)) {
        rtn += 'servers:\n';
        rtn += `- url: '${doc.alps.ext.filter(metadata_root)[0].value}'\n`;
        rtn += '\n';
    }
    // paths
    rtn += 'paths:\n';
    // gets
    coll = doc.alps.descriptor.filter(safe);
    coll.forEach(function (item) {
        rtn += `  /${item.id}:\n`;
        rtn += '    get:\n';
        rtn += `      summary: '${item.text || item.id}'\n`;
        rtn += `      operationId: ${item.id}\n`;
        rtn += '      responses:\n';
        rtn += '        200:\n';
        rtn += `          description: ${item.id}\n`;
        rtn += '          content:\n';
        rtn += '            application/json:\n';
        rtn += '              schema:\n';
        rtn += '                type: array\n';
        rtn += '                items:\n';
        rtn += `                  $ref: '??/components/schemas/${item.rt || item.returns}'\n`;
    });
    // posts
    coll = doc.alps.descriptor.filter(unsafe);
    coll.forEach(function (item) {
        rtn += `  /${item.id}:\n`;
        rtn += '    post:\n';
        rtn += `      summary: '${item.text || item.id}'\n`;
        rtn += `      operationId: ${item.id}\n`;
        rtn += '      requestBody:\n';
        rtn += '        content:\n';
        rtn += '          application/json:\n';
        rtn += '            schema:\n';
        rtn += `              $ref: '??/components/schemas/${item.rt || item.returns}'\n`;
        rtn += '      responses:\n';
        rtn += '        200:\n';
        rtn += `          description: add ${item.id}\n`;
        rtn += '          content:\n';
        rtn += '            application/json:\n';
        rtn += '              schema:\n';
        rtn += '                type: array\n';
        rtn += '                items:\n';
        rtn += `                  $ref: '??/components/schemas/${item.rt || item.returns}'\n`;
    });
    // put
    coll = doc.alps.descriptor.filter(update);
    coll.forEach(function (item) {
        rtn += `  /${item.id}:\n`;
        rtn += '    put:\n';
        rtn += `      summary: '${item.text || item.id}'\n`;
        rtn += `      operationId: ${item.id}\n`;
        rtn += '      requestBody:\n';
        rtn += '        content:\n';
        rtn += '          application/json:\n';
        rtn += '            schema:\n';
        rtn += `              $ref: '??/components/schemas/${item.rt || item.returns}'\n`;
        rtn += '      responses:\n';
        rtn += '        200:\n';
        rtn += `          description: add ${item.id}\n`;
        rtn += '          content:\n';
        rtn += '            application/json:\n';
        rtn += '              schema:\n';
        rtn += '                type: array\n';
        rtn += '                items:\n';
        rtn += `                  $ref: '??/components/schemas/${item.rt || item.returns}'\n`;
    });
    // deletes
    coll = doc.alps.descriptor.filter(remove);
    coll.forEach(function (item) {
        rtn += `  /${item.id}/{id}:\n`;
        rtn += '    delete:\n';
        rtn += `      summary: '${item.text || item.id}'\n`;
        rtn += `      operationId: ${item.id}\n`;
        rtn += '      parameters:\n';
        item.descriptor.forEach(function (prop) {
            rtn += `        - name: ${prop.href}\n`;
            rtn += '          in: path\n';
            rtn += `          description: ${prop.href} of ${item.id}\n`;
            rtn += '          required: true\n';
            rtn += '          schema:\n';
            rtn += '            type: string\n';
        });
        rtn += '      responses:\n';
        rtn += '        204:\n';
        rtn += `          description: delete ${item.id}\n`;
    });
    rtn += '\n';
    // components
    rtn += 'components:\n';
    rtn += '  schemas:\n';
    coll = doc.alps.descriptor.filter(groups);
    coll.forEach(function (item) {
        rtn += `    ${item.id}:\n`;
        if (item.text) {
            rtn += `      description: ${item.text}\n`;
        }
        rtn += '      type: object\n';
        rtn += '      properties:\n';
        item.descriptor.forEach(function (prop) {
            rtn += `          ${prop.href}:\n`;
            rtn += '            type: string\n';
            rtn += `            example: ${rString(prop.href)}\n`;
        });
    });
    // clean up doc
    rtn = rtn.replace(rxHash, '');
    rtn = rtn.replace(rxQ, '#');
    return rtn;
}
// ****************************************************
// to AsyncAPI document (incomplete)
// ****************************************************
function toAsync(doc) {
    var rtn = '';
    // preamble
    rtn += 'async: 2.0.0\n';
    rtn += '\n';
    // signature
    rtn += '?? *******************************************************************\n';
    rtn += '?? generated by "unified" from\n';
    rtn += `?? date: ${new Date()}`;
    rtn += '\n';
    rtn += '?? http://github.com/mamund/2020-11-unified\n';
    rtn += '?? *******************************************************************\n';
    rtn += '\n';
    rtn += `id: '${doc.alps.id}'\n`;
    rtn += '\n';
    // info section
    rtn += 'info:\n';
    rtn += `  title: ${doc.alps.ext.filter(metadata_title)[0].value || 'ALPS API'}\n`;
    rtn += `  description: ${doc.alps.doc.value || 'Generated from ALPS file'}\n`;
    rtn += "  version: '1.0.0'\n";
    rtn += `  baseTopic: ${doc.alps.ext.filter(metadata_name)[0].value || ''}\n`;
    rtn += `  host: ${doc.alps.ext.filter(metadata_root)[0].value || 'http://localhost:8888/root'}\n`;
    rtn += '  schemes:\n';
    rtn += "    - 'amqp'\n";
    rtn += "    - 'mqtt'\n";
    rtn += '\n';
    rtn += '# topics:\n';
    rtn += '# **** TBD ****';
    // clean up doc
    rtn = rtn.replace(rxHash, '');
    rtn = rtn.replace(rxQ, '#');
    return rtn;
}
//*******************************************
// collection filters
//*******************************************
function semantic(doc) {
    return doc.type === 'semantic';
}
function groups(doc) {
    return doc.type === 'group';
}
function safe(doc) {
    return doc.type === 'safe';
}
function unsafe(doc) {
    return doc.type === 'unsafe';
}
function idempotent(doc) {
    return doc.type === 'idempotent';
}
function remove(doc) {
    return (doc.type === 'idempotent' && (doc.tags && doc.tags.indexOf('delete') != -1));
}
function update(doc) {
    return (doc.type === 'idempotent' && (doc.tags && doc.tags.indexOf('update') != -1));
}
function metadata_title(doc) {
    return (doc.type === 'metadata' && (doc.name && doc.name === ('title')));
}
function metadata_root(doc) {
    return (doc.type === 'metadata' && (doc.name && doc.name === ('root')));
}
function metadata_name(doc) {
    return (doc.type === 'metadata' && (doc.name && doc.name === ('name')));
}
function rString(id) {
    var rtn = '';
    if (id && id.indexOf('id') != -1) {
        rtn = Math.random().toString(9).substring(2, 4) + Math.random().toString(9).substring(2, 4);
    }
    else {
        rtn = Math.random().toString(36).substring(2, 15) + Math.random().toString(36).substring(2, 15);
    }
    return rtn;
}
//# sourceMappingURL=data:application/json;base64,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