#  Copyright 2020, Olivier 'reivilibre'.
#
#  This file is part of Scone.
#
#  Scone is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  Scone is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with Scone.  If not, see <https://www.gnu.org/licenses/>.

from scone.default.steps.systemd_steps import (
    cook_systemd_daemon_reload,
    cook_systemd_enable,
    cook_systemd_start,
    cook_systemd_stop,
)
from scone.head.head import Head
from scone.head.kitchen import Kitchen, Preparation
from scone.head.recipe import Recipe, RecipeContext
from scone.head.utils import check_type, check_type_opt


class SystemdUnit(Recipe):
    """
    System unit.
    TODO(performance): make it collapsible in a way so that it can daemon-reload
        only once in most situations.

    TODO(performance): deduplication.
    """

    _NAME = "systemd"

    def __init__(self, recipe_context: RecipeContext, args: dict, head):
        super().__init__(recipe_context, args, head)

        unit = check_type(args.get("unit"), str)
        self.unit_name = unit if "." in unit else unit + ".service"
        already_installed = check_type(args.get("already_installed", False), bool)
        self.at = check_type_opt(args.get("at", None), str)

        if not (already_installed or self.at):
            # already_installed is for when the unit already exists on the system
            # and is not created by scone.
            raise ValueError(
                "Must supply either already_installed = true or "
                f"at = /path/to/{self.unit_name}"
            )

        self.enabled = check_type_opt(args.get("enabled"), bool)
        self.restart_on = check_type_opt(args.get("restart_on"), list)
        self.started = check_type_opt(args.get("started"), bool)

    def prepare(self, preparation: Preparation, head: Head) -> None:
        super().prepare(preparation, head)
        if self.at:
            # TODO(potential future): preparation.provides("systemd-unit",
            #  self.unit_name)
            preparation.needs("file", self.at)

    async def cook(self, kitchen: Kitchen) -> None:
        if self.enabled is not None or self.started is not None:
            await cook_systemd_daemon_reload(kitchen)

        if self.enabled is not None:
            await cook_systemd_enable(kitchen, self.enabled, self.unit_name)

        if self.started is not None:
            if self.started:
                await cook_systemd_start(kitchen, self.unit_name)
            else:
                await cook_systemd_stop(kitchen, self.unit_name)
