"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.QualifiedFunctionBase = exports.FunctionBase = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const event_invoke_config_1 = require("./event-invoke-config");
const event_source_mapping_1 = require("./event-source-mapping");
const function_url_1 = require("./function-url");
const lambda_generated_1 = require("./lambda.generated");
const util_1 = require("./util");
class FunctionBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Flag to delay adding a warning message until current version is invoked.
         * @internal
         */
        this._warnIfCurrentVersionCalled = false;
        /**
         * Mapping of invocation principals to grants. Used to de-dupe `grantInvoke()` calls.
         * @internal
         */
        this._invocationGrants = {};
        /**
         * Mapping of fucntion URL invocation principals to grants. Used to de-dupe `grantInvokeUrl()` calls.
         * @internal
         */
        this._functionUrlInvocationGrants = {};
    }
    /**
     * A warning will be added to functions under the following conditions:
     * - permissions that include `lambda:InvokeFunction` are added to the unqualified function.
     * - function.currentVersion is invoked before or after the permission is created.
     *
     * This applies only to permissions on Lambda functions, not versions or aliases.
     * This function is overridden as a noOp for QualifiedFunctionBase.
     */
    considerWarningOnInvokeFunctionPermissions(scope, action) {
        const affectedPermissions = ['lambda:InvokeFunction', 'lambda:*', 'lambda:Invoke*'];
        if (affectedPermissions.includes(action)) {
            if (scope.node.tryFindChild('CurrentVersion')) {
                this.warnInvokeFunctionPermissions(scope);
            }
            else {
                this._warnIfCurrentVersionCalled = true;
            }
        }
    }
    warnInvokeFunctionPermissions(scope) {
        core_1.Annotations.of(scope).addWarning([
            "AWS Lambda has changed their authorization strategy, which may cause client invocations using the 'Qualifier' parameter of the lambda function to fail with Access Denied errors.",
            "If you are using a lambda Version or Alias, make sure to call 'grantInvoke' or 'addPermission' on the Version or Alias, not the underlying Function",
            'See: https://github.com/aws/aws-cdk/issues/19273',
        ].join('\n'));
    }
    /**
     * Adds a permission to the Lambda resource policy.
     * @param id The id for the permission construct
     * @param permission The permission to grant to this Lambda function. @see Permission for details.
     */
    addPermission(id, permission) {
        var _c, _d, _e, _f, _g;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_Permission(permission);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addPermission);
            }
            throw error;
        }
        if (!this.canCreatePermissions) {
            // FIXME: @deprecated(v2) - throw an error if calling `addPermission` on a resource that doesn't support it.
            return;
        }
        const principal = this.parsePermissionPrincipal(permission.principal);
        const { sourceAccount, sourceArn } = (_c = this.parseConditions(permission.principal)) !== null && _c !== void 0 ? _c : {};
        const action = (_d = permission.action) !== null && _d !== void 0 ? _d : 'lambda:InvokeFunction';
        const scope = (_e = permission.scope) !== null && _e !== void 0 ? _e : this;
        this.considerWarningOnInvokeFunctionPermissions(scope, action);
        new lambda_generated_1.CfnPermission(scope, id, {
            action,
            principal,
            functionName: this.functionArn,
            eventSourceToken: permission.eventSourceToken,
            sourceAccount: (_f = permission.sourceAccount) !== null && _f !== void 0 ? _f : sourceAccount,
            sourceArn: (_g = permission.sourceArn) !== null && _g !== void 0 ? _g : sourceArn,
            functionUrlAuthType: permission.functionUrlAuthType,
        });
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Access the Connections object
     *
     * Will fail if not a VPC-enabled Lambda Function
     */
    get connections() {
        if (!this._connections) {
            // eslint-disable-next-line max-len
            throw new Error('Only VPC-associated Lambda Functions have security groups to manage. Supply the "vpc" parameter when creating the Lambda, or "securityGroupId" when importing it.');
        }
        return this._connections;
    }
    get latestVersion() {
        if (!this._latestVersion) {
            this._latestVersion = new LatestVersion(this);
        }
        return this._latestVersion;
    }
    /**
     * Whether or not this Lambda function was bound to a VPC
     *
     * If this is is `false`, trying to access the `connections` object will fail.
     */
    get isBoundToVpc() {
        return !!this._connections;
    }
    addEventSourceMapping(id, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_EventSourceMappingOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addEventSourceMapping);
            }
            throw error;
        }
        return new event_source_mapping_1.EventSourceMapping(this, id, {
            target: this,
            ...options,
        });
    }
    /**
     * Grant the given identity permissions to invoke this Lambda
     */
    grantInvoke(grantee) {
        const identifier = `Invoke${grantee.grantPrincipal}`; // calls the .toString() of the principal
        // Memoize the result so subsequent grantInvoke() calls are idempotent
        let grant = this._invocationGrants[identifier];
        if (!grant) {
            grant = this.grant(grantee, identifier, 'lambda:InvokeFunction', this.resourceArnsForGrantInvoke);
            this._invocationGrants[identifier] = grant;
        }
        return grant;
    }
    /**
     * Grant the given identity permissions to invoke this Lambda Function URL
     */
    grantInvokeUrl(grantee) {
        const identifier = `InvokeFunctionUrl${grantee.grantPrincipal}`; // calls the .toString() of the principal
        // Memoize the result so subsequent grantInvoke() calls are idempotent
        let grant = this._functionUrlInvocationGrants[identifier];
        if (!grant) {
            grant = this.grant(grantee, identifier, 'lambda:InvokeFunctionUrl', [this.functionArn], {
                functionUrlAuthType: function_url_1.FunctionUrlAuthType.AWS_IAM,
            });
            this._functionUrlInvocationGrants[identifier] = grant;
        }
        return grant;
    }
    addEventSource(source) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_IEventSource(source);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addEventSource);
            }
            throw error;
        }
        source.bind(this);
    }
    configureAsyncInvoke(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_EventInvokeConfigOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.configureAsyncInvoke);
            }
            throw error;
        }
        if (this.node.tryFindChild('EventInvokeConfig') !== undefined) {
            throw new Error(`An EventInvokeConfig has already been configured for the function at ${this.node.path}`);
        }
        new event_invoke_config_1.EventInvokeConfig(this, 'EventInvokeConfig', {
            function: this,
            ...options,
        });
    }
    addFunctionUrl(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_FunctionUrlOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addFunctionUrl);
            }
            throw error;
        }
        return new function_url_1.FunctionUrl(this, 'FunctionUrl', {
            function: this,
            ...options,
        });
    }
    /**
     * Returns the construct tree node that corresponds to the lambda function.
     * For use internally for constructs, when the tree is set up in non-standard ways. Ex: SingletonFunction.
     * @internal
     */
    _functionNode() {
        return this.node;
    }
    /**
     * Given the function arn, check if the account id matches this account
     *
     * Function ARNs look like this:
     *
     *   arn:aws:lambda:region:account-id:function:function-name
     *
     * ..which means that in order to extract the `account-id` component from the ARN, we can
     * split the ARN using ":" and select the component in index 4.
     *
     * @returns true if account id of function matches the account specified on the stack, false otherwise.
     *
     * @internal
     */
    _isStackAccount() {
        if (core_1.Token.isUnresolved(this.stack.account) || core_1.Token.isUnresolved(this.functionArn)) {
            return false;
        }
        return this.stack.splitArn(this.functionArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).account === this.stack.account;
    }
    grant(grantee, identifier, action, resourceArns, permissionOverrides) {
        const grant = iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: [action],
            resourceArns,
            // Fake resource-like object on which to call addToResourcePolicy(), which actually
            // calls addPermission()
            resource: {
                addToResourcePolicy: (_statement) => {
                    // Couldn't add permissions to the principal, so add them locally.
                    this.addPermission(identifier, {
                        principal: grantee.grantPrincipal,
                        action: action,
                        ...permissionOverrides,
                    });
                    const permissionNode = this._functionNode().tryFindChild(identifier);
                    if (!permissionNode && !this._skipPermissions) {
                        throw new Error('Cannot modify permission to lambda function. Function is either imported or $LATEST version.\n'
                            + 'If the function is imported from the same account use `fromFunctionAttributes()` API with the `sameEnvironment` flag.\n'
                            + 'If the function is imported from a different account and already has the correct permissions use `fromFunctionAttributes()` API with the `skipPermissions` flag.');
                    }
                    return { statementAdded: true, policyDependable: permissionNode };
                },
                node: this.node,
                stack: this.stack,
                env: this.env,
                applyRemovalPolicy: this.applyRemovalPolicy,
            },
        });
        return grant;
    }
    /**
     * Translate IPrincipal to something we can pass to AWS::Lambda::Permissions
     *
     * Do some nasty things because `Permission` supports a subset of what the
     * full IAM principal language supports, and we may not be able to parse strings
     * outright because they may be tokens.
     *
     * Try to recognize some specific Principal classes first, then try a generic
     * fallback.
     */
    parsePermissionPrincipal(principal) {
        // Try some specific common classes first.
        // use duck-typing, not instance of
        // @deprecated: after v2, we can change these to 'instanceof'
        if ('wrapped' in principal) {
            // eslint-disable-next-line dot-notation
            principal = principal['wrapped'];
        }
        if ('accountId' in principal) {
            return principal.accountId;
        }
        if ('service' in principal) {
            return principal.service;
        }
        if ('arn' in principal) {
            return principal.arn;
        }
        // Try a best-effort approach to support simple principals that are not any of the predefined
        // classes, but are simple enough that they will fit into the Permission model. Main target
        // here: imported Roles, Users, Groups.
        //
        // The principal cannot have conditions and must have a single { AWS: [arn] } entry.
        const json = principal.policyFragment.principalJson;
        if (Object.keys(principal.policyFragment.conditions).length === 0 && json.AWS) {
            if (typeof json.AWS === 'string') {
                return json.AWS;
            }
            if (Array.isArray(json.AWS) && json.AWS.length === 1 && typeof json.AWS[0] === 'string') {
                return json.AWS[0];
            }
        }
        throw new Error(`Invalid principal type for Lambda permission statement: ${principal.constructor.name}. ` +
            'Supported: AccountPrincipal, ArnPrincipal, ServicePrincipal');
    }
    parseConditions(principal) {
        if (this.isPrincipalWithConditions(principal)) {
            const conditions = principal.policyFragment.conditions;
            const conditionPairs = util_1.flatMap(Object.entries(conditions), ([operator, conditionObjs]) => Object.keys(conditionObjs).map(key => { return { operator, key }; }));
            const supportedPrincipalConditions = [{ operator: 'ArnLike', key: 'aws:SourceArn' }, { operator: 'StringEquals', key: 'aws:SourceAccount' }];
            const unsupportedConditions = conditionPairs.filter((condition) => !supportedPrincipalConditions.some((supportedCondition) => supportedCondition.operator === condition.operator && supportedCondition.key === condition.key));
            if (unsupportedConditions.length == 0) {
                return {
                    sourceAccount: conditions.StringEquals['aws:SourceAccount'],
                    sourceArn: conditions.ArnLike['aws:SourceArn'],
                };
            }
            else {
                throw new Error(`PrincipalWithConditions had unsupported conditions for Lambda permission statement: ${JSON.stringify(unsupportedConditions)}. ` +
                    `Supported operator/condition pairs: ${JSON.stringify(supportedPrincipalConditions)}`);
            }
        }
        else {
            return null;
        }
    }
    isPrincipalWithConditions(principal) {
        return 'conditions' in principal;
    }
}
exports.FunctionBase = FunctionBase;
_a = JSII_RTTI_SYMBOL_1;
FunctionBase[_a] = { fqn: "@aws-cdk/aws-lambda.FunctionBase", version: "1.153.1" };
class QualifiedFunctionBase extends FunctionBase {
    constructor() {
        super(...arguments);
        this.permissionsNode = this.node;
    }
    get latestVersion() {
        return this.lambda.latestVersion;
    }
    get resourceArnsForGrantInvoke() {
        return [this.functionArn];
    }
    configureAsyncInvoke(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_EventInvokeConfigOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.configureAsyncInvoke);
            }
            throw error;
        }
        if (this.node.tryFindChild('EventInvokeConfig') !== undefined) {
            throw new Error(`An EventInvokeConfig has already been configured for the qualified function at ${this.node.path}`);
        }
        new event_invoke_config_1.EventInvokeConfig(this, 'EventInvokeConfig', {
            function: this.lambda,
            qualifier: this.qualifier,
            ...options,
        });
    }
    considerWarningOnInvokeFunctionPermissions(_scope, _action) {
        // noOp
        return;
    }
}
exports.QualifiedFunctionBase = QualifiedFunctionBase;
_b = JSII_RTTI_SYMBOL_1;
QualifiedFunctionBase[_b] = { fqn: "@aws-cdk/aws-lambda.QualifiedFunctionBase", version: "1.153.1" };
/**
 * The $LATEST version of a function, useful when attempting to create aliases.
 */
class LatestVersion extends FunctionBase {
    constructor(lambda) {
        super(lambda, '$LATEST');
        this.version = '$LATEST';
        this.permissionsNode = this.node;
        this.canCreatePermissions = false;
        this.lambda = lambda;
    }
    get functionArn() {
        return `${this.lambda.functionArn}:${this.version}`;
    }
    get functionName() {
        return `${this.lambda.functionName}:${this.version}`;
    }
    get architecture() {
        return this.lambda.architecture;
    }
    get grantPrincipal() {
        return this.lambda.grantPrincipal;
    }
    get latestVersion() {
        return this;
    }
    get role() {
        return this.lambda.role;
    }
    get edgeArn() {
        throw new Error('$LATEST function version cannot be used for Lambda@Edge');
    }
    get resourceArnsForGrantInvoke() {
        return [this.functionArn];
    }
    addAlias(aliasName, options = {}) {
        return util_1.addAlias(this, this, aliasName, options);
    }
}
//# sourceMappingURL=data:application/json;base64,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