"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alias = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const function_base_1 = require("./function-base");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const scalable_function_attribute_1 = require("./private/scalable-function-attribute");
/**
 * A new alias to a particular version of a Lambda function.
 */
class Alias extends function_base_1.QualifiedFunctionBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.aliasName,
        });
        this.canCreatePermissions = true;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_AliasProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.lambda = props.version.lambda;
        this.aliasName = this.physicalName;
        this.version = props.version;
        this.architecture = this.lambda.architecture;
        const alias = new lambda_generated_1.CfnAlias(this, 'Resource', {
            name: this.aliasName,
            description: props.description,
            functionName: this.version.lambda.functionName,
            functionVersion: props.version.version,
            routingConfig: this.determineRoutingConfig(props),
            provisionedConcurrencyConfig: this.determineProvisionedConcurrency(props),
        });
        // Use a Service Linked Role
        // https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-service-linked-roles.html
        this.scalingRole = iam.Role.fromRoleArn(this, 'ScalingRole', this.stack.formatArn({
            service: 'iam',
            region: '',
            resource: 'role/aws-service-role/lambda.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_LambdaConcurrency',
        }));
        this.functionArn = this.getResourceArnAttribute(alias.ref, {
            service: 'lambda',
            resource: 'function',
            resourceName: `${this.lambda.functionName}:${this.physicalName}`,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
        this.qualifier = lambda_version_1.extractQualifierFromArn(alias.ref);
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        // ARN parsing splits on `:`, so we can only get the function's name from the ARN as resourceName...
        // And we're parsing it out (instead of using the underlying function directly) in order to have use of it incur
        // an implicit dependency on the resource.
        this.functionName = `${this.stack.splitArn(this.functionArn, core_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}:${this.aliasName}`;
    }
    static fromAliasAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_AliasAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromAliasAttributes);
            }
            throw error;
        }
        class Imported extends function_base_1.QualifiedFunctionBase {
            constructor() {
                super(...arguments);
                this.aliasName = attrs.aliasName;
                this.version = attrs.aliasVersion;
                this.lambda = attrs.aliasVersion.lambda;
                this.functionArn = `${attrs.aliasVersion.lambda.functionArn}:${attrs.aliasName}`;
                this.functionName = `${attrs.aliasVersion.lambda.functionName}:${attrs.aliasName}`;
                this.grantPrincipal = attrs.aliasVersion.grantPrincipal;
                this.role = attrs.aliasVersion.role;
                this.architecture = attrs.aliasVersion.lambda.architecture;
                this.canCreatePermissions = this._isStackAccount();
                this.qualifier = attrs.aliasName;
            }
        }
        return new Imported(scope, id);
    }
    get grantPrincipal() {
        return this.version.grantPrincipal;
    }
    get role() {
        return this.version.role;
    }
    metric(metricName, props = {}) {
        // Metrics on Aliases need the "bare" function name, and the alias' ARN, this differs from the base behavior.
        return super.metric(metricName, {
            dimensionsMap: {
                FunctionName: this.lambda.functionName,
                // construct the name from the underlying lambda so that alarms on an alias
                // don't cause a circular dependency with CodeDeploy
                // see: https://github.com/aws/aws-cdk/issues/2231
                Resource: `${this.lambda.functionName}:${this.aliasName}`,
            },
            ...props,
        });
    }
    /**
     * Configure provisioned concurrency autoscaling on a function alias. Returns a scalable attribute that can call
     * `scaleOnUtilization()` and `scaleOnSchedule()`.
     *
     * @param options Autoscaling options
     */
    addAutoScaling(options) {
        var _b;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_AutoScalingOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addAutoScaling);
            }
            throw error;
        }
        if (this.scalableAlias) {
            throw new Error('AutoScaling already enabled for this alias');
        }
        return this.scalableAlias = new scalable_function_attribute_1.ScalableFunctionAttribute(this, 'AliasScaling', {
            minCapacity: (_b = options.minCapacity) !== null && _b !== void 0 ? _b : 1,
            maxCapacity: options.maxCapacity,
            resourceId: `function:${this.functionName}`,
            dimension: 'lambda:function:ProvisionedConcurrency',
            serviceNamespace: appscaling.ServiceNamespace.LAMBDA,
            role: this.scalingRole,
        });
    }
    /**
     * Calculate the routingConfig parameter from the input props
     */
    determineRoutingConfig(props) {
        if (!props.additionalVersions || props.additionalVersions.length === 0) {
            return undefined;
        }
        this.validateAdditionalWeights(props.additionalVersions);
        return {
            additionalVersionWeights: props.additionalVersions.map(vw => {
                return {
                    functionVersion: vw.version.version,
                    functionWeight: vw.weight,
                };
            }),
        };
    }
    /**
     * Validate that the additional version weights make sense
     *
     * We validate that they are positive and add up to something <= 1.
     */
    validateAdditionalWeights(weights) {
        const total = weights.map(w => {
            if (w.weight < 0 || w.weight > 1) {
                throw new Error(`Additional version weight must be between 0 and 1, got: ${w.weight}`);
            }
            return w.weight;
        }).reduce((a, x) => a + x);
        if (total > 1) {
            throw new Error(`Sum of additional version weights must not exceed 1, got: ${total}`);
        }
    }
    /**
     * Validate that the provisionedConcurrentExecutions makes sense
     *
     * Member must have value greater than or equal to 1
     */
    determineProvisionedConcurrency(props) {
        if (!props.provisionedConcurrentExecutions) {
            return undefined;
        }
        if (props.provisionedConcurrentExecutions <= 0) {
            throw new Error('provisionedConcurrentExecutions must have value greater than or equal to 1');
        }
        return { provisionedConcurrentExecutions: props.provisionedConcurrentExecutions };
    }
}
exports.Alias = Alias;
_a = JSII_RTTI_SYMBOL_1;
Alias[_a] = { fqn: "@aws-cdk/aws-lambda.Alias", version: "1.153.1" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWxpYXMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhbGlhcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSxrRUFBa0U7QUFFbEUsd0NBQXdDO0FBQ3hDLHdDQUEwQztBQUkxQyxtREFBbUU7QUFDbkUscURBQXFFO0FBQ3JFLHlEQUE4QztBQUM5Qyx1RkFBa0Y7QUE2RWxGOztHQUVHO0FBQ0gsTUFBYSxLQUFNLFNBQVEscUNBQXFCO0lBcUQ5QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWlCO1FBQ3pELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyxTQUFTO1NBQzlCLENBQUMsQ0FBQztRQVJjLHlCQUFvQixHQUFZLElBQUksQ0FBQzs7Ozs7Ozs7OztRQVV0RCxJQUFJLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDO1FBQ25DLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQztRQUNuQyxJQUFJLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDN0IsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLFlBQVksQ0FBQztRQUU3QyxNQUFNLEtBQUssR0FBRyxJQUFJLDJCQUFRLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUMzQyxJQUFJLEVBQUUsSUFBSSxDQUFDLFNBQVM7WUFDcEIsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQzlCLFlBQVksRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxZQUFZO1lBQzlDLGVBQWUsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU87WUFDdEMsYUFBYSxFQUFFLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxLQUFLLENBQUM7WUFDakQsNEJBQTRCLEVBQUUsSUFBSSxDQUFDLCtCQUErQixDQUFDLEtBQUssQ0FBQztTQUMxRSxDQUFDLENBQUM7UUFFSCw0QkFBNEI7UUFDNUIsbUhBQW1IO1FBQ25ILElBQUksQ0FBQyxXQUFXLEdBQUcsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQztZQUNoRixPQUFPLEVBQUUsS0FBSztZQUNkLE1BQU0sRUFBRSxFQUFFO1lBQ1YsUUFBUSxFQUFFLG9FQUFvRTtZQUM5RSxZQUFZLEVBQUUsMkRBQTJEO1NBQzFFLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLFdBQVcsR0FBRyxJQUFJLENBQUMsdUJBQXVCLENBQUMsS0FBSyxDQUFDLEdBQUcsRUFBRTtZQUN6RCxPQUFPLEVBQUUsUUFBUTtZQUNqQixRQUFRLEVBQUUsVUFBVTtZQUNwQixZQUFZLEVBQUUsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLFlBQVksSUFBSSxJQUFJLENBQUMsWUFBWSxFQUFFO1lBQ2hFLFNBQVMsRUFBRSxnQkFBUyxDQUFDLG1CQUFtQjtTQUN6QyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsU0FBUyxHQUFHLHdDQUF1QixDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUVwRCxJQUFJLEtBQUssQ0FBQyxTQUFTLElBQUksS0FBSyxDQUFDLFNBQVMsSUFBSSxLQUFLLENBQUMsV0FBVyxJQUFJLEtBQUssQ0FBQyxhQUFhLEtBQUssU0FBUyxFQUFFO1lBQ2hHLElBQUksQ0FBQyxvQkFBb0IsQ0FBQztnQkFDeEIsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO2dCQUMxQixTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVM7Z0JBQzFCLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztnQkFDOUIsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhO2FBQ25DLENBQUMsQ0FBQztTQUNKO1FBRUQsb0dBQW9HO1FBQ3BHLGdIQUFnSDtRQUNoSCwwQ0FBMEM7UUFDMUMsSUFBSSxDQUFDLFlBQVksR0FBRyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsZ0JBQVMsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLFlBQWEsSUFBSSxJQUFJLENBQUMsU0FBUyxFQUFFLENBQUM7S0FDL0g7SUF0R00sTUFBTSxDQUFDLG1CQUFtQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNCOzs7Ozs7Ozs7O1FBQ3BGLE1BQU0sUUFBUyxTQUFRLHFDQUFxQjtZQUE1Qzs7Z0JBQ2tCLGNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO2dCQUM1QixZQUFPLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztnQkFDN0IsV0FBTSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDO2dCQUNuQyxnQkFBVyxHQUFHLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsV0FBVyxJQUFJLEtBQUssQ0FBQyxTQUFTLEVBQUUsQ0FBQztnQkFDNUUsaUJBQVksR0FBRyxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLFlBQVksSUFBSSxLQUFLLENBQUMsU0FBUyxFQUFFLENBQUM7Z0JBQzlFLG1CQUFjLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQyxjQUFjLENBQUM7Z0JBQ25ELFNBQUksR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQztnQkFDL0IsaUJBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxZQUFZLENBQUM7Z0JBRW5ELHlCQUFvQixHQUFHLElBQUksQ0FBQyxlQUFlLEVBQUUsQ0FBQztnQkFDOUMsY0FBUyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUM7WUFDakQsQ0FBQztTQUFBO1FBQ0QsT0FBTyxJQUFJLFFBQVEsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7S0FDaEM7SUF5RkQsSUFBVyxjQUFjO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUM7S0FDcEM7SUFFRCxJQUFXLElBQUk7UUFDYixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDO0tBQzFCO0lBRU0sTUFBTSxDQUFDLFVBQWtCLEVBQUUsUUFBa0MsRUFBRTtRQUNwRSw2R0FBNkc7UUFDN0csT0FBTyxLQUFLLENBQUMsTUFBTSxDQUFDLFVBQVUsRUFBRTtZQUM5QixhQUFhLEVBQUU7Z0JBQ2IsWUFBWSxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsWUFBWTtnQkFDdEMsMkVBQTJFO2dCQUMzRSxvREFBb0Q7Z0JBQ3BELGtEQUFrRDtnQkFDbEQsUUFBUSxFQUFFLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxZQUFZLElBQUksSUFBSSxDQUFDLFNBQVMsRUFBRTthQUMxRDtZQUNELEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztLQUNKO0lBRUQ7Ozs7O09BS0c7SUFDSSxjQUFjLENBQUMsT0FBMkI7Ozs7Ozs7Ozs7O1FBQy9DLElBQUksSUFBSSxDQUFDLGFBQWEsRUFBRTtZQUN0QixNQUFNLElBQUksS0FBSyxDQUFDLDRDQUE0QyxDQUFDLENBQUM7U0FDL0Q7UUFDRCxPQUFPLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSx1REFBeUIsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQzlFLFdBQVcsUUFBRSxPQUFPLENBQUMsV0FBVyxtQ0FBSSxDQUFDO1lBQ3JDLFdBQVcsRUFBRSxPQUFPLENBQUMsV0FBVztZQUNoQyxVQUFVLEVBQUUsWUFBWSxJQUFJLENBQUMsWUFBWSxFQUFFO1lBQzNDLFNBQVMsRUFBRSx3Q0FBd0M7WUFDbkQsZ0JBQWdCLEVBQUUsVUFBVSxDQUFDLGdCQUFnQixDQUFDLE1BQU07WUFDcEQsSUFBSSxFQUFFLElBQUksQ0FBQyxXQUFXO1NBQ3ZCLENBQUMsQ0FBQztLQUNKO0lBRUQ7O09BRUc7SUFDSyxzQkFBc0IsQ0FBQyxLQUFpQjtRQUM5QyxJQUFJLENBQUMsS0FBSyxDQUFDLGtCQUFrQixJQUFJLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3RFLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBRUQsSUFBSSxDQUFDLHlCQUF5QixDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBRXpELE9BQU87WUFDTCx3QkFBd0IsRUFBRSxLQUFLLENBQUMsa0JBQWtCLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFO2dCQUMxRCxPQUFPO29CQUNMLGVBQWUsRUFBRSxFQUFFLENBQUMsT0FBTyxDQUFDLE9BQU87b0JBQ25DLGNBQWMsRUFBRSxFQUFFLENBQUMsTUFBTTtpQkFDMUIsQ0FBQztZQUNKLENBQUMsQ0FBQztTQUNILENBQUM7S0FDSDtJQUVEOzs7O09BSUc7SUFDSyx5QkFBeUIsQ0FBQyxPQUF3QjtRQUN4RCxNQUFNLEtBQUssR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQzVCLElBQUksQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7Z0JBQUUsTUFBTSxJQUFJLEtBQUssQ0FBQywyREFBMkQsQ0FBQyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7YUFBRTtZQUM3SCxPQUFPLENBQUMsQ0FBQyxNQUFNLENBQUM7UUFDbEIsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBRTNCLElBQUksS0FBSyxHQUFHLENBQUMsRUFBRTtZQUNiLE1BQU0sSUFBSSxLQUFLLENBQUMsNkRBQTZELEtBQUssRUFBRSxDQUFDLENBQUM7U0FDdkY7S0FDRjtJQUVEOzs7O09BSUc7SUFDSywrQkFBK0IsQ0FBQyxLQUFpQjtRQUN2RCxJQUFJLENBQUMsS0FBSyxDQUFDLCtCQUErQixFQUFFO1lBQzFDLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBRUQsSUFBSSxLQUFLLENBQUMsK0JBQStCLElBQUksQ0FBQyxFQUFFO1lBQzlDLE1BQU0sSUFBSSxLQUFLLENBQUMsNEVBQTRFLENBQUMsQ0FBQztTQUMvRjtRQUVELE9BQU8sRUFBRSwrQkFBK0IsRUFBRSxLQUFLLENBQUMsK0JBQStCLEVBQUUsQ0FBQztLQUNuRjs7QUF0TUgsc0JBdU1DIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgYXBwc2NhbGluZyBmcm9tICdAYXdzLWNkay9hd3MtYXBwbGljYXRpb25hdXRvc2NhbGluZyc7XG5pbXBvcnQgKiBhcyBjbG91ZHdhdGNoIGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZHdhdGNoJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCB7IEFybkZvcm1hdCB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBBcmNoaXRlY3R1cmUgfSBmcm9tICcuL2FyY2hpdGVjdHVyZSc7XG5pbXBvcnQgeyBFdmVudEludm9rZUNvbmZpZ09wdGlvbnMgfSBmcm9tICcuL2V2ZW50LWludm9rZS1jb25maWcnO1xuaW1wb3J0IHsgSUZ1bmN0aW9uLCBRdWFsaWZpZWRGdW5jdGlvbkJhc2UgfSBmcm9tICcuL2Z1bmN0aW9uLWJhc2UnO1xuaW1wb3J0IHsgZXh0cmFjdFF1YWxpZmllckZyb21Bcm4sIElWZXJzaW9uIH0gZnJvbSAnLi9sYW1iZGEtdmVyc2lvbic7XG5pbXBvcnQgeyBDZm5BbGlhcyB9IGZyb20gJy4vbGFtYmRhLmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBTY2FsYWJsZUZ1bmN0aW9uQXR0cmlidXRlIH0gZnJvbSAnLi9wcml2YXRlL3NjYWxhYmxlLWZ1bmN0aW9uLWF0dHJpYnV0ZSc7XG5pbXBvcnQgeyBBdXRvU2NhbGluZ09wdGlvbnMsIElTY2FsYWJsZUZ1bmN0aW9uQXR0cmlidXRlIH0gZnJvbSAnLi9zY2FsYWJsZS1hdHRyaWJ1dGUtYXBpJztcblxuZXhwb3J0IGludGVyZmFjZSBJQWxpYXMgZXh0ZW5kcyBJRnVuY3Rpb24ge1xuICAvKipcbiAgICogTmFtZSBvZiB0aGlzIGFsaWFzLlxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSBhbGlhc05hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHVuZGVybHlpbmcgTGFtYmRhIGZ1bmN0aW9uIHZlcnNpb24uXG4gICAqL1xuICByZWFkb25seSB2ZXJzaW9uOiBJVmVyc2lvbjtcbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBgbGFtYmRhLkFsaWFzYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBbGlhc09wdGlvbnMgZXh0ZW5kcyBFdmVudEludm9rZUNvbmZpZ09wdGlvbnMge1xuICAvKipcbiAgICogRGVzY3JpcHRpb24gZm9yIHRoZSBhbGlhc1xuICAgKlxuICAgKiBAZGVmYXVsdCBObyBkZXNjcmlwdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFkZGl0aW9uYWwgdmVyc2lvbnMgd2l0aCBpbmRpdmlkdWFsIHdlaWdodHMgdGhpcyBhbGlhcyBwb2ludHMgdG9cbiAgICpcbiAgICogSW5kaXZpZHVhbCBhZGRpdGlvbmFsIHZlcnNpb24gd2VpZ2h0cyBzcGVjaWZpZWQgaGVyZSBzaG91bGQgYWRkIHVwIHRvXG4gICAqIChsZXNzIHRoYW4pIG9uZS4gQWxsIHJlbWFpbmluZyB3ZWlnaHQgaXMgcm91dGVkIHRvIHRoZSBkZWZhdWx0XG4gICAqIHZlcnNpb24uXG4gICAqXG4gICAqIEZvciBleGFtcGxlLCB0aGUgY29uZmlnIGlzXG4gICAqXG4gICAqICAgIHZlcnNpb246IFwiMVwiXG4gICAqICAgIGFkZGl0aW9uYWxWZXJzaW9uczogW3sgdmVyc2lvbjogXCIyXCIsIHdlaWdodDogMC4wNSB9XVxuICAgKlxuICAgKiBUaGVuIDUlIG9mIHRyYWZmaWMgd2lsbCBiZSByb3V0ZWQgdG8gZnVuY3Rpb24gdmVyc2lvbiAyLCB3aGlsZVxuICAgKiB0aGUgcmVtYWluaW5nIDk1JSBvZiB0cmFmZmljIHdpbGwgYmUgcm91dGVkIHRvIGZ1bmN0aW9uIHZlcnNpb24gMS5cbiAgICpcbiAgICogQGRlZmF1bHQgTm8gYWRkaXRpb25hbCB2ZXJzaW9uc1xuICAgKi9cbiAgcmVhZG9ubHkgYWRkaXRpb25hbFZlcnNpb25zPzogVmVyc2lvbldlaWdodFtdO1xuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgYSBwcm92aXNpb25lZCBjb25jdXJyZW5jeSBjb25maWd1cmF0aW9uIGZvciBhIGZ1bmN0aW9uJ3MgYWxpYXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IE5vIHByb3Zpc2lvbmVkIGNvbmN1cnJlbmN5XG4gICAqL1xuICByZWFkb25seSBwcm92aXNpb25lZENvbmN1cnJlbnRFeGVjdXRpb25zPzogbnVtYmVyO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGEgbmV3IExhbWJkYSBhbGlhc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIEFsaWFzUHJvcHMgZXh0ZW5kcyBBbGlhc09wdGlvbnMge1xuICAvKipcbiAgICogTmFtZSBvZiB0aGlzIGFsaWFzXG4gICAqL1xuICByZWFkb25seSBhbGlhc05hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogRnVuY3Rpb24gdmVyc2lvbiB0aGlzIGFsaWFzIHJlZmVycyB0b1xuICAgKlxuICAgKiBVc2UgbGFtYmRhLmN1cnJlbnRWZXJzaW9uIHRvIHJlZmVyZW5jZSBhIHZlcnNpb24gd2l0aCB5b3VyIGxhdGVzdCBjaGFuZ2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgdmVyc2lvbjogSVZlcnNpb247XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgQWxpYXNBdHRyaWJ1dGVzIHtcbiAgcmVhZG9ubHkgYWxpYXNOYW1lOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGFsaWFzVmVyc2lvbjogSVZlcnNpb247XG59XG5cbi8qKlxuICogQSBuZXcgYWxpYXMgdG8gYSBwYXJ0aWN1bGFyIHZlcnNpb24gb2YgYSBMYW1iZGEgZnVuY3Rpb24uXG4gKi9cbmV4cG9ydCBjbGFzcyBBbGlhcyBleHRlbmRzIFF1YWxpZmllZEZ1bmN0aW9uQmFzZSBpbXBsZW1lbnRzIElBbGlhcyB7XG4gIHB1YmxpYyBzdGF0aWMgZnJvbUFsaWFzQXR0cmlidXRlcyhzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhdHRyczogQWxpYXNBdHRyaWJ1dGVzKTogSUFsaWFzIHtcbiAgICBjbGFzcyBJbXBvcnRlZCBleHRlbmRzIFF1YWxpZmllZEZ1bmN0aW9uQmFzZSBpbXBsZW1lbnRzIElBbGlhcyB7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgYWxpYXNOYW1lID0gYXR0cnMuYWxpYXNOYW1lO1xuICAgICAgcHVibGljIHJlYWRvbmx5IHZlcnNpb24gPSBhdHRycy5hbGlhc1ZlcnNpb247XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgbGFtYmRhID0gYXR0cnMuYWxpYXNWZXJzaW9uLmxhbWJkYTtcbiAgICAgIHB1YmxpYyByZWFkb25seSBmdW5jdGlvbkFybiA9IGAke2F0dHJzLmFsaWFzVmVyc2lvbi5sYW1iZGEuZnVuY3Rpb25Bcm59OiR7YXR0cnMuYWxpYXNOYW1lfWA7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZnVuY3Rpb25OYW1lID0gYCR7YXR0cnMuYWxpYXNWZXJzaW9uLmxhbWJkYS5mdW5jdGlvbk5hbWV9OiR7YXR0cnMuYWxpYXNOYW1lfWA7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZ3JhbnRQcmluY2lwYWwgPSBhdHRycy5hbGlhc1ZlcnNpb24uZ3JhbnRQcmluY2lwYWw7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgcm9sZSA9IGF0dHJzLmFsaWFzVmVyc2lvbi5yb2xlO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGFyY2hpdGVjdHVyZSA9IGF0dHJzLmFsaWFzVmVyc2lvbi5sYW1iZGEuYXJjaGl0ZWN0dXJlO1xuXG4gICAgICBwcm90ZWN0ZWQgcmVhZG9ubHkgY2FuQ3JlYXRlUGVybWlzc2lvbnMgPSB0aGlzLl9pc1N0YWNrQWNjb3VudCgpO1xuICAgICAgcHJvdGVjdGVkIHJlYWRvbmx5IHF1YWxpZmllciA9IGF0dHJzLmFsaWFzTmFtZTtcbiAgICB9XG4gICAgcmV0dXJuIG5ldyBJbXBvcnRlZChzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhpcyBhbGlhcy5cbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGFsaWFzTmFtZTogc3RyaW5nO1xuICAvKipcbiAgICogQVJOIG9mIHRoaXMgYWxpYXNcbiAgICpcbiAgICogVXNlZCB0byBiZSBhYmxlIHRvIHVzZSBBbGlhcyBpbiBwbGFjZSBvZiBhIHJlZ3VsYXIgTGFtYmRhLiBMYW1iZGEgYWNjZXB0c1xuICAgKiBBUk5zIGV2ZXJ5d2hlcmUgaXQgYWNjZXB0cyBmdW5jdGlvbiBuYW1lcy5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBmdW5jdGlvbk5hbWU6IHN0cmluZztcblxuICBwdWJsaWMgcmVhZG9ubHkgbGFtYmRhOiBJRnVuY3Rpb247XG5cbiAgcHVibGljIHJlYWRvbmx5IGFyY2hpdGVjdHVyZTogQXJjaGl0ZWN0dXJlO1xuXG4gIHB1YmxpYyByZWFkb25seSB2ZXJzaW9uOiBJVmVyc2lvbjtcblxuICAvKipcbiAgICogQVJOIG9mIHRoaXMgYWxpYXNcbiAgICpcbiAgICogVXNlZCB0byBiZSBhYmxlIHRvIHVzZSBBbGlhcyBpbiBwbGFjZSBvZiBhIHJlZ3VsYXIgTGFtYmRhLiBMYW1iZGEgYWNjZXB0c1xuICAgKiBBUk5zIGV2ZXJ5d2hlcmUgaXQgYWNjZXB0cyBmdW5jdGlvbiBuYW1lcy5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBmdW5jdGlvbkFybjogc3RyaW5nO1xuXG4gIHByb3RlY3RlZCByZWFkb25seSBxdWFsaWZpZXI6IHN0cmluZztcblxuICBwcm90ZWN0ZWQgcmVhZG9ubHkgY2FuQ3JlYXRlUGVybWlzc2lvbnM6IGJvb2xlYW4gPSB0cnVlO1xuXG4gIHByaXZhdGUgc2NhbGFibGVBbGlhcz86IFNjYWxhYmxlRnVuY3Rpb25BdHRyaWJ1dGU7XG4gIHByaXZhdGUgcmVhZG9ubHkgc2NhbGluZ1JvbGU6IGlhbS5JUm9sZTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQWxpYXNQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgcGh5c2ljYWxOYW1lOiBwcm9wcy5hbGlhc05hbWUsXG4gICAgfSk7XG5cbiAgICB0aGlzLmxhbWJkYSA9IHByb3BzLnZlcnNpb24ubGFtYmRhO1xuICAgIHRoaXMuYWxpYXNOYW1lID0gdGhpcy5waHlzaWNhbE5hbWU7XG4gICAgdGhpcy52ZXJzaW9uID0gcHJvcHMudmVyc2lvbjtcbiAgICB0aGlzLmFyY2hpdGVjdHVyZSA9IHRoaXMubGFtYmRhLmFyY2hpdGVjdHVyZTtcblxuICAgIGNvbnN0IGFsaWFzID0gbmV3IENmbkFsaWFzKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIG5hbWU6IHRoaXMuYWxpYXNOYW1lLFxuICAgICAgZGVzY3JpcHRpb246IHByb3BzLmRlc2NyaXB0aW9uLFxuICAgICAgZnVuY3Rpb25OYW1lOiB0aGlzLnZlcnNpb24ubGFtYmRhLmZ1bmN0aW9uTmFtZSxcbiAgICAgIGZ1bmN0aW9uVmVyc2lvbjogcHJvcHMudmVyc2lvbi52ZXJzaW9uLFxuICAgICAgcm91dGluZ0NvbmZpZzogdGhpcy5kZXRlcm1pbmVSb3V0aW5nQ29uZmlnKHByb3BzKSxcbiAgICAgIHByb3Zpc2lvbmVkQ29uY3VycmVuY3lDb25maWc6IHRoaXMuZGV0ZXJtaW5lUHJvdmlzaW9uZWRDb25jdXJyZW5jeShwcm9wcyksXG4gICAgfSk7XG5cbiAgICAvLyBVc2UgYSBTZXJ2aWNlIExpbmtlZCBSb2xlXG4gICAgLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F1dG9zY2FsaW5nL2FwcGxpY2F0aW9uL3VzZXJndWlkZS9hcHBsaWNhdGlvbi1hdXRvLXNjYWxpbmctc2VydmljZS1saW5rZWQtcm9sZXMuaHRtbFxuICAgIHRoaXMuc2NhbGluZ1JvbGUgPSBpYW0uUm9sZS5mcm9tUm9sZUFybih0aGlzLCAnU2NhbGluZ1JvbGUnLCB0aGlzLnN0YWNrLmZvcm1hdEFybih7XG4gICAgICBzZXJ2aWNlOiAnaWFtJyxcbiAgICAgIHJlZ2lvbjogJycsXG4gICAgICByZXNvdXJjZTogJ3JvbGUvYXdzLXNlcnZpY2Utcm9sZS9sYW1iZGEuYXBwbGljYXRpb24tYXV0b3NjYWxpbmcuYW1hem9uYXdzLmNvbScsXG4gICAgICByZXNvdXJjZU5hbWU6ICdBV1NTZXJ2aWNlUm9sZUZvckFwcGxpY2F0aW9uQXV0b1NjYWxpbmdfTGFtYmRhQ29uY3VycmVuY3knLFxuICAgIH0pKTtcblxuICAgIHRoaXMuZnVuY3Rpb25Bcm4gPSB0aGlzLmdldFJlc291cmNlQXJuQXR0cmlidXRlKGFsaWFzLnJlZiwge1xuICAgICAgc2VydmljZTogJ2xhbWJkYScsXG4gICAgICByZXNvdXJjZTogJ2Z1bmN0aW9uJyxcbiAgICAgIHJlc291cmNlTmFtZTogYCR7dGhpcy5sYW1iZGEuZnVuY3Rpb25OYW1lfToke3RoaXMucGh5c2ljYWxOYW1lfWAsXG4gICAgICBhcm5Gb3JtYXQ6IEFybkZvcm1hdC5DT0xPTl9SRVNPVVJDRV9OQU1FLFxuICAgIH0pO1xuXG4gICAgdGhpcy5xdWFsaWZpZXIgPSBleHRyYWN0UXVhbGlmaWVyRnJvbUFybihhbGlhcy5yZWYpO1xuXG4gICAgaWYgKHByb3BzLm9uRmFpbHVyZSB8fCBwcm9wcy5vblN1Y2Nlc3MgfHwgcHJvcHMubWF4RXZlbnRBZ2UgfHwgcHJvcHMucmV0cnlBdHRlbXB0cyAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aGlzLmNvbmZpZ3VyZUFzeW5jSW52b2tlKHtcbiAgICAgICAgb25GYWlsdXJlOiBwcm9wcy5vbkZhaWx1cmUsXG4gICAgICAgIG9uU3VjY2VzczogcHJvcHMub25TdWNjZXNzLFxuICAgICAgICBtYXhFdmVudEFnZTogcHJvcHMubWF4RXZlbnRBZ2UsXG4gICAgICAgIHJldHJ5QXR0ZW1wdHM6IHByb3BzLnJldHJ5QXR0ZW1wdHMsXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICAvLyBBUk4gcGFyc2luZyBzcGxpdHMgb24gYDpgLCBzbyB3ZSBjYW4gb25seSBnZXQgdGhlIGZ1bmN0aW9uJ3MgbmFtZSBmcm9tIHRoZSBBUk4gYXMgcmVzb3VyY2VOYW1lLi4uXG4gICAgLy8gQW5kIHdlJ3JlIHBhcnNpbmcgaXQgb3V0IChpbnN0ZWFkIG9mIHVzaW5nIHRoZSB1bmRlcmx5aW5nIGZ1bmN0aW9uIGRpcmVjdGx5KSBpbiBvcmRlciB0byBoYXZlIHVzZSBvZiBpdCBpbmN1clxuICAgIC8vIGFuIGltcGxpY2l0IGRlcGVuZGVuY3kgb24gdGhlIHJlc291cmNlLlxuICAgIHRoaXMuZnVuY3Rpb25OYW1lID0gYCR7dGhpcy5zdGFjay5zcGxpdEFybih0aGlzLmZ1bmN0aW9uQXJuLCBBcm5Gb3JtYXQuQ09MT05fUkVTT1VSQ0VfTkFNRSkucmVzb3VyY2VOYW1lIX06JHt0aGlzLmFsaWFzTmFtZX1gO1xuICB9XG5cbiAgcHVibGljIGdldCBncmFudFByaW5jaXBhbCgpIHtcbiAgICByZXR1cm4gdGhpcy52ZXJzaW9uLmdyYW50UHJpbmNpcGFsO1xuICB9XG5cbiAgcHVibGljIGdldCByb2xlKCkge1xuICAgIHJldHVybiB0aGlzLnZlcnNpb24ucm9sZTtcbiAgfVxuXG4gIHB1YmxpYyBtZXRyaWMobWV0cmljTmFtZTogc3RyaW5nLCBwcm9wczogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zID0ge30pOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgLy8gTWV0cmljcyBvbiBBbGlhc2VzIG5lZWQgdGhlIFwiYmFyZVwiIGZ1bmN0aW9uIG5hbWUsIGFuZCB0aGUgYWxpYXMnIEFSTiwgdGhpcyBkaWZmZXJzIGZyb20gdGhlIGJhc2UgYmVoYXZpb3IuXG4gICAgcmV0dXJuIHN1cGVyLm1ldHJpYyhtZXRyaWNOYW1lLCB7XG4gICAgICBkaW1lbnNpb25zTWFwOiB7XG4gICAgICAgIEZ1bmN0aW9uTmFtZTogdGhpcy5sYW1iZGEuZnVuY3Rpb25OYW1lLFxuICAgICAgICAvLyBjb25zdHJ1Y3QgdGhlIG5hbWUgZnJvbSB0aGUgdW5kZXJseWluZyBsYW1iZGEgc28gdGhhdCBhbGFybXMgb24gYW4gYWxpYXNcbiAgICAgICAgLy8gZG9uJ3QgY2F1c2UgYSBjaXJjdWxhciBkZXBlbmRlbmN5IHdpdGggQ29kZURlcGxveVxuICAgICAgICAvLyBzZWU6IGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvYXdzLWNkay9pc3N1ZXMvMjIzMVxuICAgICAgICBSZXNvdXJjZTogYCR7dGhpcy5sYW1iZGEuZnVuY3Rpb25OYW1lfToke3RoaXMuYWxpYXNOYW1lfWAsXG4gICAgICB9LFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ29uZmlndXJlIHByb3Zpc2lvbmVkIGNvbmN1cnJlbmN5IGF1dG9zY2FsaW5nIG9uIGEgZnVuY3Rpb24gYWxpYXMuIFJldHVybnMgYSBzY2FsYWJsZSBhdHRyaWJ1dGUgdGhhdCBjYW4gY2FsbFxuICAgKiBgc2NhbGVPblV0aWxpemF0aW9uKClgIGFuZCBgc2NhbGVPblNjaGVkdWxlKClgLlxuICAgKlxuICAgKiBAcGFyYW0gb3B0aW9ucyBBdXRvc2NhbGluZyBvcHRpb25zXG4gICAqL1xuICBwdWJsaWMgYWRkQXV0b1NjYWxpbmcob3B0aW9uczogQXV0b1NjYWxpbmdPcHRpb25zKTogSVNjYWxhYmxlRnVuY3Rpb25BdHRyaWJ1dGUge1xuICAgIGlmICh0aGlzLnNjYWxhYmxlQWxpYXMpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQXV0b1NjYWxpbmcgYWxyZWFkeSBlbmFibGVkIGZvciB0aGlzIGFsaWFzJyk7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLnNjYWxhYmxlQWxpYXMgPSBuZXcgU2NhbGFibGVGdW5jdGlvbkF0dHJpYnV0ZSh0aGlzLCAnQWxpYXNTY2FsaW5nJywge1xuICAgICAgbWluQ2FwYWNpdHk6IG9wdGlvbnMubWluQ2FwYWNpdHkgPz8gMSxcbiAgICAgIG1heENhcGFjaXR5OiBvcHRpb25zLm1heENhcGFjaXR5LFxuICAgICAgcmVzb3VyY2VJZDogYGZ1bmN0aW9uOiR7dGhpcy5mdW5jdGlvbk5hbWV9YCxcbiAgICAgIGRpbWVuc2lvbjogJ2xhbWJkYTpmdW5jdGlvbjpQcm92aXNpb25lZENvbmN1cnJlbmN5JyxcbiAgICAgIHNlcnZpY2VOYW1lc3BhY2U6IGFwcHNjYWxpbmcuU2VydmljZU5hbWVzcGFjZS5MQU1CREEsXG4gICAgICByb2xlOiB0aGlzLnNjYWxpbmdSb2xlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENhbGN1bGF0ZSB0aGUgcm91dGluZ0NvbmZpZyBwYXJhbWV0ZXIgZnJvbSB0aGUgaW5wdXQgcHJvcHNcbiAgICovXG4gIHByaXZhdGUgZGV0ZXJtaW5lUm91dGluZ0NvbmZpZyhwcm9wczogQWxpYXNQcm9wcykge1xuICAgIGlmICghcHJvcHMuYWRkaXRpb25hbFZlcnNpb25zIHx8IHByb3BzLmFkZGl0aW9uYWxWZXJzaW9ucy5sZW5ndGggPT09IDApIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuXG4gICAgdGhpcy52YWxpZGF0ZUFkZGl0aW9uYWxXZWlnaHRzKHByb3BzLmFkZGl0aW9uYWxWZXJzaW9ucyk7XG5cbiAgICByZXR1cm4ge1xuICAgICAgYWRkaXRpb25hbFZlcnNpb25XZWlnaHRzOiBwcm9wcy5hZGRpdGlvbmFsVmVyc2lvbnMubWFwKHZ3ID0+IHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICBmdW5jdGlvblZlcnNpb246IHZ3LnZlcnNpb24udmVyc2lvbixcbiAgICAgICAgICBmdW5jdGlvbldlaWdodDogdncud2VpZ2h0LFxuICAgICAgICB9O1xuICAgICAgfSksXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBWYWxpZGF0ZSB0aGF0IHRoZSBhZGRpdGlvbmFsIHZlcnNpb24gd2VpZ2h0cyBtYWtlIHNlbnNlXG4gICAqXG4gICAqIFdlIHZhbGlkYXRlIHRoYXQgdGhleSBhcmUgcG9zaXRpdmUgYW5kIGFkZCB1cCB0byBzb21ldGhpbmcgPD0gMS5cbiAgICovXG4gIHByaXZhdGUgdmFsaWRhdGVBZGRpdGlvbmFsV2VpZ2h0cyh3ZWlnaHRzOiBWZXJzaW9uV2VpZ2h0W10pIHtcbiAgICBjb25zdCB0b3RhbCA9IHdlaWdodHMubWFwKHcgPT4ge1xuICAgICAgaWYgKHcud2VpZ2h0IDwgMCB8fCB3LndlaWdodCA+IDEpIHsgdGhyb3cgbmV3IEVycm9yKGBBZGRpdGlvbmFsIHZlcnNpb24gd2VpZ2h0IG11c3QgYmUgYmV0d2VlbiAwIGFuZCAxLCBnb3Q6ICR7dy53ZWlnaHR9YCk7IH1cbiAgICAgIHJldHVybiB3LndlaWdodDtcbiAgICB9KS5yZWR1Y2UoKGEsIHgpID0+IGEgKyB4KTtcblxuICAgIGlmICh0b3RhbCA+IDEpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgU3VtIG9mIGFkZGl0aW9uYWwgdmVyc2lvbiB3ZWlnaHRzIG11c3Qgbm90IGV4Y2VlZCAxLCBnb3Q6ICR7dG90YWx9YCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFZhbGlkYXRlIHRoYXQgdGhlIHByb3Zpc2lvbmVkQ29uY3VycmVudEV4ZWN1dGlvbnMgbWFrZXMgc2Vuc2VcbiAgICpcbiAgICogTWVtYmVyIG11c3QgaGF2ZSB2YWx1ZSBncmVhdGVyIHRoYW4gb3IgZXF1YWwgdG8gMVxuICAgKi9cbiAgcHJpdmF0ZSBkZXRlcm1pbmVQcm92aXNpb25lZENvbmN1cnJlbmN5KHByb3BzOiBBbGlhc1Byb3BzKTogQ2ZuQWxpYXMuUHJvdmlzaW9uZWRDb25jdXJyZW5jeUNvbmZpZ3VyYXRpb25Qcm9wZXJ0eSB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKCFwcm9wcy5wcm92aXNpb25lZENvbmN1cnJlbnRFeGVjdXRpb25zKSB7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5wcm92aXNpb25lZENvbmN1cnJlbnRFeGVjdXRpb25zIDw9IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcigncHJvdmlzaW9uZWRDb25jdXJyZW50RXhlY3V0aW9ucyBtdXN0IGhhdmUgdmFsdWUgZ3JlYXRlciB0aGFuIG9yIGVxdWFsIHRvIDEnKTtcbiAgICB9XG5cbiAgICByZXR1cm4geyBwcm92aXNpb25lZENvbmN1cnJlbnRFeGVjdXRpb25zOiBwcm9wcy5wcm92aXNpb25lZENvbmN1cnJlbnRFeGVjdXRpb25zIH07XG4gIH1cbn1cblxuLyoqXG4gKiBBIHZlcnNpb24vd2VpZ2h0IHBhaXIgZm9yIHJvdXRpbmcgdHJhZmZpYyB0byBMYW1iZGEgZnVuY3Rpb25zXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVmVyc2lvbldlaWdodCB7XG4gIC8qKlxuICAgKiBUaGUgdmVyc2lvbiB0byByb3V0ZSB0cmFmZmljIHRvXG4gICAqL1xuICByZWFkb25seSB2ZXJzaW9uOiBJVmVyc2lvbjtcblxuICAvKipcbiAgICogSG93IG11Y2ggd2VpZ2h0IHRvIGFzc2lnbiB0byB0aGlzIHZlcnNpb24gKDAuLjEpXG4gICAqL1xuICByZWFkb25seSB3ZWlnaHQ6IG51bWJlcjtcbn1cbiJdfQ==