# -*- coding: utf-8 -*-
"""
@File        : department_funcs.py
@Author      : yu wen yang
@Time        : 2022/5/13 10:12 上午
@Description :
"""
import math

from django.db.models import Max
from django.db import transaction

from departments.models import Department, DepartmentUserMap
from departments.forms.forms import (
    DepartmentForms,
)
from utils.error_response import Response


def get_departments(organization_id: int) -> dict:
    """
    通过组织id获取全量部门
    :param organization_id:
    :return:
    """

    instance = Department.objects.filter(
        organization_id=organization_id,
        path__isnull=True,
        is_deleted=False
    )
    items = list()

    for i in instance:
        items.append(i.department_tree())

    return Response(data=items)


def get_sub_department(organization_id: int, department_id=None) -> dict:
    """
    通过部门id获取部门下的子部门, 默认获取所有的一级部门
    :param organization_id: 组织id
    :param department_id: 部门id
    :return:
    """

    if not department_id:
        department_instance = Department.objects.filter(
            organization_id=organization_id, is_deleted=False, path__isnull=True
        )
        department_items = []
        for department in department_instance:
            department_items.append({
                "id": department.id,
                "name": department.name,
                "charger_id": department.charger_id,
                "path": department.path,

            })
        return Response(data=department_items)

    instance = Department.objects.get(
        pk=department_id,
        organization_id=organization_id,
        is_deleted=False
    )
    department_path = instance.path if instance.path else list()
    department_path.append(department_id)
    department = Department.objects.filter(
        organization_id=organization_id,
        is_deleted=False,
        path=department_path
    )
    return Response(data=[{"id": i.id, "name": i.name, "charger_id": i.charger_id, "path": i.path} for i in department])


def create_department(data: dict) -> dict:
    """
    新建部门
    :param data:
    :return:
    """

    """
    1. 校验需要创建企业的数据, 判断数据合法性
    2. 构建新建部门path(如果父级部门有path, 将父级部门id加入父级path中;如果父级部门没有path, 则path=父级部门)
    3. 找到数据表中最大的排序数, 将这个数字加1并且创建新的部门.
    """
    # 校验数据
    checked = DepartmentForms(data)
    # 判断数据合法性
    if not checked.is_valid():
        return Response(code=500, msg="error", data=checked.errors)
    # 构造path
    parent_id = data.pop("parent_id")
    parent = Department.objects.get(pk=parent_id)
    if parent.path:
        path = parent.path
        path.append(parent.id)
    else:
        path = [parent.id]
    # 找到最大排序数
    show_order = Department.objects.aggregate(num=Max('show_order'))["num"]
    data["path"] = path
    data["show_order"] = show_order + 1

    department_instance = Department.objects.create(**data)
    return Response(data={'id': department_instance.pk})


def get_department_object(organization_id: int, pk: int):
    """
    获取部门详情
    :param organization_id: 组织id
    :param pk: 部门id
    :return:
    """
    try:
        instance = Department.objects.get(pk=pk, organization_id=organization_id, is_deleted=False)
        return Response(
            data={"id": instance.id, "name": instance.name, "path": instance.path, "charger_id": instance.charger_id}
        )
    except Department.DoesNotExist:
        return Response(code=500, msg="error", data=f"the department can't be found  id: {pk}")


@transaction.atomic
def update_department(pk: int, data: dict):
    """
    修改部门内容
    :param pk:
    :param data:
    :return:
    """
    checked = DepartmentForms(data)
    # 判断数据合法性
    if not checked.is_valid():
        return Response(code=500, msg="error", data=checked.errors)
    organization_id = data["organization_id"]
    try:
        instance = Department.objects.get(pk=pk, organization_id=organization_id, is_deleted=False)
        parent_id = data["parent_id"]
        self_path = instance.id

        # 查询当前部门的子级部门(path包含当前部门id的部门)
        son_departments = Department.objects.filter(
            path__contains=[self_path, ]
        ).values_list("id", flat=True)

        # 如果父部门选择了自己或者自己的子级部门抛出异常
        if int(parent_id) == self_path or int(parent_id) in list(son_departments):
            raise ValueError("该父部门不可选择")

        parent = Department.objects.get(organization_id=organization_id, pk=parent_id)
        if parent.path:
            path = parent.path
            path.append(parent.id)
        else:
            path = [parent.id]

        instance.path = path
        instance.name = data["name"]
        instance.charger_id = data["charger_id"]
        instance.save()
        return Response(data="修改成功")
    except Exception as err:
        raise ValueError(f"{str(err)}  id: {pk}")


def delete_department(organization_id: int, pk: int):
    """
    删除部门
    :param organization_id:
    :param pk:
    :return:
    """
    Department.objects.filter(organization_id=organization_id, pk=pk).update(is_deleted=True)
    return Response(data="删除成功")


@transaction.atomic
def add_department_user_map(organization_id: int, department_id: int, user_ids: list):
    """
    添加部门-用户的映射关系
    :param organization_id:
    :param department_id:
    :param user_ids:
    :return:
    """
    if not Department.objects.filter(
            organization_id=organization_id, pk=department_id
    ).exists():
        return Response(code=400, msg="error", data="当前组织没有该部门")

    department_user_list = [
        DepartmentUserMap(department_id=department_id, user_id=user) for user in user_ids
    ]
    try:
        DepartmentUserMap.objects.bulk_create(department_user_list)
    except Exception as err:
        raise Exception(err)
    return Response(data="添加成功")


def get_department_user(organization_id: int, department_id: int, page: int, size: int):
    """
    通过部门获取用户
    :param organization_id: 组织id
    :param department_id: 部门id
    :param page: 页码
    :param size: 每页数量
    :return:
    """
    user_ids = DepartmentUserMap.objects.filter(
        department__organization_id=organization_id,
        department__is_deleted=False,
        department_id=department_id
    )
    total = user_ids.count()
    max_page = math.ceil(total / size)
    items = [
        item for item in user_ids.order_by("-id").values_list("user_id", flat=True)[(page-1)*size: page*size]
    ] if page <= max_page else []

    return Response(data={
        "total": total,
        "page_range": [i for i in range(1, max_page+1)],
        "current_page": page,
        "list": items
    })


def delete_department_user_map(organization_id: int, department_id: int, user_ids: list):
    """
    添加部门-用户的映射关系
    :param organization_id:
    :param department_id:
    :param user_ids:
    :return:
    """
    if not Department.objects.filter(
            organization_id=organization_id, pk=department_id
    ).exists():
        return Response(code=400, msg="error", data="当前组织没有该部门")

    DepartmentUserMap.objects.filter(
        department__organization_id=organization_id,
        user_id__in=user_ids
    ).delete()

    return Response(data="移除成功")
