"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrowApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const lambda = require("aws-cdk-lib/aws-lambda");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const logs = require("aws-cdk-lib/aws-logs");
/**
 * For copying shared code to all paths
 */
const fse = require("fs-extra");
const DEFAULT_LAMBDA_CODE = `
exports.handler = async function (event, context, callback) {
  try {
    const data = {
      statusCode: 201,
    };
    return data;
  } catch (uncaughtError) {
    console.error(uncaughtError);
    throw uncaughtError;
  }
}
`;
/**
 * @experimental
 */
class CrowApi extends constructs_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        const { sourceDirectory = 'src', sharedDirectory = 'shared', useAuthorizerLambda = false, authorizerDirectory = 'authorizer', authorizerConfiguration = {}, createApiKey = false, logRetention = logs.RetentionDays.ONE_WEEK, databaseTables: databaseTablesUsed = {}, apiGatewayConfiguration = {}, apiGatewayName = 'crow-api', lambdaConfigurations = {}, } = props;
        const specialDirectories = [
            sharedDirectory,
            authorizerDirectory,
        ];
        const defaultLambda = new lambda.Function(this, 'default-crow-lambda', {
            runtime: lambda.Runtime.NODEJS_12_X,
            code: new lambda.InlineCode(DEFAULT_LAMBDA_CODE),
            handler: 'index.handler',
            logRetention,
        });
        // API Gateway log group
        const gatewayLogGroup = new logs.LogGroup(this, 'api-access-logs', {
            retention: logs.RetentionDays.ONE_WEEK,
        });
        // The API Gateway itself
        const gateway = new apigateway.LambdaRestApi(this, apiGatewayName, {
            handler: defaultLambda,
            proxy: false,
            deploy: true,
            deployOptions: {
                loggingLevel: apigateway.MethodLoggingLevel.ERROR,
                accessLogDestination: new apigateway.LogGroupLogDestination(gatewayLogGroup),
            },
            apiKeySourceType: createApiKey ? apigateway.ApiKeySourceType.HEADER : undefined,
            ...apiGatewayConfiguration,
        });
        if (createApiKey) {
            // API Key
            const apiKey = gateway.addApiKey('api-key');
            const usagePlan = new apigateway.UsagePlan(this, 'usage-plan', {
                throttle: {
                    burstLimit: 5000,
                    rateLimit: 10000,
                },
                apiStages: [
                    {
                        api: gateway,
                        stage: gateway.deploymentStage,
                    },
                ],
            });
            usagePlan.addApiKey(apiKey);
        }
        let tokenAuthorizer;
        if (useAuthorizerLambda) {
            const fullAuthorizerDirectory = `${sourceDirectory}/${authorizerDirectory}`;
            copySharedDirectory(fullAuthorizerDirectory);
            const authorizerLambda = new lambda.Function(this, 'authorizer-lambda', {
                runtime: lambda.Runtime.NODEJS_14_X,
                code: lambda.Code.fromAsset(fullAuthorizerDirectory),
                handler: 'index.handler',
                logRetention,
            });
            this.authorizerLambda = authorizerLambda;
            const tokenAuthorizerConfiguration = {
                handler: authorizerLambda,
                resultsCacheTtl: cdk.Duration.seconds(3600),
                ...authorizerConfiguration,
            };
            tokenAuthorizer = new apigateway.TokenAuthorizer(this, 'token-authorizer', tokenAuthorizerConfiguration);
        }
        // Returns child directories given the path of a parent
        function getDirectoryChildren(parentDirectory) {
            const directories = fse.readdirSync(parentDirectory, { withFileTypes: true })
                .filter((dirent) => dirent.isDirectory())
                .map((dirent) => dirent.name);
            return directories;
        }
        // Copies shared directory to a given target directory
        function copySharedDirectory(targetPath) {
            const sourceSharedDirectory = `${sourceDirectory}/${sharedDirectory}`;
            const targetSharedDirectory = `${targetPath}/${sharedDirectory}`;
            // NOTE this is file I/O so it takes a while
            fse.emptyDirSync(targetSharedDirectory);
            // Copy shared code to target path
            try {
                fse.copySync(sourceSharedDirectory, targetSharedDirectory);
            }
            catch (err) {
                // console.error(err);
                console.error(`There was an error copying the shared directory to ${targetSharedDirectory}`);
                // Let this pass and not disrupt the entire application
            }
        }
        // Given a path, look for crow.json and return configuration
        function getConfiguration(path) {
            const configurationFile = `${path}/crow.json`;
            try {
                if (fse.existsSync(configurationFile)) {
                    const configuration = fse.readJsonSync(configurationFile);
                    return configuration;
                }
            }
            catch (err) {
                console.error(err);
                // Don't want to crash synth if config file isn't present
            }
            return {};
        }
        function bundleLambdaProps(userConfiguration, codePath, apiPath) {
            const { lambdaConfiguration, databaseTables, } = userConfiguration;
            const propConfiguration = lambdaConfigurations[apiPath] || {};
            const lambdaProps = {
                runtime: lambda.Runtime.NODEJS_14_X,
                code: lambda.Code.fromAsset(codePath),
                handler: 'index.handler',
                logRetention,
                environment: {},
                ...lambdaConfiguration,
                ...propConfiguration,
            };
            if (databaseTables) {
                const tableEnvironmentVariables = {};
                Object.entries(databaseTables).forEach(([table, environmentVariableName]) => {
                    var _b;
                    tableEnvironmentVariables[environmentVariableName] = (_b = databaseTablesUsed[table]) === null || _b === void 0 ? void 0 : _b.tableName;
                });
                const environmentWithTables = {
                    // Let any manual environment variables take precedence over
                    //   automated ones
                    ...tableEnvironmentVariables,
                    ...lambdaProps.environment,
                };
                lambdaProps.environment = environmentWithTables;
            }
            return lambdaProps;
        }
        function grantTablePermissions(newLambda, userConfiguration) {
            const { databaseTables, } = userConfiguration;
            if (!databaseTables) {
                return;
            }
            Object.keys(databaseTables).forEach((table) => {
                var _b;
                (_b = databaseTablesUsed[table]) === null || _b === void 0 ? void 0 : _b.grantFullAccess(newLambda);
            });
        }
        const root = sourceDirectory;
        const verbs = ['get', 'post', 'put', 'delete'];
        const graph = {};
        const lambdasByPath = {};
        // Initialize with root
        graph['/'] = {
            resource: gateway.root,
            path: root,
            paths: [],
            verbs: [],
        };
        // First is directory path, second is API path
        const nodes = [[root, '/']];
        // BFS that creates API Gateway structure and copies shared code
        while (nodes.length) {
            const [directoryPath, apiPath] = nodes.shift() || ['i hate', 'typescript'];
            const children = getDirectoryChildren(directoryPath);
            // console.log(`${apiPath}'s children are: ${children}`);
            // Don't have to worry about previously visited nodes since this is a file structure...unless there are symlinks?
            children.forEach((child) => {
                const newDirectoryPath = `${directoryPath}/${child}`;
                // If we're on the root path, don't separate with a slash (/)
                //   because it ends up looking like //child-path
                const newApiPath = apiPath === '/' ? `/${child}` : `${apiPath}/${child}`;
                if (verbs.includes(child)) {
                    // If directory is a verb, we don't traverse it anymore
                    //   and need to create an API Gateway method and Lambda
                    const configuration = getConfiguration(newDirectoryPath);
                    const lambdaProps = bundleLambdaProps(configuration, newDirectoryPath, newApiPath);
                    const { useAuthorizerLambda: authorizerLambdaConfigured } = configuration;
                    copySharedDirectory(newDirectoryPath);
                    const newLambda = new lambda.Function(this, newDirectoryPath, lambdaProps);
                    grantTablePermissions(newLambda, configuration);
                    const methodConfiguration = {};
                    if (authorizerLambdaConfigured && useAuthorizerLambda) {
                        methodConfiguration.authorizationType = apigateway.AuthorizationType.CUSTOM;
                        methodConfiguration.authorizer = tokenAuthorizer;
                    }
                    graph[apiPath].resource.addMethod(child.toUpperCase(), new apigateway.LambdaIntegration(newLambda), methodConfiguration);
                    graph[apiPath].verbs.push(child);
                    lambdasByPath[newApiPath] = newLambda;
                }
                else if (specialDirectories.includes(child)) {
                    // The special directories should not result in an API path
                    // This means the API also cannot have a resource with the
                    //   same name
                }
                else {
                    // If directory is not a verb, create new API Gateway resource
                    //   for use by verb directory later
                    const newResource = graph[apiPath].resource
                        .resourceForPath(child);
                    nodes.push([newDirectoryPath, newApiPath]);
                    // Add child to parent's paths
                    graph[apiPath].paths.push(child);
                    // Initialize graph to include child
                    graph[newApiPath] = {
                        resource: newResource,
                        path: newDirectoryPath,
                        paths: [],
                        verbs: [],
                    };
                }
            });
        }
        // console.log(graph);
        // Expose API Gateway
        this.gateway = gateway;
        this.lambdaFunctions = lambdasByPath;
    }
}
exports.CrowApi = CrowApi;
_a = JSII_RTTI_SYMBOL_1;
CrowApi[_a] = { fqn: "crow-api.CrowApi", version: "0.1.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbmRleC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLG1DQUFtQztBQUNuQywyQ0FBdUM7QUFFdkMsaURBQWlEO0FBQ2pELHlEQUF5RDtBQUN6RCw2Q0FBNkM7QUFFN0M7O0dBRUc7QUFDSCxnQ0FBZ0M7QUFFaEMsTUFBTSxtQkFBbUIsR0FBRzs7Ozs7Ozs7Ozs7O0NBWTNCLENBQUM7Ozs7QUEyREYsTUFBYSxPQUFRLFNBQVEsc0JBQVM7Ozs7SUFNcEMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFvQjtRQUM1RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sRUFDSixlQUFlLEdBQUcsS0FBSyxFQUN2QixlQUFlLEdBQUcsUUFBUSxFQUMxQixtQkFBbUIsR0FBRyxLQUFLLEVBQzNCLG1CQUFtQixHQUFHLFlBQVksRUFDbEMsdUJBQXVCLEdBQUcsRUFBRSxFQUM1QixZQUFZLEdBQUcsS0FBSyxFQUNwQixZQUFZLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FBQyxRQUFRLEVBQzFDLGNBQWMsRUFBRSxrQkFBa0IsR0FBRyxFQUFFLEVBQ3ZDLHVCQUF1QixHQUFHLEVBQUUsRUFDNUIsY0FBYyxHQUFHLFVBQVUsRUFDM0Isb0JBQW9CLEdBQUcsRUFBRSxHQUMxQixHQUFHLEtBQUssQ0FBQztRQUVWLE1BQU0sa0JBQWtCLEdBQUc7WUFDekIsZUFBZTtZQUNmLG1CQUFtQjtTQUNwQixDQUFDO1FBRUYsTUFBTSxhQUFhLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRTtZQUNyRSxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1lBQ25DLElBQUksRUFBRSxJQUFJLE1BQU0sQ0FBQyxVQUFVLENBQUMsbUJBQW1CLENBQUM7WUFDaEQsT0FBTyxFQUFFLGVBQWU7WUFDeEIsWUFBWTtTQUNiLENBQUMsQ0FBQztRQUVILHdCQUF3QjtRQUN4QixNQUFNLGVBQWUsR0FBRyxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO1lBQ2pFLFNBQVMsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLFFBQVE7U0FDdkMsQ0FBQyxDQUFDO1FBRUgseUJBQXlCO1FBQ3pCLE1BQU0sT0FBTyxHQUFHLElBQUksVUFBVSxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQ2pFLE9BQU8sRUFBRSxhQUFhO1lBQ3RCLEtBQUssRUFBRSxLQUFLO1lBQ1osTUFBTSxFQUFFLElBQUk7WUFDWixhQUFhLEVBQUU7Z0JBQ2IsWUFBWSxFQUFFLFVBQVUsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLO2dCQUNqRCxvQkFBb0IsRUFBRSxJQUFJLFVBQVUsQ0FBQyxzQkFBc0IsQ0FBQyxlQUFlLENBQUM7YUFDN0U7WUFDRCxnQkFBZ0IsRUFBRSxZQUFZLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDL0UsR0FBRyx1QkFBdUI7U0FDM0IsQ0FBQyxDQUFDO1FBRUgsSUFBSSxZQUFZLEVBQUU7WUFDaEIsVUFBVTtZQUNWLE1BQU0sTUFBTSxHQUFHLE9BQU8sQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUM7WUFDNUMsTUFBTSxTQUFTLEdBQUcsSUFBSSxVQUFVLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxZQUFZLEVBQUU7Z0JBQzdELFFBQVEsRUFBRTtvQkFDUixVQUFVLEVBQUUsSUFBSTtvQkFDaEIsU0FBUyxFQUFFLEtBQUs7aUJBQ2pCO2dCQUNELFNBQVMsRUFBRTtvQkFDVDt3QkFDRSxHQUFHLEVBQUUsT0FBTzt3QkFDWixLQUFLLEVBQUUsT0FBTyxDQUFDLGVBQWU7cUJBQy9CO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1lBQ0gsU0FBUyxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQztTQUM3QjtRQUVELElBQUksZUFBdUMsQ0FBQztRQUM1QyxJQUFJLG1CQUFtQixFQUFFO1lBQ3ZCLE1BQU0sdUJBQXVCLEdBQUcsR0FBRyxlQUFlLElBQUksbUJBQW1CLEVBQUUsQ0FBQztZQUU1RSxtQkFBbUIsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1lBRTdDLE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxtQkFBbUIsRUFBRTtnQkFDdEUsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztnQkFDbkMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLHVCQUF1QixDQUFDO2dCQUNwRCxPQUFPLEVBQUUsZUFBZTtnQkFDeEIsWUFBWTthQUNiLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxnQkFBZ0IsQ0FBQztZQUV6QyxNQUFNLDRCQUE0QixHQUFHO2dCQUNuQyxPQUFPLEVBQUUsZ0JBQWdCO2dCQUN6QixlQUFlLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDO2dCQUMzQyxHQUFHLHVCQUF1QjthQUMzQixDQUFDO1lBQ0YsZUFBZSxHQUFHLElBQUksVUFBVSxDQUFDLGVBQWUsQ0FDOUMsSUFBSSxFQUNKLGtCQUFrQixFQUNsQiw0QkFBNEIsQ0FDN0IsQ0FBQztTQUNIO1FBRUQsdURBQXVEO1FBQ3ZELFNBQVMsb0JBQW9CLENBQUMsZUFBdUI7WUFDbkQsTUFBTSxXQUFXLEdBQUcsR0FBRyxDQUFDLFdBQVcsQ0FBQyxlQUFlLEVBQUUsRUFBRSxhQUFhLEVBQUUsSUFBSSxFQUFFLENBQUM7aUJBQzFFLE1BQU0sQ0FBQyxDQUFDLE1BQVcsRUFBRSxFQUFFLENBQUMsTUFBTSxDQUFDLFdBQVcsRUFBRSxDQUFDO2lCQUM3QyxHQUFHLENBQUMsQ0FBQyxNQUFXLEVBQUUsRUFBRSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUNyQyxPQUFPLFdBQVcsQ0FBQztRQUNyQixDQUFDO1FBRUQsc0RBQXNEO1FBQ3RELFNBQVMsbUJBQW1CLENBQUMsVUFBa0I7WUFDN0MsTUFBTSxxQkFBcUIsR0FBRyxHQUFHLGVBQWUsSUFBSSxlQUFlLEVBQUUsQ0FBQztZQUN0RSxNQUFNLHFCQUFxQixHQUFHLEdBQUcsVUFBVSxJQUFJLGVBQWUsRUFBRSxDQUFDO1lBQ2pFLDRDQUE0QztZQUM1QyxHQUFHLENBQUMsWUFBWSxDQUNkLHFCQUFxQixDQUN0QixDQUFDO1lBQ0Ysa0NBQWtDO1lBQ2xDLElBQUk7Z0JBQ0YsR0FBRyxDQUFDLFFBQVEsQ0FDVixxQkFBcUIsRUFDckIscUJBQXFCLENBQ3RCLENBQUM7YUFDSDtZQUFDLE9BQU8sR0FBRyxFQUFFO2dCQUNaLHNCQUFzQjtnQkFDdEIsT0FBTyxDQUFDLEtBQUssQ0FBQyxzREFBc0QscUJBQXFCLEVBQUUsQ0FBQyxDQUFDO2dCQUM3Rix1REFBdUQ7YUFDeEQ7UUFDSCxDQUFDO1FBRUQsNERBQTREO1FBQzVELFNBQVMsZ0JBQWdCLENBQUMsSUFBWTtZQUNwQyxNQUFNLGlCQUFpQixHQUFHLEdBQUcsSUFBSSxZQUFZLENBQUM7WUFDOUMsSUFBSTtnQkFDRixJQUFJLEdBQUcsQ0FBQyxVQUFVLENBQUMsaUJBQWlCLENBQUMsRUFBRTtvQkFDckMsTUFBTSxhQUFhLEdBQUcsR0FBRyxDQUFDLFlBQVksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO29CQUMxRCxPQUFPLGFBQWEsQ0FBQztpQkFDdEI7YUFDRjtZQUFDLE9BQU8sR0FBRyxFQUFFO2dCQUNaLE9BQU8sQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQ25CLHlEQUF5RDthQUMxRDtZQUVELE9BQU8sRUFBRSxDQUFDO1FBQ1osQ0FBQztRQUVELFNBQVMsaUJBQWlCLENBQUMsaUJBQWlDLEVBQUUsUUFBZ0IsRUFBRSxPQUFlO1lBQzdGLE1BQU0sRUFDSixtQkFBbUIsRUFDbkIsY0FBYyxHQUNmLEdBQUcsaUJBQWlCLENBQUM7WUFFdEIsTUFBTSxpQkFBaUIsR0FBRyxvQkFBb0IsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUM7WUFFOUQsTUFBTSxXQUFXLEdBQUc7Z0JBQ2xCLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7Z0JBQ25DLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUM7Z0JBQ3JDLE9BQU8sRUFBRSxlQUFlO2dCQUN4QixZQUFZO2dCQUNaLFdBQVcsRUFBRSxFQUFFO2dCQUNmLEdBQUcsbUJBQW1CO2dCQUN0QixHQUFHLGlCQUFpQjthQUNyQixDQUFDO1lBRUYsSUFBSSxjQUFjLEVBQUU7Z0JBQ2xCLE1BQU0seUJBQXlCLEdBQVEsRUFBRSxDQUFDO2dCQUMxQyxNQUFNLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsS0FBSyxFQUFFLHVCQUF1QixDQUFDLEVBQUUsRUFBRTs7b0JBQzFFLHlCQUF5QixDQUFDLHVCQUF1QixDQUFDLFNBQUcsa0JBQWtCLENBQUMsS0FBSyxDQUFDLDBDQUFFLFNBQVMsQ0FBQztnQkFDNUYsQ0FBQyxDQUFDLENBQUM7Z0JBRUgsTUFBTSxxQkFBcUIsR0FBRztvQkFDNUIsNERBQTREO29CQUM1RCxtQkFBbUI7b0JBQ25CLEdBQUcseUJBQXlCO29CQUM1QixHQUFHLFdBQVcsQ0FBQyxXQUFXO2lCQUMzQixDQUFDO2dCQUNGLFdBQVcsQ0FBQyxXQUFXLEdBQUcscUJBQXFCLENBQUM7YUFDakQ7WUFFRCxPQUFPLFdBQVcsQ0FBQztRQUNyQixDQUFDO1FBRUQsU0FBUyxxQkFBcUIsQ0FBQyxTQUEyQixFQUFFLGlCQUFpQztZQUMzRixNQUFNLEVBQ0osY0FBYyxHQUNmLEdBQUcsaUJBQWlCLENBQUM7WUFFdEIsSUFBSSxDQUFDLGNBQWMsRUFBRTtnQkFDbkIsT0FBTTthQUNQO1lBQ0QsTUFBTSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxLQUFLLEVBQUUsRUFBRTs7Z0JBQzVDLE1BQUEsa0JBQWtCLENBQUMsS0FBSyxDQUFDLDBDQUFFLGVBQWUsQ0FBQyxTQUFTLEVBQUU7WUFDeEQsQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDO1FBRUQsTUFBTSxJQUFJLEdBQUcsZUFBZSxDQUFDO1FBQzdCLE1BQU0sS0FBSyxHQUFHLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDL0MsTUFBTSxLQUFLLEdBQVksRUFBRSxDQUFDO1FBQzFCLE1BQU0sYUFBYSxHQUFrQixFQUFFLENBQUM7UUFFeEMsdUJBQXVCO1FBQ3ZCLEtBQUssQ0FBQyxHQUFHLENBQUMsR0FBRztZQUNYLFFBQVEsRUFBRSxPQUFPLENBQUMsSUFBSTtZQUN0QixJQUFJLEVBQUUsSUFBSTtZQUNWLEtBQUssRUFBRSxFQUFFO1lBQ1QsS0FBSyxFQUFFLEVBQUU7U0FDVixDQUFDO1FBQ0YsOENBQThDO1FBQzlDLE1BQU0sS0FBSyxHQUF1QixDQUFDLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFFaEQsZ0VBQWdFO1FBQ2hFLE9BQU8sS0FBSyxDQUFDLE1BQU0sRUFBRTtZQUNuQixNQUFNLENBQUMsYUFBYSxFQUFFLE9BQU8sQ0FBQyxHQUFHLEtBQUssQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLFFBQVEsRUFBRSxZQUFZLENBQUMsQ0FBQztZQUMzRSxNQUFNLFFBQVEsR0FBVSxvQkFBb0IsQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUM1RCx5REFBeUQ7WUFFekQsaUhBQWlIO1lBQ2pILFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxLQUFLLEVBQUUsRUFBRTtnQkFFekIsTUFBTSxnQkFBZ0IsR0FBRyxHQUFHLGFBQWEsSUFBSSxLQUFLLEVBQUUsQ0FBQztnQkFDckQsNkRBQTZEO2dCQUM3RCxpREFBaUQ7Z0JBQ2pELE1BQU0sVUFBVSxHQUFHLE9BQU8sS0FBSyxHQUFHLENBQUMsQ0FBQyxDQUFDLElBQUksS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDLEdBQUcsT0FBTyxJQUFJLEtBQUssRUFBRSxDQUFDO2dCQUV6RSxJQUFJLEtBQUssQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLEVBQUU7b0JBQ3pCLHVEQUF1RDtvQkFDdkQsd0RBQXdEO29CQUV4RCxNQUFNLGFBQWEsR0FBbUIsZ0JBQWdCLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztvQkFDekUsTUFBTSxXQUFXLEdBQUcsaUJBQWlCLENBQUMsYUFBYSxFQUFFLGdCQUFnQixFQUFFLFVBQVUsQ0FBQyxDQUFDO29CQUNuRixNQUFNLEVBQUUsbUJBQW1CLEVBQUUsMEJBQTBCLEVBQUUsR0FBRyxhQUFhLENBQUM7b0JBRTFFLG1CQUFtQixDQUFDLGdCQUFnQixDQUFDLENBQUM7b0JBRXRDLE1BQU0sU0FBUyxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsV0FBVyxDQUFDLENBQUM7b0JBRTNFLHFCQUFxQixDQUFDLFNBQVMsRUFBRSxhQUFhLENBQUMsQ0FBQztvQkFFaEQsTUFBTSxtQkFBbUIsR0FBeUIsRUFBRSxDQUFDO29CQUNyRCxJQUFJLDBCQUEwQixJQUFJLG1CQUFtQixFQUFFO3dCQUNyRCxtQkFBbUIsQ0FBQyxpQkFBaUIsR0FBRyxVQUFVLENBQUMsaUJBQWlCLENBQUMsTUFBTSxDQUFDO3dCQUM1RSxtQkFBbUIsQ0FBQyxVQUFVLEdBQUcsZUFBZSxDQUFDO3FCQUNsRDtvQkFFRCxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FDL0IsS0FBSyxDQUFDLFdBQVcsRUFBRSxFQUNuQixJQUFJLFVBQVUsQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTLENBQUMsRUFDM0MsbUJBQW1CLENBQ3BCLENBQUM7b0JBQ0YsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7b0JBQ2pDLGFBQWEsQ0FBQyxVQUFVLENBQUMsR0FBRyxTQUFTLENBQUM7aUJBRXZDO3FCQUFNLElBQUksa0JBQWtCLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxFQUFFO29CQUM3QywyREFBMkQ7b0JBQzNELDBEQUEwRDtvQkFDMUQsY0FBYztpQkFDZjtxQkFBTTtvQkFDTCw4REFBOEQ7b0JBQzlELG9DQUFvQztvQkFFcEMsTUFBTSxXQUFXLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLFFBQVE7eUJBQ3hDLGVBQWUsQ0FBQyxLQUFLLENBQUMsQ0FBQztvQkFFMUIsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLGdCQUFnQixFQUFFLFVBQVUsQ0FBQyxDQUFDLENBQUM7b0JBRTNDLDhCQUE4QjtvQkFDOUIsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7b0JBRWpDLG9DQUFvQztvQkFDcEMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxHQUFHO3dCQUNsQixRQUFRLEVBQUUsV0FBVzt3QkFDckIsSUFBSSxFQUFFLGdCQUFnQjt3QkFDdEIsS0FBSyxFQUFFLEVBQUU7d0JBQ1QsS0FBSyxFQUFFLEVBQUU7cUJBQ1YsQ0FBQztpQkFFSDtZQUNILENBQUMsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxzQkFBc0I7UUFFdEIscUJBQXFCO1FBQ3JCLElBQUksQ0FBQyxPQUFPLEdBQUcsT0FBTyxDQUFDO1FBQ3ZCLElBQUksQ0FBQyxlQUFlLEdBQUcsYUFBYSxDQUFDO0lBQ3ZDLENBQUM7O0FBelJILDBCQTBSQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNkayBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IElUYWJsZSB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1keW5hbW9kYic7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxhbWJkYSc7XG5pbXBvcnQgKiBhcyBhcGlnYXRld2F5IGZyb20gJ2F3cy1jZGstbGliL2F3cy1hcGlnYXRld2F5JztcbmltcG9ydCAqIGFzIGxvZ3MgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxvZ3MnO1xuXG4vKipcbiAqIEZvciBjb3B5aW5nIHNoYXJlZCBjb2RlIHRvIGFsbCBwYXRoc1xuICovXG5pbXBvcnQgKiBhcyBmc2UgZnJvbSAnZnMtZXh0cmEnO1xuXG5jb25zdCBERUZBVUxUX0xBTUJEQV9DT0RFID0gYFxuZXhwb3J0cy5oYW5kbGVyID0gYXN5bmMgZnVuY3Rpb24gKGV2ZW50LCBjb250ZXh0LCBjYWxsYmFjaykge1xuICB0cnkge1xuICAgIGNvbnN0IGRhdGEgPSB7XG4gICAgICBzdGF0dXNDb2RlOiAyMDEsXG4gICAgfTtcbiAgICByZXR1cm4gZGF0YTtcbiAgfSBjYXRjaCAodW5jYXVnaHRFcnJvcikge1xuICAgIGNvbnNvbGUuZXJyb3IodW5jYXVnaHRFcnJvcik7XG4gICAgdGhyb3cgdW5jYXVnaHRFcnJvcjtcbiAgfVxufVxuYDtcblxuZXhwb3J0IGludGVyZmFjZSBMYW1iZGFzQnlQYXRoIHtcbiAgW3BhdGg6IHN0cmluZ106IGxhbWJkYS5JRnVuY3Rpb24sXG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgQ3Jvd1RhYmxlc1VzZWQge1xuICBbdGFibGVOYW1lOiBzdHJpbmddOiBJVGFibGVcbn1cblxuLy8gZXhwb3J0IGludGVyZmFjZSBJQ3Jvd0xhbWJkYUZ1bmN0aW9uUHJvcHMgZXh0ZW5kcyBPbWl0PFxuLy8gICBsYW1iZGEuRnVuY3Rpb25Qcm9wcyxcbi8vICAgJ3J1bnRpbWUnIHwgJ2NvZGUnIHwgJ2hhbmRsZXInXG4vLyA+IHtcbi8vICAgcnVudGltZT86IGxhbWJkYS5SdW50aW1lLFxuLy8gICBjb2RlPzogbGFtYmRhLkNvZGUsXG4vLyAgIGhhbmRsZXI/OiBzdHJpbmcsXG4vLyB9XG5cbi8vIGV4cG9ydCBpbnRlcmZhY2UgSUxhbWJkYUNvbmZpZ3Mge1xuLy8gICBbbGFtYmRhcGF0aDogc3RyaW5nXTogSUNyb3dMYW1iZGFGdW5jdGlvblByb3BzLFxuLy8gfVxuXG5leHBvcnQgaW50ZXJmYWNlIElDcm93QXBpUHJvcHMge1xuICBzb3VyY2VEaXJlY3Rvcnk/OiBzdHJpbmcsXG4gIHNoYXJlZERpcmVjdG9yeT86IHN0cmluZyxcbiAgdXNlQXV0aG9yaXplckxhbWJkYT86IGJvb2xlYW4sXG4gIGF1dGhvcml6ZXJEaXJlY3Rvcnk/OiBzdHJpbmcsXG4gIGF1dGhvcml6ZXJDb25maWd1cmF0aW9uPzogYXBpZ2F0ZXdheS5Ub2tlbkF1dGhvcml6ZXJQcm9wcyxcbiAgY3JlYXRlQXBpS2V5PzogYm9vbGVhbixcbiAgbG9nUmV0ZW50aW9uPzogbG9ncy5SZXRlbnRpb25EYXlzLFxuICBkYXRhYmFzZVRhYmxlcz86IENyb3dUYWJsZXNVc2VkLFxuICBhcGlHYXRld2F5Q29uZmlndXJhdGlvbj86IHN0cmluZyxcbiAgYXBpR2F0ZXdheU5hbWU/OiBzdHJpbmcsXG4gIGxhbWJkYUNvbmZpZ3VyYXRpb25zPzogYW55LFxufVxuXG5pbnRlcmZhY2UgQ3Jvd0pzb25Db25maWcge1xuICBkYXRhYmFzZVRhYmxlcz86IG9iamVjdCxcbiAgbGFtYmRhQ29uZmlndXJhdGlvbj86IG9iamVjdCxcbiAgdXNlQXV0aG9yaXplckxhbWJkYT86IGJvb2xlYW4sXG59XG5cbmludGVyZmFjZSBGU0dyYXBoTm9kZSB7XG4gIHJlc291cmNlOiBhcGlnYXRld2F5LklSZXNvdXJjZSxcbiAgcGF0aDogc3RyaW5nLFxuICBwYXRoczogc3RyaW5nW10sXG4gIHZlcmJzOiBzdHJpbmdbXSxcbn1cblxuaW50ZXJmYWNlIEZTR3JhcGgge1xuICBbcGF0aDogc3RyaW5nXTogRlNHcmFwaE5vZGUsXG59XG5cbmludGVyZmFjZSBDcm93QXBpTWV0aG9kT3B0aW9ucyB7XG4gIGF1dGhvcml6YXRpb25UeXBlPzogYXBpZ2F0ZXdheS5BdXRob3JpemF0aW9uVHlwZSxcbiAgYXV0aG9yaXplcj86IGFwaWdhdGV3YXkuSUF1dGhvcml6ZXIsXG59XG5cbmV4cG9ydCBjbGFzcyBDcm93QXBpIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgcHVibGljIGF1dGhvcml6ZXJMYW1iZGEhOiBsYW1iZGEuSUZ1bmN0aW9uO1xuICBwdWJsaWMgZ2F0ZXdheSE6IGFwaWdhdGV3YXkuSVJlc3RBcGk7XG4gIHB1YmxpYyBsYW1iZGFGdW5jdGlvbnMhOiBMYW1iZGFzQnlQYXRoO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogSUNyb3dBcGlQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCB7XG4gICAgICBzb3VyY2VEaXJlY3RvcnkgPSAnc3JjJyxcbiAgICAgIHNoYXJlZERpcmVjdG9yeSA9ICdzaGFyZWQnLFxuICAgICAgdXNlQXV0aG9yaXplckxhbWJkYSA9IGZhbHNlLFxuICAgICAgYXV0aG9yaXplckRpcmVjdG9yeSA9ICdhdXRob3JpemVyJyxcbiAgICAgIGF1dGhvcml6ZXJDb25maWd1cmF0aW9uID0ge30sXG4gICAgICBjcmVhdGVBcGlLZXkgPSBmYWxzZSxcbiAgICAgIGxvZ1JldGVudGlvbiA9IGxvZ3MuUmV0ZW50aW9uRGF5cy5PTkVfV0VFSyxcbiAgICAgIGRhdGFiYXNlVGFibGVzOiBkYXRhYmFzZVRhYmxlc1VzZWQgPSB7fSxcbiAgICAgIGFwaUdhdGV3YXlDb25maWd1cmF0aW9uID0ge30sXG4gICAgICBhcGlHYXRld2F5TmFtZSA9ICdjcm93LWFwaScsXG4gICAgICBsYW1iZGFDb25maWd1cmF0aW9ucyA9IHt9LFxuICAgIH0gPSBwcm9wcztcblxuICAgIGNvbnN0IHNwZWNpYWxEaXJlY3RvcmllcyA9IFtcbiAgICAgIHNoYXJlZERpcmVjdG9yeSxcbiAgICAgIGF1dGhvcml6ZXJEaXJlY3RvcnksXG4gICAgXTtcblxuICAgIGNvbnN0IGRlZmF1bHRMYW1iZGEgPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHRoaXMsICdkZWZhdWx0LWNyb3ctbGFtYmRhJywge1xuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEyX1gsXG4gICAgICBjb2RlOiBuZXcgbGFtYmRhLklubGluZUNvZGUoREVGQVVMVF9MQU1CREFfQ09ERSksXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICBsb2dSZXRlbnRpb24sXG4gICAgfSk7XG5cbiAgICAvLyBBUEkgR2F0ZXdheSBsb2cgZ3JvdXBcbiAgICBjb25zdCBnYXRld2F5TG9nR3JvdXAgPSBuZXcgbG9ncy5Mb2dHcm91cCh0aGlzLCAnYXBpLWFjY2Vzcy1sb2dzJywge1xuICAgICAgcmV0ZW50aW9uOiBsb2dzLlJldGVudGlvbkRheXMuT05FX1dFRUssXG4gICAgfSk7XG5cbiAgICAvLyBUaGUgQVBJIEdhdGV3YXkgaXRzZWxmXG4gICAgY29uc3QgZ2F0ZXdheSA9IG5ldyBhcGlnYXRld2F5LkxhbWJkYVJlc3RBcGkodGhpcywgYXBpR2F0ZXdheU5hbWUsIHtcbiAgICAgIGhhbmRsZXI6IGRlZmF1bHRMYW1iZGEsXG4gICAgICBwcm94eTogZmFsc2UsXG4gICAgICBkZXBsb3k6IHRydWUsXG4gICAgICBkZXBsb3lPcHRpb25zOiB7XG4gICAgICAgIGxvZ2dpbmdMZXZlbDogYXBpZ2F0ZXdheS5NZXRob2RMb2dnaW5nTGV2ZWwuRVJST1IsXG4gICAgICAgIGFjY2Vzc0xvZ0Rlc3RpbmF0aW9uOiBuZXcgYXBpZ2F0ZXdheS5Mb2dHcm91cExvZ0Rlc3RpbmF0aW9uKGdhdGV3YXlMb2dHcm91cCksXG4gICAgICB9LFxuICAgICAgYXBpS2V5U291cmNlVHlwZTogY3JlYXRlQXBpS2V5ID8gYXBpZ2F0ZXdheS5BcGlLZXlTb3VyY2VUeXBlLkhFQURFUiA6IHVuZGVmaW5lZCxcbiAgICAgIC4uLmFwaUdhdGV3YXlDb25maWd1cmF0aW9uLFxuICAgIH0pO1xuXG4gICAgaWYgKGNyZWF0ZUFwaUtleSkge1xuICAgICAgLy8gQVBJIEtleVxuICAgICAgY29uc3QgYXBpS2V5ID0gZ2F0ZXdheS5hZGRBcGlLZXkoJ2FwaS1rZXknKTtcbiAgICAgIGNvbnN0IHVzYWdlUGxhbiA9IG5ldyBhcGlnYXRld2F5LlVzYWdlUGxhbih0aGlzLCAndXNhZ2UtcGxhbicsIHtcbiAgICAgICAgdGhyb3R0bGU6IHtcbiAgICAgICAgICBidXJzdExpbWl0OiA1MDAwLFxuICAgICAgICAgIHJhdGVMaW1pdDogMTAwMDAsXG4gICAgICAgIH0sXG4gICAgICAgIGFwaVN0YWdlczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIGFwaTogZ2F0ZXdheSxcbiAgICAgICAgICAgIHN0YWdlOiBnYXRld2F5LmRlcGxveW1lbnRTdGFnZSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSk7XG4gICAgICB1c2FnZVBsYW4uYWRkQXBpS2V5KGFwaUtleSk7XG4gICAgfVxuXG4gICAgbGV0IHRva2VuQXV0aG9yaXplcjogYXBpZ2F0ZXdheS5JQXV0aG9yaXplcjtcbiAgICBpZiAodXNlQXV0aG9yaXplckxhbWJkYSkge1xuICAgICAgY29uc3QgZnVsbEF1dGhvcml6ZXJEaXJlY3RvcnkgPSBgJHtzb3VyY2VEaXJlY3Rvcnl9LyR7YXV0aG9yaXplckRpcmVjdG9yeX1gO1xuXG4gICAgICBjb3B5U2hhcmVkRGlyZWN0b3J5KGZ1bGxBdXRob3JpemVyRGlyZWN0b3J5KTtcblxuICAgICAgY29uc3QgYXV0aG9yaXplckxhbWJkYSA9IG5ldyBsYW1iZGEuRnVuY3Rpb24odGhpcywgJ2F1dGhvcml6ZXItbGFtYmRhJywge1xuICAgICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTRfWCxcbiAgICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGZ1bGxBdXRob3JpemVyRGlyZWN0b3J5KSxcbiAgICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgICBsb2dSZXRlbnRpb24sXG4gICAgICB9KTtcbiAgICAgIHRoaXMuYXV0aG9yaXplckxhbWJkYSA9IGF1dGhvcml6ZXJMYW1iZGE7XG5cbiAgICAgIGNvbnN0IHRva2VuQXV0aG9yaXplckNvbmZpZ3VyYXRpb24gPSB7XG4gICAgICAgIGhhbmRsZXI6IGF1dGhvcml6ZXJMYW1iZGEsXG4gICAgICAgIHJlc3VsdHNDYWNoZVR0bDogY2RrLkR1cmF0aW9uLnNlY29uZHMoMzYwMCksXG4gICAgICAgIC4uLmF1dGhvcml6ZXJDb25maWd1cmF0aW9uLFxuICAgICAgfTtcbiAgICAgIHRva2VuQXV0aG9yaXplciA9IG5ldyBhcGlnYXRld2F5LlRva2VuQXV0aG9yaXplcihcbiAgICAgICAgdGhpcyxcbiAgICAgICAgJ3Rva2VuLWF1dGhvcml6ZXInLFxuICAgICAgICB0b2tlbkF1dGhvcml6ZXJDb25maWd1cmF0aW9uXG4gICAgICApO1xuICAgIH1cblxuICAgIC8vIFJldHVybnMgY2hpbGQgZGlyZWN0b3JpZXMgZ2l2ZW4gdGhlIHBhdGggb2YgYSBwYXJlbnRcbiAgICBmdW5jdGlvbiBnZXREaXJlY3RvcnlDaGlsZHJlbihwYXJlbnREaXJlY3Rvcnk6IHN0cmluZykge1xuICAgICAgY29uc3QgZGlyZWN0b3JpZXMgPSBmc2UucmVhZGRpclN5bmMocGFyZW50RGlyZWN0b3J5LCB7IHdpdGhGaWxlVHlwZXM6IHRydWUgfSlcbiAgICAgICAgLmZpbHRlcigoZGlyZW50OiBhbnkpID0+IGRpcmVudC5pc0RpcmVjdG9yeSgpKVxuICAgICAgICAubWFwKChkaXJlbnQ6IGFueSkgPT4gZGlyZW50Lm5hbWUpO1xuICAgICAgcmV0dXJuIGRpcmVjdG9yaWVzO1xuICAgIH1cblxuICAgIC8vIENvcGllcyBzaGFyZWQgZGlyZWN0b3J5IHRvIGEgZ2l2ZW4gdGFyZ2V0IGRpcmVjdG9yeVxuICAgIGZ1bmN0aW9uIGNvcHlTaGFyZWREaXJlY3RvcnkodGFyZ2V0UGF0aDogc3RyaW5nKSB7XG4gICAgICBjb25zdCBzb3VyY2VTaGFyZWREaXJlY3RvcnkgPSBgJHtzb3VyY2VEaXJlY3Rvcnl9LyR7c2hhcmVkRGlyZWN0b3J5fWA7XG4gICAgICBjb25zdCB0YXJnZXRTaGFyZWREaXJlY3RvcnkgPSBgJHt0YXJnZXRQYXRofS8ke3NoYXJlZERpcmVjdG9yeX1gO1xuICAgICAgLy8gTk9URSB0aGlzIGlzIGZpbGUgSS9PIHNvIGl0IHRha2VzIGEgd2hpbGVcbiAgICAgIGZzZS5lbXB0eURpclN5bmMoXG4gICAgICAgIHRhcmdldFNoYXJlZERpcmVjdG9yeSxcbiAgICAgICk7XG4gICAgICAvLyBDb3B5IHNoYXJlZCBjb2RlIHRvIHRhcmdldCBwYXRoXG4gICAgICB0cnkge1xuICAgICAgICBmc2UuY29weVN5bmMoXG4gICAgICAgICAgc291cmNlU2hhcmVkRGlyZWN0b3J5LFxuICAgICAgICAgIHRhcmdldFNoYXJlZERpcmVjdG9yeSxcbiAgICAgICAgKTtcbiAgICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgICAvLyBjb25zb2xlLmVycm9yKGVycik7XG4gICAgICAgIGNvbnNvbGUuZXJyb3IoYFRoZXJlIHdhcyBhbiBlcnJvciBjb3B5aW5nIHRoZSBzaGFyZWQgZGlyZWN0b3J5IHRvICR7dGFyZ2V0U2hhcmVkRGlyZWN0b3J5fWApO1xuICAgICAgICAvLyBMZXQgdGhpcyBwYXNzIGFuZCBub3QgZGlzcnVwdCB0aGUgZW50aXJlIGFwcGxpY2F0aW9uXG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gR2l2ZW4gYSBwYXRoLCBsb29rIGZvciBjcm93Lmpzb24gYW5kIHJldHVybiBjb25maWd1cmF0aW9uXG4gICAgZnVuY3Rpb24gZ2V0Q29uZmlndXJhdGlvbihwYXRoOiBzdHJpbmcpOiBDcm93SnNvbkNvbmZpZyB8IG9iamVjdCB7XG4gICAgICBjb25zdCBjb25maWd1cmF0aW9uRmlsZSA9IGAke3BhdGh9L2Nyb3cuanNvbmA7XG4gICAgICB0cnkge1xuICAgICAgICBpZiAoZnNlLmV4aXN0c1N5bmMoY29uZmlndXJhdGlvbkZpbGUpKSB7XG4gICAgICAgICAgY29uc3QgY29uZmlndXJhdGlvbiA9IGZzZS5yZWFkSnNvblN5bmMoY29uZmlndXJhdGlvbkZpbGUpO1xuICAgICAgICAgIHJldHVybiBjb25maWd1cmF0aW9uO1xuICAgICAgICB9XG4gICAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgICAgY29uc29sZS5lcnJvcihlcnIpO1xuICAgICAgICAvLyBEb24ndCB3YW50IHRvIGNyYXNoIHN5bnRoIGlmIGNvbmZpZyBmaWxlIGlzbid0IHByZXNlbnRcbiAgICAgIH1cblxuICAgICAgcmV0dXJuIHt9O1xuICAgIH1cblxuICAgIGZ1bmN0aW9uIGJ1bmRsZUxhbWJkYVByb3BzKHVzZXJDb25maWd1cmF0aW9uOiBDcm93SnNvbkNvbmZpZywgY29kZVBhdGg6IHN0cmluZywgYXBpUGF0aDogc3RyaW5nKSB7XG4gICAgICBjb25zdCB7XG4gICAgICAgIGxhbWJkYUNvbmZpZ3VyYXRpb24sXG4gICAgICAgIGRhdGFiYXNlVGFibGVzLFxuICAgICAgfSA9IHVzZXJDb25maWd1cmF0aW9uO1xuXG4gICAgICBjb25zdCBwcm9wQ29uZmlndXJhdGlvbiA9IGxhbWJkYUNvbmZpZ3VyYXRpb25zW2FwaVBhdGhdIHx8IHt9O1xuXG4gICAgICBjb25zdCBsYW1iZGFQcm9wcyA9IHtcbiAgICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzE0X1gsXG4gICAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChjb2RlUGF0aCksXG4gICAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgICAgbG9nUmV0ZW50aW9uLFxuICAgICAgICBlbnZpcm9ubWVudDoge30sIC8vIEluaXRpYWxpemUgdGhpcyB0byBhbGxvdyBzcHJlYWRpbmcgbGF0ZXJcbiAgICAgICAgLi4ubGFtYmRhQ29uZmlndXJhdGlvbiwgLy8gTGV0IHVzZXIgb3ZlcnJpZGUgYW55IGRlZmF1bHRzXG4gICAgICAgIC4uLnByb3BDb25maWd1cmF0aW9uLCAvLyBMZXQgcHJvcCBjb25maWd1cmF0aW9uIG92ZXJyaWRlIGFueXRoaW5nXG4gICAgICB9O1xuXG4gICAgICBpZiAoZGF0YWJhc2VUYWJsZXMpIHtcbiAgICAgICAgY29uc3QgdGFibGVFbnZpcm9ubWVudFZhcmlhYmxlczogYW55ID0ge307XG4gICAgICAgIE9iamVjdC5lbnRyaWVzKGRhdGFiYXNlVGFibGVzKS5mb3JFYWNoKChbdGFibGUsIGVudmlyb25tZW50VmFyaWFibGVOYW1lXSkgPT4ge1xuICAgICAgICAgIHRhYmxlRW52aXJvbm1lbnRWYXJpYWJsZXNbZW52aXJvbm1lbnRWYXJpYWJsZU5hbWVdID0gZGF0YWJhc2VUYWJsZXNVc2VkW3RhYmxlXT8udGFibGVOYW1lO1xuICAgICAgICB9KTtcblxuICAgICAgICBjb25zdCBlbnZpcm9ubWVudFdpdGhUYWJsZXMgPSB7XG4gICAgICAgICAgLy8gTGV0IGFueSBtYW51YWwgZW52aXJvbm1lbnQgdmFyaWFibGVzIHRha2UgcHJlY2VkZW5jZSBvdmVyXG4gICAgICAgICAgLy8gICBhdXRvbWF0ZWQgb25lc1xuICAgICAgICAgIC4uLnRhYmxlRW52aXJvbm1lbnRWYXJpYWJsZXMsXG4gICAgICAgICAgLi4ubGFtYmRhUHJvcHMuZW52aXJvbm1lbnQsXG4gICAgICAgIH07XG4gICAgICAgIGxhbWJkYVByb3BzLmVudmlyb25tZW50ID0gZW52aXJvbm1lbnRXaXRoVGFibGVzO1xuICAgICAgfVxuXG4gICAgICByZXR1cm4gbGFtYmRhUHJvcHM7XG4gICAgfVxuXG4gICAgZnVuY3Rpb24gZ3JhbnRUYWJsZVBlcm1pc3Npb25zKG5ld0xhbWJkYTogbGFtYmRhLklGdW5jdGlvbiwgdXNlckNvbmZpZ3VyYXRpb246IENyb3dKc29uQ29uZmlnKSB7XG4gICAgICBjb25zdCB7XG4gICAgICAgIGRhdGFiYXNlVGFibGVzLFxuICAgICAgfSA9IHVzZXJDb25maWd1cmF0aW9uO1xuXG4gICAgICBpZiAoIWRhdGFiYXNlVGFibGVzKSB7XG4gICAgICAgIHJldHVyblxuICAgICAgfVxuICAgICAgT2JqZWN0LmtleXMoZGF0YWJhc2VUYWJsZXMpLmZvckVhY2goKHRhYmxlKSA9PiB7XG4gICAgICAgIGRhdGFiYXNlVGFibGVzVXNlZFt0YWJsZV0/LmdyYW50RnVsbEFjY2VzcyhuZXdMYW1iZGEpO1xuICAgICAgfSk7XG4gICAgfVxuXG4gICAgY29uc3Qgcm9vdCA9IHNvdXJjZURpcmVjdG9yeTtcbiAgICBjb25zdCB2ZXJicyA9IFsnZ2V0JywgJ3Bvc3QnLCAncHV0JywgJ2RlbGV0ZSddO1xuICAgIGNvbnN0IGdyYXBoOiBGU0dyYXBoID0ge307XG4gICAgY29uc3QgbGFtYmRhc0J5UGF0aDogTGFtYmRhc0J5UGF0aCA9IHt9O1xuXG4gICAgLy8gSW5pdGlhbGl6ZSB3aXRoIHJvb3RcbiAgICBncmFwaFsnLyddID0ge1xuICAgICAgcmVzb3VyY2U6IGdhdGV3YXkucm9vdCxcbiAgICAgIHBhdGg6IHJvb3QsXG4gICAgICBwYXRoczogW10sXG4gICAgICB2ZXJiczogW10sXG4gICAgfTtcbiAgICAvLyBGaXJzdCBpcyBkaXJlY3RvcnkgcGF0aCwgc2Vjb25kIGlzIEFQSSBwYXRoXG4gICAgY29uc3Qgbm9kZXM6IFtzdHJpbmcsIHN0cmluZ11bXSA9IFtbcm9vdCwgJy8nXV07XG5cbiAgICAvLyBCRlMgdGhhdCBjcmVhdGVzIEFQSSBHYXRld2F5IHN0cnVjdHVyZSBhbmQgY29waWVzIHNoYXJlZCBjb2RlXG4gICAgd2hpbGUgKG5vZGVzLmxlbmd0aCkge1xuICAgICAgY29uc3QgW2RpcmVjdG9yeVBhdGgsIGFwaVBhdGhdID0gbm9kZXMuc2hpZnQoKSB8fCBbJ2kgaGF0ZScsICd0eXBlc2NyaXB0J107XG4gICAgICBjb25zdCBjaGlsZHJlbjogYW55W10gPSBnZXREaXJlY3RvcnlDaGlsZHJlbihkaXJlY3RvcnlQYXRoKTtcbiAgICAgIC8vIGNvbnNvbGUubG9nKGAke2FwaVBhdGh9J3MgY2hpbGRyZW4gYXJlOiAke2NoaWxkcmVufWApO1xuXG4gICAgICAvLyBEb24ndCBoYXZlIHRvIHdvcnJ5IGFib3V0IHByZXZpb3VzbHkgdmlzaXRlZCBub2RlcyBzaW5jZSB0aGlzIGlzIGEgZmlsZSBzdHJ1Y3R1cmUuLi51bmxlc3MgdGhlcmUgYXJlIHN5bWxpbmtzP1xuICAgICAgY2hpbGRyZW4uZm9yRWFjaCgoY2hpbGQpID0+IHtcblxuICAgICAgICBjb25zdCBuZXdEaXJlY3RvcnlQYXRoID0gYCR7ZGlyZWN0b3J5UGF0aH0vJHtjaGlsZH1gO1xuICAgICAgICAvLyBJZiB3ZSdyZSBvbiB0aGUgcm9vdCBwYXRoLCBkb24ndCBzZXBhcmF0ZSB3aXRoIGEgc2xhc2ggKC8pXG4gICAgICAgIC8vICAgYmVjYXVzZSBpdCBlbmRzIHVwIGxvb2tpbmcgbGlrZSAvL2NoaWxkLXBhdGhcbiAgICAgICAgY29uc3QgbmV3QXBpUGF0aCA9IGFwaVBhdGggPT09ICcvJyA/IGAvJHtjaGlsZH1gIDogYCR7YXBpUGF0aH0vJHtjaGlsZH1gO1xuXG4gICAgICAgIGlmICh2ZXJicy5pbmNsdWRlcyhjaGlsZCkpIHtcbiAgICAgICAgICAvLyBJZiBkaXJlY3RvcnkgaXMgYSB2ZXJiLCB3ZSBkb24ndCB0cmF2ZXJzZSBpdCBhbnltb3JlXG4gICAgICAgICAgLy8gICBhbmQgbmVlZCB0byBjcmVhdGUgYW4gQVBJIEdhdGV3YXkgbWV0aG9kIGFuZCBMYW1iZGFcblxuICAgICAgICAgIGNvbnN0IGNvbmZpZ3VyYXRpb246IENyb3dKc29uQ29uZmlnID0gZ2V0Q29uZmlndXJhdGlvbihuZXdEaXJlY3RvcnlQYXRoKTtcbiAgICAgICAgICBjb25zdCBsYW1iZGFQcm9wcyA9IGJ1bmRsZUxhbWJkYVByb3BzKGNvbmZpZ3VyYXRpb24sIG5ld0RpcmVjdG9yeVBhdGgsIG5ld0FwaVBhdGgpO1xuICAgICAgICAgIGNvbnN0IHsgdXNlQXV0aG9yaXplckxhbWJkYTogYXV0aG9yaXplckxhbWJkYUNvbmZpZ3VyZWQgfSA9IGNvbmZpZ3VyYXRpb247XG5cbiAgICAgICAgICBjb3B5U2hhcmVkRGlyZWN0b3J5KG5ld0RpcmVjdG9yeVBhdGgpO1xuXG4gICAgICAgICAgY29uc3QgbmV3TGFtYmRhID0gbmV3IGxhbWJkYS5GdW5jdGlvbih0aGlzLCBuZXdEaXJlY3RvcnlQYXRoLCBsYW1iZGFQcm9wcyk7XG5cbiAgICAgICAgICBncmFudFRhYmxlUGVybWlzc2lvbnMobmV3TGFtYmRhLCBjb25maWd1cmF0aW9uKTtcblxuICAgICAgICAgIGNvbnN0IG1ldGhvZENvbmZpZ3VyYXRpb246IENyb3dBcGlNZXRob2RPcHRpb25zID0ge307XG4gICAgICAgICAgaWYgKGF1dGhvcml6ZXJMYW1iZGFDb25maWd1cmVkICYmIHVzZUF1dGhvcml6ZXJMYW1iZGEpIHtcbiAgICAgICAgICAgIG1ldGhvZENvbmZpZ3VyYXRpb24uYXV0aG9yaXphdGlvblR5cGUgPSBhcGlnYXRld2F5LkF1dGhvcml6YXRpb25UeXBlLkNVU1RPTTtcbiAgICAgICAgICAgIG1ldGhvZENvbmZpZ3VyYXRpb24uYXV0aG9yaXplciA9IHRva2VuQXV0aG9yaXplcjtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICBncmFwaFthcGlQYXRoXS5yZXNvdXJjZS5hZGRNZXRob2QoXG4gICAgICAgICAgICBjaGlsZC50b1VwcGVyQ2FzZSgpLFxuICAgICAgICAgICAgbmV3IGFwaWdhdGV3YXkuTGFtYmRhSW50ZWdyYXRpb24obmV3TGFtYmRhKSxcbiAgICAgICAgICAgIG1ldGhvZENvbmZpZ3VyYXRpb24sXG4gICAgICAgICAgKTtcbiAgICAgICAgICBncmFwaFthcGlQYXRoXS52ZXJicy5wdXNoKGNoaWxkKTtcbiAgICAgICAgICBsYW1iZGFzQnlQYXRoW25ld0FwaVBhdGhdID0gbmV3TGFtYmRhO1xuXG4gICAgICAgIH0gZWxzZSBpZiAoc3BlY2lhbERpcmVjdG9yaWVzLmluY2x1ZGVzKGNoaWxkKSkge1xuICAgICAgICAgIC8vIFRoZSBzcGVjaWFsIGRpcmVjdG9yaWVzIHNob3VsZCBub3QgcmVzdWx0IGluIGFuIEFQSSBwYXRoXG4gICAgICAgICAgLy8gVGhpcyBtZWFucyB0aGUgQVBJIGFsc28gY2Fubm90IGhhdmUgYSByZXNvdXJjZSB3aXRoIHRoZVxuICAgICAgICAgIC8vICAgc2FtZSBuYW1lXG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgLy8gSWYgZGlyZWN0b3J5IGlzIG5vdCBhIHZlcmIsIGNyZWF0ZSBuZXcgQVBJIEdhdGV3YXkgcmVzb3VyY2VcbiAgICAgICAgICAvLyAgIGZvciB1c2UgYnkgdmVyYiBkaXJlY3RvcnkgbGF0ZXJcblxuICAgICAgICAgIGNvbnN0IG5ld1Jlc291cmNlID0gZ3JhcGhbYXBpUGF0aF0ucmVzb3VyY2VcbiAgICAgICAgICAgIC5yZXNvdXJjZUZvclBhdGgoY2hpbGQpO1xuXG4gICAgICAgICAgbm9kZXMucHVzaChbbmV3RGlyZWN0b3J5UGF0aCwgbmV3QXBpUGF0aF0pO1xuXG4gICAgICAgICAgLy8gQWRkIGNoaWxkIHRvIHBhcmVudCdzIHBhdGhzXG4gICAgICAgICAgZ3JhcGhbYXBpUGF0aF0ucGF0aHMucHVzaChjaGlsZCk7XG5cbiAgICAgICAgICAvLyBJbml0aWFsaXplIGdyYXBoIHRvIGluY2x1ZGUgY2hpbGRcbiAgICAgICAgICBncmFwaFtuZXdBcGlQYXRoXSA9IHtcbiAgICAgICAgICAgIHJlc291cmNlOiBuZXdSZXNvdXJjZSxcbiAgICAgICAgICAgIHBhdGg6IG5ld0RpcmVjdG9yeVBhdGgsXG4gICAgICAgICAgICBwYXRoczogW10sXG4gICAgICAgICAgICB2ZXJiczogW10sXG4gICAgICAgICAgfTtcblxuICAgICAgICB9XG4gICAgICB9KTtcbiAgICB9XG5cbiAgICAvLyBjb25zb2xlLmxvZyhncmFwaCk7XG5cbiAgICAvLyBFeHBvc2UgQVBJIEdhdGV3YXlcbiAgICB0aGlzLmdhdGV3YXkgPSBnYXRld2F5O1xuICAgIHRoaXMubGFtYmRhRnVuY3Rpb25zID0gbGFtYmRhc0J5UGF0aDtcbiAgfVxufVxuIl19