'''
# Crow API

Crow API is an AWS CDK construct meant to speed up the time to market for an API. Crow lets you build an API intuitively based on the file structure of a project.

```sh
npm install --save crow-api
```

## Getting Started

[Start your application as a normal CDK app](https://docs.aws.amazon.com/cdk/latest/guide/getting_started.html)

```sh
npm install -g aws-cdk
cdk bootstrap # If this is your first cdk app, you will need to bootstrap your AWS account
cdk init app --language javascript
```

Next, install the Crow API package

```sh
npm install --save crow-api
```

In the `lib/` folder generated by the `cdk`, there should be a single file named `<your-app>-stack.js`. Create your Crow API construct inside of that file like so

```javascript
const cdk = require('@aws-cdk/core');
const { CrowApi } = require('crow-api');

class MyAppStack extends cdk.Stack {
  /**
   *
   * @param {cdk.Construct} scope
   * @param {string} id
   * @param {cdk.StackProps=} props
   */
  constructor(scope, id, props) {
    super(scope, id, props);

    // The code that defines your stack goes here
    new CrowApi(this, 'my-api', {});
  }
}

module.exports = { MyAppStack }
```

Your API will start to take shape as you create folders to define paths and methods (see Example File Structure below). To deploy your API, simply run `cdk synth` and `cdk deploy`. Follow the instructions as they are prompted, and you will end up receiving a URL where your API now lives.

## Example File Structure

The following file structure is present in this repository and is meant to be an example for setting up an API.

```
| src/
  | authorizer/
    | index.js
  | v1/
    | book/
      | get/
        | index.js
      | post/
        | index.js
        | crow.json
      | chapters/
        | get/
          | index.js
          | crow.json
    | authors/
      | get/
        | index.js
      | post/
        | index.js
        | crow.json
```

The preceding file structure will create an API with the following routes:

* GET /v1/book
* POST /v1/book
* GET /v1/book/chapters
* GET /v1/authors
* POST /v1/authors

There needs to be an `index.js` file inside of a folder named after an HTTP method in order for a path to be created. The `index.js` file needs to export a `handler` method that will process the payload and return.

## Crow API Configuration (using `crow.json`)

Crow looks for a special file in verb folders (`get/`, `post/`, etc.) called `crow.json`. This file is meant to contain configuration items that make it easy to pass props and other configuration to the AWS CDK constructs used by Crow. An example `crow.json` file might look something like this

```json
{
  "databaseTables": {
    "primaryTable": "PRIMARY_TABLE_NAME"
  },
  "lambdaConfiguration": {
    "memorySize": 256
  },
  "useAuthorizerLambda": true
}
```

The keys accepted are outlined below.

#### `databaseTables`

The keys of the `databaseTables` object need to also be passed in as props to the `CrowApi` construct (see `databaseTables` under "Crow API Props" for more detail). The key name is what is used to match with the `databaseTables` prop keys ([see `databaseTables` below](#databaseTables-1)) and the value is the name of the environment variable which will be present in the Lambda function running your code.

#### `lambdaConfiguration`

The `lambdaConfiguration` object is passed directly in to the `Lambda.Function` construct which will be running this directories code. Anything available in the [class constructor for the `Function`](https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-lambda.Function.html) can be overridden.

For configuration items that are not simple types (i.e. timeout must be type `Duration`) use the [`lambdaConfigurations`](#lambdaConfigurations) prop.

**Note:**

Be care with this configuration item as all configuration here takes precedence over Crow defaults. I suggest not using this configuration item unless you are experienced with the AWS CDK and Lambda.

#### `useAuthorizerLambda`

The `useAuthorizerLambda` boolean tells Crow which methods should first have requests authorized using the authorizer Lambda. This configuration item goes hand-in-hand with the `CrowApi` construct props: [`useAuthorizerLambda`](#useAuthorizerLambda-1), [`authorizerDirectory`](#authorizerDirectory), and [`authorizerConfiguration`](#authorizerConfiguration). If this configuration item is `false`, the method will be open.

## Crow API Props

Crow API takes in a few props to help you customize away from defaults.

#### `sourceDirectory`

By default, Crow walks through the `src` directory in the root of the repository to determine routes and methods, but you can change the top level directory by passing in the `sourceDirectory` prop. The string passed in should not start with or end with a slash (`/`). For example, `src`, `api/src`, or `source` are all valid options to pass in through that prop.

#### `sharedDirectory`

By default, Crow copies the `shared` directory in the source directory of the repository to all routes and methods, but you can change the name of the shared directory by passing in the `sharedDirectory` prop. The string passed in should not start with or end with a slash (`/`) and must be a direct child of the source directory. For example, `common` or `utils` are valid but `shared/utils` is not.

**Note:**

Each route is responsible for its own dependencies including those used by the shared code. I know this is weird, but I have not taken the time to come up with a convenient way to merge two modules' dependencies yet. This means there should be no dependencies saved in the shared module, please do all of that in the target module.

#### `useAuthorizerLambda`

Crow will create and attach an authorizer Lambda to specific methods if requested. The `useAuthorizerLambda` prop tells the `CrowApi` Construct that it should create an authorizer Lambda and accepts a boolean value. This is `false` by default.

#### `authorizerDirectory`

Crow will allow for a Lambda authorizer to be created and used by specific methods if requested. The `authorizerDirectory` prop tells Crow where to find the code for the Lambda authorizer **within the source directory which can be specified in the `sourceDirectory` prop**. It expects to find an `index.js` file that exports a `handler` function within the `authorizerDirectory`.

By default, Crow expects to find a directory called `src/authorizer` containing the authorizer Lambda source if the `useAuthorizerLambda` prop is `true`. If a different directory within the source directory should be looked at for this code, it should be specified by passing in a string to the `authorizerDirectory` prop. The string passed in should not start with nor end with a slash (`/`). For example, `auth` or `authLambdaSrc` are valid.

#### `authorizerConfiguration`

The `authorizerConfiguration` prop is passed directly to the `APIGateway.TokenAuthorizer` construct which will be in charge of your API's authorization. Anything available in the [class constructor for the `TokenAuthorizer`](https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-apigateway.TokenAuthorizer.html) can be overridden.

**Note:**

Be care with this configuration item as all configuration here takes precedence over Crow defaults. I suggest not using this configuration item unless you are experienced with the AWS CDK, API Gateway, and Lambda.

#### `createApiKey`

By default, Crow does not create an API key associated with the API. If an API key is desired, pass in the `createApiKey` prop as `true`.

#### `logRetention`

By default, Crow creates log groups for resources it creates and sets the log retention to one week. If a different retention is desired pass in the `logRetention` prop of [enum type `RetentionDays`](https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-logs.RetentionDays.html). This involves importing the proper package like so

```javascript
const cdk = require('@aws-cdk/core');
const logs = require('@aws-cdk/aws-logs');
const { CrowApi } = require('crow-api');

class MyAppStack extends cdk.Stack {
  /**
   *
   * @param {cdk.Construct} scope
   * @param {string} id
   * @param {cdk.StackProps=} props
   */
  constructor(scope, id, props) {
    super(scope, id, props);

    // The code that defines your stack goes here
    new CrowApi(this, 'my-api', {
      logRetention: logs.RetentionDays.ONE_MONTH,
    });
  }
}

module.exports = { MyAppStack }
```

#### `databaseTables`

Crow accepts an object of Dynamo DB tables that are used by the individual handlers. The key needs to be the same name as what is given in the `crow.json` file of the handler. The value for that key needs to be the name of the environment variable in which the table name will be injected. The following is a valid example of the `databaseTables` prop.

```javascript
#!/usr/bin/env node

const cdk = require('@aws-cdk/core');
const { TablesStack } = require('../lib/tables-stack');
const { CrowApiStack } = require('../lib/crow-api-stack');

const devEnvironment = {
  account: '123456789012',
  region: 'us-east-1',
};

const app = new cdk.App();

const tables = new TablesStack(app, 'TablesStack', {
  env: devEnvironment,
});


new CrowApiStack(app, 'CrowApiStack', {
  env: devEnvironment,
  sourceDirectory: 'src',
  sharedDirectory: 'utils',
  createApiKey: true,
  databaseTables: {
    primaryTable: tables.primaryTable,
  },
});
```

It is expected that the `TablesStack` outputs a Dynamo DB table as `this.primaryTable`, which is then passed to Crow through the `databaseTables` prop. For an example of this in action, see the [`tables-stack.js`](./lib/tables-stack.js) file in the `lib/` folder of this repo.

#### `apiGatewayConfiguration`

This props allows for more complex overrides to the API Gateway that fronts your API. The configuration allowed is exactly the same as the [LambdaRestApi props](https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-apigateway.LambdaRestApi.html).

**Note:**

Be care with this configuration item as all configuration here takes precedence over Crow defaults. I suggest not using this configuration item unless you are experienced with the AWS CDK and API Gateway.

An example of this prop might look like the following:

```javascript
#!/usr/bin/env node

const cdk = require('@aws-cdk/core');
const apigateway = require('@aws-cdk/aws-apigateway');
const { TablesStack } = require('../lib/tables-stack');
const { CrowApiStack } = require('../lib/crow-api-stack');

const devEnvironment = {
  account: '123456789012',
  region: 'us-east-1',
};

const app = new cdk.App();

const tables = new TablesStack(app, 'TablesStack', {
  env: devEnvironment,
});


new CrowApiStack(app, 'CrowApiStack', {
  env: devEnvironment,
  sourceDirectory: 'src',
  sharedDirectory: 'utils',
  createApiKey: true,
  databaseTables: {
    primaryTable: tables.primaryTable,
  },
  apiGatewayConfiguration: {
    endpointConfiguration: {
      types: [apigateway.EndpointType.REGIONAL],
    },
  },
});
```

#### `apiGatewayName`

This is a simple prop that names the API Gateway. This is how the API will be identified in the AWS console. The value should be a string without spaces and defaults to `crow-api`.

#### `lambdaConfigurations`

This props allows for more complex overrides to Lambda functions. The prop is an object with keys corresponding to the API path of a Lambda function and a value corresponding to the configuration that should be applied to the Lambda. This prop will override any configuration set in the [`lambdaConfiguration`](#lambdaConfiguration) in the `crow.json` file. The configuration allowed is exactly the same as the [Lambda Function props](https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-lambda.Function.html).

An example of this prop might look like the following:

```javascript
#!/usr/bin/env node

const cdk = require('@aws-cdk/core');
const apigateway = require('@aws-cdk/aws-apigateway');
const { TablesStack } = require('../lib/tables-stack');
const { CrowApiStack } = require('../lib/crow-api-stack');

const devEnvironment = {
  account: '123456789012',
  region: 'us-east-1',
};

const app = new cdk.App();

const tables = new TablesStack(app, 'TablesStack', {
  env: devEnvironment,
});


new CrowApiStack(app, 'CrowApiStack', {
  env: devEnvironment,
  sourceDirectory: 'src',
  sharedDirectory: 'utils',
  createApiKey: true,
  databaseTables: {
    primaryTable: tables.primaryTable,
  },
  apiGatewayConfiguration: {
    endpointConfiguration: {
      types: [apigateway.EndpointType.REGIONAL],
    },
  },
  lambdaConfigurations: {
    '/v1/book/get': {
      timeout: cdk.Duration.seconds(5),
    },
  },
});
```
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from ._jsii import *

import aws_cdk.aws_apigateway
import aws_cdk.aws_lambda
import aws_cdk.aws_logs
import constructs


class CrowApi(
    constructs.Construct,
    metaclass=jsii.JSIIMeta,
    jsii_type="crow-api.CrowApi",
):
    '''
    :stability: experimental
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        props: "ICrowApiProps",
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param props: -

        :stability: experimental
        '''
        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="authorizerLambda")
    def authorizer_lambda(self) -> aws_cdk.aws_lambda.IFunction:
        '''
        :stability: experimental
        '''
        return typing.cast(aws_cdk.aws_lambda.IFunction, jsii.get(self, "authorizerLambda"))

    @authorizer_lambda.setter
    def authorizer_lambda(self, value: aws_cdk.aws_lambda.IFunction) -> None:
        jsii.set(self, "authorizerLambda", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="gateway")
    def gateway(self) -> aws_cdk.aws_apigateway.IRestApi:
        '''
        :stability: experimental
        '''
        return typing.cast(aws_cdk.aws_apigateway.IRestApi, jsii.get(self, "gateway"))

    @gateway.setter
    def gateway(self, value: aws_cdk.aws_apigateway.IRestApi) -> None:
        jsii.set(self, "gateway", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="lambdaFunctions")
    def lambda_functions(self) -> "LambdasByPath":
        '''
        :stability: experimental
        '''
        return typing.cast("LambdasByPath", jsii.get(self, "lambdaFunctions"))

    @lambda_functions.setter
    def lambda_functions(self, value: "LambdasByPath") -> None:
        jsii.set(self, "lambdaFunctions", value)


@jsii.data_type(
    jsii_type="crow-api.CrowTablesUsed",
    jsii_struct_bases=[],
    name_mapping={},
)
class CrowTablesUsed:
    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CrowTablesUsed(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.interface(jsii_type="crow-api.ICrowApiProps")
class ICrowApiProps(typing_extensions.Protocol):
    '''
    :stability: experimental
    '''

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="apiGatewayConfiguration")
    def api_gateway_configuration(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        ...

    @api_gateway_configuration.setter
    def api_gateway_configuration(self, value: typing.Optional[builtins.str]) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="apiGatewayName")
    def api_gateway_name(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        ...

    @api_gateway_name.setter
    def api_gateway_name(self, value: typing.Optional[builtins.str]) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="authorizerConfiguration")
    def authorizer_configuration(
        self,
    ) -> typing.Optional[aws_cdk.aws_apigateway.TokenAuthorizerProps]:
        '''
        :stability: experimental
        '''
        ...

    @authorizer_configuration.setter
    def authorizer_configuration(
        self,
        value: typing.Optional[aws_cdk.aws_apigateway.TokenAuthorizerProps],
    ) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="authorizerDirectory")
    def authorizer_directory(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        ...

    @authorizer_directory.setter
    def authorizer_directory(self, value: typing.Optional[builtins.str]) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="createApiKey")
    def create_api_key(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        ...

    @create_api_key.setter
    def create_api_key(self, value: typing.Optional[builtins.bool]) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="databaseTables")
    def database_tables(self) -> typing.Optional[CrowTablesUsed]:
        '''
        :stability: experimental
        '''
        ...

    @database_tables.setter
    def database_tables(self, value: typing.Optional[CrowTablesUsed]) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="lambdaConfigurations")
    def lambda_configurations(self) -> typing.Any:
        '''
        :stability: experimental
        '''
        ...

    @lambda_configurations.setter
    def lambda_configurations(self, value: typing.Any) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="logRetention")
    def log_retention(self) -> typing.Optional[aws_cdk.aws_logs.RetentionDays]:
        '''
        :stability: experimental
        '''
        ...

    @log_retention.setter
    def log_retention(
        self,
        value: typing.Optional[aws_cdk.aws_logs.RetentionDays],
    ) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sharedDirectory")
    def shared_directory(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        ...

    @shared_directory.setter
    def shared_directory(self, value: typing.Optional[builtins.str]) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceDirectory")
    def source_directory(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        ...

    @source_directory.setter
    def source_directory(self, value: typing.Optional[builtins.str]) -> None:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="useAuthorizerLambda")
    def use_authorizer_lambda(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        ...

    @use_authorizer_lambda.setter
    def use_authorizer_lambda(self, value: typing.Optional[builtins.bool]) -> None:
        ...


class _ICrowApiPropsProxy:
    '''
    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "crow-api.ICrowApiProps"

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="apiGatewayConfiguration")
    def api_gateway_configuration(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "apiGatewayConfiguration"))

    @api_gateway_configuration.setter
    def api_gateway_configuration(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "apiGatewayConfiguration", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="apiGatewayName")
    def api_gateway_name(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "apiGatewayName"))

    @api_gateway_name.setter
    def api_gateway_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "apiGatewayName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="authorizerConfiguration")
    def authorizer_configuration(
        self,
    ) -> typing.Optional[aws_cdk.aws_apigateway.TokenAuthorizerProps]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[aws_cdk.aws_apigateway.TokenAuthorizerProps], jsii.get(self, "authorizerConfiguration"))

    @authorizer_configuration.setter
    def authorizer_configuration(
        self,
        value: typing.Optional[aws_cdk.aws_apigateway.TokenAuthorizerProps],
    ) -> None:
        jsii.set(self, "authorizerConfiguration", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="authorizerDirectory")
    def authorizer_directory(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "authorizerDirectory"))

    @authorizer_directory.setter
    def authorizer_directory(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "authorizerDirectory", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="createApiKey")
    def create_api_key(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "createApiKey"))

    @create_api_key.setter
    def create_api_key(self, value: typing.Optional[builtins.bool]) -> None:
        jsii.set(self, "createApiKey", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="databaseTables")
    def database_tables(self) -> typing.Optional[CrowTablesUsed]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[CrowTablesUsed], jsii.get(self, "databaseTables"))

    @database_tables.setter
    def database_tables(self, value: typing.Optional[CrowTablesUsed]) -> None:
        jsii.set(self, "databaseTables", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="lambdaConfigurations")
    def lambda_configurations(self) -> typing.Any:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Any, jsii.get(self, "lambdaConfigurations"))

    @lambda_configurations.setter
    def lambda_configurations(self, value: typing.Any) -> None:
        jsii.set(self, "lambdaConfigurations", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="logRetention")
    def log_retention(self) -> typing.Optional[aws_cdk.aws_logs.RetentionDays]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[aws_cdk.aws_logs.RetentionDays], jsii.get(self, "logRetention"))

    @log_retention.setter
    def log_retention(
        self,
        value: typing.Optional[aws_cdk.aws_logs.RetentionDays],
    ) -> None:
        jsii.set(self, "logRetention", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sharedDirectory")
    def shared_directory(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sharedDirectory"))

    @shared_directory.setter
    def shared_directory(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "sharedDirectory", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceDirectory")
    def source_directory(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceDirectory"))

    @source_directory.setter
    def source_directory(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "sourceDirectory", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="useAuthorizerLambda")
    def use_authorizer_lambda(self) -> typing.Optional[builtins.bool]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "useAuthorizerLambda"))

    @use_authorizer_lambda.setter
    def use_authorizer_lambda(self, value: typing.Optional[builtins.bool]) -> None:
        jsii.set(self, "useAuthorizerLambda", value)

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ICrowApiProps).__jsii_proxy_class__ = lambda : _ICrowApiPropsProxy


@jsii.data_type(
    jsii_type="crow-api.LambdasByPath",
    jsii_struct_bases=[],
    name_mapping={},
)
class LambdasByPath:
    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LambdasByPath(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


__all__ = [
    "CrowApi",
    "CrowTablesUsed",
    "ICrowApiProps",
    "LambdasByPath",
]

publication.publish()
