"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.flatten = exports.GitHubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cx_api_1 = require("aws-cdk-lib/cx-api");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const helpers_internal_1 = require("aws-cdk-lib/pipelines/lib/helpers-internal");
const decamelize = require("decamelize");
const YAML = require("yaml");
const github = require("./workflows-model");
const CDKOUT_ARTIFACT = 'cdk.out';
const RUNS_ON = 'ubuntu-latest';
/**
 * CDK Pipelines for GitHub workflows.
 *
 * @stability stable
 */
class GitHubWorkflow extends pipelines_1.PipelineBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h;
        super(scope, id, props);
        this.cdkCliVersion = props.cdkCliVersion;
        this.preSynthed = (_b = props.preSynthed) !== null && _b !== void 0 ? _b : false;
        this.buildContainer = props.buildContainer;
        this.preBuildSteps = (_c = props.preBuildSteps) !== null && _c !== void 0 ? _c : [];
        this.postBuildSteps = (_d = props.postBuildSteps) !== null && _d !== void 0 ? _d : [];
        this.awsCredentials = (_e = props.awsCredentials) !== null && _e !== void 0 ? _e : {
            accessKeyId: 'AWS_ACCESS_KEY_ID',
            secretAccessKey: 'AWS_SECRET_ACCESS_KEY',
        };
        this.workflowPath = (_f = props.workflowPath) !== null && _f !== void 0 ? _f : '.github/workflows/deploy.yml';
        if (!this.workflowPath.endsWith('.yml') && !this.workflowPath.endsWith('.yaml')) {
            throw new Error('workflow file is expected to be a yaml file');
        }
        if (!this.workflowPath.includes('.github/workflows/')) {
            throw new Error('workflow files must be stored in the \'.github/workflows\' directory of your repository');
        }
        this.workflowName = (_g = props.workflowName) !== null && _g !== void 0 ? _g : 'deploy';
        this.workflowTriggers = (_h = props.workflowTriggers) !== null && _h !== void 0 ? _h : {
            push: { branches: ['main'] },
            workflowDispatch: {},
        };
    }
    /**
     * Implemented by subclasses to do the actual pipeline construction.
     *
     * @stability stable
     */
    doBuildPipeline() {
        const app = aws_cdk_lib_1.Stage.of(this);
        if (!app) {
            throw new Error('');
        }
        const cdkoutDir = app === null || app === void 0 ? void 0 : app.outdir;
        const jobs = new Array();
        const structure = new helpers_internal_1.PipelineGraph(this, {
            selfMutation: false,
            publishTemplate: true,
            prepareStep: false,
        });
        for (const stageNode of flatten(structure.graph.sortedChildren())) {
            if (!helpers_internal_1.isGraph(stageNode)) {
                throw new Error(`Top-level children must be graphs, got '${stageNode}'`);
            }
            const tranches = stageNode.sortedLeaves();
            for (const tranche of tranches) {
                for (const node of tranche) {
                    const job = this.jobForNode(node, {
                        assemblyDir: cdkoutDir,
                        structure,
                    });
                    if (job) {
                        jobs.push(job);
                    }
                }
            }
        }
        // convert jobs to a map and make sure there are no duplicates
        const jobmap = {};
        for (const job of jobs) {
            if (job.id in jobmap) {
                throw new Error(`duplicate job id ${job.id}`);
            }
            jobmap[job.id] = snakeCaseKeys(job.definition);
        }
        const workflow = {
            name: this.workflowName,
            on: snakeCaseKeys(this.workflowTriggers, '_'),
            jobs: jobmap,
        };
        // write as a yaml file
        const yaml = YAML.stringify(workflow, {
            indent: 2,
        });
        // eslint-disable-next-line no-console
        console.error(`writing ${this.workflowPath}`);
        fs_1.mkdirSync(path.dirname(this.workflowPath), { recursive: true });
        fs_1.writeFileSync(this.workflowPath, yaml);
    }
    /**
     * Make an action from the given node and/or step
     */
    jobForNode(node, options) {
        var _b, _c;
        switch ((_b = node.data) === null || _b === void 0 ? void 0 : _b.type) {
            // Nothing for these, they are groupings (shouldn't even have popped up here)
            case 'group':
            case 'stack-group':
            case undefined:
                throw new Error(`jobForNode: did not expect to get group nodes: ${(_c = node.data) === null || _c === void 0 ? void 0 : _c.type}`);
            case 'self-update':
                throw new Error('github workflows does not support self mutation');
            case 'publish-assets':
                return this.jobForAssetPublish(node, node.data.assets, options);
            case 'prepare':
                throw new Error('"prepare" is not supported by GitHub worflows');
            case 'execute':
                return this.jobForDeploy(node, node.data.stack, node.data.captureOutputs);
            case 'step':
                if (node.data.isBuildStep) {
                    return this.jobForBuildStep(node, node.data.step);
                }
                else if (node.data.step instanceof pipelines_1.ShellStep) {
                    return this.jobForScriptStep(node, node.data.step);
                }
                else {
                    throw new Error(`unsupported step type: ${node.data.step.constructor.name}`);
                }
        }
    }
    jobForAssetPublish(node, assets, options) {
        const installSuffix = this.cdkCliVersion ? `@${this.cdkCliVersion}` : '';
        const cdkoutDir = options.assemblyDir;
        // create one file and make one step
        const relativeToAssembly = (p) => path.posix.join(cdkoutDir, path.relative(path.resolve(cdkoutDir), p));
        const fileContents = ['set -x'].concat(assets.map((asset) => {
            return `npx cdk-assets --path "${relativeToAssembly(asset.assetManifestPath)}" --verbose publish "${asset.assetSelector}"`;
        }));
        const publishStepFile = path.join(cdkoutDir, `publish-${node.uniqueId}-step.sh`);
        fs_1.mkdirSync(path.dirname(publishStepFile), { recursive: true });
        fs_1.writeFileSync(publishStepFile, fileContents.join('\n'), { encoding: 'utf-8' });
        const publishStep = {
            name: `Publish ${node.uniqueId}`,
            run: `/bin/bash ./cdk.out/${path.relative(cdkoutDir, publishStepFile)}`,
        };
        return {
            id: node.uniqueId,
            definition: {
                name: `Publish Assets ${node.uniqueId}`,
                needs: this.renderDependencies(node),
                permissions: {
                    contents: github.JobPermission.READ,
                },
                runsOn: RUNS_ON,
                steps: [
                    ...this.stepsToDownloadAssembly(cdkoutDir),
                    {
                        name: 'Install',
                        run: `npm install --no-save cdk-assets${installSuffix}`,
                    },
                    ...this.stepsToConfigureAws({ region: 'us-west-2' }),
                    publishStep,
                ],
            },
        };
    }
    jobForDeploy(node, stack, _captureOutputs) {
        const region = stack.region;
        const account = stack.account;
        if (!region || !account) {
            throw new Error('"account" and "region" are required');
        }
        if (!stack.templateUrl) {
            throw new Error(`unable to determine template URL for stack ${stack.stackArtifactId}`);
        }
        const resolve = (s) => {
            return cx_api_1.EnvironmentPlaceholders.replace(s, {
                accountId: account,
                region: region,
                partition: 'aws',
            });
        };
        const params = {
            'name': stack.stackName,
            'template': resolve(stack.templateUrl),
            'no-fail-on-empty-changeset': '1',
        };
        if (stack.executionRoleArn) {
            params['role-arn'] = resolve(stack.executionRoleArn);
        }
        const assumeRoleArn = stack.assumeRoleArn ? resolve(stack.assumeRoleArn) : undefined;
        return {
            id: node.uniqueId,
            definition: {
                name: `Deploy ${stack.stackArtifactId}`,
                permissions: { contents: github.JobPermission.NONE },
                needs: this.renderDependencies(node),
                runsOn: RUNS_ON,
                steps: [
                    ...this.stepsToConfigureAws({ region, assumeRoleArn }),
                    {
                        uses: 'aws-actions/aws-cloudformation-github-deploy@v1',
                        with: params,
                    },
                ],
            },
        };
    }
    jobForBuildStep(node, step) {
        if (!(step instanceof pipelines_1.ShellStep)) {
            throw new Error('synthStep must be a ScriptStep');
        }
        if (step.inputs.length > 0) {
            throw new Error('synthStep cannot have inputs');
        }
        if (step.outputs.length > 1) {
            throw new Error('synthStep must have a single output');
        }
        if (!step.primaryOutput) {
            throw new Error('synthStep requires a primaryOutput which contains cdk.out');
        }
        const cdkOut = step.outputs[0];
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: 'Synthesize',
                permissions: {
                    contents: github.JobPermission.READ,
                },
                runsOn: RUNS_ON,
                needs: this.renderDependencies(node),
                env: step.env,
                container: this.buildContainer,
                steps: [
                    ...this.stepsToCheckout(),
                    ...this.preBuildSteps,
                    ...installSteps,
                    {
                        name: 'Build',
                        run: step.commands.join('\n'),
                    },
                    ...this.postBuildSteps,
                    ...this.stepsToUploadAssembly(cdkOut.directory),
                ],
            },
        };
    }
    jobForScriptStep(node, step) {
        if (Object.keys(step.envFromCfnOutputs).length > 0) {
            throw new Error('"envFromOutputs" is not supported');
        }
        const downloadInputs = new Array();
        const uploadOutputs = new Array();
        for (const input of step.inputs) {
            downloadInputs.push({
                uses: 'actions/download-artifact@v2',
                with: {
                    name: input.fileSet.id,
                    path: input.directory,
                },
            });
        }
        for (const output of step.outputs) {
            uploadOutputs.push({
                uses: 'actions/upload-artifact@v2.1.1',
                with: {
                    name: output.fileSet.id,
                    path: output.directory,
                },
            });
        }
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: step.id,
                permissions: {
                    contents: github.JobPermission.READ,
                },
                runsOn: RUNS_ON,
                needs: this.renderDependencies(node),
                env: step.env,
                steps: [
                    ...downloadInputs,
                    ...installSteps,
                    { run: step.commands.join('\n') },
                    ...uploadOutputs,
                ],
            },
        };
    }
    stepsToConfigureAws({ region, assumeRoleArn }) {
        const params = {
            'aws-access-key-id': `\${{ secrets.${this.awsCredentials.accessKeyId} }}`,
            'aws-secret-access-key': `\${{ secrets.${this.awsCredentials.secretAccessKey} }}`,
            'aws-region': region,
            'role-skip-session-tagging': true,
            'role-duration-seconds': 30 * 60,
        };
        if (this.awsCredentials.sessionToken) {
            params['aws-session-token'] = `\${{ secrets.${this.awsCredentials.sessionToken} }}`;
        }
        if (assumeRoleArn) {
            params['role-to-assume'] = assumeRoleArn;
            params['role-external-id'] = 'Pipeline';
        }
        return [
            {
                uses: 'aws-actions/configure-aws-credentials@v1',
                with: params,
            },
        ];
    }
    stepsToDownloadAssembly(targetDir) {
        if (this.preSynthed) {
            return this.stepsToCheckout();
        }
        return [{
                name: `Download ${CDKOUT_ARTIFACT}`,
                uses: 'actions/download-artifact@v2',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: targetDir,
                },
            }];
    }
    stepsToCheckout() {
        return [{
                name: 'Checkout',
                uses: 'actions/checkout@v2',
            }];
    }
    stepsToUploadAssembly(dir) {
        if (this.preSynthed) {
            return [];
        }
        return [{
                name: `Upload ${CDKOUT_ARTIFACT}`,
                uses: 'actions/upload-artifact@v2.1.1',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: dir,
                },
            }];
    }
    renderDependencies(node) {
        const deps = new Array();
        for (const d of node.allDeps) {
            if (d instanceof helpers_internal_1.Graph) {
                deps.push(...d.allLeaves().nodes);
            }
            else {
                deps.push(d);
            }
        }
        return deps.map(x => x.uniqueId);
    }
}
exports.GitHubWorkflow = GitHubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GitHubWorkflow[_a] = { fqn: "cdk-pipelines-github.GitHubWorkflow", version: "0.0.72" };
function snakeCaseKeys(obj, sep = '-') {
    if (typeof obj !== 'object' || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(o => snakeCaseKeys(o, sep));
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (typeof v === 'object' && v != null) {
            v = snakeCaseKeys(v);
        }
        result[decamelize(k, { separator: sep })] = v;
    }
    return result;
}
function* flatten(xs) {
    for (const x of xs) {
        for (const y of x) {
            yield y;
        }
    }
}
exports.flatten = flatten;
//# sourceMappingURL=data:application/json;base64,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