class Table(object):
    """The Table class is the basic table class."""
    name = ""
    table_content = [[]]
    """
    The table_content holds the table data
    """

    def display(self, divider=", "):
        """
        Use display for pretty-printing to the console.

        :param divider: Used to customize the divider between the columns.
        :return: A string with a pretty-print of your table.
        """
        string = ""
        for row in self.table_content:
            for record in range(int(len(row))):
                string = string + str(row[record])
                if int(len(row)) != record + 1:
                    string = string + str(divider)
            string = string + "\n"
        return string

    def get_col(self, col_name):
        """
        Used to get all data in one column.
        :param col_name: The name of the column you wish to get.
        :return: A list of all data in one column.
        """
        col_position = self.table_content[0].index(col_name)

        col = []

        for i in range(int(len(self.table_content))-1):
            col.append(self.table_content[i+1][col_position])

        return col

    def add_expand(self, new_col_name, old_col_name, action, value):
        """
        Used to add a column that is based on another column.
        :param new_col_name: The name for you new column.
        :param old_col_name: The name for the column you wish to expand from.
        :param action: What change you want from your old column to your new column. Can be "+" (or "add"), "-" (or subtract), "*" (or multiply), "/" (or divide), or "join" for combining two strings.
        :param value: What difference from the old column you wish to make. For example, if your action was "*", then and your value was "2", everything would get multiplied by two.
        :return:
        """
        result = []


        for i in self.get_col(old_col_name):
            if action == "+" or action == "add":
                result.append(float(i) + float(value))

            elif action == "-" or action == "subtract":
                result.append(float(i) - float(value))

            elif action == "*" or action == "multiply":
                result.append(float(i) * float(value))

            elif action == "/" or action == "divide":
                result.append(float(i) / float(value))

            elif action == "join":
                result.append(str(i) + str(value))

            else:
                raise Exception(f"Unknown action '{action}'")

        self.add_col(new_col_name, result)

    def add_row(self, new_content):
        """
        Used to add a row to your table.
        :param new_content: A list of the row you wish to add.
        :return:
        """
        self.table_content.append(new_content)

    def add_col(self, col_name, default_value=""):
        """
        Used to add a column to your table.
        :param col_name: The name of your new column.
        :param default_value: The default value all your new records will get in that column when created or a list of specific values that will go into your column.
        :return:
        """
        self.table_content[0].append(col_name)

        for i in range(int(len(self.table_content)) - 1):
            if type(default_value) is str:
                self.table_content[i + 1].append(default_value)
            elif type(default_value) is list:

                self.table_content[i + 1].append(default_value[i])

    def edit_row(self, row_num, col_name, new_value):
        """
        Used to edit a record of your table.
        :param row_num: The row number of the record that you wish to edit.
        :param col_name: The column name of your record that you wish to edit.
        :param new_value: The rew record value.
        :return:
        """
        self.table_content[row_num][self.table_content[0].index(col_name)] = new_value

    def filter(self, col_name, value, type="exact", search_start=1, search_end="END", add_headers_to_result=True, legacy=False):
        """
        Uesed to filter your table.
        :param col_name: The column you wish to use to filter.
        :param value: The value you wish to use to filter with.
        :param type: The filter type. Can be "exact" (same), "iexact" (not case-sensitive), "greaterthan", or "lessthan".
        :param search_start: The row to start filtering at.
        :param search_end: The row to stop filtering at. Type "END" to stop at the end.
        :param add_headers_to_result: If True, your table headers will be included in the result.
        :param legacy: If True, will return a list instead of a Table object.
        :return: A Table object with the filtered results (unless specified otherwise).
        """
        if search_end == "END":
            search_end = int(len(self.table_content))

        result_list = []
        for i in self.table_content[search_start:search_end]:

            if type == "exact":
                if i[self.table_content[0].index(col_name)] == value:
                    result_list.append(i)
            elif type == "iexact":
                if i[self.table_content[0].index(col_name)].lower() == value.lower():
                    result_list.append(i)

            elif type == "greaterthan":
                if float(i[self.table_content[0].index(col_name)]) > float(value):
                    result_list.append(i)

            elif type == "lessthan":
                if float(i[self.table_content[0].index(col_name)]) < float(value):
                    result_list.append(i)

            else:
                raise Exception(f'Could not find filter method "{type}"')

        if add_headers_to_result:
            result_list.insert(0, self.table_content[0])

        if legacy:
            return result_list
        else:
            temp_table = Table()
            temp_table.table_content = result_list
            return temp_table

    def save(self, path, divider=","):
        content = self.display(divider)
        with open(path, 'w') as csv:
            csv.write(content)


class CsvTable(Table):
    def __init__(self, csv_path, divider=","):
        self.csv_path = csv_path

        with open(csv_path) as csv_file:
            csv_content = csv_file.read()

        csv_content = csv_content.split("\n")

        csv_content_list = []
        for row in csv_content:
            csv_content_list.append(row.split(divider))

        self.table_content = csv_content_list
