# Set pylint configuration for this file
# pylint: disable=missing-docstring, protected-access

# import general packages
import unittest
import os
import datetime

from ..proto_builder import ProtoBuilder
from ...fdl_parser.fdl_parser import FDLParser

__VERSION__ = '0.2.0'


class TestProtoBuilderGenerate(unittest.TestCase):

    def setUp(self) -> None:
        self.maxDiff = None  # pylint: disable=invalid-name

    def test_command_simple(self):
        # set up the ProtoBuilder object used for the validation
        fdl_parser = FDLParser(fdl_filename=os.path.join(os.path.dirname(__file__),
                                                         'fdl',
                                                         'Command_Simple.sila.xml'))
        proto_builder = ProtoBuilder(fdl_parser=fdl_parser)

        code = proto_builder._generate_proto()

        self.assertEqual(
            code,
            self._get_header() + "\n\n" +
            self._read_proto_file('Command_Simple.proto')
        )

    def test_command_observable_intermediate(self):
        # set up the ProtoBuilder object used for the validation
        fdl_parser = FDLParser(fdl_filename=os.path.join(os.path.dirname(__file__),
                                                         'fdl',
                                                         'Command_Observable_Intermediate.sila.xml'))
        proto_builder = ProtoBuilder(fdl_parser=fdl_parser)

        code = proto_builder._generate_proto()

        self.assertEqual(
            code,
            self._get_header() + "\n\n" +
            self._read_proto_file('Command_Observable_Intermediate.proto')
        )

    def test_command_observable_no_intermediate(self):
        # set up the ProtoBuilder object used for the validation
        fdl_parser = FDLParser(fdl_filename=os.path.join(os.path.dirname(__file__),
                                                         'fdl',
                                                         'Command_Observable_NoIntermediate.sila.xml'))
        proto_builder = ProtoBuilder(fdl_parser=fdl_parser)

        code = proto_builder._generate_proto()

        self.assertEqual(
            code,
            self._get_header() + "\n\n" +
            self._read_proto_file('Command_Observable_NoIntermediate.proto')
        )

    def test_property_simple(self):
        # set up the ProtoBuilder object used for the validation
        fdl_parser = FDLParser(fdl_filename=os.path.join(os.path.dirname(__file__),
                                                         'fdl',
                                                         'Property_Simple.sila.xml'))
        proto_builder = ProtoBuilder(fdl_parser=fdl_parser)

        code = proto_builder._generate_proto()

        self.assertEqual(
            code,
            self._get_header() + "\n\n" +
            self._read_proto_file('Property_Simple.proto')
        )

    def test_property_observable(self):
        # set up the ProtoBuilder object used for the validation
        fdl_parser = FDLParser(fdl_filename=os.path.join(os.path.dirname(__file__),
                                                         'fdl',
                                                         'Property_Observable.sila.xml'))
        proto_builder = ProtoBuilder(fdl_parser=fdl_parser)

        code = proto_builder._generate_proto()

        self.assertEqual(
            code,
            self._get_header() + "\n\n" +
            self._read_proto_file('Property_Observable.proto')
        )

    @staticmethod
    def _read_proto_file(filename: str) -> str:

        file = os.path.join(os.path.dirname(__file__), 'fdl', filename)
        with open(file, 'r') as input_file:
            return input_file.read()

    @staticmethod
    def _get_header():
        return TEMPLATE_HEADER.format(
            generator='sila2lib.proto_builder.proto_builder',
            version=__VERSION__,
            date=datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')
        ).strip()


TEMPLATE_HEADER = """
// This file is automatically generated by {generator} version {version}
// :generation date: {date}
//
// ---- PLEASE DO NOT MODIFY MANUALLY !! ---
""".strip()
