# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['tinytorchtest']

package_data = \
{'': ['*']}

install_requires = \
['toml>=0.10.2,<0.11.0', 'torch>=1.11.0,<2.0.0']

setup_kwargs = {
    'name': 'tinytorchtest',
    'version': '1.1.0',
    'description': 'A tiny test suite for pytorch based machine learning models.',
    'long_description': '# Tiny Torchtest\n\n![coverage](.coverage.svg)\n\nA Tiny Test Suite for pytorch based Machine Learning models, inspired by\n[mltest](https://github.com/Thenerdstation/mltest/blob/master/mltest/mltest.py).\nChase Roberts lists out 4 basic tests in his [medium\npost](https://medium.com/@keeper6928/mltest-automatically-test-neural-network-models-in-one-function-call-eb6f1fa5019d)\nabout mltest. tinytorchtest is mostly a pytorch port of mltest (which was\nwritten for tensorflow).\n\n--- \n\nForked from [BrainPugh](https://github.com/BrianPugh/torchtest) who\nforked the repo from\n[suriyadeepan](https://github.com/suriyadeepan/torchtest).\n\nNotable changes:\n\n-   Support for models to have multiple positional arguments.\n\n-   Support for unsupervised learning.\n\n- \tObject orientated implementation.\n\n- \tEasily reproducible tests - thanks to the object orientated implementation!\n\n-   Fewer requirements (due to streamlining testing).\n\n-   More comprehensive internal unit tests.\n\n-   This repository is still active. I\'ve created an\n    [issue](https://github.com/suriyadeepan/torchtest/issues/6) to\n    double check but it looks like the original maintainer is no longer\n    actioning pull requests.\n\n---\n\n# Installation\n\n``` bash\npip install --upgrade tinytorchtest\n```\n\n# Usage\n\n``` python\n# imports for examples\nimport torch\nimport torch.nn as nn\n```\n\n## Variables Change\n\n``` python\nfrom tinytorchtest import tinytorchtest as ttt\n\n# We\'ll be using a simple linear model\nmodel = nn.Linear(20, 2)\n\n# For this example, we\'ll pretend we have a classification problem\n# and create some random inputs and outputs.\ninputs = torch.randn(20, 20)\ntargets = torch.randint(0, 2, (20,)).long()\nbatch = [inputs, targets]\n\n# Next we\'ll need a loss function\nloss_fn = nn.functional.cross_entropy()\n\n# ... and an optimisation function\noptim = torch.optim.Adam(model.parameters())\n\n# Lets set up the test object\ntest = ttt.TinyTorchTest(model, loss_fn, optim, batch)\n\n# Now we\'ve got our tiny test object, lets run some tests!\n# What are the variables?\nprint(\'Our list of parameters\', [ np[0] for np in model.named_parameters() ])\n\n# Do they change after a training step?\n#  Let\'s run a train step and see\ntest.test(test_vars_change=True)\n```\n\n``` python\n""" FAILURE """\n# Let\'s try to break this, so the test fails\nparams_to_train = [ np[1] for np in model.named_parameters() if np[0] is not \'bias\' ]\n# Run test now\ntest.test(test_vars_change=True)\n# YES! bias did not change\n```\n\n## Variables Don\'t Change\n\n``` python\n# What if bias is not supposed to change, by design?\n#  Let\'s test to see if bias remains the same after training\ntest.test(non_train_vars=[(\'bias\', model.bias)])\n# It does! Good. Now, let\'s move on.\n```\n\n## Output Range\n\n``` python\n# NOTE : bias is fixed (not trainable)\ntest.test(output_range=(-2, 2), test_output_range=True)\n\n# Seems to work...\n```\n\n``` python\n""" FAILURE """\n#  Let\'s tweak the model to fail the test.\nmodel.bias = nn.Parameter(2 + torch.randn(2, ))\n\n# We\'ll still use the same loss function, optimiser and batch\n# from earlier; however this time we\'ve tweaked the bias of the model.\n# As it\'s a new model, we\'ll need a new tiny test object.\ntest = ttt.TinyTorchTest(model , loss_fn, optim, batch)\n\ntest.test(output_range=(-1, 1), test_output_range=True)\n\n# As expected, it fails; yay!\n```\n\n## NaN Tensors\n\n``` python\n""" FAILURE """\n\n# Again, keeping everything the same but tweaking the model\nmodel.bias = nn.Parameter(float(\'NaN\') * torch.randn(2, ))\n\ntest = ttt.TinyTorchTest(model , loss_fn, optim, batch)\n\ntest.test(test_nan_vals=True)\n# This test should fail as we\'ve got \'NaN\' values in the outputs.\n```\n\n## Inf Tensors\n\n``` python\n""" FAILURE """\nmodel.bias = nn.Parameter(float(\'Inf\') * torch.randn(2, ))\n\ntest = ttt.TinyTorchTest(model , loss_fn, optim, batch)\n\ntest.test(test_inf_vals=True)\n# Again, this will fail as we\'ve now got \'Inf\' values in our model outputs.\n```\n\n## Multi-argument models\n``` python\n# Everything we\'ve done works for models with multi-arguments\n\n# Let\'s define a network that takes some input features along \n# with a 3D spacial coordinate and predicts a single value.\n# Sure, we could perform the concatenation before we pass \n# our inputs to the model but let\'s say that it\'s much easier to\n# do it this way. Maybe as you\'re working tightly with other codes\n# and you want to match your inputs with the other code.\nclass MutliArgModel(torch.nn.Module):\n\tdef __init__(self):\n\t\tself.layers = torch.nn.Linear(8, 1)\n\tdef foward(self, data, x, y, z):\n\t\tinputs = torch.cat((data, x, y, z), dim=1)\n\t\treturn self.layers(nn_input)\nmodel = MultiArgModel()\n\n# This looks a bit more like a regression problem so we\'ll redefine our loss \n# function to be something more appropriate.\nloss_fn = torch.nn.MSELoss()\n\n# We\'ll stick with the Adam optimiser but for completeness lets redefine it below\noptim = Adam(model.parameters())\n\n# We\'ll also need some new data for this model\ninputs = (\n\ttorch.rand(10, 5), # data\n\ttorch.rand(10, 1), # x\n\ttorch.rand(10, 1), # y\n\ttorch.rand(10, 1), # z\n)\noutputs = torch.rand(10,1)\nbatch = [inputs, outputs]\n\t\t\n# Next we initialise our tiny test object\ntest = ttt.TinyTorchTest(model , loss_fn, optim, batch)\n\n# Now lets run some tests\ntest.test(\n\ttrain_vars=list(model.named_parameters()),\n\ttest_vars_change=True,\n\ttest_inf_vals=True,\n\ttest_nan_vals=True,\n)\n# Great! Everything works as before but with models that take multiple inputs.\n```\n\n## Unsupervised learning\n\n``` python\n# We\'ve looked a lot at supervised learning examples\n# but what about unsupervised learning?\n\n# Lets define a simple model\nmodel = nn.Linear(20, 2)\n\n# Now our inputs - notice there are no labels so we just have inputs in our batch\nbatch = torch.randn(20, 20)\n\n# Here we\'ll write a very basic loss function that represents a reconstruction loss.\n# This is actually a mean absolute distance loss function.\n# This would typically be used for something like an auto-encoder.\n# The important thing to note is tinytorchtest expects the loss to be loss(outputs, inputs).\ndef loss_fn(output, input):\n\treturn torch.mean(torch.abs(output - input))\n\n# We set supervised to false, to let the test suite\n# know that there aren\'t any targets or correct labels.\ntest = ttt.TinyTorchTest(model , loss_fn, optim, batch, supervised=False)\n\n# Now lets run some tests\ntest.test(\n\ttrain_vars=list(model.named_parameters()),\n\ttest_vars_change=True,\n\ttest_inf_vals=True,\n\ttest_nan_vals=True,\n)\n# Great! Everything works as before but with unsupervised models.\n```\n\n## Testing the GPU\n\n``` python\n# Some models really need GPU availability.\n# We can get our test suite to fail when the GPU isn\'t available.\n\n# Sticking with the unsupervised example\ntest = ttt.TinyTorchTest(model , loss_fn, optim, batch, supervised=False)\n\n# Now lets make sure the GPU is available.\ntest.test(test_gpu_available=True)\n# This test will fail if the GPU isn\'t available. Your CPU can thank you later.\n\n# We can also explitly ask that our model and tensors be moved to the GPU\ntest = ttt.TinyTorchTest(model , loss_fn, optim, batch, supervised=False, device=\'cuda:0\')\n\n# Now all future tests will be run on the GPU\n```\n\n## Reproducible tests\n\n``` python\n# When unit testing our models it\'s good practice to have reproducable results.\n# For this, we can spefiy a seed when getting our tiny test object.\ntest = ttt.TinyTorchTest(model, loss_fn, optim, batch, seed=42)\n\n# This seed will be called before running each test so the results should always be the same\n# regardless of the order they are called.\n\n```\n\n# Debugging\n\n``` bash\ntorchtest\\torchtest.py", line 151, in _var_change_helper\nassert not torch.equal(p0, p1)\nRuntimeError: Expected object of backend CPU but got backend CUDA for argument #2 \'other\'\n```\n\nWhen you are making use of a GPU, you should explicitly specify\n`device=cuda:0`. By default `device` is set to `cpu`. See [issue\n#1](https://github.com/suriyadeepan/torchtest/issues/1) for more\ninformation.\n\n``` python\ntest = ttt.TinyTorchTest(model , loss_fn, optim, batch, device=\'cuda:0\')\n```\n\n# Citation\n\n``` tex\n@misc{abdrysdale2022\n  author = {Alex Drysdale},\n  title = {tinytorchtest},\n  year = {2022},\n  publisher = {GitHub},\n  journal = {GitHub repository},\n  howpublished = {\\url{https://github.com/abdrysdale/tinytorchtest}},\n  commit = {4c39c52f27aad1fe9bcc7fbb2525fe1292db81b7}\n }\n@misc{Ram2019,\n  author = {Suriyadeepan Ramamoorthy},\n  title = {torchtest},\n  year = {2019},\n  publisher = {GitHub},\n  journal = {GitHub repository},\n  howpublished = {\\url{https://github.com/suriyadeepan/torchtest}},\n  commit = {42ba442e54e5117de80f761a796fba3589f9b223}\n}\n```\n',
    'author': 'Alex',
    'author_email': 'adrysdale@protonmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/abdrysdale/tinytorchtest',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
